/* GNOME DB library
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-ui.h"

#define SHORTCUT_ICON_LIST_WIDGET "SHORTCUT_IconListWidget"

typedef struct
{
  gchar*                  label;
  GdaUIShortcutSelectFunc select_func;
  gpointer                user_data;
} item_data_t;

/*
 * Private functions
 */

/*
 * Callbacks
 */
static void
select_icon_cb (GnomeIconList *icon_list, gint num, GdkEvent *event, GtkWidget *shortcut)
{
  item_data_t* item_data;

  g_return_if_fail(GNOME_IS_ICON_LIST(icon_list));
  g_return_if_fail(GTK_IS_WIDGET(shortcut));

  item_data = (item_data_t *) gnome_icon_list_get_icon_data(GNOME_ICON_LIST(icon_list), num);
  if (item_data && item_data->select_func)
    {
      item_data->select_func(shortcut, item_data->label, item_data->user_data);
    }
}

/*
 * Public functions
 */
GtkWidget *
gda_ui_shortcut_new (void)
{
  GtkWidget* shortcut;
  GtkWidget* icon_list;

  shortcut = gda_ui_new_scrolled_window_widget();
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(shortcut),
  				 GTK_POLICY_ALWAYS,
				 GTK_POLICY_ALWAYS);
  icon_list = gda_ui_new_icon_list_widget();
  gtk_signal_connect(GTK_OBJECT(icon_list),
		     "select_icon",
		     GTK_SIGNAL_FUNC(select_icon_cb),
		     shortcut);
  gtk_object_set_data(GTK_OBJECT(shortcut),
		      SHORTCUT_ICON_LIST_WIDGET,
		      (gpointer) icon_list);
  gtk_container_add(GTK_CONTAINER(shortcut), icon_list);

  gtk_widget_show_all(shortcut);
  return shortcut;
}

void
gda_ui_shortcut_append (GtkWidget *shortcut,
			const gchar *str,
			const gchar *icon_file,
			GdaUIShortcutSelectFunc select_func,
			gpointer user_data)
{
  GtkWidget* icon_list;

  g_return_if_fail(GTK_IS_WIDGET(shortcut));
  g_return_if_fail(str != NULL);

  icon_list = gtk_object_get_data(GTK_OBJECT(shortcut), SHORTCUT_ICON_LIST_WIDGET);
  if (GNOME_IS_ICON_LIST(icon_list))
    {
      gint         pos;
      item_data_t* item_data;

      /* create private structure */
      item_data = g_new0(item_data_t, 1);
      item_data->label = g_strdup(str);
      item_data->select_func = select_func;
      item_data->user_data = user_data;

      gnome_icon_list_freeze(GNOME_ICON_LIST(icon_list));
      pos = gnome_icon_list_append(GNOME_ICON_LIST(icon_list), icon_file, str);
      gnome_icon_list_set_icon_data_full(GNOME_ICON_LIST(icon_list),
					 pos,
					 (gpointer) item_data,
					 (GtkDestroyNotify) g_free);
      gnome_icon_list_thaw(GNOME_ICON_LIST(icon_list));
    }
}

void
gda_ui_shortcut_remove (GtkWidget *shortcut, const gchar *str)
{
  gint       pos;
  GtkWidget* icon_list;

  g_return_if_fail(GTK_IS_WIDGET(shortcut));
  g_return_if_fail(str != NULL);

  icon_list = gtk_object_get_data(GTK_OBJECT(shortcut), SHORTCUT_ICON_LIST_WIDGET);
  if (GNOME_IS_ICON_LIST(icon_list))
    {
      for (pos = 0; pos < GNOME_ICON_LIST(icon_list)->icons; pos++)
	{
	  item_data_t* item_data;

	  item_data = (item_data_t *) gnome_icon_list_get_icon_data(GNOME_ICON_LIST(icon_list), pos);
	  if (item_data && item_data->label && !strcmp(item_data->label, str))
	    {
	      gnome_icon_list_freeze(GNOME_ICON_LIST(icon_list));
	      gnome_icon_list_remove(GNOME_ICON_LIST(icon_list), pos);
	      gnome_icon_list_thaw(GNOME_ICON_LIST(icon_list));
	      break;
	    }
	}
    }
}
