/* GNOME DB libary
 * Copyright (C) 1998,1999 Michael Lausch
 * Copyright (C) 1999, 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-ui.h"

enum
{
  GNOME_DB_NEXT,
  GNOME_DB_PREV,
  LAST_SIGNAL
};

static gint
gnome_db_error_signals[LAST_SIGNAL] = {0,};

static void gnome_db_error_class_init      (GnomeDbErrorClass* klass);
static void gnome_db_error_init            (GnomeDbError* error);
static void display_current_error          (GnomeDbError* error);

static void
gnome_db_error_real_prev(GnomeDbError* error)
{
  if (!error->position)
    return;
  error->position--;
  error->error_ptr = g_list_previous(error->error_ptr);
  display_current_error(error);
}

void
gnome_db_error_real_next(GnomeDbError* error)
{
  if (!g_list_next(error->error_ptr))
    return;
  error->position++;
  error->error_ptr = g_list_next(error->error_ptr);
  display_current_error(error);
}


static void
display_current_error(GnomeDbError* error)
{
  Gda_Error * gda_error;
  const gchar*      sqlstate;
  gchar       position[128];

  if (!error->error_ptr)
    return;
  gda_error = error->error_ptr->data;

  gtk_text_set_editable(GTK_TEXT(error->description), TRUE);
  gtk_entry_set_editable(GTK_ENTRY(error->source), TRUE);
  gtk_entry_set_editable(GTK_ENTRY(error->sqlstate), TRUE);
  gtk_text_freeze(GTK_TEXT(error->description));
  gtk_text_set_editable(GTK_TEXT(error->description), TRUE);
  gtk_text_set_point(GTK_TEXT(error->description), 0);
  gtk_text_forward_delete(GTK_TEXT(error->description),
			  gtk_text_get_length(GTK_TEXT(error->description)));
  
  gtk_text_insert(GTK_TEXT(error->description),
		  0,
		  0,
		  0,
		  gda_error_description(gda_error),
		  strlen(gda_error_description(gda_error)));
  gtk_entry_set_text(GTK_ENTRY(error->source), gda_error_source(gda_error));

  sqlstate = gda_error_sqlstate(gda_error);
  if (sqlstate)
    gtk_entry_set_text(GTK_ENTRY(error->sqlstate), gda_error_sqlstate(gda_error));
  else
    gtk_editable_delete_text(GTK_EDITABLE(error->sqlstate), 0, 100);
  gtk_text_set_editable(GTK_TEXT(error->description), FALSE);
  gtk_entry_set_editable(GTK_ENTRY(error->source), FALSE);
  gtk_entry_set_editable(GTK_ENTRY(error->sqlstate), FALSE);
  gtk_text_thaw(GTK_TEXT(error->description));
  
  if (gda_error->helpurl)
    gtk_widget_set_sensitive(GTK_WIDGET(error->help_button), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(error->help_button), FALSE);

  
  g_snprintf(position, sizeof(position), _("Error %d of %d"), error->position + 1,
	     g_list_length(error->cnc->errors_head));
  gtk_label_set_text(GTK_LABEL(error->position_label), position);
}

guint
gnome_db_error_get_type(void)
{
  static guint db_error_type = 0;

  if (!db_error_type)
    {
      GtkTypeInfo db_error_info =
      {
	"GnomeDbError",
	sizeof (GnomeDbError),
	sizeof(GnomeDbErrorClass),
	(GtkClassInitFunc)gnome_db_error_class_init,
	(GtkObjectInitFunc)gnome_db_error_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      db_error_type = gtk_type_unique(gtk_vbox_get_type(), &db_error_info);
    }
  return db_error_type;
}

static void
gnome_db_error_class_init(GnomeDbErrorClass* klass)
{
  GtkObjectClass* object_class;

  object_class = (GtkObjectClass*) klass;

  gnome_db_error_signals[GNOME_DB_PREV] = gtk_signal_new("previous",
						         GTK_RUN_LAST,
						         object_class->type,
						         GTK_SIGNAL_OFFSET(GnomeDbErrorClass, prev),
						         gtk_signal_default_marshaller,
						         GTK_TYPE_INT, 0);
  gnome_db_error_signals[GNOME_DB_NEXT] = gtk_signal_new("next",
						         GTK_RUN_LAST,
						         object_class->type,
						         GTK_SIGNAL_OFFSET(GnomeDbErrorClass, next),
						         gtk_signal_default_marshaller,
						         GTK_TYPE_INT, 0);
		      
  gtk_object_class_add_signals(object_class, gnome_db_error_signals, LAST_SIGNAL);
  
  klass->next = gnome_db_error_real_next;
  klass->prev = gnome_db_error_real_prev;
}
      

#define XTABLE  (GTK_EXPAND|GTK_FILL)
#define YTABLE  (GTK_FILL)

#define LABEL_XTABLE (GTK_FILL)
#define LABEL_YTABLE (GTK_FILL)

static void
gnome_db_error_init (GnomeDbError* error)
{
  GtkWidget* label;
  GtkWidget* table;
  GtkWidget* vscroll;
  GtkWidget* text_table;
  
  table = gtk_table_new(3, 4, FALSE);

  label = gda_ui_new_label_widget(_("Description"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
		   LABEL_XTABLE, LABEL_YTABLE, 3, 3);

  error->description = gtk_text_new(0, 0);
  gtk_text_set_word_wrap(GTK_TEXT(error->description),
			 TRUE);
  gtk_text_set_line_wrap(GTK_TEXT(error->description),
			 TRUE);

  gtk_widget_set_usize(error->description, -2, 10 + 2 * gdk_string_height(error->description->style->font, ""));
  
  vscroll = gtk_vscrollbar_new(GTK_TEXT(error->description)->vadj);
  text_table = gtk_table_new(1, 2, FALSE);
  gtk_table_attach(GTK_TABLE(text_table), error->description, 0, 1, 0, 1,
		   GTK_FILL|GTK_EXPAND|GTK_SHRINK, GTK_FILL|GTK_EXPAND|GTK_SHRINK, 3, 3);
  gtk_table_attach(GTK_TABLE(text_table), vscroll, 1, 2, 0, 1,
		   GTK_FILL|GTK_SHRINK, GTK_FILL|GTK_EXPAND|GTK_SHRINK, 3, 3);
  gtk_widget_show(vscroll);
  gtk_widget_show(error->description);

  gtk_table_attach(GTK_TABLE(table), text_table, 1, 2, 0, 1,
		   XTABLE, GTK_EXPAND|YTABLE, 3, 3);
  gtk_widget_show(text_table);

  error->help_button = gda_ui_new_button_widget_with_pixmap(_("Show Help"),
                                                            GNOME_STOCK_BUTTON_HELP);
  gtk_table_attach(GTK_TABLE(table), error->help_button, 3, 4, 0, 1,
		   GTK_FILL, GTK_EXPAND|GTK_FILL, 3, 3);
  gtk_widget_show(error->help_button);

  label = gda_ui_new_label_widget(_("Source"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
		   LABEL_XTABLE, LABEL_YTABLE, 3, 3);
  
  error->source = gda_ui_new_entry_widget(0, FALSE);
  gtk_widget_set_usize(error->source, gdk_string_width(error->source->style->font, "M") * 40, -2);
  gtk_table_attach(GTK_TABLE(table), error->source, 1, 2, 1, 2,
		   XTABLE, YTABLE, 3, 3);
  
  label = gda_ui_new_label_widget(_("sqlstate"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
		   LABEL_XTABLE, LABEL_YTABLE, 3, 3);
  
  error->sqlstate = gda_ui_new_entry_widget(0, FALSE);
  gtk_table_attach(GTK_TABLE(table), error->sqlstate, 1, 2, 2, 3,
		   XTABLE, YTABLE, 3, 3);

  error->position_label = gda_ui_new_label_widget(_("Error 0 of 0"));
  gtk_misc_set_alignment(GTK_MISC(error->position_label), 0.5, 0.5);
  gtk_table_attach(GTK_TABLE(table), error->position_label, 0, 2, 3, 4,
		   LABEL_XTABLE, LABEL_YTABLE, 3, 3);
  
  gtk_box_pack_start(GTK_BOX(error), table, TRUE, TRUE, 0);
  gtk_widget_show(table);
}

/**
 * gnome_db_error_new
 * @cnc: Gda_Connection to show error from
 *
 * Create a new GnomeDbError widget, which is a special widget used to
 * traverse errors occurred for a given connection. It is a simple container
 * widget that you can pack into any widget you want.
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget*
gnome_db_error_new (Gda_Connection* cnc)
{
  
  GtkWidget* widget;
  GnomeDbError* error;
  
  widget = gtk_type_new(gnome_db_error_get_type());
  GTK_BOX(widget)->homogeneous = 0;
  
  error = GNOME_DB_ERROR(widget);
  error->cnc = cnc;
  gtk_widget_hide(widget);
  
  return widget;
}

/**
 * gnome_db_error_show
 * @error_widget: the GnomeDbError widget
 *
 * Make the given GnomeDbError widget visible
 */
void
gnome_db_error_show (GnomeDbError* error_widget)
{
  GnomeDbError* error = GNOME_DB_ERROR(error_widget);

  error->error_ptr = error->cnc->errors_head;
  if (!error->error_ptr)
    return;
  error->position = 0;
  display_current_error(error);
  
  gtk_widget_show(GTK_WIDGET(error_widget));
}

/**
 * gnome_db_error_prev
 * @error_widget: the GnomeDbError widget
 *
 * Display the previous error in the given GnomeDbError widget
 */
void
gnome_db_error_prev(GnomeDbError* error_widget)
{
  gtk_signal_emit(GTK_OBJECT(error_widget), gnome_db_error_signals[GNOME_DB_PREV]);
}

/**
 * gnome_db_error_next
 * @error_widget: the GnomeDbError widget
 *
 * Display the next error in the given GnomeDbError widget
 */
void
gnome_db_error_next(GnomeDbError* error_widget)
{
  gtk_signal_emit(GTK_OBJECT(error_widget), gnome_db_error_signals[GNOME_DB_NEXT]);
}
