/* GNOME DB Common Library
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-common.h"

/**
 * gda_server_new:
 *
 * Allocates memory for a new Gda_Server object and initializes struct 
 * members.
 *
 * Returns: a pointer to a new Gda_Server object.
 */
Gda_Server *
gda_server_new (void)
{
  Gda_Server* retval;

  retval = g_new0(Gda_Server, 1);
  return retval;
}

/**
 * gda_server_copy:
 * @server: the server to be copies.
 *
 * Make a deep copy of all the data needed to describe a gGDA server.
 *
 * Returns: a pointer to the newly allocated server object
 */
Gda_Server *
gda_server_copy (Gda_Server* server)
{
  Gda_Server* retval;

  retval = gda_server_new();
  
  if (server->name)
    retval->name = g_strdup(server->name);

  if (server->comment)
    retval->comment = g_strdup(server->comment);

  if (server->location)
    retval->location = g_strdup(server->location);
  if (server->repo_id)
    retval->repo_id = g_strdup(server->repo_id);
  
  return retval;
}

/**
 * gda_server_free:
 * @server: the server to de-allocate.
 *
 * Frees the memory allocated with gda_server_new() and the memory
 * allocated to struct members.
 */
void
gda_server_free (Gda_Server* server)
{
  if (server->name) g_free((gpointer) server->name);
  if (server->comment) g_free((gpointer) server->comment);
  if (server->location) g_free((gpointer) server->location);
  if (server->repo_id) g_free((gpointer) server->repo_id);
  g_free(server);
}

/**
 * gda_server_list:
 *
 * Searches the GNORBA database for GDA servers and returns a Glist of 
 * Gda_Server structs. 
 *
 * Returns: a GList of GDA servers structs
 */
GList *
gda_server_list (void)
{
  GList*              retval = 0;
#if USING_OAF
  OAF_ServerInfoList* servlist;
#else
  GoadServerList*     servlist;
  GoadServer*         slist;
#endif
  gint                i;
  Gda_Server*         server;
  
  servlist = goad_server_list_get();
  slist = servlist->list;
  for (i = 0; slist[i].repo_id; i++)
    {
      int j;
      gboolean is_match = FALSE;

      for(j = 0; slist[i].repo_id[j] && !is_match; j++)
	if (strcmp(slist[i].repo_id[j], "IDL:GDA/ConnectionFactory:1.0")  == 0)
	  {
	    is_match = TRUE;
	    break;
	  }
      if (is_match)
	{
	  fprintf(stderr,"Found server '%s'\n", slist[i].server_id);
	  server = gda_server_new();
	  server->name     = g_strdup(slist[i].server_id);
	  server->location = g_strdup(slist[i].location_info);
	  server->comment  = g_strdup(slist[i].description);
	  server->repo_id  = g_strdup(*(slist[i].repo_id));
	  server->type     = slist[i].type;
	  retval = g_list_append(retval, server);
	}
    }
  goad_server_list_free(servlist);
  return retval;
}

/**
 * gda_server_free_list
 * @list: list of #Gda_Server structures
 *
 * Frees a list of #Gda_Server structures previously returned by
 * a call to #gda_server_list
 */
void
gda_server_free_list (GList *list)
{
  GList* node;

  while ((node = g_list_first(list)))
    {
      Gda_Server* server = (Gda_Server *) node->data;
      list = g_list_remove(list, (gpointer) server);
      gda_server_free(server);
    }
}

/**
 * gda_server_find_by_name
 * @provider: provider name
 *
 * Returns a #Gda_Server structure describing the given provider. This function
 * searches all the servers present on your system
 * and tries to find the specified provider.
 *
 * Returns: a pointer to the server structure, or NULL on error
 */
Gda_Server *
gda_server_find_by_name (const gchar *provider)
{
  GList*      list;
  GList*      node;
  Gda_Server* server = NULL;

  g_return_val_if_fail(provider, NULL);

  list = gda_server_list();
  node = g_list_first(list);
  while (node)
    {
      if (!strcmp(provider, GDA_SERVER_NAME((Gda_Server *) node->data)))
        {
          server = gda_server_copy((Gda_Server *) node->data);
          break;
        }
      node = g_list_next(node);
    }
  gda_server_free_list(list);
  return (server);
}

/**
 * gda_list_datasources:
 *
 * Lists all datasources configured on the system.
 *
 * Returns a GList with the names of all data sources configured.
 */
GList *
gda_list_datasources (void)
{
  GList* res = 0;
  GList* dsns;
  GList* node;

  dsns = node = gda_dsn_list();
  while (node)
    {
      Gda_Dsn* dsn = (Gda_Dsn *) node->data;
      if (dsn)
	{
	  res = g_list_append(res, g_strdup(GDA_DSN_GDA_NAME(dsn)));
	}
      node = g_list_next(node);
    }
  gda_dsn_free_list(dsns);

  return res;
}

/**
 * gda_list_datasources_for_provider:
 * @provider: the provider which should be used to look for datasources
 *
 * Returns: a GList of all datasources available to a specific @provider.
 */
GList *
gda_list_datasources_for_provider (gchar* provider)
{
  GList* res = 0;
  GList* dsns;
  GList* node;

  dsns = node = gda_dsn_list();
  while (node)
    {
      Gda_Dsn* dsn = (Gda_Dsn *) node->data;
      if (dsn && !strcmp(GDA_DSN_PROVIDER(dsn), provider))
	{
	  res = g_list_append(res, g_strdup(GDA_DSN_GDA_NAME(dsn)));
	}
      node = g_list_next(node);
    }
  gda_dsn_free_list(dsns);

  return res;
}

static gchar *
get_config_string (const gchar *format, ...)
{
  gchar   buffer[2048];
  va_list args;

  g_return_val_if_fail(format, 0);

  va_start(args, format);
  vsprintf(buffer, format, args);
  va_end(args);

  return gnome_config_get_string(buffer);
}

/**
 * gda_dsn_list
 *
 * Returns a list of all available data sources. The returned value is
 * a GList of #Gda_Dsn structures
 */
GList *
gda_dsn_list (void)
{
  gpointer gda_iterator;
  GList*   datasources = 0;
  gchar*   gda_name;
  gchar*   section_name;
  gchar*   global_gdalib;
  gchar*   str;
  
  /* first get local data sources */
  gda_iterator = gnome_config_init_iterator("/gdalib/Datasources");
  while ((gda_iterator = gnome_config_iterator_next(gda_iterator, &gda_name, &section_name)))
    {
      Gda_Dsn* dsn = gda_dsn_new();

      dsn->gda_name = g_strdup(section_name);
      dsn->provider = get_config_string("/gdalib/%s/Provider", dsn->gda_name);
      dsn->dsn_str = get_config_string("/gdalib/%s/DSN", dsn->gda_name);
      dsn->description = get_config_string("/gdalib/%s/Description", dsn->gda_name);
      dsn->username = get_config_string("/gdalib/%s/Username", dsn->gda_name);
      dsn->config = get_config_string("/gdalib/%s/Configurator", dsn->gda_name);
      dsn->is_global = FALSE;

      datasources = g_list_append(datasources, (gpointer) dsn);
    }

  /* then, global data sources */
  global_gdalib = g_strdup_printf("=%s/gdalib=", GDA_CONFIG_DIR);
  str = g_strdup_printf("%s/Datasources", global_gdalib);
  gda_iterator = gnome_config_init_iterator(str);
  while ((gda_iterator = gnome_config_iterator_next(gda_iterator, &gda_name, &section_name)))
    {
      Gda_Dsn* dsn = NULL;
      GList*   node;

      /* look if the DSN already exists in user config */
      node = g_list_first(datasources);
      while (node)
	{
	  Gda_Dsn* dsn_tmp = (Gda_Dsn *) node->data;
	  if (dsn_tmp)
	    {
	      if (!g_strcasecmp(GDA_DSN_GDA_NAME(dsn_tmp), gda_name))
		{
		  /* only fill info not supplied in the user config */
		  if (!dsn_tmp->provider)
		    dsn_tmp->provider = get_config_string("%s/%s/Provider", global_gdalib, dsn->gda_name);
		  if (!dsn_tmp->dsn_str)
		    dsn->dsn_str = get_config_string("%s/%s/DSN", global_gdalib, dsn->gda_name);
		  if (!dsn_tmp->description)
		    dsn->description = get_config_string("%s/%s/Description", global_gdalib, dsn->gda_name);
		  if (!dsn_tmp->username)
		    dsn->username = get_config_string("%s/%s/Username", global_gdalib, dsn->gda_name);
		  if (!dsn_tmp->config)
		    dsn->config = get_config_string("%s/%s/Configurator", global_gdalib, dsn->gda_name);
		  dsn->is_global = TRUE;

		  dsn = dsn_tmp;
		  break;
		}
	    }
	  node = g_list_next(node);
	}
      if (!dsn)
	{
	  dsn = gda_dsn_new();

	  dsn->gda_name = g_strdup(section_name);
	  dsn->provider = get_config_string("%s/%s/Provider", global_gdalib, dsn->gda_name);
	  dsn->dsn_str = get_config_string("%s/%s/DSN", global_gdalib, dsn->gda_name);
	  dsn->description = get_config_string("%s/%s/Description", global_gdalib, dsn->gda_name);
	  dsn->username = get_config_string("%s/%s/Username", global_gdalib, dsn->gda_name);
	  dsn->config = get_config_string("%s/%s/Configurator", global_gdalib, dsn->gda_name);
	  dsn->is_global = TRUE;

	  datasources = g_list_append(datasources, (gpointer) dsn);
	}
      node = g_list_next(node);
    }
  g_free((gpointer) str);
  g_free((gpointer) global_gdalib);
  return datasources;
}

/**
 * gda_dsn_free
 * @dsn: data source
 */
void
gda_dsn_free (Gda_Dsn *dsn)
{
  g_return_if_fail(dsn != NULL);

  if (dsn->gda_name) g_free((gpointer) dsn->gda_name);
  if (dsn->provider) g_free((gpointer) dsn->provider);
  if (dsn->dsn_str) g_free((gpointer) dsn->dsn_str);
  if (dsn->description) g_free((gpointer) dsn->description);
  if (dsn->username) g_free((gpointer) dsn->username);
  if (dsn->config) g_free((gpointer) dsn->config);
  g_free((gpointer) dsn);
}

/**
 * gda_dsn_find_by_name
 * @dsn_name: data source name
 */
Gda_Dsn *
gda_dsn_find_by_name (const gchar *dsn_name)
{
  GList*   list;
  gboolean found = FALSE;
  Gda_Dsn* rc = NULL;

  g_return_val_if_fail(dsn_name != NULL, NULL);

  list = gda_dsn_list();
  while (list)
    {
      Gda_Dsn* dsn = (Gda_Dsn *) list->data;
      if (dsn && !found)
	{
	  if (!g_strcasecmp(GDA_DSN_GDA_NAME(dsn), dsn_name))
	    {
	      rc = dsn;
	      found = TRUE;
	    }
	  else gda_dsn_free(dsn);
	}
      else gda_dsn_free(dsn);
      list = g_list_next(list);
    }
  g_list_free(g_list_first(list));
  return rc;
}

/**
 * gda_dsn_set_name
 * @dsn: data source
 * @name: new data source name
 */
void
gda_dsn_set_name (Gda_Dsn *dsn, const gchar *name)
{
  g_return_if_fail(dsn != NULL);
  g_return_if_fail(name != NULL);

  if (dsn->gda_name) g_free((gpointer) dsn->gda_name);
  dsn->gda_name = g_strdup(name);
}

/**
 * gda_dsn_set_provider
 * @dsn: data source
 * @provider: provider name
 */
void
gda_dsn_set_provider (Gda_Dsn *dsn, const gchar *provider)
{
  g_return_if_fail(dsn != NULL);
  g_return_if_fail(provider != NULL);

  /* FIXME: should check if the provider does exist */
  if (dsn->provider) g_free((gpointer) dsn->provider);
  dsn->provider = g_strdup(provider);
}

/**
 * gda_dsn_set_dsn
 * @dsn: data source
 * @dsn_str: DSN connection string
 */
void
gda_dsn_set_dsn (Gda_Dsn *dsn, const gchar *dsn_str)
{
  g_return_if_fail(dsn != NULL);
  g_return_if_fail(dsn_str != NULL);

  if (dsn->dsn_str) g_free((gpointer) dsn->dsn_str);
  dsn->dsn_str = g_strdup(dsn_str);
}

/**
 * gda_dsn_set_description
 * @dsn: data source
 * @description: DSN description
 */
void
gda_dsn_set_description (Gda_Dsn *dsn, const gchar *description)
{
  g_return_if_fail(dsn != NULL);
  g_return_if_fail(description != NULL);

  if (dsn->description) g_free((gpointer) dsn->description);
  dsn->description = g_strdup(description);
}

/**
 * gda_dsn_set_username
 * @dsn: data source
 * @username: user name
 */
void
gda_dsn_set_username (Gda_Dsn *dsn, const gchar *username)
{
  g_return_if_fail(dsn != NULL);
  g_return_if_fail(username != NULL);

  if (dsn->username) g_free((gpointer) dsn->username);
  dsn->username = g_strdup(username);
}

/**
 * gda_dsn_set_config
 * @dsn: data source
 * @config: configurator
 */
void
gda_dsn_set_config (Gda_Dsn *dsn, const gchar *config)
{
  g_return_if_fail(dsn != NULL);
  g_return_if_fail(config != NULL);

  if (dsn->config) g_free((gpointer) dsn->config);
  dsn->config = g_strdup(config);
}

/**
 * gda_dsn_set_global
 * @dsn: data source
 * @is_global: global flag
 */
void
gda_dsn_set_global (Gda_Dsn *dsn, gboolean is_global)
{
  g_return_if_fail(dsn != NULL);
  dsn->is_global = is_global;
}

/**
 * gda_dsn_save
 * @dsn: data source
 *
 * Saves the given data source into the GDA configuration
 */
gboolean
gda_dsn_save (Gda_Dsn *dsn)
{
  g_return_val_if_fail(dsn != NULL, FALSE);

  if (dsn->gda_name)
    {
      gchar* config_prefix;
      gchar* tmp;

      if (dsn->is_global)
	config_prefix = g_strdup_printf("=%s/gdalib=", GDA_CONFIG_DIR);
      else config_prefix = g_strdup("/gdalib");

      tmp = g_strdup_printf("%s/Datasources/%s", config_prefix, dsn->gda_name);
      gnome_config_set_string(tmp, GDA_DSN_GDA_NAME(dsn));
      g_free((gpointer) tmp);

      tmp = g_strdup_printf("%s/%s/Provider", config_prefix, dsn->gda_name);
      gnome_config_set_string(tmp, GDA_DSN_PROVIDER(dsn));
      g_free((gpointer) tmp);

      tmp = g_strdup_printf("%s/%s/DSN", config_prefix, dsn->gda_name);
      gnome_config_set_string(tmp, GDA_DSN_DSN(dsn));
      g_free((gpointer) tmp);

      tmp = g_strdup_printf("%s/%s/Description", config_prefix, dsn->gda_name);
      gnome_config_set_string(tmp, GDA_DSN_DESCRIPTION(dsn));
      g_free((gpointer) tmp);

      tmp = g_strdup_printf("%s/%s/Username", config_prefix, dsn->gda_name);
      gnome_config_set_string(tmp, GDA_DSN_USERNAME(dsn));
      g_free((gpointer) tmp);

      tmp = g_strdup_printf("%s/%s/Configurator", config_prefix, dsn->gda_name);
      gnome_config_set_string(tmp, GDA_DSN_CONFIG(dsn));
      g_free((gpointer) tmp);

      gnome_config_sync();
      g_free((gpointer) config_prefix);
      return TRUE;
    }
  else g_warning("data source has no name");
  return FALSE;
}

/**
 * gda_dsn_remove
 * @dsn: data source
 */
gboolean
gda_dsn_remove (Gda_Dsn *dsn)
{
  gchar* config_prefix;
  gchar* tmp;

  g_return_val_if_fail(dsn != NULL, FALSE);

  if (dsn->is_global)
    config_prefix = g_strdup_printf("=%s/gdalib=", GDA_CONFIG_DIR);
  else config_prefix = g_strdup("/gdalib");

  tmp = g_strdup_printf("%s/Datasources/%s", config_prefix, GDA_DSN_GDA_NAME(dsn));
  gnome_config_clean_key(tmp);
  g_free((gpointer) tmp);

  tmp = g_strdup_printf("%s/%s", config_prefix, GDA_DSN_GDA_NAME(dsn));
  gnome_config_clean_section(tmp);
  gnome_config_sync();
  g_free((gpointer) tmp);

  g_free((gpointer) config_prefix);
  return TRUE;
}

/**
 * gda_dsn_free_list
 */
void
gda_dsn_free_list (GList *list)
{
  GList* node;

  g_return_if_fail(list);

  while ((node = g_list_first(list)))
    {
      Gda_Dsn* dsn = (Gda_Dsn *) node->data;
      list = g_list_remove(list, (gpointer) dsn);
      gda_dsn_free(dsn);
    }
}

