/* GNOME DB Front End
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-fe.h"
#include <stdio.h>

static void activate_export_cb   (GtkWidget *w, gpointer data);
static void activate_import_cb   (GtkWidget *w, gpointer data);
static void grid_loaded_cb       (GnomeDbGrid *grid, gpointer data);
static void grid_row_selected_cb (GnomeDbGrid *grid, gint row, gpointer data);

static void clear_sql_cb         (GtkWidget *w, gpointer data);
static void copy_sql_cb          (GtkWidget *w, gpointer data);
static void cut_sql_cb           (GtkWidget *w, gpointer data);
static void open_sql_cb          (GtkWidget *w, gpointer data);
static void paste_sql_cb         (GtkWidget *w, gpointer data);
static void run_sql_cb           (GtkWidget *w, gpointer data);
static void save_sql_cb          (GtkWidget *w, gpointer data);
static void select_sql_cb        (GtkWidget *w, gpointer data);

static GList*      connections = NULL;
static GnomeUIInfo cnc_toolbox[] =
{
  { GNOME_APP_UI_ITEM, N_("SQL"), N_("SQL Window"),
    fe_activate_connection_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Batch"), N_("Batch commands window"),
    fe_activate_connection_batch_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TIMER, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Queries"), N_("Query designer"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_INDEX, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Browser"), N_("Browse Database"),
    fe_activate_connection_browser_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PROP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Export"), N_("Export database"),
    activate_export_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_REDO, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Import"), N_("Import data into current database"),
    activate_import_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_UNDO, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Disconnect"), N_("Close connection"),
    fe_close_connection_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_STOP, 0, 0, NULL },
  GNOMEUIINFO_END
};
static GnomeUIInfo sql_toolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Execute"), N_("Execute current command"),
    run_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Select"), N_("Select from previous commands"),
    select_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_UP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Open"), N_("Load file into buffer"),
    open_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), N_("Save current file"),
    save_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
  GNOMEUIINFO_END
};
static GnomeUIInfo sql_toolbar2[] =
{
  { GNOME_APP_UI_ITEM, N_("Cut"), N_("Cut selected text"),
    cut_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_CUT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selected text"),
    copy_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste clipboard contents"),
    paste_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PASTE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear window contents"),
    clear_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  GNOMEUIINFO_END
};
static GnomeUIInfo export_toolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Start"), N_("Start export"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear all fields"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Select all"), N_("Select all objects in database"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_FORWARD, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Unselect"), N_("Clear object selection"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_BACK, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
fill_feature_list (Gda_Connection *cnc, GtkCList *clist)
{
  register gint     cnt;
  gchar*            row[2];
  struct
  {
    gchar*                 label;
    GDA_Connection_Feature feature;
  } features[] =
  {
    { N_("Foreign keys"), GDA_Connection_FEATURE_FOREIGN_KEYS },
    { N_("Object Identifiers"), GDA_Connection_FEATURE_OBJECT_ID },
    { N_("Object Inheritance"), GDA_Connection_FEATURE_INHERITANCE },
    { N_("Sequences"), GDA_Connection_FEATURE_SEQUENCES },
    { N_("SQL (Standard)"), GDA_Connection_FEATURE_SQL },
    { N_("SQL Sub-Selects"), GDA_Connection_FEATURE_SQL_SUBSELECT },
    { N_("Stored Procedures"), GDA_Connection_FEATURE_PROCS },
    { N_("Transactions"), GDA_Connection_FEATURE_TRANSACTIONS },
    { N_("Triggers"), GDA_Connection_FEATURE_TRIGGERS },
    { N_("Views"), GDA_Connection_FEATURE_VIEWS },
    { N_("XML Queries"), GDA_Connection_FEATURE_XML_QUERIES }
  };
  GtkWidget* on_pixmap;
  GtkWidget* off_pixmap;

  g_return_if_fail(IS_GDA_CONNECTION(cnc));
  g_return_if_fail(GTK_IS_CLIST(clist));

  /* create pixmaps */
  on_pixmap = gda_ui_get_pixmap(GNOME_STOCK_MENU_BOOK_OPEN);
  off_pixmap = gda_ui_get_pixmap(GNOME_STOCK_MENU_BOOK_GREEN);

  row[0] = (gchar *) g_malloc0(32);

  gtk_clist_freeze(clist);
  for (cnt = 0; cnt < sizeof(features) / sizeof(features[0]); cnt ++)
    {
      strcpy(row[0], features[cnt].label);
      row[1] = "";
      gtk_clist_append(clist, row);
      if (gda_connection_supports(cnc, features[cnt].feature))
        {
          gtk_clist_set_pixmap(clist, cnt, 1,
                               GNOME_PIXMAP(on_pixmap)->pixmap,
                               GNOME_PIXMAP(on_pixmap)->mask);
        }
      else
        {
          gtk_clist_set_pixmap(clist, cnt, 1,
                               GNOME_PIXMAP(off_pixmap)->pixmap,
                               GNOME_PIXMAP(off_pixmap)->mask);
        }
    }

  gtk_clist_thaw(clist);
  g_free((gpointer) row[0]);
}

static void
init_browser_window (connection_data_t *cnc_data)
{
  g_return_if_fail(cnc_data != NULL);

  if (!cnc_data->browser_tab)
    {
      cnc_data->browser_tab = gda_ui_new_table_widget(2, 4, FALSE);
      cnc_data->browser = gnome_db_browser_new(cnc_data->cnc);
      gtk_widget_show(cnc_data->browser);
      gtk_table_attach(GTK_TABLE(cnc_data->browser_tab), cnc_data->browser, 1, 2, 0, 4,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       3, 3);
      gtk_notebook_append_page(GTK_NOTEBOOK(cnc_data->widget),
			       cnc_data->browser_tab,
			       gtk_label_new(_("Browser")));
    }
}

static void
init_export_window (connection_data_t *cnc_data)
{
  g_return_if_fail(cnc_data != NULL);

  if (!cnc_data->export_tab)
    {
      GtkWidget* scroll;
      GtkWidget* toolbar;
      GtkWidget* label;

      cnc_data->export_tab = gda_ui_new_table_widget(4, 4, FALSE);

      scroll = gda_ui_new_scrolled_window_widget();
      gtk_table_attach(GTK_TABLE(cnc_data->export_tab), scroll, 0, 1, 0, 4,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       3, 3);
      cnc_data->export_tree = gda_ui_new_tree_widget();
      gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll),
					    cnc_data->export_tree);

      /* add toolbar */
      toolbar = gda_ui_new_toolbar_widget(GTK_ORIENTATION_HORIZONTAL,
					  GTK_TOOLBAR_ICONS,
					  export_toolbar,
                                          NULL);
      gtk_table_attach(GTK_TABLE(cnc_data->export_tab), toolbar, 1, 4, 0, 1,
		       GTK_FILL,
		       GTK_FILL,
		       3, 3);

      label = gda_ui_new_label_widget(_("File name"));
      gtk_table_attach(GTK_TABLE(cnc_data->export_tab), label, 1, 2, 1, 2,
		       GTK_SHRINK,
		       GTK_SHRINK,
		       3, 3);
      cnc_data->export_filename_entry = gda_ui_new_file_entry_widget("FE_Export_history");
      gtk_table_attach(GTK_TABLE(cnc_data->export_tab), cnc_data->export_filename_entry,
		       2, 3, 1, 2,
		       GTK_SHRINK,
		       GTK_SHRINK,
		       3, 3);

      /* create log window */
      scroll = gda_ui_new_scrolled_window_widget();
      gtk_table_attach(GTK_TABLE(cnc_data->export_tab), scroll, 1, 4, 3, 4,
		       GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		       GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		       3, 3);
      cnc_data->export_log = gda_ui_new_text_widget();
      gtk_container_add(GTK_CONTAINER(scroll), cnc_data->export_log);

      /* add tab */
      gtk_notebook_append_page(GTK_NOTEBOOK(cnc_data->widget),
			       cnc_data->export_tab,
			       gtk_label_new(_("Export")));
    }
}

static void
init_import_window (connection_data_t *cnc_data)
{
  g_return_if_fail(cnc_data != NULL);

  if (!cnc_data->import_tab)
    {
      cnc_data->import_tab = gda_ui_new_table_widget(2, 4, FALSE);
      gtk_notebook_append_page(GTK_NOTEBOOK(cnc_data->widget),
			       cnc_data->import_tab,
			       gtk_label_new(_("Import")));
    }
}

static void
init_sql_window (connection_data_t *cnc_data)
{
  GtkWidget* scroll;
  GtkWidget* toolbar;

  g_return_if_fail(cnc_data != NULL);

  cnc_data->sql_tab = gda_ui_new_table_widget(3, 6, FALSE);

  scroll = gda_ui_new_scrolled_window_widget();
  gtk_table_attach(GTK_TABLE(cnc_data->sql_tab), scroll, 2, 3, 0, 1,
		   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		   3, 3);
  cnc_data->cmd_area = gda_ui_new_text_widget();
  gtk_container_add(GTK_CONTAINER(scroll), cnc_data->cmd_area);

  cnc_data->grid = gda_ui_new_grid_widget(NULL);
  gtk_signal_connect(GTK_OBJECT(cnc_data->grid),
		     "loaded",
		     GTK_SIGNAL_FUNC(grid_loaded_cb),
		     NULL);
  gtk_signal_connect(GTK_OBJECT(cnc_data->grid),
		     "select_row",
		     GTK_SIGNAL_FUNC(grid_row_selected_cb),
		     NULL);
  gtk_signal_connect(GTK_OBJECT(cnc_data->grid),
		     "unselect_row",
		     GTK_SIGNAL_FUNC(grid_row_selected_cb),
		     NULL);
  gtk_table_attach(GTK_TABLE(cnc_data->sql_tab), cnc_data->grid, 0, 3, 1, 6,
		   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
		   3, 3);

  /* add toolbars */
  toolbar = gda_ui_new_toolbar_widget(GTK_ORIENTATION_VERTICAL,
				      GTK_TOOLBAR_ICONS,
				      sql_toolbar,
                                      (gpointer) cnc_data);
  gtk_table_attach(GTK_TABLE(cnc_data->sql_tab), toolbar, 0, 1, 0, 1,
		   GTK_FILL,
		   GTK_FILL,
		   3, 3);

  toolbar = gda_ui_new_toolbar_widget(GTK_ORIENTATION_VERTICAL,
				      GTK_TOOLBAR_ICONS,
				      sql_toolbar2,
                                      (gpointer) cnc_data);
  gtk_table_attach(GTK_TABLE(cnc_data->sql_tab), toolbar, 1, 2, 0, 1,
		   GTK_FILL,
		   GTK_FILL,
		   3, 3);
  gtk_notebook_append_page(GTK_NOTEBOOK(cnc_data->widget),
			   cnc_data->sql_tab,
			   gtk_label_new("SQL"));
}

/*
 * Callbacks
 */
static void
activate_export_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = fe_connection_get_current();

  if (cnc_data)
    {
      init_export_window(cnc_data);
      gtk_notebook_set_page(GTK_NOTEBOOK(cnc_data->widget),
			    gtk_notebook_page_num(GTK_NOTEBOOK(cnc_data->widget),
						  cnc_data->export_tab));
    }
}

static void
activate_import_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = fe_connection_get_current();

  if (cnc_data)
    {
      init_import_window(cnc_data);
      gtk_notebook_set_page(GTK_NOTEBOOK(cnc_data->widget),
			    gtk_notebook_page_num(GTK_NOTEBOOK(cnc_data->widget),
						  cnc_data->import_tab));
    }
}

static void
clear_sql_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = (connection_data_t *) data;

  if (cnc_data)
    {
      if (GTK_IS_EDITABLE(cnc_data->cmd_area))
	{
	  gtk_editable_delete_text(GTK_EDITABLE(cnc_data->cmd_area),
				   0,
				   gtk_text_get_length(GTK_TEXT(cnc_data->cmd_area)));
	}
      if (GNOME_DB_IS_GRID(cnc_data->grid))
	gnome_db_grid_set_recordset(GNOME_DB_GRID(cnc_data->grid), NULL);
    }
}

static void
copy_sql_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = (connection_data_t *) data;

  if (cnc_data && GTK_IS_EDITABLE(cnc_data->cmd_area))
    {
      gtk_editable_copy_clipboard(GTK_EDITABLE(cnc_data->cmd_area));
    }
}

static void
cut_sql_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = (connection_data_t *) data;

  if (cnc_data && GTK_IS_EDITABLE(cnc_data->cmd_area))
    {
      gtk_editable_cut_clipboard(GTK_EDITABLE(cnc_data->cmd_area));
    }
}

static void
grid_loaded_cb (GnomeDbGrid *grid, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_GRID(grid));

  gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
				_("%d rows"),
				gnome_db_grid_get_row_count(grid));
}

static void
grid_row_selected_cb (GnomeDbGrid *grid, gint row, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_GRID(grid));

  gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
				_("Row %d - %d rows selected"),
				row, gnome_db_grid_get_selected_rows(grid));
}

static void
open_sql_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      gchar* filename = gda_ui_select_file(_("Select file"));
      if (filename)
	{
	  FILE* fp;

	  fp = fopen(filename, "r");
	  if (fp)
	    {
	      gchar  buffer[513];

	      /* read file contents */
	      gtk_editable_delete_text(GTK_EDITABLE(cnc_data->cmd_area),
				       0,
				       gtk_text_get_length(GTK_TEXT(cnc_data->cmd_area)));
	      while (!feof(fp))
		{
		  memset(buffer, '\0', sizeof(buffer));
		  fread((const void *) buffer, sizeof(buffer) - 1, 1, fp);
		  if (ferror(fp))
		    {
		      gda_ui_show_error(_("Error reading from file\n'%s'"), filename);
		      break;
		    }
		  buffer[sizeof(buffer) - 1] = '\0';
		  gtk_text_insert(GTK_TEXT(cnc_data->cmd_area), NULL, NULL, NULL,
				  buffer, sizeof(buffer) - 1);
		  
		}
	      fclose(fp);
	    }
	  else gda_ui_show_error(_("Could not open file\n'%s'"), filename);
	  g_free((gpointer) filename);
	}
    }
}

static void
paste_sql_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = (connection_data_t *) data;

  if (cnc_data && GTK_IS_EDITABLE(cnc_data->cmd_area))
    {
      gtk_editable_paste_clipboard(GTK_EDITABLE(cnc_data->cmd_area));
    }
}

static void
run_sql_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      gchar* sql = gtk_editable_get_chars(GTK_EDITABLE(cnc_data->cmd_area), 0, -1);
      if (sql)
	{
	  gulong         reccount;
	  Gda_Recordset* recset = gda_connection_execute(cnc_data->cnc, sql, &reccount, 0);
	  if (recset)
	    {
	      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					    _("Command succeeded"));
	    }
	  else
	    {
	      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					    _("Command failed"));
	    }
	  gnome_db_grid_set_recordset(GNOME_DB_GRID(cnc_data->grid), recset);
	  fe_config_add_command((const gchar *) sql);
	}
    }
}

static void
save_sql_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      gchar* filename = gda_ui_select_file(_("Select file"));
      if (filename)
	{
	  gchar* txt;

	  txt = gtk_editable_get_chars(GTK_EDITABLE(cnc_data->cmd_area),
				       0,
				       gtk_text_get_length(GTK_TEXT(cnc_data->cmd_area)));
	  if (txt)
	    {
	      FILE* fp = fopen(filename, "w");
	      if (fp)
		{
		  fwrite((void *) txt, strlen(txt), 1, fp);
		  if (ferror(fp))
		    gda_ui_show_error(_("Error writing to file\n'%s'"), filename);
		  fclose(fp);
		}
	      else gda_ui_show_error(_("Could not open file\n'%s'"), filename);
	    }
	  g_free((gpointer) filename);
	}
    }
}

static void
select_sql_cb (GtkWidget *w, gpointer data)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      GtkWidget* dialog;
      GtkWidget* scroll;
      GtkWidget* clist;
      GList*     cmds;
      gint       btn;
      gint       cnt = 1;

      /* create dialog */
      dialog = gnome_dialog_new(_("Select Command"),
				GNOME_STOCK_BUTTON_OK,
				GNOME_STOCK_BUTTON_CANCEL,
				NULL);
      gtk_widget_set_usize(dialog, 400, 300);
      scroll = gda_ui_new_scrolled_window_widget();
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scroll, 1, 1, 0);
      clist = gda_ui_new_clist_widget(NULL, 2);
      gtk_container_add(GTK_CONTAINER(scroll), clist);

      /* fill list of commands */
      cmds = fe_config_get_last_commands();
      while (cmds)
	{
	  gchar* row[2];

	  row[0] = g_strdup_printf("%d", cnt);
	  row[1] = (gchar *) cmds->data;
	  gtk_clist_append(GTK_CLIST(clist), row);
	  cmds = g_list_next(cmds);
	  g_free((gpointer) row[0]);
	  cnt++;
	}

      /* run dialog */
      btn = gnome_dialog_run(GNOME_DIALOG(dialog));
      if (btn == 0)
	{
	  gchar* txt = NULL;
	  guint  item = GPOINTER_TO_UINT(GTK_CLIST(clist)->selection->data);
	  
	  if (gtk_clist_get_text(GTK_CLIST(clist), item, 1, &txt))
	    {
	      gtk_editable_delete_text(GTK_EDITABLE(cnc_data->cmd_area),
				       0,
				       gtk_text_get_length(GTK_TEXT(cnc_data->cmd_area)));
	      gtk_text_insert(GTK_TEXT(cnc_data->cmd_area),
			      NULL, NULL, NULL,
			      txt, strlen(txt));
	    }
	}
      gnome_dialog_close(GNOME_DIALOG(dialog));
    }
}

/*
 * Public functions
 */
void
fe_connection_open (const gchar *str, Gda_Connection *cnc)
{
  connection_data_t* cnc_data;
  gchar*             label;

  g_return_if_fail(IS_GDA_CONNECTION(cnc));
  g_return_if_fail(gda_connection_is_open(cnc));

  cnc_data = g_new0(connection_data_t, 1);
  cnc_data->cnc = cnc;

  /* create widget */
  cnc_data->widget = gda_ui_new_notebook_widget();
  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(cnc_data->widget), FALSE);
  gtk_object_set_data(GTK_OBJECT(cnc_data->widget), FE_CONNECTION_DATA, (gpointer) cnc_data);

  init_sql_window(cnc_data);

  cnc_data->gda_name = g_strdup(str);
  label = g_strdup_printf("%s@%s", gda_connection_get_user(cnc), cnc_data->gda_name);
  fe_workarea_add_tab(label, cnc_data->widget, cnc_toolbox);
  g_free((gpointer) label);
  connections = g_list_append(connections, (gpointer) cnc_data);
}

connection_data_t *
fe_connection_get_current (void)
{
  connection_data_t* cnc_data;
  GtkWidget*         w;

  w = fe_workarea_get_current_tab();
  if (w)
    {
      cnc_data = (connection_data_t *) gtk_object_get_data(GTK_OBJECT(w), FE_CONNECTION_DATA);
      if (cnc_data && cnc_data->widget == w)
	return cnc_data;
    }
  return NULL;
}

void
fe_connection_disconnect (void)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      gda_connection_close(cnc_data->cnc);
      gda_connection_free(cnc_data->cnc);

      fe_workarea_remove_tab(cnc_data->widget);
      connections = g_list_remove(connections, (gpointer) cnc_data);
      g_free((gpointer) cnc_data->gda_name);
      g_free((gpointer) cnc_data);

      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container), _("Ready"));
    }
}

void
fe_connection_show_errors (Gda_Connection *cnc)
{
  static GtkWidget* error_dialog = NULL;

  g_return_if_fail(IS_GDA_CONNECTION(cnc));

  if (!error_dialog)
    {
      error_dialog = gnome_db_errordlg_new(cnc, _("Error Viewer"));
    }
  gnome_db_errordlg_show_errors(GNOME_DB_ERRORDLG(error_dialog));
}

void
fe_connection_begin_transaction (void)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      if (gda_connection_supports(cnc_data->cnc, GDA_Connection_FEATURE_TRANSACTIONS))
	{
	  if (gda_connection_begin_transaction(cnc_data->cnc) == 0)
	    {
	      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					_("Transaction started"));
	    }
	  else
	    {
	      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					_("Begin transaction failed"));
	    }
	}
      else
	{
	  gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					_("Data source does not support transactions"));
	}
    }
}

void
fe_connection_commit_transaction (void)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      if (gda_connection_supports(cnc_data->cnc, GDA_Connection_FEATURE_TRANSACTIONS))
	{
	  if (gda_connection_commit_transaction(cnc_data->cnc) == 0)
	    {
	      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					_("Commit executed successfully"));
	    }
	  else
	    {
	      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					_("Commit transaction failed"));
	    }
	}
      else
	{
	  gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					_("Data source does not support transactions"));
	}
    }
}

void
fe_connection_rollback_transaction (void)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      if (gda_connection_supports(cnc_data->cnc, GDA_Connection_FEATURE_TRANSACTIONS))
	{
	  if (gda_connection_rollback_transaction(cnc_data->cnc) == 0)
	    {
	      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					_("Rollback executed successfully"));
	    }
	  else
	    {
	      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					_("Rollback transaction failed"));
	    }
	}
      else
	{
	  gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_container),
					_("Data source does not support transactions"));
	}
    }
}

void
fe_connection_show_browser_window (void)
{
  connection_data_t* cnc_data = fe_connection_get_current();

  if (cnc_data)
    {
      init_browser_window(cnc_data);
      gtk_notebook_set_page(GTK_NOTEBOOK(cnc_data->widget),
			    gtk_notebook_page_num(GTK_NOTEBOOK(cnc_data->widget),
						  cnc_data->browser_tab));
    }
}

void
fe_connection_show_sql_window (void)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      gtk_notebook_set_page(GTK_NOTEBOOK(cnc_data->widget),
			    gtk_notebook_page_num(GTK_NOTEBOOK(cnc_data->widget),
						  cnc_data->sql_tab));
    }
}

void
fe_connection_open_config (void)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      GtkWidget* widget;
      gchar*     goad_id;

      goad_id = g_strdup_printf("control:%s-config",
				gda_connection_get_provider(cnc_data->cnc));
      widget = gnome_db_control_widget_new(goad_id);
      if (GNOME_DB_IS_CONTROL_WIDGET(widget))
	{
	  GtkWidget* dialog;
	  GtkWidget* frame;
	  gboolean   true = true;

	  /* set widget properties */
	  gtk_widget_show(widget);
          gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(widget),
					       GNOME_DB_CONTROL_PROP_DSN,
					       (gconstpointer) cnc_data->gda_name);
	  gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(widget),
					       GNOME_DB_CONTROL_PROP_IDENT_NAME,
					       (gconstpointer) gda_connection_get_user(cnc_data->cnc));
	  gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(widget),
					       GNOME_DB_CONTROL_PROP_USER_NAME,
					       (gconstpointer) gda_connection_get_user(cnc_data->cnc));
	  gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(widget),
					       GNOME_DB_CONTROL_PROP_IDENT_PASS,
					       (gconstpointer) gda_connection_get_password(cnc_data->cnc));
	  gnome_db_control_widget_set_prop_ptr(GNOME_DB_CONTROL_WIDGET(widget),
					       GNOME_DB_CONTROL_PROP_START,
					       (gconstpointer) &true);

	  dialog = gnome_dialog_new(cnc_data->gda_name, GNOME_STOCK_BUTTON_CLOSE, NULL);
	  gnome_dialog_set_close(GNOME_DIALOG(dialog), TRUE);

	  frame = gda_ui_new_frame_widget(NULL);
	  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
	  gtk_container_add(GTK_CONTAINER(frame), widget);
	  gtk_widget_show(dialog);
	}
      else gda_ui_show_error(_("Could not activate module %s"), goad_id);
      g_free((gpointer) goad_id);
    }
}

void
fe_connection_view_properties (void)
{
  connection_data_t* cnc_data = fe_connection_get_current();
  if (cnc_data)
    {
      GtkWidget *dialog, *frame, *table, *label, *entry, *button, *clist, *scroll, *frame2;
      gchar      txt[256];

      /* create dialog */
      dialog = gnome_dialog_new(_("Connection Properties"),
				GNOME_STOCK_BUTTON_CLOSE,
				NULL);
      frame = gda_ui_new_frame_widget(_("Properties"));
      gtk_container_set_border_width(GTK_CONTAINER(frame), 3);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 0, 0, 0);
      table = gda_ui_new_table_widget(3, 7, 0);
      gtk_container_add(GTK_CONTAINER(frame), table);

      label = gda_ui_new_label_widget(_("GDA Name"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), cnc_data->gda_name);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("Provider"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_provider(cnc_data->cnc));
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("DSN"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_dsn(cnc_data->cnc));
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("Command Timeout"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      sprintf(txt, "%ld", gda_connection_get_cmd_timeout(cnc_data->cnc));
      gtk_entry_set_text(GTK_ENTRY(entry), txt);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("Connect Timeout"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      sprintf(txt, "%ld", gda_connection_get_connect_timeout(cnc_data->cnc));
      gtk_entry_set_text(GTK_ENTRY(entry), txt);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 4, 5, GTK_FILL, GTK_FILL, 3, 3);

      label = gda_ui_new_label_widget(_("Trace File"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 5, 6, GTK_FILL, GTK_FILL, 3, 3);
      button = gda_ui_new_button_widget_with_pixmap(_("View"), GNOME_STOCK_MENU_PROP);
      gtk_table_attach(GTK_TABLE(table), button, 2, 3, 5, 6, GTK_SHRINK, GTK_SHRINK, 3, 3);

      label = gda_ui_new_label_widget(_("Version"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 6, 7, GTK_FILL, GTK_FILL, 3, 3);
      entry = gda_ui_new_entry_widget(0, FALSE);
      gtk_entry_set_text(GTK_ENTRY(entry), gda_connection_get_version(cnc_data->cnc));
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 6, 7, GTK_FILL, GTK_FILL, 3, 3);

      frame2 = gda_ui_new_frame_widget(_("Provider Features"));
      gtk_container_set_border_width(GTK_CONTAINER(frame2), 3);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame2, 1, 1, 0);
      scroll = gda_ui_new_scrolled_window_widget();
      gtk_container_add(GTK_CONTAINER(frame2), scroll);
      clist = gda_ui_new_clist_widget(0, 2);
      gtk_container_add(GTK_CONTAINER(scroll), clist);
      fill_feature_list(cnc_data->cnc, GTK_CLIST(clist));

      /* run the dialog */
      gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
    }
}
