/* GNOME DB Front End
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-fe.h"

#define FE_WORKAREA_TAB_DATA "FE_Workarea_TabData"

typedef struct
{
  gchar*        title;
  GtkWidget*    widget;
  GnomeUIInfo** toolbox;
} workarea_tab_data_t;

static GtkWidget* work_area = NULL;
static GtkWidget* toolbox = NULL;
static GtkWidget* notebook = NULL;
static GList*     tabs = NULL;

/*
 * Private functions
 */
static workarea_tab_data_t *
find_tab_by_widget (GtkWidget *widget)
{
  GList* node = g_list_first(tabs);

  g_return_val_if_fail(GTK_IS_WIDGET(widget), NULL);

  while (node)
    {
      workarea_tab_data_t* tab_data = (workarea_tab_data_t *) node->data;
      if (tab_data && tab_data->widget == widget)
	return tab_data;
      node = g_list_next(node);
    }
  return NULL;
}

static void
set_toolbox (GnomeUIInfo *info[])
{
  g_return_if_fail(info != NULL);

  if (toolbox) gtk_widget_destroy(toolbox);

  toolbox = gda_ui_new_toolbar_widget(GTK_ORIENTATION_VERTICAL,
				      GTK_TOOLBAR_BOTH,
				      info,
                                      NULL);
  gtk_table_attach(GTK_TABLE(work_area), toolbox, 3, 4, 0, 2,
		   GTK_FILL, GTK_FILL, 3, 3);
}

/*
 * Callbacks
 */
static void
notebook_switch_page_cb (GtkNotebook *nt, GtkNotebookPage *page, guint num, gpointer data)
{
  GtkWidget* widget;

  g_return_if_fail(GTK_IS_NOTEBOOK(nt));

  widget = gtk_notebook_get_nth_page(GTK_NOTEBOOK(nt), num);
  if (widget)
    {
      workarea_tab_data_t* tab_data;

      tab_data = (workarea_tab_data_t *) gtk_object_get_data(GTK_OBJECT(widget),
							     FE_WORKAREA_TAB_DATA);
      if (tab_data)
	{
	  set_toolbox(tab_data->toolbox);
	}
    }
}

/*
 * Public functions
 */
GtkWidget *
fe_workarea_create (void)
{
  if (!work_area)
    {
      work_area = gda_ui_new_table_widget(4, 2, FALSE);

      notebook = gda_ui_new_notebook_widget();
      gtk_signal_connect(GTK_OBJECT(notebook),
			 "switch_page",
			 GTK_SIGNAL_FUNC(notebook_switch_page_cb),
			 NULL);
      gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_BOTTOM);
      gtk_table_attach(GTK_TABLE(work_area), notebook, 0, 3, 0, 2,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       3, 3);
    }
  return work_area;
}

void
fe_workarea_add_tab (const gchar *title, GtkWidget *w, GnomeUIInfo *toolbar[])
{
  workarea_tab_data_t* data;

  data = g_new0(workarea_tab_data_t, 1);
  data->title = title ? g_strdup(title) : NULL;
  data->widget = w;
  data->toolbox = toolbar;

  gtk_widget_show(data->widget);
  gtk_object_set_data(GTK_OBJECT(data->widget), FE_WORKAREA_TAB_DATA, (gpointer) data);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
			   data->widget,
			   gtk_label_new(title));
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
			gtk_notebook_page_num(GTK_NOTEBOOK(notebook), data->widget));

  set_toolbox(data->toolbox);
  tabs = g_list_append(tabs, (gpointer) data);
}

void
fe_workarea_remove_tab (GtkWidget *widget)
{
  workarea_tab_data_t* tab_data;

  g_return_if_fail(GTK_IS_WIDGET(widget));

  tab_data = find_tab_by_widget(widget);
  if (tab_data)
    {
      gtk_notebook_remove_page(GTK_NOTEBOOK(notebook),
			       gtk_notebook_page_num(GTK_NOTEBOOK(notebook),
						     tab_data->widget));
      tabs = g_list_remove(tabs, (gpointer) tab_data);
      if (tab_data->title) g_free((gpointer) tab_data->title);
      g_free((gpointer) tab_data);

      /* update tool box */
      widget = fe_workarea_get_current_tab();
      if (widget)
	{
	  tab_data = find_tab_by_widget(widget);
	  if (tab_data)
	    {
	      set_toolbox(tab_data->toolbox);
	    }
	}
      else if (toolbox)
	{
	  gtk_widget_destroy(toolbox);
	  toolbox = NULL;
	}
    }
}

void
fe_workarea_activate_tab (GtkWidget *widget)
{
  workarea_tab_data_t* tab_data;

  g_return_if_fail(GTK_IS_WIDGET(widget));

  tab_data = find_tab_by_widget(widget);
  if (tab_data)
    {
      gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
			    gtk_notebook_page_num(GTK_NOTEBOOK(notebook),
						  widget));
    }
}

GtkWidget *
fe_workarea_get_current_tab (void)
{
  guint num = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
  return gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), num);
}

