/* GNOME DB Postgres Config Component
 * Copyright (C) 2000 Rodrigo Moya
 * Copyright (C) 2000 Vivien Malerba
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-postgres-config-users.h"

/*
 * Actual work of building the widget in the component. The info
 * struct will be filled at that time
 */
GtkWidget* create_user_list_config(PostgresPropsInfo *info);


/* to set/reset the contents of the widget */
static void users_list_start_reset(PostgresPropsInfo *info); 


/* 
 * Prototypes for the CB functions of that component
 */
static void users_list_clist_sel_cb(GtkWidget *wid, gint row, gint column, 
			     GdkEventButton *event, PostgresPropsInfo *info);
static void users_list_new_user_cb(GtkWidget *btn, PostgresPropsInfo *info);
static void users_list_modif_user_cb(GtkWidget *btn, PostgresPropsInfo *info);
static void users_list_del_user_cb(GtkWidget *btn, PostgresPropsInfo *info);
static void users_list_date_cb_cb(GtkWidget *btn, PostgresPropsInfo *info);
static void refresh_component(PostgresPropsInfo *info);


/*
 * Implementation 
 */
GtkWidget * create_users_list_widget_cb (void)
{
  GtkWidget *vb;
  GtkWidget *w;
  PostgresPropsInfo *info;

  vb = gtk_vbox_new(FALSE, GNOME_PAD);

  info = g_new0(PostgresPropsInfo, 1);
  info->ident = TRUE; /* default */
  gtk_object_set_data(GTK_OBJECT(vb), "propinfo", info);
  info->refresh_component = refresh_component;

  /* declaring the widget for updates */
  controls_list = g_slist_append(controls_list, info);

  w = create_user_id_config(info);
  gtk_widget_show(w);
  gtk_box_pack_start(GTK_BOX(vb), w, FALSE, FALSE, 0);
  info->ident_widget = w;

  w = create_user_list_config(info);
  gtk_widget_show(w);
  gtk_box_pack_start(GTK_BOX(vb), w, TRUE, TRUE, 0);

  g_print("New users list  widget: %p\n", vb);
  gtk_signal_connect(GTK_OBJECT(vb), "destroy",
		     GTK_SIGNAL_FUNC(component_widget_destroy_cb), NULL);
  return vb;
}

/*
 * Function to set the properties at the creation of the bonobo component
 */
void  users_list_new_instance_cb(GnomeDbControl *control,
				 GnomeDbControlInstance *control_instance,
				 gpointer data)
{
  PostgresPropsInfo *info;

  /* fetching the info */
  info = gtk_object_get_data(GTK_OBJECT(control_instance->widget), "propinfo");
  /* info is the control instance's data */
  gnome_db_control_set_instance_data(control_instance, info);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_DSN, _("DSN for the connection"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_IDENT_NAME, _("User identity"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_IDENT_PASS, _("User passord"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_IDENT_SHOW, _("Show/hide the user id"),
				GNOME_DB_CONTROL_ARG_BOOLEAN, NULL);
  gnome_db_control_add_property(control_instance,GNOME_DB_CONTROL_PROP_REFRESH, 
				_("Give it TRUE to reset/start the component"),
				GNOME_DB_CONTROL_ARG_BOOLEAN, NULL);
  gnome_db_control_add_property(control_instance,
				GNOME_DB_CONTROL_PROP_USER_NAME, _("User name setting"),
				GNOME_DB_CONTROL_ARG_STRING, NULL);
}


/* 
 * Actual creation of the widgets 
 */
GtkWidget* create_user_list_config(PostgresPropsInfo *info)
{
  GtkWidget *frame, *table, *vb, *wid, *clist, *bb, *sw, *hb;
  gchar *titles[]={N_("Name"), N_("Group(s)"), N_("Create DB"), 
		   N_("Create users"), N_("Valid until")};
  gchar *title[]={N_("Group(s)")};
  gint i;
  time_t now;

  frame = gtk_frame_new(_("Known database users"));
  
  vb = gtk_vbox_new(FALSE, GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(vb), GNOME_PAD);
  gtk_container_add(GTK_CONTAINER(frame), vb);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), 
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(vb), sw, TRUE, TRUE, 0);

  clist = gtk_clist_new_with_titles(5, titles);
  for (i=0; i<5; i++)
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), i, TRUE);
  gtk_container_add(GTK_CONTAINER(sw), clist);
  gtk_clist_set_sort_column(GTK_CLIST(clist), 0);
  gtk_clist_set_auto_sort(GTK_CLIST(clist), TRUE);
  gtk_signal_connect(GTK_OBJECT(clist), "select_row",
		     GTK_SIGNAL_FUNC(users_list_clist_sel_cb), info);
  info->u.users.clist = clist;
  gtk_object_set_data(GTK_OBJECT(info->u.users.clist), "selsysid", 
		      GINT_TO_POINTER(-1));

  table = gtk_table_new(5, 3, FALSE);
  gtk_box_pack_start(GTK_BOX(vb), table, FALSE, TRUE, 0);

  wid = gtk_label_new(_("Name"));
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 0, 1, 0, 1);
  wid = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 1, 2, 0, 1);
  info->u.users.name_entry = wid;

  wid = gtk_label_new(_("Password"));
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 0, 1, 1, 2);
  wid = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 1, 2, 1, 2);
  gtk_entry_set_visibility(GTK_ENTRY(wid), FALSE);
  info->u.users.password_entry = wid;

  wid = gtk_label_new(_("Create DB"));
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 0, 1, 2, 3);
  wid = gtk_check_button_new();
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 1, 2, 2, 3);
  info->u.users.createdb_cb = wid;

  wid = gtk_label_new(_("Create users"));
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 0, 1, 3, 4);
  wid = gtk_check_button_new();
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 1, 2, 3, 4);
  info->u.users.createusers_cb = wid;

  wid = gtk_label_new(_("Valid until"));
  gtk_table_attach_defaults(GTK_TABLE(table), wid, 0, 1, 4, 5);
  hb = gtk_hbox_new(FALSE, 0);
  wid = gtk_check_button_new();
  gtk_box_pack_start(GTK_BOX(hb), wid, FALSE, FALSE, 0);
  info->u.users.date_cb = wid;
  gtk_signal_connect(GTK_OBJECT(wid), "toggled",
		     GTK_SIGNAL_FUNC(users_list_date_cb_cb), info);
  now = time(NULL);
  wid = gnome_date_edit_new(now, FALSE, FALSE);
  gtk_widget_set_sensitive(wid, FALSE);
  gtk_box_pack_start(GTK_BOX(hb), wid, TRUE, TRUE, 0);
  gtk_table_attach_defaults(GTK_TABLE(table), hb, 1, 2, 4, 5);
  info->u.users.date = wid;

  gtk_table_set_col_spacing(GTK_TABLE(table), 1, GNOME_PAD);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), 
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_table_attach_defaults(GTK_TABLE(table), sw, 2, 3, 0, 5);

  wid = gtk_clist_new_with_titles(1, title);
  gtk_container_add(GTK_CONTAINER(sw), wid);
  gtk_clist_column_titles_passive(GTK_CLIST(wid));
  info->u.users.group_clist = wid;

  bb = gtk_hbutton_box_new();
  gtk_box_pack_start(GTK_BOX(vb), bb, FALSE, TRUE, 0);
  
  wid = gtk_button_new_with_label(_("Save as new user"));
  gtk_container_add(GTK_CONTAINER(bb), wid);
  gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		     GTK_SIGNAL_FUNC(users_list_new_user_cb), info);

  wid = gtk_button_new_with_label(_("Save modified user"));
  gtk_container_add(GTK_CONTAINER(bb), wid);
  gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		     GTK_SIGNAL_FUNC(users_list_modif_user_cb), info);

  wid = gtk_button_new_with_label(_("Delete user"));
  gtk_container_add(GTK_CONTAINER(bb), wid);
  gtk_signal_connect(GTK_OBJECT(wid), "clicked",
		     GTK_SIGNAL_FUNC(users_list_del_user_cb), info);

  gtk_widget_show_all(vb);
  /* other settings */
  info->ident = TRUE;
  info->running = FALSE;
  info->dsn = NULL;

  return frame;
}


/* 
 * Get properties replacement function 
 */
void users_list_get_property_cb(GnomeDbControl *control,
				GnomeDbControlInstance *control_instance,
				const gchar *name, 
				gpointer data)
{
  PostgresPropsInfo *info;
  gchar *str;
  gboolean done=FALSE;

  info = gnome_db_control_get_instance_data(control_instance);
  if (!strcmp(name, GNOME_DB_CONTROL_PROP_DSN))
    {
      if (info->dsn)
	gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_DSN, info->dsn);
      else
	gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_DSN, "");
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_NAME))
    {
      str = gtk_entry_get_text(GTK_ENTRY(info->user_name));
      gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_IDENT_NAME, str);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_PASS))
    {
      str = gtk_entry_get_text(GTK_ENTRY(info->user_passwd));
      gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_IDENT_PASS, str);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_SHOW))
    {
      gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_IDENT_SHOW, 
				    &(info->ident));
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_REFRESH))
    {
      gnome_db_control_set_property(control_instance, GNOME_DB_CONTROL_PROP_REFRESH, 
				    &(info->running));
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_USER_NAME))
    {
      gnome_db_control_set_property(control_instance, 
				    GNOME_DB_CONTROL_PROP_USER_NAME,
				    gtk_entry_get_text(GTK_ENTRY(info->u.users.name_entry)));
      done = TRUE;
    }

  if (!done)
    g_warning("Unknown property '%s'\n", name);
}

/* 
 * Set properties replacement function 
 */
void users_list_set_property_cb(GnomeDbControl *control,
				GnomeDbControlInstance *control_instance,
				const gchar *name,
				gconstpointer value,
				gpointer data)
{
  PostgresPropsInfo *info;
  gchar *str;
  gint sysid;
  PGresult *res;
  PGconn *conn;
  gboolean error=FALSE, done=FALSE;

  info = gnome_db_control_get_instance_data(control_instance);
  if (!strcmp(name, GNOME_DB_CONTROL_PROP_DSN))
    {
      if (info->dsn)
	g_free(info->dsn);
      info->dsn = g_strdup((gchar *) value);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_NAME))
    {
      gtk_entry_set_text(GTK_ENTRY(info->user_name), (gchar *) value);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_PASS))
    {
      gtk_entry_set_text(GTK_ENTRY(info->user_passwd), (gchar *) value);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_IDENT_SHOW))
    {
      info->ident = *((gboolean *) value);

      if (info->ident)
	gtk_widget_show(info->ident_widget);
      else
	gtk_widget_hide(info->ident_widget);
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_REFRESH))
    {
      if (*((gboolean *) value))
	{
	  refresh_component(info);
	  info->running = TRUE;
	}
      done = TRUE;
    }

  if (!done && !strcmp(name, GNOME_DB_CONTROL_PROP_USER_NAME))
    {
      conn = get_setup_conn(info);
      sysid = -1;
      if (conn)
	{
	  str = g_strdup_printf("SELECT usesysid FROM pg_user "
				"where usename = '%s'", (gchar *) value);
	  res = PQexec(conn, str);
	  g_free(str);
	  if (is_pgresult_valid(res))
	    {
	      if (PQntuples(res) > 0)
		sysid = atoi(PQgetvalue(res, 0, 0));
	      else
		error = TRUE;
	      PQclear(res);
	    }
	  else
	    error = TRUE;
	  PQfinish(conn);
	}
      else
	error = TRUE;
      
      if (!error && (sysid>-1))
	{
	  gint row;
	  row = gtk_clist_find_row_from_data(GTK_CLIST(info->u.users.clist),
					     GINT_TO_POINTER(sysid));
	  if (row >= 0)
	    gtk_clist_select_row(GTK_CLIST(info->u.users.clist), row, 0);
	}
      done = TRUE;
    }

  if (!done)
    g_warning("Unknown property '%s'\n", name);
}


/*
 * Refreshment function for the widget
 */
static void refresh_component(PostgresPropsInfo *info)
{
  users_list_start_reset(info);
}


static void users_list_start_reset(PostgresPropsInfo *info)
{
  PGresult *users;
  PGconn *conn;
  guint nb, i;
  gchar *text[5], datestr[30];
  struct tm *stm;
  gint row, sysid, tmpsysid;
  gpointer ptr;

  conn = get_setup_conn(info);
  if (conn)
    {
      PQexec(conn, "SET DATESTYLE TO 'ISO'");
      users = PQexec(conn, "SELECT usename, usecreatedb, "
		     "usesuper, valuntil, usesysid FROM pg_user");
      if (is_pgresult_valid(users))
	{
	  /* preserve the selection, if possible */
	  sysid = -1;
	  ptr = gtk_object_get_data(GTK_OBJECT(info->u.users.clist), 
				    "selsysid");
	  sysid = GPOINTER_TO_INT(ptr);
	  gtk_clist_freeze(GTK_CLIST(info->u.users.clist));
	  gtk_object_set_data(GTK_OBJECT(info->u.users.clist), "selsysid", 
			      GINT_TO_POINTER(-1));
	  gtk_clist_clear(GTK_CLIST(info->u.users.clist));
	  nb = PQntuples(users);
	  for (i=0; i<nb; i++)
	    {
	      text[0] = PQgetvalue(users, i, 0);
	      text[1] = "FIXME";
	      if (*PQgetvalue(users, i, 1) == 't')
		text[2] = _("Yes");
	      else
		text[2] = _("No");
	      if (*PQgetvalue(users, i, 2) == 't')
		text[3] = _("Yes");
	      else
		text[3] = _("No");
	      if (!PQgetisnull(users, i, 3))
		{
		  stm = str_to_tmstruct_timestamp(PQgetvalue(users, i, 3));
		  strftime(datestr, 29, "%x", stm);
		  text[4] = datestr;
		  g_free(stm);
		}
	      else
		text[4] = "";
	      row = gtk_clist_append(GTK_CLIST(info->u.users.clist), text);
	      tmpsysid = atoi(PQgetvalue(users, i, 4));
	      gtk_clist_set_row_data(GTK_CLIST(info->u.users.clist), row, 
				     GINT_TO_POINTER(tmpsysid));
	    }
	  gtk_clist_thaw(GTK_CLIST(info->u.users.clist));
	  PQclear(users);
	  /* restoring the selection, if possible */
	  row = gtk_clist_find_row_from_data(GTK_CLIST(info->u.users.clist),
					     GINT_TO_POINTER(sysid));
	  if (row >= 0)
	    gtk_clist_select_row(GTK_CLIST(info->u.users.clist), row, 0);
	}
      PQfinish(conn);
    }
}

void users_list_clist_sel_cb(GtkWidget *wid, gint row, gint column, 
			     GdkEventButton *event, PostgresPropsInfo *info)
{
  gchar *uname, *query;
  PGresult *user;
  PGconn *conn;
  struct tm *stm;
  time_t tmt;
  gint sysid;

  /* setting the selection */
  sysid = GPOINTER_TO_INT
    (gtk_clist_get_row_data(GTK_CLIST(info->u.users.clist), row));
  gtk_object_set_data(GTK_OBJECT(info->u.users.clist), "selsysid",
		      GINT_TO_POINTER(sysid));

  gtk_clist_get_text(GTK_CLIST(info->u.users.clist), row, 0, &uname);
  conn = get_setup_conn(info);
  if (conn)
    {
      query = g_strdup_printf("SELECT usecreatedb, usesuper, "
			      "valuntil FROM pg_user where usename = '%s'",
			      uname);
      PQexec(conn, "SET DATESTYLE TO 'ISO'");
      user = PQexec(conn, query);
      g_free(query);
      if (is_pgresult_valid(user))
	{
	  gtk_entry_set_text(GTK_ENTRY(info->u.users.name_entry), uname);
	  gtk_entry_set_text(GTK_ENTRY(info->u.users.password_entry), "");
	  /* FIXME: work with groups */
	  if (*PQgetvalue(user, 0, 0) == 't')
	    gtk_toggle_button_set_active
	      (GTK_TOGGLE_BUTTON(info->u.users.createdb_cb), TRUE);
	  else
	    gtk_toggle_button_set_active
	      (GTK_TOGGLE_BUTTON(info->u.users.createdb_cb), FALSE);

	  if (*PQgetvalue(user, 0, 1) == 't')
	    gtk_toggle_button_set_active
	      (GTK_TOGGLE_BUTTON(info->u.users.createusers_cb), TRUE);
	  else
	    gtk_toggle_button_set_active
	      (GTK_TOGGLE_BUTTON(info->u.users.createusers_cb), FALSE);
	  if (!PQgetisnull(user, 0, 2))
	    {
	      stm = str_to_tmstruct_timestamp(PQgetvalue(user, 0, 2));

	      tmt = mktime(stm);
	      g_free(stm);
	      gnome_date_edit_set_time(GNOME_DATE_EDIT(info->u.users.date),
				       tmt);
	      gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON(info->u.users.date_cb), TRUE);
	    }
	  else
	    {
	      gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON(info->u.users.date_cb), FALSE);
	    }
	  PQclear(user);
	}
      PQfinish(conn);
    }
}

void users_list_new_user_cb(GtkWidget *btn, PostgresPropsInfo *info)
{
  GString *query;
  gchar *uname;
  PGresult *user;
  PGconn *conn;

  conn = get_setup_conn(info);
  if (conn)
    {
      query = g_string_new("CREATE USER ");
      /* user name */
      uname =g_strdup(gtk_entry_get_text(GTK_ENTRY(info->u.users.name_entry)));
      uname = escape_chars(uname);
      g_string_sprintfa(query, "%s ", uname);
      g_free(uname);
      /* user password */
      uname =g_strdup
	(gtk_entry_get_text(GTK_ENTRY(info->u.users.password_entry)));
      if (*uname != 0)
	{
	  uname = escape_chars(uname);
	  g_string_sprintfa(query, "WITH PASSWORD %s ", uname);
	}
      g_free(uname);      

      if (GTK_TOGGLE_BUTTON(info->u.users.createdb_cb)->active)
	g_string_append(query, "CREATEDB ");
      else
	g_string_append(query, "NOCREATEDB ");
      if (GTK_TOGGLE_BUTTON(info->u.users.createusers_cb)->active)
	g_string_append(query, "CREATEUSER ");
      else
	g_string_append(query, "NOCREATEUSER ");
      /* FIXME: insert the group stuff */
      
      /* valid until information */
      if (GTK_TOGGLE_BUTTON(info->u.users.date_cb)->active)
	{
	  time_t tmt;
	  struct tm *stm;

	  tmt = gnome_date_edit_get_date(GNOME_DATE_EDIT(info->u.users.date));
	  stm = localtime(&tmt);
	  uname = g_strdup_printf("%d-%02d-%02d", stm->tm_year+1900, 
				  stm->tm_mon+1, stm->tm_mday);
	  g_string_sprintfa(query, "VALID UNTIL '%s'", uname);
	  g_free(uname);
	}
      g_print("QUERY: %s\n", query->str);
      user = PQexec(conn, query->str);
      g_string_free(query, TRUE);
      if (! is_pgresult_valid(user))
	{
	  g_print("%s\n", PQresultErrorMessage(user));
	}
      else
	{ /* reload the contents of the clist */
	  PQclear(user);
	  users_list_start_reset(info);
	}
      PQfinish(conn);
    }
  require_components_refresh(info);
}

void users_list_modif_user_cb(GtkWidget *btn, PostgresPropsInfo *info)
{
  gint sysid;
  gpointer ptr;
  GString *query;
  gchar *uname;
  PGresult *user, *res;
  PGconn *conn;
  gboolean error = FALSE;

  sysid = -1;
  ptr = gtk_object_get_data(GTK_OBJECT(info->u.users.clist), 
			    "selsysid");
  sysid = GPOINTER_TO_INT(ptr);
  if (sysid >= 0) /* otherwise don't do anything */
    {
      /* get the user name */
      conn = get_setup_conn(info);
      if (conn)
	{
	  uname = g_strdup_printf("SELECT usename FROM pg_user WHERE "
				  "usesysid = %d",sysid); 
	  user = PQexec(conn, uname);
	  g_free(uname);
	  uname = NULL;
	  if (is_pgresult_valid(user))
	    {
	      if (PQntuples(user)==1)
		{
		  uname = PQgetvalue(user, 0, 0);
		  if (strcmp(uname, gtk_entry_get_text
			     (GTK_ENTRY(info->u.users.name_entry))))
		    {
		      /* change the user name here */
		      gchar *newname;
		      newname = g_strdup(gtk_entry_get_text
					(GTK_ENTRY(info->u.users.name_entry)));
		      query = g_string_new("UPDATE pg_shadow set usename=");
		      newname = escape_chars(newname);
		      g_string_sprintfa(query, "'%s' WHERE usesysid=%d",
					newname, sysid);
		      g_print("ALTER USER name Query: %s\n", query->str);
		      res = PQexec(conn, query->str);
		      g_string_free(query, TRUE);
		      uname = newname;
		      if (!is_pgresult_valid(res))
			{
			  g_print("%s\n", PQresultErrorMessage(res));
			  error = TRUE;
			}
		      else
			PQclear(res);
		    }
		  else
		    uname = g_strdup(uname);
		  /* uname contains the user name (allocated mem) */
		}
	      else  /* user can't be found anymore! */
		error = TRUE;
	      PQclear(user);
	    }

	  if (error)
	    {
	      if (uname) g_free(uname);
	    }
	  else
	    {
	      /* let's continue to update the `uname` user */
	      query = g_string_new("ALTER USER ");
	      g_string_sprintfa(query, "%s ", uname);
	      g_free(uname);
	      if (*gtk_entry_get_text(GTK_ENTRY(info->u.users.password_entry))
		  != 0)
		g_string_sprintfa(query, "WITH PASSWORD %s ", 
				  gtk_entry_get_text
				  (GTK_ENTRY(info->u.users.password_entry)));
	      if (GTK_TOGGLE_BUTTON(info->u.users.createdb_cb)->active)
		g_string_append(query, "CREATEDB ");
	      else
		g_string_append(query, "NOCREATEDB ");
	      if (GTK_TOGGLE_BUTTON(info->u.users.createusers_cb)->active)
		g_string_append(query, "CREATEUSER ");
	      else
		g_string_append(query, "NOCREATEUSER ");
	      if (GTK_TOGGLE_BUTTON(info->u.users.date_cb)->active)
		{
		  time_t tmt;
		  struct tm *stm;
		  gchar *uname;

		  tmt = gnome_date_edit_get_date
		    (GNOME_DATE_EDIT(info->u.users.date));
		  stm = localtime(&tmt);
		  uname = g_strdup_printf("%d-%02d-%02d", stm->tm_year+1900, 
					  stm->tm_mon+1, stm->tm_mday);
		  g_string_sprintfa(query, "VALID UNTIL '%s'", uname);
		  g_free(uname);		  
		}
	      else
		{ /* if valid until was set and is not anymore */
		  /* FIXME: find the godd query for this */
		}
	      res = PQexec(conn, query->str);
	      if (!is_pgresult_valid(user))
		{
		  g_print("%s\n", PQresultErrorMessage(user));
		}
	      else
		{ /* reload the contents of the clist */
		  users_list_start_reset(info);
		}		  	      
	      g_string_free(query, TRUE);
	    }
	  PQfinish(conn);
	}
    }
  require_components_refresh(info);
}

void users_list_del_user_cb(GtkWidget *btn, PostgresPropsInfo *info)
{
  gint sysid;
  gpointer ptr;
  gchar *uname, *query;
  PGresult *user;
  PGconn *conn;

  sysid = -1;
  ptr = gtk_object_get_data(GTK_OBJECT(info->u.users.clist), 
			    "selsysid");
  sysid = GPOINTER_TO_INT(ptr);
  if (sysid >= 0) /* otherwise don't do anything */
    {
      /* get the user name */
      conn = get_setup_conn(info);
      if (conn)
	{
	  uname = g_strdup_printf("SELECT usename FROM pg_user WHERE "
				  "usesysid = %d",sysid); 
	  user = PQexec(conn, uname);
	  g_free(uname);
	  if (is_pgresult_valid(user))
	    {
	      if (PQntuples(user)==1)
		{
		  uname = g_strdup(PQgetvalue(user, 0, 0));
		  PQclear(user);
		  /* delete that user now */
		  query = g_strdup_printf("DROP USER %s", uname);
		  g_free(uname);
		  g_print("DELETE USER query is: %s\n", query);
		  user = PQexec(conn, query);
		  if (!is_pgresult_valid(user))
		    {
		      g_print("%s\n", PQresultErrorMessage(user));
		    }
		  else
		    { /* reload the contents of the clist */
		      users_list_start_reset(info);
		    }		  
		  g_free(query);
		}
	      else
		{ /* user can't be found anymore! */
		  PQclear(user);
		  PQfinish(conn);
		  return;
		}
	    }
	  PQfinish(conn);
	}
    }
  require_components_refresh(info);
}

void users_list_date_cb_cb(GtkWidget *btn, PostgresPropsInfo *info)
{
  gtk_widget_set_sensitive(info->u.users.date, 
			   GTK_TOGGLE_BUTTON(info->u.users.date_cb)->active);
}
