/* GNOME DB libary
 * Copyright (C) 2000 Vivien Malerba
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome.h>
#include <bonobo.h>
#include <libgnorba/gnorba.h>
#include <gda-ui.h>
#include <gnome-db-control-widget.h>

static GtkWidget *nb = NULL; /* main notebook for components */
GSList *controls = NULL; /* a hold on all the opened controls */

/* CB to handle the closing of the properties dialog */
static void props_dlg_clicked_cb(GnomeDialog *dialog, gint button_number,
				 GtkWidget *bonobo_control)
{
  GSList *list;

  switch(button_number)
    {
    case 1: /* close */
      gtk_object_set_data(GTK_OBJECT(bonobo_control), "propsdlg", NULL);
      /* FIXME: free all the "prop_name" values that was allocated for each of 
	 the buttons */
      gnome_dialog_close(dialog);
      break;
    case 0: /* refresh */
      list = gtk_object_get_data(GTK_OBJECT(bonobo_control), "getbtns");
      while (list)
	{
	  gtk_button_clicked(GTK_BUTTON(list->data));
	  list = g_slist_next(list);
	}
      break;
    }
}

static void set_property_cb(GtkWidget *wid, BonoboPropertyBagClient *prop)
{
  gchar *pname;
  GtkWidget *pwid;
  BonoboArgType tc;

  pname = gtk_object_get_data(GTK_OBJECT(wid), "prop_name");
  pwid = gtk_object_get_data(GTK_OBJECT(wid), "prop_wid");
  if (GTK_IS_TOGGLE_BUTTON(pwid))
    {
      bonobo_property_bag_client_set_value_gboolean(prop, pname,
						    GTK_TOGGLE_BUTTON(pwid)->active);
    }
  if (GTK_IS_ENTRY(pwid))
    {
      gchar *textval;
      textval = gtk_entry_get_text(GTK_ENTRY(pwid));
      tc = bonobo_property_bag_client_get_property_type(prop, pname);
      switch(tc->kind)
	{
	case CORBA_tk_string :
	  bonobo_property_bag_client_set_value_string(prop, pname, textval);
	  break;
	case CORBA_tk_short :
	  bonobo_property_bag_client_set_value_gint(prop, pname, 
						    atoi(textval));
	  break;
	case CORBA_tk_double :
	  bonobo_property_bag_client_set_value_gdouble(prop, pname,
						      atof(textval));
	  break;
	case CORBA_tk_float :
	  bonobo_property_bag_client_set_value_gfloat(prop, pname,
						      atof(textval));
	  break;
	case CORBA_tk_long :
	  bonobo_property_bag_client_set_value_glong(prop, pname, 
						     atol(textval));
	  break;
	default :
	  g_warning("Unhandled type %d", tc->kind);
	}
    }
}

static void write_prop_in_widget(GtkWidget *pwid, 
				 BonoboPropertyBagClient *prop, gchar *pname)
{
  BonoboArgType tc;

  if (GTK_IS_TOGGLE_BUTTON(pwid))
    {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pwid),
	    bonobo_property_bag_client_get_value_gboolean(prop, pname));
    }
  if (GTK_IS_ENTRY(pwid))
    {
      gchar *textval=NULL;
      tc = bonobo_property_bag_client_get_property_type(prop, pname);
      switch(tc->kind)
	{
	case CORBA_tk_string :
	  textval = g_strdup
	    (bonobo_property_bag_client_get_value_string(prop, pname));
	  break;
	case CORBA_tk_short :
	  textval = g_strdup_printf("%d",
		 bonobo_property_bag_client_get_value_gint(prop, pname)); 
	  break;
	case CORBA_tk_double :
	  textval = g_strdup_printf("%f",
		 bonobo_property_bag_client_get_value_gdouble(prop, pname)); 
	  break;
	case CORBA_tk_float :
	  textval = g_strdup_printf("%f",
		 bonobo_property_bag_client_get_value_gfloat(prop, pname)); 
	  break;
	case CORBA_tk_long :
	  textval = g_strdup_printf("%f",
		 bonobo_property_bag_client_get_value_glong(prop, pname)); 
	  break;
	default :
	  g_warning("Unhandled type %d", tc->kind);
	}
      gtk_entry_set_text(GTK_ENTRY(pwid), textval);
      g_free(textval);
    }  
}

static void get_property_cb(GtkWidget *wid, BonoboPropertyBagClient *prop)
{
  gchar *pname;
  GtkWidget *pwid;

  pname = gtk_object_get_data(GTK_OBJECT(wid), "prop_name");
  pwid = gtk_object_get_data(GTK_OBJECT(wid), "prop_wid");
  write_prop_in_widget(pwid, prop, pname);
}

/* displays a dialog with all the properties */
static void props_cb(GtkWidget *button, gpointer data)
{
  GList *list, *ptr;
  gint page, nbre, i;
  GtkWidget *control_widget;
  BonoboPropertyBagClient *prop;
  GtkWidget *sw, *dlg, *table=NULL;
  gchar *str, *goadid;
  GSList *getbtns;

  page = gtk_notebook_get_current_page(GTK_NOTEBOOK(nb));
  control_widget = gtk_notebook_get_nth_page(GTK_NOTEBOOK(nb), page);
  if (control_widget)
    {
      if (!gtk_object_get_data(GTK_OBJECT(control_widget), "propsdlg"))
	{
	  prop = GNOME_DB_CONTROL_WIDGET(control_widget)->prop;
	  
	  list = bonobo_property_bag_client_get_property_names(prop);
	  nbre = g_list_length(list);
	  
	  ptr = list;
	  if (! ptr)
	    {
	      gnome_ok_dialog("No property set for this component!\n");
	      return;
	    }
	  
	  /* table widget */
	  table = gtk_table_new(nbre, 4, FALSE);
	  gtk_table_set_col_spacing(GTK_TABLE(table), 0, GNOME_PAD);
	  i = 0;
	  getbtns = NULL;
	  while (ptr)
	    {
	      gchar *name = (gchar *)ptr->data;
	      GtkWidget *wid, *entry=NULL;
	      BonoboArgType tc;
	      gboolean bool_val;

	      g_print("Property: %s is ", name);
	      tc = bonobo_property_bag_client_get_property_type(prop, name);

	      /* widgets */
	      wid = gtk_label_new(name);
	      gtk_table_attach_defaults(GTK_TABLE(table), wid, 0, 1, i, i+1);
	      /* the entry depend on the property type */
	      switch (tc->kind)
		{
		case CORBA_tk_boolean :
		  entry = gtk_check_button_new();
		  g_print("Boolean\n");
		  bool_val = 
		    bonobo_property_bag_client_get_value_gboolean(prop, name);
		  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(entry), 
					       bool_val);
		  break;
		case CORBA_tk_string :
		  entry = gtk_entry_new();
		  g_print("String\n");
		  break;
		case CORBA_tk_short :
		  entry = gtk_entry_new();
		  g_print("Short\n");
		  break;
		case CORBA_tk_double :
		case CORBA_tk_float :
		  entry = gtk_entry_new();
		  g_print("Float\n");
		  break;
		case CORBA_tk_long :
		  g_print("Long\n");
		  break;
		default :
		  entry = gtk_entry_new();
		  g_print("Unknown\n");
		  g_warning("Unhandled type %d", tc->kind);
		}
	      gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, i, i+1);
	      write_prop_in_widget(entry, prop, name);

	      /* get val button */
	      wid = gtk_button_new_with_label("Get value");
	      gtk_object_set_data(GTK_OBJECT(wid), "prop_name", 
				  g_strdup(name));
	      gtk_object_set_data(GTK_OBJECT(wid), "prop_wid", entry);
	      gtk_signal_connect(GTK_OBJECT(wid), "clicked",
				 GTK_SIGNAL_FUNC(get_property_cb), prop);
	      gtk_table_attach(GTK_TABLE(table), wid, 2, 3, i, i+1, 0, 0, 
			       GNOME_PAD/2, GNOME_PAD/2);
	      getbtns = g_slist_append(getbtns, wid);

	      /* set val button */
	      wid = gtk_button_new_with_label("Set value");
	      gtk_object_set_data(GTK_OBJECT(wid), "prop_name", 
				  g_strdup(name));
	      gtk_object_set_data(GTK_OBJECT(wid), "prop_wid", entry);
	      gtk_signal_connect(GTK_OBJECT(wid), "clicked",
				 GTK_SIGNAL_FUNC(set_property_cb), prop);
	      gtk_table_attach(GTK_TABLE(table), wid, 3, 4, i, i+1, 0, 0,
			       GNOME_PAD/2, GNOME_PAD/2);
	      ptr = g_list_next(ptr);
	      i++;
	    }
	  
	  /* freeing the list */
	  while (list)
	    {
	      g_free(list->data);
	      list = g_list_remove_link(list, list);
	    }
	  
	  /* display a dlg with the table inside */
	  sw = gtk_scrolled_window_new(NULL, NULL);
	  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
					 GTK_POLICY_AUTOMATIC, 
					 GTK_POLICY_AUTOMATIC);
	  gtk_container_set_border_width(GTK_CONTAINER(sw), GNOME_PAD/2);
	  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sw), 
						table);
	  goadid = GNOME_DB_CONTROL_WIDGET(control_widget)->goad_id;
	  str = g_strdup_printf("Properties for %s", goadid);
	  dlg = gnome_dialog_new(str, _("Refresh all"), _("Ok"), NULL);
	  gtk_widget_set_usize(GTK_WIDGET(dlg), 450, 300);
	  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dlg)->vbox), sw, TRUE, 
			     TRUE, 0);
	  gtk_signal_connect(GTK_OBJECT(dlg), "clicked",
			     GTK_SIGNAL_FUNC(props_dlg_clicked_cb), 
			     control_widget);
	  gtk_widget_show_all(dlg);
	  gtk_object_set_data(GTK_OBJECT(control_widget), "propsdlg", dlg);
	  gtk_object_set_data(GTK_OBJECT(control_widget), "getbtns", getbtns);
	}
      else
	{
	  gdk_window_raise(GTK_WIDGET(gtk_object_get_data
		 (GTK_OBJECT(control_widget), "propsdlg"))->window);
	}
    }
}

static void close_control(GtkWidget *control_widget, gboolean info_in_dlg)
{
  if (GNOME_DB_CONTROL_WIDGET(control_widget)->bonobo_widget)
    {
      GnomeDialog *dlg;

      /* properties dialog */
      dlg = gtk_object_get_data(GTK_OBJECT(control_widget), "propsdlg");
      if (dlg)
	gnome_dialog_close(dlg);

      /* destroy the widget */
      gtk_object_destroy(GTK_OBJECT(control_widget));

      controls = g_slist_remove(controls, control_widget);
      if (info_in_dlg)
	gnome_ok_dialog("Bonobo control destroyed!\n");
    }
}

static void exit_cb(GtkWidget *wid, gpointer data)
{
  while(controls)
    {
      g_print("Destroying bonobo control %s\n", 
	      GNOME_DB_CONTROL_WIDGET(controls->data)->goad_id);
      close_control(GTK_WIDGET(controls->data), FALSE);
      controls = g_slist_remove_link(controls, controls);
    }
  gtk_main_quit();
}


static void close_clicked(GtkWidget *button, BonoboPropertyBagClient *prop)
{
  gint page;
  GtkWidget *control_widget;

  page = gtk_notebook_get_current_page(GTK_NOTEBOOK(nb));
  control_widget = gtk_notebook_get_nth_page(GTK_NOTEBOOK(nb), page);
  if (control_widget)
    {
      close_control(control_widget, TRUE);
      gtk_notebook_remove_page(GTK_NOTEBOOK(nb), page);
    }
}

static void reset_cb(GtkWidget *button, BonoboPropertyBagClient *prop)
{
  gint page;
  GtkWidget *control_widget;
  gchar *str;

  page = gtk_notebook_get_current_page(GTK_NOTEBOOK(nb));
  control_widget = gtk_notebook_get_nth_page(GTK_NOTEBOOK(nb), page);

  if (control_widget)
    {
      gnome_db_control_widget_set_prop(GNOME_DB_CONTROL_WIDGET(control_widget),
				       GNOME_DB_CONTROL_PROP_START, TRUE);
    }
}


void display_bonobo_widget(GtkObject *obj, gpointer unused_data)
{
  gchar *goadid;
  const gchar* ids[] = { "IDL:BonoboControl/gnome-db-config:1.0", NULL };
  GtkWidget *control_widget;
  GtkWidget *label;

  goadid = gnome_bonobo_select_goad_id(_("Select config component"), ids);
  if (goadid)
    {
      control_widget = gnome_db_control_widget_new(goadid);
  
      if (!GNOME_DB_CONTROL_WIDGET(control_widget)->bonobo_widget)
	{
	  gchar *str;
	  str = g_strdup_printf("Could not open bonobo control:\n%s", goadid);
	  gnome_error_dialog(str);
	  g_free(str);
	}
      else
	{
	  label = gtk_label_new(goadid);
	  gtk_notebook_append_page(GTK_NOTEBOOK(nb), control_widget, label);
	  gtk_notebook_set_page(GTK_NOTEBOOK(nb), 
				g_list_length(GTK_NOTEBOOK(nb)->children));
	  gtk_widget_show(control_widget);
	  controls = g_slist_append(controls, control_widget);
	}  
    }
}



void about_cb (GtkWidget *w, gpointer data)
{
  gda_ui_show_about_dialog(_("GNOME-DB Controls browser"));
}

void init_menus(GnomeApp *app)
{
    /* file menu */
  GnomeUIInfo file_menu []= {
    GNOMEUIINFO_MENU_EXIT_ITEM(exit_cb, NULL),
    GNOMEUIINFO_END
  };

  /* about menu */
  GnomeUIInfo helpmenu[] = {
    GNOMEUIINFO_MENU_ABOUT_ITEM(about_cb, NULL),
    GNOMEUIINFO_END
  };
  
  /* shell for the components menu */
  GnomeUIInfo comps_menu []= {
    {
      GNOME_APP_UI_ITEM, N_("New control"), NULL, 
      display_bonobo_widget,
      NULL, NULL, 
      GNOME_APP_PIXMAP_NONE, NULL,
      0, 0, NULL
    },
    {
      GNOME_APP_UI_ITEM, N_("Close control"), NULL, 
      close_clicked,
      NULL, NULL, 
      GNOME_APP_PIXMAP_NONE, NULL,
      0, 0, NULL
    },
    GNOMEUIINFO_END
  };


  /* ALL menus */
  GnomeUIInfo main_menu []= {
    GNOMEUIINFO_MENU_FILE_TREE(file_menu),
    GNOMEUIINFO_SUBTREE(N_("Controls"), &comps_menu),
    GNOMEUIINFO_MENU_HELP_TREE(helpmenu),
    GNOMEUIINFO_END
  };
  gnome_app_create_menus(app, main_menu);
}

int main(int argc, char *argv[])
{
  GtkWidget *app, *vbox, *bb, *button, *frame;
  CORBA_Environment ev;
  CORBA_ORB orb;

  CORBA_exception_init(&ev);
  
  gnome_CORBA_init("GNOME-DB Components Browser", "1.0", &argc, argv, 0, &ev);
  orb = gnome_CORBA_ORB();
  if (bonobo_init(orb, NULL, NULL) == FALSE)
    g_error("Unable to init bonobo");
  
  app = gnome_app_new("Bonobo:Clock", NULL);
  init_menus(GNOME_APP(app));
  gtk_widget_set_usize(app, 500, 550);

  nb = gtk_notebook_new();
  gtk_notebook_set_scrollable(GTK_NOTEBOOK(nb), TRUE);

  vbox = gtk_vbox_new(FALSE, GNOME_PAD);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), GNOME_PAD);
  gtk_box_pack_start(GTK_BOX(vbox), nb, TRUE, TRUE, 1);
  
  frame = gtk_frame_new(_("Control settings"));
  gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, TRUE, 1);

  bb = gtk_hbutton_box_new();
  gtk_container_set_border_width(GTK_CONTAINER(bb), GNOME_PAD/2);
  gtk_container_add(GTK_CONTAINER(frame), bb);  

  button = gtk_button_new_with_label("Properties");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(props_cb), NULL);
  gtk_container_add(GTK_CONTAINER(bb), button);

  button = gtk_button_new_with_label("Start/Reset");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(reset_cb), NULL);
  gtk_container_add(GTK_CONTAINER(bb), button);
  
  button = gtk_button_new_with_label("Close");
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(close_clicked), NULL);
  gtk_container_add(GTK_CONTAINER(bb), button);
  
  gnome_app_set_contents(GNOME_APP(app), vbox);
  gtk_widget_show_all(app);
  
  bonobo_main();
  return 0;
}
