/* GNOME DB components libary
 * Copyright (C) 2000 Rodrigo Moya
 * Copyright (C) 2000 Vivien Malerba
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome-db.h>
#include <GnomeDB.h>

/*
 * GnomeDbControlWidget object signals
 */
enum
{
  GNOME_DB_CONTROL_WIDGET_LAST_SIGNAL
};

static gint gnome_db_control_widget_signals[GNOME_DB_CONTROL_WIDGET_LAST_SIGNAL] = { 0, };

/*
 * Callbacks
 */
void
focus_changed_cb (GtkWidget *w, GdkEventAny *event, GnomeDbControlWidget *wid)
{
  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid));
  
  if (GTK_WIDGET_HAS_FOCUS(wid->bonobo_widget)) gnome_db_control_widget_activate(wid);
  else gnome_db_control_widget_deactivate(wid);
}

/*
 * GnomeDbControlWidget object interface
 */
static void
gnome_db_control_widget_class_init (GnomeDbControlWidgetClass *klass)
{
  GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);
}

static void
gnome_db_control_widget_init (GnomeDbControlWidget *widget)
{
  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(widget));

  widget->bonobo_widget = NULL;
  widget->gnomedb_control = CORBA_OBJECT_NIL;
}

GtkType
gnome_db_control_widget_get_type (void)
{
  static GtkType db_control_widget_type = 0;

  if (!db_control_widget_type)
    {
      GtkTypeInfo db_control_widget_info =
      {
        "GnomeDbControlWidget",
        sizeof (GnomeDbControlWidget),
        sizeof (GnomeDbControlWidgetClass),
        (GtkClassInitFunc) gnome_db_control_widget_class_init,
        (GtkObjectInitFunc) gnome_db_control_widget_init,
        (GtkArgSetFunc) NULL,
        (GtkArgSetFunc) NULL
      };
      db_control_widget_type = gtk_type_unique(gtk_vbox_get_type(),
					                           &db_control_widget_info);
    }
  return db_control_widget_type;
}

/**
 * gnome_db_control_widget_new
 */
GtkWidget *
gnome_db_control_widget_new (const gchar *id, Bonobo_UIContainer uic)
{
  GnomeDbControlWidget* widget;

  g_return_val_if_fail(id != NULL, NULL);

  widget = GNOME_DB_CONTROL_WIDGET(gtk_type_new(gnome_db_control_widget_get_type()));
  widget->goad_id = g_strdup(id);

  /* create the BonoboWidget */
  widget->bonobo_widget = BONOBO_WIDGET(bonobo_widget_new_control(id, uic));
  if (BONOBO_IS_WIDGET(widget->bonobo_widget))
    {
      BonoboControlFrame *cf;

      /* make the control merge */
      gtk_signal_connect(GTK_OBJECT(widget->bonobo_widget),
                         "focus_in_event",
                         GTK_SIGNAL_FUNC(focus_changed_cb),
                         widget);
      gtk_signal_connect(GTK_OBJECT(widget->bonobo_widget),
                         "focus_out_event",
                         GTK_SIGNAL_FUNC(focus_changed_cb),
                         widget);
      gnome_db_control_widget_activate(widget);
      
      gtk_widget_show(GTK_WIDGET(widget->bonobo_widget));
      gtk_box_pack_start(GTK_BOX(widget), GTK_WIDGET(widget->bonobo_widget), 1, 1, 0);
      
      /* setting the property bag here */
      cf = bonobo_widget_get_control_frame(BONOBO_WIDGET(widget->bonobo_widget));
      widget->prop = bonobo_control_frame_get_control_property_bag(cf, NULL);
    }
  else 
    {
      g_warning("Could not load %s", id);
      widget->bonobo_widget = NULL;
      widget->prop = NULL;
    }
  return GTK_WIDGET(widget);
}

static void
gnome_db_control_widget_destroy (GnomeDbControlWidget *widget)
{
  BonoboControlFrame* frame;

  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(widget));
  g_return_if_fail(BONOBO_IS_WIDGET(widget->bonobo_widget));

  g_free(widget->goad_id);
  frame = bonobo_widget_get_control_frame(BONOBO_WIDGET(widget->bonobo_widget));
  /* FIXME: is that the correct way to get rid of a bonobo control? */
  bonobo_control_frame_control_deactivate(frame);
}

gboolean
gnome_db_control_widget_supported (const gchar *goad_id)
{
  const gchar* ids[] = { "IDL:BonoboControl/gnome-db-config:1.0", NULL };
  GList*       servers, *list;
  gboolean     found = FALSE;

  servers = bonobo_directory_get_server_list(ids);
  list = servers;
  while (!found && list)
    {
      if (!strcmp(bonobo_directory_get_server_info_id(list->data), goad_id))
        found = TRUE;
      list = g_list_next(list);
    }
  bonobo_directory_free_server_list(servers);
  return found;
}

/**
 * gnome_db_control_widget_is_alive
 */
gboolean
gnome_db_control_widget_is_alive (GnomeDbControlWidget *wid)
{
  BonoboControlFrame *control_frame;
  CORBA_Object corba_object;
  CORBA_Environment ev;
  gboolean is_dead;

  g_return_val_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid), FALSE);

  control_frame = bonobo_widget_get_control_frame(wid->bonobo_widget);
  corba_object = bonobo_control_frame_get_control(control_frame);

  CORBA_exception_init(&ev);
  is_dead = CORBA_Object_non_existent(corba_object, &ev);
  CORBA_exception_free(&ev);

  return is_dead;
}

/**
 * gnome_db_control_widget_activate
 */
void
gnome_db_control_widget_activate (GnomeDbControlWidget *wid)
{
  BonoboControlFrame* control_frame;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid));
  
  control_frame = bonobo_widget_get_control_frame(wid->bonobo_widget);
  bonobo_control_frame_set_autoactivate(control_frame, FALSE);
  bonobo_control_frame_control_activate(control_frame);
}

/**
 * gnome_db_control_widget_deactivate
 */
void
gnome_db_control_widget_deactivate (GnomeDbControlWidget *wid)
{
  BonoboControlFrame* control_frame;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid));
  
  control_frame = bonobo_widget_get_control_frame(wid->bonobo_widget);
  bonobo_control_frame_set_autoactivate(control_frame, FALSE);
  bonobo_control_frame_control_deactivate(control_frame);
}

/* 
 * Functions to manipulate controls from the properties.
 */
gboolean 
gnome_db_control_widget_prop_supported (GnomeDbControlWidget *wid, 
                                        const gchar *prop_name)
{
  if (bonobo_property_bag_client_get_property(wid->prop, prop_name, NULL) != 
      CORBA_OBJECT_NIL)
    return TRUE;
  else
    return FALSE;
}

void
gnome_db_control_widget_set_prop_ptr (GnomeDbControlWidget *wid, 
                                      const gchar *prop_name,
                                      gconstpointer value)
{
  CORBA_TypeCode ptype;

  ptype = bonobo_property_bag_client_get_property_type(wid->prop, prop_name, NULL);
  switch (ptype->kind)
    {
    case CORBA_tk_string:
      bonobo_property_bag_client_set_value_string(wid->prop, prop_name,
						  (gchar *) value, NULL);
      break;
    case CORBA_tk_boolean:
      bonobo_property_bag_client_set_value_gboolean(wid->prop, prop_name,
						    *((gboolean*) value), NULL);
      break;
    case CORBA_tk_float:
      bonobo_property_bag_client_set_value_gfloat(wid->prop, prop_name,
						  *((gfloat *) value), NULL);
      break;
    case CORBA_tk_double:
      bonobo_property_bag_client_set_value_gdouble(wid->prop, prop_name,
						  *((gdouble *) value), NULL);
      break;
    case CORBA_tk_long:
      bonobo_property_bag_client_set_value_glong(wid->prop, prop_name,
						 *((glong *) value), NULL);
      break;
    default:
      g_warning("Unsupported type of property");
    }
}

void
gnome_db_control_widget_set_prop (GnomeDbControlWidget *wid, 
                                  const gchar *prop_name, 
                                  ...)
{
  CORBA_TypeCode ptype;
  va_list pa;

  gchar *mstr;
  gboolean mbool;
  gfloat mfloat;
  gdouble mdouble;
  glong mlong;

  ptype = bonobo_property_bag_client_get_property_type(wid->prop, prop_name, NULL);
  va_start(pa, prop_name);
  switch (ptype->kind)
    {
    case CORBA_tk_string:
      mstr = va_arg(pa, gchar *);
      bonobo_property_bag_client_set_value_string(wid->prop, prop_name,
                                                  mstr, NULL);
      break;
    case CORBA_tk_boolean:
      mbool = va_arg(pa, gboolean);
      bonobo_property_bag_client_set_value_gboolean(wid->prop, prop_name,
                                                    mbool, NULL);
      break;
    case CORBA_tk_float:
      mfloat = va_arg(pa, gdouble);
      bonobo_property_bag_client_set_value_gfloat(wid->prop, prop_name,
						  mfloat, NULL);
      break;
    case CORBA_tk_double:
      mdouble = va_arg(pa, gdouble);
      bonobo_property_bag_client_set_value_gdouble(wid->prop, prop_name,
						  mdouble, NULL);
      break;
    case CORBA_tk_long:
      mlong = va_arg(pa, glong);
      bonobo_property_bag_client_set_value_glong(wid->prop, prop_name,
						 mlong, NULL);
      break;
    default:
      g_warning("Unsupported type of property");
    }
  va_end(pa);
}

void
gnome_db_control_widget_get_prop (GnomeDbControlWidget *wid,
                                  const gchar *prop_name,
                                  gpointer *value)
{
  CORBA_TypeCode ptype;

  ptype = bonobo_property_bag_client_get_property_type(wid->prop, prop_name, NULL);
  switch (ptype->kind)
    {
    case CORBA_tk_string:
      *((gchar **) value) = 
	bonobo_property_bag_client_get_value_string(wid->prop, prop_name, NULL);
      break;
    case CORBA_tk_boolean:
      *((gboolean *) value) = 
	bonobo_property_bag_client_get_value_gboolean(wid->prop, prop_name, NULL);
      break;
    case CORBA_tk_float:
      *((gfloat *) value) = 
	bonobo_property_bag_client_get_value_gfloat(wid->prop, prop_name, NULL);
      break;
    case CORBA_tk_double:
      *((gdouble *) value) = 
	bonobo_property_bag_client_get_value_gdouble(wid->prop, prop_name, NULL);
      break;
    case CORBA_tk_long:
      *((glong *) value) = 
	bonobo_property_bag_client_get_value_glong(wid->prop, prop_name, NULL);
      break;
    default:
      g_warning("Unsupported type of property");
    }
}

/**
 * gnome_db_control_widget_get_gnomedb_control
 *
 */
static CORBA_Object
gnome_db_control_widget_get_gnomedb_control (GnomeDbControlWidget *wid)
{
  CORBA_Environment ev;
  
  g_return_val_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid), CORBA_OBJECT_NIL);
  
  CORBA_exception_init(&ev);
  if (CORBA_Object_is_nil(wid->gnomedb_control, &ev))
    {
      BonoboObjectClient* object_client;
      
      object_client = bonobo_widget_get_server(wid->bonobo_widget);
      //wid->gnomedb_control = (GNOME_DB_Control) bonobo_object_client_query_interface(object_client,
      //                                                                              GNOME_DB_REPO_CONTROL,
      //                                                                              NULL);
      wid->gnomedb_control = BONOBO_OBJECT(object_client)->corba_objref;
    }
  CORBA_exception_free(&ev);
  return wid->gnomedb_control;
}

/**
 * gnome_db_control_widget_get_version
 * @wid: control widget
 *
 * Return the version supported by the underlying control.
 */
gchar *
gnome_db_control_widget_get_version (GnomeDbControlWidget *wid)
{
  GNOME_DB_Control gnomedb_control;
  
  g_return_val_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid), NULL);
  g_return_val_if_fail(BONOBO_IS_WIDGET(wid->bonobo_widget), NULL);
  
  gnomedb_control = gnome_db_control_widget_get_gnomedb_control(wid);
  if (gnomedb_control != CORBA_OBJECT_NIL)
    {
      CORBA_Environment ev;
      CORBA_char*       retval;
      gchar*            version = NULL;
      
      CORBA_exception_init(&ev);
      retval = GNOME_DB_Control_getVersion(gnomedb_control, &ev);
      if (ev._major == CORBA_NO_EXCEPTION) version = g_strdup(retval);
      CORBA_exception_free(&ev);
      return version;
    }
  
  return NULL;
}

/**
 * gnome_db_control_instance_get_authors
 */
gchar *
gnome_db_control_instance_get_authors (GnomeDbControlWidget *wid)
{
  GNOME_DB_Control gnomedb_control;
  
  g_return_val_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid), NULL);
  g_return_val_if_fail(BONOBO_IS_WIDGET(wid->bonobo_widget), NULL);
  
  gnomedb_control = gnome_db_control_widget_get_gnomedb_control(wid);
  if (gnomedb_control != CORBA_OBJECT_NIL)
    {
      CORBA_Environment ev;
      CORBA_char*       retval;
      gchar*            version = NULL;
      
      CORBA_exception_init(&ev);
      retval = GNOME_DB_Control_getAuthors(gnomedb_control, &ev);
      if (ev._major == CORBA_NO_EXCEPTION) version = g_strdup(retval);
      CORBA_exception_free(&ev);
      return version;
    }
  
  return NULL;
}

/**
 * gnome_db_control_widget_run
 * @wid: control widget
 */
void
gnome_db_control_widget_run (GnomeDbControlWidget *wid)
{
  GNOME_DB_Control gnomedb_control;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid));
  g_return_if_fail(BONOBO_IS_WIDGET(wid->bonobo_widget));
  
  gnomedb_control = gnome_db_control_widget_get_gnomedb_control(wid);
  if (gnomedb_control != CORBA_OBJECT_NIL)
    {
      CORBA_Environment ev;
      
      CORBA_exception_init(&ev);
      GNOME_DB_Control_run(gnomedb_control, &ev);
      CORBA_exception_free(&ev);
    }
}

/**
 * gnome_db_control_widget_do_print
 * @wid: control widget
 */
void
gnome_db_control_widget_do_print (GnomeDbControlWidget *wid)
{
  GNOME_DB_Control gnomedb_control;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid));
  g_return_if_fail(BONOBO_IS_WIDGET(wid->bonobo_widget));
  
  gnomedb_control = gnome_db_control_widget_get_gnomedb_control(wid);
  if (gnomedb_control != CORBA_OBJECT_NIL)
    {
      CORBA_Environment ev;
      
      CORBA_exception_init(&ev);
      GNOME_DB_Control_doPrint(gnomedb_control, &ev);
      CORBA_exception_free(&ev);
    }
}

/**
 * gnome_db_control_widget_do_cut_clipboard
 * @wid: control widget
 */
void
gnome_db_control_widget_do_cut_clipboard (GnomeDbControlWidget *wid)
{
  GNOME_DB_Control gnomedb_control;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid));
  g_return_if_fail(BONOBO_IS_WIDGET(wid->bonobo_widget));
  
  gnomedb_control = gnome_db_control_widget_get_gnomedb_control(wid);
  if (gnomedb_control != CORBA_OBJECT_NIL)
    {
      CORBA_Environment ev;
      
      CORBA_exception_init(&ev);
      GNOME_DB_Control_doCutClipboard(gnomedb_control, &ev);
      CORBA_exception_free(&ev);
    }
}

/**
 * gnome_db_control_widget_do_copy_clipboard
 * @wid: control widget
 */
void
gnome_db_control_widget_do_copy_clipboard (GnomeDbControlWidget *wid)
{
  GNOME_DB_Control gnomedb_control;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid));
  g_return_if_fail(BONOBO_IS_WIDGET(wid->bonobo_widget));
  
  gnomedb_control = gnome_db_control_widget_get_gnomedb_control(wid);
  if (gnomedb_control != CORBA_OBJECT_NIL)
    {
      CORBA_Environment ev;
      
      CORBA_exception_init(&ev);
      GNOME_DB_Control_doCopyClipboard(gnomedb_control, &ev);
      CORBA_exception_free(&ev);
    }
}

/**
 * gnome_db_control_widget_do_paste_clipboard
 * @wid: control widget
 */
void
gnome_db_control_widget_do_paste_clipboard (GnomeDbControlWidget *wid)
{
  GNOME_DB_Control gnomedb_control;
  
  g_return_if_fail(GNOME_DB_IS_CONTROL_WIDGET(wid));
  g_return_if_fail(BONOBO_IS_WIDGET(wid->bonobo_widget));
  
  gnomedb_control = gnome_db_control_widget_get_gnomedb_control(wid);
  if (gnomedb_control != CORBA_OBJECT_NIL)
    {
      CORBA_Environment ev;
      
      CORBA_exception_init(&ev);
      GNOME_DB_Control_doPasteClipboard(gnomedb_control, &ev);
      CORBA_exception_free(&ev);
    }
}
