/*  GNOME-DB Components
 *  Copyrigth (C) 2000 Rodrigo Moya
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gnomedb-com.h>

typedef struct {
	GList *connections;
	GdaConnection *selected_cnc;
	GtkWidget *toplevel;
	GtkWidget *cnc_list;
	GtkWidget *notebook;
	GtkWidget *filename_entry;
	GtkWidget *export_table_data;
	GtkWidget *table_list;
	GtkWidget *selected_tables;
	GtkWidget *log_output;
} _export_instance_data_t;

static void connection_selected_cb (GtkMenuItem *menu_item, gpointer user_data);

static GnomeDbControl* export_control = NULL;

/*
 * Private functions
 */

static void
add_connection_cb (GdaConnectionPool *pool,
		   GdaConnection *cnc,
		   const gchar *gda_name,
		   const gchar *username,
		   const gchar *password,
		   gpointer user_data)
{
	GtkWidget *menu_item;
	GtkWidget *menu = (GtkWidget *) user_data;

	g_return_if_fail(IS_GDA_CONNECTION(cnc));
	g_return_if_fail(GTK_IS_MENU(menu));

	if (gda_name) {
		gchar *tmp_str = g_strdup(gda_name);

		menu_item = gtk_menu_item_new_with_label(tmp_str);
		gtk_object_set_data_full(GTK_OBJECT(menu_item),
					 "GNOMEDB_MenuItemLabel",
					 (gpointer) tmp_str,
					 (GtkDestroyNotify) g_free);
		gtk_object_set_data(GTK_OBJECT(menu_item), "COMPONENTS_Export_GdaConnection", cnc);
		gtk_signal_connect(GTK_OBJECT(menu_item),
				   "activate",
				   GTK_SIGNAL_FUNC(connection_selected_cb),
				   gtk_object_get_data(GTK_OBJECT(menu), "COMPONENTS_Export_InstanceData"));
		gtk_widget_show(menu_item);
		gtk_menu_append(GTK_MENU(menu), menu_item);
	}
}

static void
fill_connection_list (_export_instance_data_t *instance_data)
{
	GtkWidget *cnc_list;
	GtkWidget *menu;

	g_return_if_fail(instance_data != NULL);

	if (GTK_IS_OPTION_MENU(instance_data->cnc_list)) {
		gtk_option_menu_remove_menu(GTK_OPTION_MENU(instance_data->cnc_list));
		menu = gtk_menu_new();
		gtk_object_set_data(GTK_OBJECT(menu), "COMPONENTS_Export_InstanceData", instance_data);
		gda_connection_pool_foreach(components_connection_get_pool(),
					    (GdaConnectionPoolForeachFunc) add_connection_cb,
					    (gpointer) menu);

		gtk_option_menu_set_menu(GTK_OPTION_MENU(instance_data->cnc_list), menu);
	}
}

static void
refresh_lists (_export_instance_data_t *instance_data, GdaConnection *cnc)
{
	GdaRecordset *recset;

	g_return_if_fail(instance_data != NULL);

	/* fill in the tables list */
	recset = gda_connection_open_schema(cnc, GDA_Connection_GDCN_SCHEMA_TABLES,
					    GDA_Connection_no_CONSTRAINT);
	gnome_db_list_set_recordset(GNOME_DB_LIST(instance_data->table_list), recset, 0);
	gnome_db_clear_clist(GTK_CLIST(instance_data->selected_tables));

	instance_data->selected_cnc = cnc;
}

/*
 * Callbacks
 */

static void
add_all_tables_cb (GtkButton *button, gpointer user_data)
{
       	_export_instance_data_t *instance_data = (_export_instance_data_t *) user_data;

	g_return_if_fail(instance_data != NULL);
}

static void
add_table_cb (GtkButton *button, gpointer user_data)
{
	gchar *table_name;
	_export_instance_data_t *instance_data = (_export_instance_data_t *) user_data;

	g_return_if_fail(instance_data != NULL);

	table_name = gnome_db_list_get_string(GNOME_DB_LIST(instance_data->table_list));
	if (table_name) {
		gchar *row[2] = { "", ""};

		/* FIXME: check if the table is already selected */
		row[1] = table_name;
		gtk_clist_append(GTK_CLIST(instance_data->selected_tables), row);
	}
}

static void
connect_button_clicked_cb (GtkButton *button, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *login;
	_export_instance_data_t *instance_data = (_export_instance_data_t *) user_data;

	if (instance_data) {
		/* create login dialog box */
		dialog = gnome_dialog_new(_("Open connection"),
					  GNOME_STOCK_BUTTON_OK,
					  GNOME_STOCK_BUTTON_CANCEL,
					  NULL);
		gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
		login = gnome_db_login_new(NULL);
		gtk_widget_show(login);
		gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), login, 1, 1, 0);

		/* run dialog */
		if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
			GdaConnection *cnc;

			/* open the connection */
			cnc = components_connection_open(
				gnome_db_login_get_gda_name(GNOME_DB_LOGIN(login)),
				gnome_db_login_get_username(GNOME_DB_LOGIN(login)),
				gnome_db_login_get_password(GNOME_DB_LOGIN(login)));
			if (cnc) {
				instance_data->connections = g_list_append(instance_data->connections, cnc);
				fill_connection_list(instance_data);
				refresh_lists(instance_data, cnc);
			}
		}
		gnome_dialog_close(GNOME_DIALOG(dialog));
	}
}

static void
connection_selected_cb (GtkMenuItem *menu_item, gpointer user_data)
{
	GdaConnection *cnc;
	_export_instance_data_t *instance_data = (_export_instance_data_t *) user_data;

	g_return_if_fail(GTK_IS_MENU_ITEM(menu_item));
	g_return_if_fail(instance_data != NULL);

	cnc = (GdaConnection *) gtk_object_get_data(GTK_OBJECT(menu_item),
						    "COMPONENTS_Export_GdaConnection");
	if (IS_GDA_CONNECTION(cnc)) {
		refresh_lists(instance_data, cnc);
	}
}

static void
remove_table_cb (GtkButton *button, gpointer user_data)
{
	GList *l;
	_export_instance_data_t *instance_data = (_export_instance_data_t *) user_data;

	g_return_if_fail(instance_data != NULL);

	gtk_clist_freeze(GTK_CLIST(instance_data->selected_tables));
	for (l = g_list_last(GTK_CLIST(instance_data->selected_tables)->selection);
	     l != NULL;
	     l = g_list_previous(l)) {
		gtk_clist_remove(GTK_CLIST(instance_data->selected_tables),
				 GPOINTER_TO_INT(l->data));
	}
	gtk_clist_thaw(GTK_CLIST(instance_data->selected_tables));
}

static void
widget_destroyed_cb (GtkObject *object, gpointer user_data)
{
	_export_instance_data_t *instance_data = (_export_instance_data_t *) user_data;

	if (instance_data) {
		/* close all the connections we opened */
		if (instance_data->connections) {
			GList *l;

			while ((l = g_list_first(instance_data->connections))) {
				GdaConnection *cnc = (GdaConnection *) l->data;
				instance_data->connections = g_list_remove(instance_data->connections, cnc);
				components_connection_close(cnc);
			}
			instance_data->connections = NULL;
		}

		g_free((gpointer) instance_data);
	}
}

/*
 * Public functions
 */
BonoboObject *
components_export_new (void)
{
	GtkWidget *export;
	GnomeDbControl *control;

	/* create the widget */
	export = components_export_create_widget();
	control = gnome_db_control_new(export);
	if (!GNOME_DB_IS_CONTROL(control)) {
		gtk_widget_destroy(export);
		control = NULL;
	}

	return BONOBO_OBJECT(control);
}

GtkWidget *
components_export_create_widget (void)
{
	GtkWidget *export;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *table;
	GtkWidget *scroll;
	_export_instance_data_t *instance_data;

	export = gnome_db_new_table_widget(4, 4, FALSE);

	instance_data = g_new0(_export_instance_data_t, 1);
	instance_data->toplevel = export;

	gtk_signal_connect(GTK_OBJECT(export),
			   "destroy",
			   GTK_SIGNAL_FUNC(widget_destroyed_cb),
			   (gpointer) instance_data);
	
	/* create the rest of the widgets */
	label = gnome_db_new_label_widget(_("Database"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach(GTK_TABLE(export), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	instance_data->cnc_list = gnome_db_new_option_menu_widget();
	gnome_db_set_widget_tooltip(instance_data->cnc_list,
				    _("Select the database you want to export. If you want"
				      ", you can also open a new connection by pressing"
				      " the 'Connect' button"));
	fill_connection_list(instance_data);
	gtk_table_attach(GTK_TABLE(export), instance_data->cnc_list, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	button = gnome_db_new_button_widget_with_pixmap("", GNOME_STOCK_MENU_CONVERT);
	gnome_db_set_widget_tooltip(button, _("Press this button if you want to open a"
					      " new connection to the database to be exported"));
	gtk_signal_connect(GTK_OBJECT(button),
			   "clicked",
			   GTK_SIGNAL_FUNC(connect_button_clicked_cb),
			   (gpointer) instance_data);
	gtk_table_attach(GTK_TABLE(export), button, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	button = gnome_db_new_button_widget_with_pixmap("", GNOME_STOCK_MENU_FORWARD);
	gnome_db_set_widget_tooltip(button, _("Press this button to start the export process"
					      " with the options/objects you've selected"));
	gtk_table_attach(GTK_TABLE(export), button, 2, 3, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

	instance_data->notebook = gnome_db_new_notebook_widget();
	gtk_table_attach(GTK_TABLE(export), instance_data->notebook, 0, 4, 2, 4,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 3, 3);

	/* create the 'Options' tab */
	table = gnome_db_new_table_widget(6, 3, FALSE);
	gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->notebook),
				 table,
				 gtk_label_new(_("Options")));

	label = gnome_db_new_label_widget(_("Output file"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	instance_data->filename_entry = gnome_db_new_file_entry_widget("GNOME_DB_Export_OutputFile");
	gnome_db_set_widget_tooltip(instance_data->filename_entry,
				    _("Select the file to which you want to save the"
				      " database contents"));
	gtk_table_attach(GTK_TABLE(table), instance_data->filename_entry, 1, 3, 0, 1,
			 GTK_FILL, GTK_FILL, 3, 3);

	instance_data->export_table_data =
		gnome_db_new_check_button_widget(_("Export table data"), TRUE);
	gnome_db_set_widget_tooltip(instance_data->export_table_data,
				    _("When selected, the export process also saves"
				      " the exported tables' data. If not selected,"
				      " only the table structure is exported"));
	gtk_table_attach(GTK_TABLE(table), instance_data->export_table_data, 0, 2, 1, 2,
			 GTK_FILL, GTK_FILL, 3, 3);

	/* create the 'Objects' tab */
	table = gnome_db_new_table_widget(6, 3, FALSE);
	gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->notebook),
				 table,
				 gtk_label_new(_("Objects")));

	instance_data->table_list = gnome_db_list_new(NULL, 0);
	gtk_widget_show(instance_data->table_list);
	gtk_table_attach(GTK_TABLE(table), instance_data->table_list, 0, 1, 0, 6,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 3, 3);

	button = gnome_db_new_button_widget(">>");
	gtk_signal_connect(GTK_OBJECT(button),
			   "clicked",
			   GTK_SIGNAL_FUNC(add_all_tables_cb),
			   (gpointer) instance_data);
	gnome_db_set_widget_tooltip(button, _("Select all objects in the left list"));
	gtk_table_attach(GTK_TABLE(table), button, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	button = gnome_db_new_button_widget(">");
	gtk_signal_connect(GTK_OBJECT(button),
			   "clicked",
			   GTK_SIGNAL_FUNC(add_table_cb),
			   (gpointer) instance_data);
	gnome_db_set_widget_tooltip(button, _("Select only highlighted objects"));
	gtk_table_attach(GTK_TABLE(table), button, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

	button = gnome_db_new_button_widget("<");
	gtk_signal_connect(GTK_OBJECT(button),
			   "clicked",
			   GTK_SIGNAL_FUNC(remove_table_cb),
			   (gpointer) instance_data);
	gnome_db_set_widget_tooltip(button, _("Unselect only highlighted objects"));
	gtk_table_attach(GTK_TABLE(table), button, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

	button = gnome_db_new_button_widget("<<");
	gnome_db_set_widget_tooltip(button, _("Unselect all objects in the right list"));
	gtk_table_attach(GTK_TABLE(table), button, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 3, 3);

	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 2, 3, 0, 6,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 3, 3);
	instance_data->selected_tables = gnome_db_new_clist_widget(NULL, 2);
	gtk_container_add(GTK_CONTAINER(scroll), instance_data->selected_tables);

	/* create the 'Log' tab */
	table = gnome_db_new_table_widget(3, 8, FALSE);
	gtk_notebook_append_page(GTK_NOTEBOOK(instance_data->notebook),
				 table,
				 gtk_label_new(_("Log")));

	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(table), scroll, 0, 2, 0, 8,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			 3, 3);
	instance_data->log_output = gnome_db_new_text_widget();
	gtk_text_set_editable(GTK_TEXT(instance_data->log_output), FALSE);
	gtk_container_add(GTK_CONTAINER(scroll), instance_data->log_output);

	return export;
}
