/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

// Owner:  miket

#include <AGUserConfig.h>
#include <AGServerConfig.h>
#include <AGUtil.h>

/* Version 0 - original creation */
#define RECORD_VERSION_0 (0)

/* Version 1 - reset cookies at next sync */
#define RECORD_VERSION_1 (1)

/* This is the version number that is written to new records */
#define CURRENT_RECORD_VERSION (RECORD_VERSION_1)

/* Forward declaration. */
static AGServerConfig * getServerByNameAndPort(AGArray * array,
                                               AGUserConfig * userConfig,
                                               char * serverName,
                                               int32 serverPort,
                                               AGUserConfigFindMode mode);
static void copyServerGroup(AGArray ** dst, AGArray * src);

static int32 getNextUID(AGUserConfig * userConfig)
{
    return userConfig->nextUID++;
}

void AGUserConfigInit(AGUserConfig * userConfig)
{
    bzero(userConfig, sizeof(AGUserConfig));
    if (NULL != userConfig) {
        userConfig->nextUID = 1;
        userConfig->servers = AGArrayNew(AGUnownedPointerElements, 1);
        userConfig->serversToAdd = NULL;
        userConfig->serversToDelete = NULL;
        userConfig->dirty = TRUE;
    }
}

AGUserConfig * AGUserConfigNew()
{
    AGUserConfig * result = (AGUserConfig *)malloc(sizeof(AGUserConfig));
    if (NULL != result)
        AGUserConfigInit(result);
    return result;
}

static void finalizeServerGroup(AGArray * array)
{
    if (NULL != array) {
        int i, n;
        n = AGArrayCount(array);
        for (i = 0; i < n; ++i) {
            AGServerConfig * sc;
            sc = (AGServerConfig*)AGArrayElementAt(array, i);
            AGServerConfigFree(sc);
        }
        AGArrayFree(array);
    }
}

void AGUserConfigFinalize(AGUserConfig * userConfig)
{
    if (NULL != userConfig) {
        finalizeServerGroup(userConfig->servers);
        finalizeServerGroup(userConfig->serversToAdd);
        finalizeServerGroup(userConfig->serversToDelete);
        bzero(userConfig, sizeof(AGUserConfig));
    }
}

void AGUserConfigFree(AGUserConfig * userConfig)
{
    if (NULL != userConfig) {
        AGUserConfigFinalize(userConfig);
        free(userConfig);
        userConfig = NULL;
    }
}

static void copyServerGroup(AGArray ** dst, AGArray * src)
{
    if (NULL != src) {
        int32 i, n;
        n = AGArrayCount(src);
        *dst = AGArrayNew(AGUnownedPointerElements, n);
        for (i = 0; i < n; ++i)
            AGArrayAppend(*dst, AGServerConfigDup((AGServerConfig *)
                AGArrayElementAt(src, i)));
    }
}

AGUserConfig *AGUserConfigCopy(AGUserConfig *dstConfig, 
                               AGUserConfig *srcConfig)
{
    dstConfig->dirty = srcConfig->dirty;
    dstConfig->nextUID = srcConfig->nextUID;

    copyServerGroup(&dstConfig->servers, srcConfig->servers);
    copyServerGroup(&dstConfig->serversToAdd, srcConfig->serversToAdd);
    copyServerGroup(&dstConfig->serversToDelete, srcConfig->serversToDelete);

    return dstConfig;
}

AGUserConfig *AGUserConfigDup(AGUserConfig *srcConfig)
{
    AGUserConfig * dstConfig;
    
    if (NULL == srcConfig)
        return NULL;

    dstConfig = (AGUserConfig *)malloc(sizeof(AGUserConfig));

    bzero(dstConfig, sizeof(AGUserConfig));
    AGUserConfigCopy(dstConfig, srcConfig);
    return dstConfig;
}

static AGBool serverConfigUidIsUnique(AGArray * array,
                                      int32 uid)
{
    int32 i, n;
    n = AGArrayCount(array);
    for (i = 0; i < n; ++i) {
        if (((AGServerConfig *)AGArrayElementAt(array, i))->uid == uid)
            return FALSE;
    }
    return TRUE;
}

static int32 addServer(AGArray * array,
                       AGUserConfig * userConfig,
                       AGServerConfig * serverConfig)
{
    if (NULL == serverConfig)
        serverConfig = AGServerConfigNew();

    if (NULL == serverConfig)
        return 0;

    if (0 == serverConfig->uid)
        serverConfig->uid = getNextUID(userConfig);

    /* Ensure that the new serverConfig's UID is indeed unique. */
    while (!serverConfigUidIsUnique(userConfig->servers, serverConfig->uid))
        ++serverConfig->uid;
    while (!serverConfigUidIsUnique(array, serverConfig->uid))
        ++serverConfig->uid;

    AGArrayAppend(array, serverConfig);

    userConfig->dirty = TRUE;

    return serverConfig->uid;
}

int32 AGUserConfigAddServer(AGUserConfig * userConfig,
                            AGServerConfig * serverConfig)
{
    return addServer(userConfig->servers, userConfig, serverConfig);
}

int32 AGUserConfigAddServerToDevice(AGUserConfig * userConfig,
                                    AGServerConfig * serverConfig)
{
    if (NULL == userConfig->servers)
        userConfig->servers = AGArrayNew(AGUnownedPointerElements, 1);
    return addServer(userConfig->servers, userConfig, serverConfig);
}

int32 AGUserConfigAddServerToDesktop(AGUserConfig * userConfig,
                                     AGServerConfig * serverConfig)
{
    if (NULL == userConfig->serversToAdd)
        userConfig->serversToAdd = AGArrayNew(AGUnownedPointerElements, 1);
    return addServer(userConfig->serversToAdd, userConfig, serverConfig);
}

static AGServerConfig * getServerByNameAndPort(AGArray * array,
                                               AGUserConfig * userConfig,
                                               char * serverName,
                                               int32 serverPort,
                                               AGUserConfigFindMode mode)
{
    int32 n;
    AGServerConfig * result = NULL;

    n = AGArrayCount(array);
    while (n--) {
        result = (AGServerConfig *)AGArrayElementAt(array, n);
        if (serverPort == result->serverPort &&
            result->serverName && 
            !strcmp(serverName, result->serverName))
            return result;
    }
    result = NULL;

    if (mode == AGUSERCONFIG_CREATE || mode == AGUSERCONFIG_CREATETEMP) {
        result = AGServerConfigNew();
        if (NULL != result) {
            result->dbconfigs = AGArrayNew(AGUnownedPointerElements, 1);
            addServer(array, userConfig, result);
            result->serverName = strdup(serverName);
            result->serverPort = serverPort;
            if (mode == AGUSERCONFIG_CREATETEMP)
                result->uid = 0;
        }
    }
    return result;
}

AGServerConfig * AGUserConfigGetServerByNameAndPort(AGUserConfig * userConfig,
                                                    char * serverName,
                                                    int32 serverPort,
                                                    AGUserConfigFindMode mode)
{
    AGServerConfig * result;
    result = getServerByNameAndPort(userConfig->servers, userConfig,
        serverName, serverPort, mode);
    if (NULL == result && NULL != userConfig->serversToAdd)
        result = getServerByNameAndPort(userConfig->serversToAdd, userConfig,
            serverName, serverPort, mode);
    return result;
}

static AGServerConfig * getServerByIndex(AGArray * array,
                                         int32 index)
{
    int32 n;
    n = AGArrayCount(array);
    return (index < 0 || index >= n)
        ? NULL
        : (AGServerConfig *)AGArrayElementAt(array, index);
}

AGServerConfig * AGUserConfigGetServerByIndex(AGUserConfig * userConfig,
                                              int32 index)
{
    return getServerByIndex(userConfig->servers, index);
}

AGServerConfig * AGUserConfigGetServerByName(AGUserConfig * userConfig,
                                             char * serverName,
                                             AGUserConfigFindMode mode)
{
    int32 n;
    AGServerConfig * result = NULL;

    n = AGUserConfigCount(userConfig);

    while (n--) {
        result = AGUserConfigGetServerByIndex(userConfig, n);
        if (!strcmp(result->serverName, serverName))
            return result;
    }
    result = NULL;
    if (mode == AGUSERCONFIG_CREATE || mode == AGUSERCONFIG_CREATETEMP) {
        result = AGServerConfigNew();
        if (NULL != result) {
            result->dbconfigs = AGArrayNew(AGUnownedPointerElements, 1);
            addServer(userConfig->servers, userConfig, result);
            result->serverName = strdup(serverName);
            if (mode == AGUSERCONFIG_CREATETEMP)
                result->uid = 0;
        }
    }
    return result;
}

int32 AGUserConfigCount(AGUserConfig * userConfig)
{
    if (NULL != userConfig)
        return AGArrayCount(userConfig->servers);
    else
        return 0;
}

/* ----------------------------------------------------------------------------
    AGServerConfig * AGUserConfigEnumerate(AGUserConfig * userConfig,
                                           AGUserConfigEnumerateState ** state)

    Return a list of servers that will be synchronized at the next sync. In
    other words, enumerates the servers that are (a) both (1) in the current
    list and (2) not scheduled for deletion, or (b) scheduled for addition.

    Call with state set to NULL.  Keep calling until function returns NULL.

*/
AGServerConfig * AGUserConfigEnumerate(AGUserConfig * userConfig,
                                       AGUserConfigEnumerateState ** state)
{
    if (NULL == *state) {
        *state = (AGUserConfigEnumerateState *)
            malloc(sizeof(AGUserConfigEnumerateState));
        (*state)->count = (*state)->addCount = 0;
    }

    /* If we're in the process of returning servers from the server list,
    return the next one that's not scheduled for deletion, and check to see
    whether we're at the end of the list. */
    while ((*state)->count >= 0) {
        AGServerConfig * sc = NULL;
        if ((*state)->count >= AGArrayCount(userConfig->servers))
            (*state)->count = -1;
        else {
            sc = (AGServerConfig *)
                AGArrayElementAt(userConfig->servers, (*state)->count);
            (*state)->count++;
            if (NULL == getServerByNameAndPort(userConfig->serversToDelete,
                userConfig, sc->serverName, sc->serverPort, AGUSERCONFIG_FIND))
                return sc;
        }
    }

    /* If we have reached this point, we have exhausted the list of servers,
    so we should return servers that are scheduled to be added. */
    while ((*state)->addCount >= 0) {
        AGServerConfig * sc = NULL;
        if ((*state)->addCount >= AGArrayCount(userConfig->serversToAdd))
            (*state)->addCount = -1;
        else {
            sc = (AGServerConfig *)
                AGArrayElementAt(userConfig->serversToAdd,
                    (*state)->addCount);
            (*state)->addCount++;
            return sc;
        }
    }

    /* If we have reached this point, we have returned all servers in the 
    server list and all servers in the list of servers that are scheduled to
    be added.  So we should free our state structure. */
    free(*state);

    return NULL;
}

AGServerConfig * AGUserConfigGetServerByUID(AGUserConfig * userConfig,
                                            int32 uid,
                                            AGUserConfigFindMode mode)
{
    AGServerConfig * result = NULL;

    if (uid > 0) {
        int32 n;
        n = AGArrayCount(userConfig->servers);
        while (n--) {
            result = (AGServerConfig *)
                AGArrayElementAt(userConfig->servers, n);
            if (uid == result->uid)
                return result;
        }
        n = AGArrayCount(userConfig->serversToAdd);
        while (n--) {
            result = (AGServerConfig *)
                AGArrayElementAt(userConfig->serversToAdd, n);
            if (uid == result->uid)
                return result;
        }
        result = NULL;
    }
    if (mode == AGUSERCONFIG_CREATE || mode == AGUSERCONFIG_CREATETEMP) {
        result = AGServerConfigNew();
        if (NULL != result) {
            result->dbconfigs = AGArrayNew(AGUnownedPointerElements, 1);
            addServer(userConfig->servers, userConfig, result);
            if (mode == AGUSERCONFIG_CREATETEMP)
                result->uid = 0;
            else
                if (uid != 0)
                    result->uid = uid;
        }
    }
    return result;
}

static void removeServer(AGArray * array,
                         AGServerConfig * serverConfig)
{
    int32 n = 0;
    while (1) {
        n = AGArrayIndexOf(array, serverConfig, n);
        if (n >= 0)
            AGArrayRemoveAt(array, n);
        else
            break;
    }
}

void AGUserConfigRemoveServer(AGUserConfig * userConfig,
                              AGServerConfig * serverConfig)
{
    removeServer(userConfig->servers, serverConfig);
    userConfig->dirty = TRUE;
}

void AGUserConfigRemoveServerFromDesktop(AGUserConfig * userConfig,
                                         char * serverName,
                                         int16 serverPort)
{
    if (NULL != userConfig->serversToAdd) {
        /* If we're deleting a server that is scheduled for addition,
        just call it a draw. */
        AGServerConfig * sc = getServerByNameAndPort(userConfig->serversToAdd,
            userConfig, serverName, serverPort, AGUSERCONFIG_FIND);
        if (NULL != sc) {
            removeServer(userConfig->serversToAdd, sc);
            AGServerConfigFree(sc);
            userConfig->dirty = TRUE;
            return;
        }
    }

    if (NULL != userConfig->servers) {
        /* We're deleting a server that OUGHT to exist. So remove it from the
        current list and move it to the list of servers scheduled for
        deletion. */
        AGServerConfig * sc = getServerByNameAndPort(userConfig->servers,
            userConfig, serverName, serverPort, AGUSERCONFIG_FIND);
        if (NULL != sc) {
            removeServer(userConfig->servers, sc);
            if (NULL == userConfig->serversToDelete)
                userConfig->serversToDelete =
                    AGArrayNew(AGUnownedPointerElements, 1);
                AGArrayAppend(userConfig->serversToDelete, sc);
            userConfig->dirty = TRUE;
        }
    }
}

void AGUserConfigRemoveServerFromDevice(AGUserConfig * userConfig,
                                        char * serverName,
                                        int16 serverPort)
{
    if (NULL != userConfig->servers) {
        AGServerConfig * sc = getServerByNameAndPort(userConfig->servers,
            userConfig, serverName, serverPort, AGUSERCONFIG_FIND);
        if (NULL != sc) {
            removeServer(userConfig->servers, sc);
            AGServerConfigFree(sc);
            userConfig->dirty = TRUE;
            return;
        }
    }
}

static void readServerGroup(AGArray ** array, AGReader * r)
{
    int32 i, n;
    n = AGReadCompactInt(r);
    *array = AGArrayNew(AGUnownedPointerElements, 1);
    for (i = 0; i < n; ++i) {
        AGServerConfig * sc = AGServerConfigNewAndReadData(r);
        AGArrayAppend(*array, sc);
    }
}

void AGUserConfigReadData(AGUserConfig * userConfig, AGReader *r)
{
    int16 version;
    version = AGReadCompactInt(r);
    userConfig->nextUID = AGReadCompactInt(r);
    readServerGroup(&userConfig->servers, r);
    readServerGroup(&userConfig->serversToAdd, r);
    readServerGroup(&userConfig->serversToDelete, r);
    userConfig->dirty = FALSE;
    if (RECORD_VERSION_0 == version)
        userConfig->resetCookiesAtNextSync = FALSE;
    else
        userConfig->resetCookiesAtNextSync = AGReadBoolean(r);
    if (RECORD_VERSION_1 == version)
        return;
}

AGUserConfig * AGUserConfigNewAndReadData(AGReader *r)
{
    AGUserConfig * result;

    result = (AGUserConfig *)malloc(sizeof(AGUserConfig));

    bzero(result, sizeof(AGUserConfig));
    AGUserConfigReadData(result, r);
    return result;
}

ExportFunc AGUserConfig * AGUserConfigInitAndReadData(AGUserConfig * userConfig,
                                                    AGReader *r)
{
    bzero(userConfig, sizeof(AGUserConfig));
    AGUserConfigReadData(userConfig, r);
    return userConfig;
}

static void writeServerGroup(AGArray * array, AGWriter * w)
{
    int32 i, n;

    n = AGArrayCount(array);
    AGWriteCompactInt(w, n);
    for (i = 0; i < n; ++i) {
        AGServerConfigWriteData((AGServerConfig *)
            AGArrayElementAt(array, i), w);
    }
}

void AGUserConfigWriteData(AGUserConfig * userConfig, AGWriter *w)
{
    AGWriteCompactInt(w, CURRENT_RECORD_VERSION);
    AGWriteCompactInt(w, userConfig->nextUID);
    writeServerGroup(userConfig->servers, w);
    writeServerGroup(userConfig->serversToAdd, w);
    writeServerGroup(userConfig->serversToDelete, w);
    AGWriteBoolean(w, userConfig->resetCookiesAtNextSync);
    userConfig->dirty = FALSE;
}

#ifndef REMOVE_SYNCHRONIZE_FEATURE
static AGUserConfig * synchronize(AGUserConfig * agreed,
                                  AGUserConfig * device,
                                  AGUserConfig * desktop)
{
    AGUserConfig * result;

    /* Begin with a brand-new UserConfig structure. */
    result = AGUserConfigNew();

    if (NULL != result) {

        int i, n;

        /* Add and synchronize the servers in the device's list that are not
        scheduled for deletion. */
        i = 0;

        if (NULL != device) {

            n = AGArrayCount(device->servers);

            while (i < n) {

                AGServerConfig * devsc = NULL;
                AGServerConfig * asc = NULL;
                AGServerConfig * desksc = NULL;

                devsc = AGUserConfigGetServerByIndex(device, i);

                if (!AGServerConfigIsValid(devsc)) {
                    ++i;
                    continue;
                }

                if (NULL != desktop) {

                    if (NULL != getServerByNameAndPort(
                        desktop->serversToDelete,
                        desktop,
                        devsc->serverName,
                        devsc->serverPort,
                        AGUSERCONFIG_FIND)) {

                        ++i;
                        continue;

                    }

                    desksc = getServerByNameAndPort(desktop->servers,
                        desktop,
                        devsc->serverName,
                        devsc->serverPort,
                        AGUSERCONFIG_FIND);

                }

                if (NULL != agreed)
                    asc = getServerByNameAndPort(agreed->servers,
                    agreed,
                    devsc->serverName,
                    devsc->serverPort,
                    AGUSERCONFIG_FIND);
            
                /* The server in the device list hasn't been deleted, so
                add the synchronized version of it. */
                if (NULL == asc || NULL == desksc)
                    addServer(result->servers,
                        result,
                        AGServerConfigDup(devsc));
                else
                    addServer(result->servers,
                        result,
                        AGServerConfigSynchronize(asc, devsc, desksc));

                ++i;

            }

        } else {

            /* Device was hard reset.  Copy over desktop list. */

            if (NULL != desktop)
                n = AGArrayCount(desktop->servers);
            else
                n = 0;

            while (i < n) {

                AGServerConfig * desksc = NULL;

                desksc = getServerByIndex(desktop->servers, i);

                addServer(result->servers,
                    result,
                    AGServerConfigDup(desksc));

                ++i;

            }

        }

        /* Add the servers in the desktop's add list. No need to 
        synchronize because they're new. */
        /* pending(miket) what about when user adds same server/port
        on device and desktop before a single sync? */
        if (NULL != desktop) {

            i = 0;

            n = AGArrayCount(desktop->serversToAdd);

            while (i < n) {

                addServer(result->servers,
                    result,
                    AGServerConfigDup(
                        getServerByIndex(desktop->serversToAdd, i)));

                ++i;
            }

        }

    }

    return result;

}

AGUserConfig * AGUserConfigSynchronize(AGUserConfig *agreed,
                                       AGUserConfig *device,
                                       AGUserConfig *desktop)
{
    AGBool resetCookies = FALSE;
    AGUserConfig * result;
    
    /*
    
    If deviceConfig is NULL, then we'll assume the device has experienced
    a hard reset.  Reset cookies and nonces.
    
    Do the same if the user requested a cookie reset on the desktop.

    */
    if (NULL != device)
        resetCookies = device->resetCookiesAtNextSync;
    else
        resetCookies = TRUE;

    if (NULL != desktop)
        resetCookies = resetCookies || desktop->resetCookiesAtNextSync;

    result = synchronize(agreed, device, desktop);

    if (resetCookies) {

        int i, n;
        n = AGArrayCount(result->servers);
        for (i = 0; i < n; ++i) {
            AGServerConfig * sc;
            sc = AGUserConfigGetServerByIndex(result, i);
            AGServerConfigResetCookie(sc);
            AGServerConfigResetNonce(sc);
        }

    }

    return result;
}
#endif /* #ifndef REMOVE_SYNCHRONIZE_FEATURE */

void AGUserConfigResetCookies(AGUserConfig *config)
{
    AGUserConfigEnumerateState * state = NULL;
    AGServerConfig * sc = NULL;
    do {
        sc = AGUserConfigEnumerate(config, &state);
        if (NULL != sc)
            AGServerConfigResetCookie(sc);
    } while (NULL != sc);
}
