/*
* Gnome-Scan
* Copyright (C) Étienne Bersac 2007 <bersace03@laposte.net>
* 
* gnome-scan is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
* 
* gnome-scan is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public
* License along with gnome-scan.  If not, write to:
* 	The Free Software Foundation, Inc.,
* 	51 Franklin Street, Fifth Floor
* 	Boston, MA  02110-1301, USA.
*/

#include <glib/gi18n.h>
#include "gnome-scan-private.h"
#include "gnome-scan-preview-plugin-area.h"
#include "gnome-scan-preview-area.h"


/* AREA */
/* draw, move and resize area */
GS_DEFINE_PREVIEW_PLUGIN (GnomeScanPreviewPluginArea, gnome_scan_preview_plugin_area,
                          "origin,page-orientation");

#define	GS_ANCHOR_NONE		-1
#define	GS_ANCHOR_OUTSIDE	-2

typedef enum {
    GSPPA_MOVE	    = 1 << 0,
    GSPPA_RESIZE	= 1 << 1,	/* implies GSPPA_MOVE */
    GSPPA_CREATE	= 1 << 2
} CursorFlag;


static void	gsppa_send_area (GnomeScanPreviewPluginArea *gsppa);

/* CURSOR / ANCHOR handling */
/* This part include function related to position detection, cursor
selection, etc. */

static inline gboolean
gsppa_point_is_in_rect(GdkPoint *point,
                       GdkRectangle *rect)
{
    return rect->x <= point->x && point->x <= rect->x+rect->width
        && rect->y <= point->y && point->y <= rect->y+rect->height;
}

static GdkRectangle*
gsppa_get_rect_for_anchor (GdkRectangle* r,
                           GtkAnchorType anchor)
{
#define	l	12
    GdkRectangle *a = g_new0 (GdkRectangle, 1);
    a->width = l;
    a->height = l;
    
    switch (anchor) {
        case GTK_ANCHOR_CENTER:
            a->width = r->width - l;
        a->height = r->height - l;
        a->x = r->x + l;
        a->y = r->y + l;
        break;
        case GTK_ANCHOR_NORTH:
            a->x = r->x + l;
        a->y = r->y;
        a->width = r->width - l;
        break;
        case GTK_ANCHOR_NORTH_WEST:
            a->x = r->x;
        a->y = r->y;
        break;
        case GTK_ANCHOR_NORTH_EAST:
            a->x = r->x + r->width;
        a->y = r->y;
        break;
        case GTK_ANCHOR_SOUTH:
            a->x = r->x + l;
        a->y = r->y + r->height;
        a->width = r->width - l;
        break;
        case GTK_ANCHOR_SOUTH_WEST:
            a->x = r->x;
        a->y = r->y + r->height;
        break;
        case GTK_ANCHOR_SOUTH_EAST:
            a->x = r->x + r->width;
        a->y = r->y + r->height;
        break;
        case GTK_ANCHOR_WEST:
            a->x = r->x;
        a->y = r->y;
        a->height = r->height - l;
        break;
        case GTK_ANCHOR_EAST:
            a->x = r->x + r->width;
        a->y = r->y;
        a->height = r->height - l;
        break;
    }
    a->x-=l/2;
    a->y-=l/2;
#undef l
    return a;
}

static GtkAnchorType
gsppa_get_anchor_type (GdkPoint *point,
                       GdkRectangle *roi,
                       GdkRectangle *window)
{
    GtkAnchorType at, res = GS_ANCHOR_NONE;
    GdkRectangle *ar;
    
    if (gsppa_point_is_in_rect (point, window)) {
        for (at = GTK_ANCHOR_CENTER; at <= GTK_ANCHOR_EAST; at++) {
            ar = gsppa_get_rect_for_anchor (roi, at);
            if (gsppa_point_is_in_rect (point, ar)) {
                res = at;
                g_free (ar);
                break;
            }
            g_free (ar);
        }
    }
    else {
        res = GS_ANCHOR_OUTSIDE;
    }
    return res;
}

static GdkCursorType
gsppa_get_cursor_for_anchor (GtkAnchorType at)
{
    GdkCursorType ct = GDK_LAST_CURSOR;
    switch (at) {
        case GTK_ANCHOR_NORTH:
            ct = GDK_TOP_SIDE;
        break;
        case GTK_ANCHOR_NORTH_EAST:
            ct = GDK_TOP_RIGHT_CORNER;
        break;
        case GTK_ANCHOR_NORTH_WEST:
            ct = GDK_TOP_LEFT_CORNER;
        break;
        case GTK_ANCHOR_SOUTH:
            ct = GDK_BOTTOM_SIDE;
        break;
        case GTK_ANCHOR_SOUTH_EAST:
            ct = GDK_BOTTOM_RIGHT_CORNER;
        break;
        case GTK_ANCHOR_SOUTH_WEST:
            ct = GDK_BOTTOM_LEFT_CORNER;
        break;
        case GTK_ANCHOR_EAST:
            ct = GDK_RIGHT_SIDE;
        break;
        case GTK_ANCHOR_WEST:
            ct = GDK_LEFT_SIDE;
        break;
        case GTK_ANCHOR_CENTER:
            ct = GDK_FLEUR;
        break;
        case GS_ANCHOR_NONE:
            ct = GDK_CROSSHAIR;
        break;
        default:
            ct = GDK_ARROW;
        break;
    }
    
    return ct;
}

/* update cursor if needed */
static void
gsppa_set_cursor (GnomeScanPreviewPluginArea *gsppa,
                  GdkPoint *point,
                  CursorFlag flag)
{
    GnomeScanPreviewPlugin *gspp = GNOME_SCAN_PREVIEW_PLUGIN (gsppa);
#define ct	gsppa->ct
#define	r	gsppa->selection
#define	cursor	gsppa->cursor
#define	at	gsppa->anchor_type
    at = gsppa_get_anchor_type(point, &r, &gsppa->window);
    
    switch (at) {
        case GTK_ANCHOR_CENTER:
            at = flag & GSPPA_MOVE ? at : GS_ANCHOR_OUTSIDE;
        break;
        case GS_ANCHOR_NONE:
            at = flag & GSPPA_CREATE ? at : GS_ANCHOR_OUTSIDE;
        break;
        default:
            at = flag & GSPPA_RESIZE ? at : GTK_ANCHOR_CENTER;
        break;
    }
    
    ct = gsppa_get_cursor_for_anchor (at);
    
    if (!cursor || cursor->type != ct) {
        if (cursor) {
            gdk_cursor_unref (cursor);
        }
        cursor = gdk_cursor_new (ct);
        gdk_window_set_cursor (gspp->preview_area->window,
                               cursor);
    }
#undef	cursor
#undef	r
#undef	ct
}

static inline CursorFlag
gsppa_get_cursor_flag_for_paper_size (GtkPaperSize *ps)
{
    return (g_str_equal (gtk_paper_size_get_name (ps), "maximal") ? 0 : GSPPA_MOVE)
        | (g_str_equal (gtk_paper_size_get_name (ps), "manual") ? GSPPA_CREATE | GSPPA_RESIZE : 0);
}


/* ACTION (moving, resizeing, creating) */
/* This part handle user events and drawing (here are optimization) */

/* draw the selected area */
static void
gnome_scan_preview_plugin_area_draw_buffer (GnomeScanPreviewPlugin *gspp,
                                            GtkWidget *gspa,
                                            cairo_t *cr)
{
    GnomeScanPreviewPluginArea *gsppa = GNOME_SCAN_PREVIEW_PLUGIN_AREA (gspp);
    GtkStateType state = GTK_WIDGET_STATE (gspa);
    GtkStyle *style = gspa->style;
    gdouble dashes[] = {
        style->ythickness * 4.,
        style->ythickness * 4.
    };
    
#define	m	gsppa->window
#define	r	gsppa->selection
    
    /* fade unselected area */
        cairo_move_to (cr, 0, 0);
    cairo_line_to (cr, m.width, 0);
    cairo_line_to (cr, m.width, m.height);
    cairo_line_to (cr, 0, m.height);
    cairo_line_to (cr, 0, r.y + 1);
    cairo_line_to (cr, r.x + 1, r.y + 1);
    cairo_move_to (cr, r.x + 1, r.y + 1);
    cairo_rel_line_to (cr, 0, r.height - 2);
    cairo_rel_line_to (cr, r.width - 2, 0);
    cairo_rel_line_to (cr, 0, -(r.height - 2));
    cairo_line_to (cr, 0, r.y + 1);
    cairo_line_to (cr, 0, 0);
    cairo_close_path (cr);
    cairo_clip (cr);
    
    gdk_cairo_set_source_color (cr, &style->base[state]);
    cairo_paint_with_alpha (cr, .5);
    
    /* dashed rectangle */
        gdk_cairo_rectangle (cr, &r);
    gdk_cairo_set_source_color (cr, &style->text[state]);
    cairo_set_line_width (cr, style->xthickness);
    cairo_set_dash (cr, dashes, G_N_ELEMENTS (dashes), 0.);
    cairo_stroke (cr);
    
#undef	m
#undef	r
}

static gboolean
gsppa_motion_notify_event (GtkWidget *wid, GdkEventMotion *event, GnomeScanPreviewPluginArea *gsppa)
{
#define	r	gsppa->selection
#define	o	gsppa->opposite_point
#define	w	gsppa->window
#define	gspp	GNOME_SCAN_PREVIEW_PLUGIN (gsppa)
    gint x, y, t;
    GdkRectangle ir, or = r;	/* store ol roi */
        
        if (event->is_hint) {
            gdk_window_get_pointer(wid->window, &x, &y, NULL);
        }
    else {
        x = event->x;
        y = event->y;
    }
    
    /* if button 1 is clicked */
        if (event->state & GDK_BUTTON1_MASK) {
            
            /* compute new roi depending on cursor position and action (defined by cursor type) */
                switch (gsppa->ct) {
                    case GDK_CROSSHAIR:	/* create a new ROI */
                        
                        r.x = MIN (o.x, x);
                    r.y = MIN (o.y, y);
                    r.width = MAX (o.x, x) - r.x;
                    r.height = MAX (o.y, y) - r.y;
                    break;
                    case GDK_FLEUR: /* move roi (denying moving outside) */
                        t = r.x + x - o.x;
                    r.x = 0 <= t && t <= w.width - r.width ? t : r.x;
                    t = r.y + y - o.y;
                    r.y = 0 <= t && t <= w.height - r.height ? t : r.y;
                    o.x = x;
                    o.y = y;
                    break;
                    
                    case GDK_TOP_LEFT_CORNER:
                        r.x = MIN (o.x, x);
                    r.width = ABS (o.x - x);
                    /* continue */
                        case GDK_TOP_SIDE:
                        r.y = MIN (o.y, y);
                    r.height = ABS (o.y - y);
                    break;
                    case GDK_TOP_RIGHT_CORNER:
                        r.x = MIN (o.x, x);
                    r.y = MIN (o.y, y);
                    r.width = ABS (o.x - x);
                    r.height = ABS (o.y - y);
                    break;
                    
                    case GDK_BOTTOM_RIGHT_CORNER:
                        r.x = MIN (o.x, x);
                    r.width = ABS (x - o.x);
                    /* continue */
                        case GDK_BOTTOM_SIDE:
                        r.y = MIN (o.y, y);
                    r.height = ABS (y - o.y);
                    break;
                    case GDK_BOTTOM_LEFT_CORNER:
                        r.x = MIN (o.x, x);
                    r.y = MIN (o.y, y);
                    r.width = ABS (o.x - x);
                    r.height = ABS (o.y - y);
                    break;
                    
                    case GDK_RIGHT_SIDE:
                        r.x = MIN (o.x, x);
                    r.width = ABS (x - o.x);
                    break;
                    case GDK_LEFT_SIDE:
                        r.x = MIN (o.x, x);
                    r.width = ABS (o.x - x);
                    break;
                    default:
                        break;
                }
            
            r.width = MIN (r.width, w.width - r.x);
            r.height = MIN (r.height, w.height - r.y);
            
            /* TODO : optimize even more ir per operation */
                gdk_rectangle_union (&r, &or, &ir);
            ir.x--;
            ir.y--;
            ir.width+=2;
            ir.height+=2;
            gnome_scan_preview_area_update (GNOME_SCAN_PREVIEW_AREA (gspp->preview_area),
                                            &ir);
            
        }
    else {
        GdkPoint point = {x, y};
        gsppa_set_cursor (gsppa, &point, gsppa->cf);
    }
    
    gdk_event_request_motions(event);
    
    return TRUE;
#undef	gspp
#undef	r
#undef	o
#undef	e
#undef	w
}

static gboolean
gsppa_button_press_event (GtkWidget *widget, GdkEventButton *event, GnomeScanPreviewPluginArea *gsppa)
{
#define r	gsppa->selection
#define	o	gsppa->opposite_point
    
    /* set the "opposite_point". opposite_point is the opposite point relative
        to the cursor for depending on the movement */
        switch (gsppa->ct) {
            case GDK_CROSSHAIR:
                case GDK_FLEUR:
                o.x = event->x;
            o.y = event->y;
            break;
            
            case GDK_TOP_SIDE:
                case GDK_TOP_LEFT_CORNER:
                case GDK_LEFT_SIDE:
                o.x = r.x + r.width;
            o.y = r.y + r.height;
            break;
            case GDK_TOP_RIGHT_CORNER:
                o.x = r.x;
            o.y = r.y + r.height;
            break;
            
            case GDK_BOTTOM_SIDE:
                case GDK_BOTTOM_RIGHT_CORNER:
                case GDK_RIGHT_SIDE:
                o.x = r.x;
            o.y = r.y;
            break;
            case GDK_BOTTOM_LEFT_CORNER:
                o.x = r.x + r.width;
            o.y = r.y;
            break;
            default:
                break;
        }
    
    gdk_pointer_grab(widget->window, TRUE,
                     GDK_POINTER_MOTION_MASK |
                     GDK_POINTER_MOTION_HINT_MASK |
                     GDK_BUTTON_RELEASE_MASK,
                     widget->window, NULL,
                     event->time);
#undef	o
#undef	r	
    return FALSE;
}

static gboolean
gsppa_button_release_event (GtkWidget *widget, GdkEventButton *event, GnomeScanPreviewPluginArea *gsppa)
{
#define	r	gsppa->selection
#define	o	gsppa->opposite_point
#define	e	gsppa->end_point
    gdk_pointer_ungrab (event->time);
    
    GdkPoint point = {event->x, event->y};
    gsppa_set_cursor (gsppa, &point, gsppa->cf);
    
    gsppa_send_area (gsppa);
    
    return FALSE;
#undef	r
#undef	o
#undef	e
}


/* AREA HANDLING */

/* store the internal area in the settings at the correct unit */
static void
gsppa_send_area (GnomeScanPreviewPluginArea *gsppa)
{
#define	r	gsppa->selection
#define s	gspp->settings
    GnomeScanPreviewPlugin *gspp = GNOME_SCAN_PREVIEW_PLUGIN (gsppa);
    gdouble res = gnome_scan_preview_area_get_resolution (GNOME_SCAN_PREVIEW_AREA (gspp->preview_area));
    
    /* origin */
    GSPoint *o = gnome_scan_settings_get_pointer (s, "origin");
    o->x = gs_convert_to_mm (r.x, GS_UNIT_PIXEL, res);
    o->y = gs_convert_to_mm(r.y, GS_UNIT_PIXEL, res);
    
    /* paper-size (if manual) */
    GtkPaperSize *p = gnome_scan_settings_get_boxed (s, "paper-size",
                                                     GTK_TYPE_PAPER_SIZE);
    if (g_str_equal (gtk_paper_size_get_name(p), "manual")) {
        g_debug (G_STRLOC ": new manual paper-size : %.2fx%.2f mm",
                 gs_convert (r.width, GS_UNIT_PIXEL, GS_UNIT_MM, res),
                 gs_convert (r.height, GS_UNIT_PIXEL, GS_UNIT_MM, res));
        g_boxed_free (GTK_TYPE_PAPER_SIZE, p);
		/* translator: Manual is the name of user defined paper size. */
        p = gtk_paper_size_new_custom ("manual", _("Manual"),
                                       gs_convert (MIN(r.width, r.height), GS_UNIT_PIXEL, GS_UNIT_MM, res),
                                       gs_convert (MAX(r.width, r.height), GS_UNIT_PIXEL, GS_UNIT_MM, res),
                                       GTK_UNIT_MM);
		/* update orientation according to area extent */
		GtkPageOrientation or = r.width > r.height ? GTK_PAGE_ORIENTATION_LANDSCAPE : GTK_PAGE_ORIENTATION_PORTRAIT;
		gnome_scan_settings_set_enum (s, "page-orientation",
									  GTK_TYPE_PAGE_ORIENTATION, or);
    }
    else if (g_str_equal (gtk_paper_size_get_name (p), "maximal")) {
		/* set origin to (0,0) for maximal paper-size */
        o->x = o->y = 0.;
    }
    
    gnome_scan_settings_set_pointer (s, "origin", o);
    gnome_scan_settings_set_boxed (s, "paper-size", GTK_TYPE_PAPER_SIZE, p);
    gnome_scan_preview_plugin_area_changed (gspp);
    g_boxed_free (GTK_TYPE_PAPER_SIZE, p);
    
#undef	s
#undef	r
}


#define	connect(s)	gsppa->s##_handler = g_signal_connect (gspp->preview_area, #s, G_CALLBACK (gsppa_##s), gsppa)

static void
gsppa_set_roi (GnomeScanPreviewPlugin *gspp)
{
    GSPoint *p;
    GtkPaperSize *ps;
    GtkPageOrientation o;
    gint tmp;
    
    GnomeScanPreviewPluginArea *gsppa = GNOME_SCAN_PREVIEW_PLUGIN_AREA (gspp);
    gdouble res = gnome_scan_preview_area_get_resolution (GNOME_SCAN_PREVIEW_AREA (gspp->preview_area));
#define	m	gsppa->window
#define	r	gsppa->selection
    
    p = gnome_scan_settings_get_pointer (gspp->settings, "origin");
    o = gnome_scan_settings_get_enum(gspp->settings, "page-orientation",
                                     GTK_TYPE_PAGE_ORIENTATION);
    if (!p) {
        p = g_new0 (GSPoint, 1);
        gnome_scan_settings_set_pointer (gspp->settings, "origin", p);
    }
    
    r.x = gs_convert_from_mm (p->x, GS_UNIT_PIXEL, res);
    r.y = gs_convert_from_mm (p->y, GS_UNIT_PIXEL, res);
    
    ps = gnome_scan_settings_get_boxed (gspp->settings, "paper-size",
                                        GTK_TYPE_PAPER_SIZE);
    
    r.width = (gint) gs_convert_from_mm (gtk_paper_size_get_width (ps, GTK_UNIT_MM), GS_UNIT_PIXEL, res);
    r.height = (gint) gs_convert_from_mm (gtk_paper_size_get_height (ps, GTK_UNIT_MM), GS_UNIT_PIXEL, res);
    
    switch(o) {
        case GTK_PAGE_ORIENTATION_LANDSCAPE:
        case GTK_PAGE_ORIENTATION_REVERSE_LANDSCAPE:
            tmp = r.width;
            r.width = MAX(r.width, r.height);
            r.height = MIN(tmp, r.height);
            break;
        case GTK_PAGE_ORIENTATION_PORTRAIT:
        case GTK_PAGE_ORIENTATION_REVERSE_PORTRAIT:
            tmp = r.width;
            r.width = MIN(r.width, r.height);
            r.height = MAX(tmp, r.height);
            break;
    }
    
    gsppa->cf = gsppa_get_cursor_flag_for_paper_size (ps);
    
    gnome_scan_preview_area_get_size (GNOME_SCAN_PREVIEW_AREA (gspp->preview_area),
                                      &m.width,
                                      &m.height);
    
#undef	m
#undef	r
}

static void
gnome_scan_preview_plugin_area_init (GnomeScanPreviewPlugin *gspp)
{
    GnomeScanPreviewPluginArea *gsppa = GNOME_SCAN_PREVIEW_PLUGIN_AREA (gspp);
    gtk_widget_add_events (GTK_WIDGET (gspp->preview_area),
                           GDK_POINTER_MOTION_MASK |
                           GDK_POINTER_MOTION_HINT_MASK |
                           GDK_BUTTON_PRESS_MASK);
    connect(button_press_event);
    connect(button_release_event);
    connect(motion_notify_event);
}

#undef	connect
#define disconnect(s)	g_signal_handler_disconnect (gspp->preview_area, gsppa->s##_handler)

static void
gnome_scan_preview_plugin_area_finalize	(GnomeScanPreviewPlugin *gspp)
{
    GnomeScanPreviewPluginArea *gsppa = GNOME_SCAN_PREVIEW_PLUGIN_AREA (gspp);
    gtk_widget_destroy (gsppa->button);
    disconnect(button_press_event);
    disconnect(button_release_event);
    disconnect(motion_notify_event);
}

#undef	disconnect


#ifndef	NO_BUTTON
static void
gsppa_select_all_clicked (GtkButton *button, GnomeScanPreviewPluginArea *gsppa)
{
#define	m	gsppa->window
#define	r	gsppa->selection
    GnomeScanPreviewPlugin *gspp = GNOME_SCAN_PREVIEW_PLUGIN (gsppa);
    r.x = 0;
    r.y = 0;
    r.width = m.width;
    r.height = m.height;
    gnome_scan_preview_area_update (GNOME_SCAN_PREVIEW_AREA (gspp->preview_area), NULL);
    gsppa_send_area (gsppa);
#undef	m
#undef	r
}
#endif

static void
gnome_scan_preview_plugin_area_build_ui	(GnomeScanPreviewPlugin *gspp, GtkBox *box)
{
    GnomeScanPreviewPluginArea *gsppa = GNOME_SCAN_PREVIEW_PLUGIN_AREA (gspp);
    gsppa->button = gtk_button_new_with_mnemonic (_("Select _All"));
    g_signal_connect (gsppa->button, "clicked",
                      G_CALLBACK (gsppa_select_all_clicked),
                      gsppa);
    gtk_container_add (GTK_CONTAINER (box), gsppa->button);
}

static void
gnome_scan_preview_plugin_area_changed (GnomeScanPreviewPlugin *gspp)
{
    gsppa_set_roi (gspp);
}

