/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace@gnome.org>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <glib/gi18n.h>
#include <gnome-scan-module.h>
#include <gnome-scan-param-specs.h>
#include <gtk/gtk.h>
#include <gtk/gtkprintjob.h>
#include "flegita-sink.h"
#include "flegita-pspec.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), FLEGITA_TYPE_SINK, FlegitaSinkPrivate))

typedef struct _FlegitaSinkPrivate FlegitaSinkPrivate;

struct _FlegitaSinkPrivate
{
  /* pspecs */
  GParamSpec		*action_spec;
  GParamSpec		*output_filename_spec;
  GParamSpec		*png_compression_spec;

  /* actions */
  FlegitaAction action;

  /* save */
  GeglNode		*output;
  GeglProcessor	*processor;
  gchar			*filename;
  guint			png_compression;

  /* print */
  GtkPrintJob	*job;
  cairo_t		*cr;
};

static GnomeScanSinkClass* parent_class = NULL;

static void		fs_configure		(GnomeScanPlugin *plugin,
						 GnomeScanSettings *settings);
static GList*		fs_get_child_nodes	(GnomeScanPlugin *plugin,
						 GeglNode *root);
static void		fs_configure_frame	(GnomeScanPlugin *plugin);
static gboolean		fs_work			(GnomeScanPlugin *plugin,
						 gdouble *progress);
static void		fs_end_frame		(GnomeScanPlugin *plugin);
static void		fs_end_scan		(GnomeScanPlugin *plugin);
#if 0
static void		fs_draw_page		(GtkPrintOperation *op,
						 GtkPrintContext *context,
						 gint page_no,
						 FlegitaSink *sink);
#endif

G_DEFINE_TYPE (FlegitaSink, flegita_sink, GNOME_TYPE_SCAN_SINK)

GS_DEFINE_QUARK(flegita_group_png_options, N_("PNG Options"))
#define	FLEGITA_PARAM_GROUP_PNG_OPTIONS	(flegita_group_png_options_quark())

static void
flegita_sink_init (FlegitaSink *object)
{
  FlegitaSinkPrivate *priv = GET_PRIVATE (object);
  GParamSpec *pspec;
  GSList *formats = NULL;
  GValue *min, *step, *max, *val;
	
  static gchar *png_mime[] = {
    "image/png",
    NULL
  };
  static gchar *png_exts[] = {
    "png",
    NULL
  };
	
  /* FORMAT */
	formats = g_slist_append (formats,
							  gnome_scan_format_new ("png",
													 GETTEXT_PACKAGE,
													 N_("PNG picture"),
													 png_mime,
													 png_exts));
	
	pspec = priv->output_filename_spec = 
		flegita_param_spec_output_filename ("output-filename",
											N_("File"),
											N_("Output filename"),
											GS_PARAM_GROUP_SINK_FRONT,
											formats,
											G_PARAM_WRITABLE);
  gs_param_spec_set_domain (pspec, GETTEXT_PACKAGE);
  gs_param_spec_set_unit (pspec, GS_UNIT_NONE);
  gnome_scan_plugin_params_add (GNOME_SCAN_PLUGIN (object),
				pspec);
	
  /* PNG COMPRESSION */
#define flegita_g_value_new_int(var, val)	\
  var = g_new0(GValue,1);			\
  g_value_init (var, G_TYPE_INT);		\
  g_value_set_int (var, val); 
	
  flegita_g_value_new_int(min, 0);
  flegita_g_value_new_int(step, 1);
  flegita_g_value_new_int(max, 9);
  flegita_g_value_new_int(val, 9);
#undef flegita_g_value_new_int
	
  pspec = priv->png_compression_spec = 
    gs_param_spec_range("png-compression-level", N_("PNG Compression Level"),
						/* translator: this explain the PNG compression level */
						N_("Higher level means lower file size, but takes more "
						   "time to save"),
			FLEGITA_PARAM_GROUP_PNG_OPTIONS,
			min, max, step, val, G_PARAM_WRITABLE);
  gs_param_spec_set_domain (pspec, GETTEXT_PACKAGE);
  gs_param_spec_set_unit (pspec, GS_UNIT_NONE);
  gnome_scan_plugin_params_add (GNOME_SCAN_PLUGIN (object),
				pspec);

}

static void
flegita_sink_finalize (GObject *object)
{
  /* TODO: Add deinitalization code here */

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
flegita_sink_class_init (FlegitaSinkClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  parent_class = GNOME_SCAN_SINK_CLASS (g_type_class_peek_parent (klass));
  GnomeScanPluginClass *plugin_class = GNOME_SCAN_PLUGIN_CLASS (klass);
	
  g_type_class_add_private (object_class, sizeof (FlegitaSinkPrivate));
  plugin_class->configure	= fs_configure;
  plugin_class->get_child_nodes	= fs_get_child_nodes;
  plugin_class->configure_frame	= fs_configure_frame;
  plugin_class->work		= fs_work;
  plugin_class->end_frame	= fs_end_frame;
  plugin_class->end_scan	= fs_end_scan;

  object_class->finalize = flegita_sink_finalize;
}

GnomeScanSink*
flegita_sink_new ()
{
  GObject *object = g_object_new (FLEGITA_TYPE_SINK,
				  "name", "flegita",
				  "blurb", _("Various common action on scan output "
					     "not related to specific software"),
				  NULL);
  return GNOME_SCAN_SINK (object);
}


/* INTERNALS */

static gchar*
fs_get_filename (gchar *filename,
		 gint count,
		 gchar *suffix)
{
  if (count) {
    filename = g_strdup_printf ("%s-%i.%s",
				filename,
				count,
				suffix);
  }
  else {
    filename = g_strdup_printf ("%s.%s",
				filename,
				suffix);
  }

  return filename;
}

/* find the next file no available, starting from count. */
static gint
fs_get_next_file_no (gchar *filename,
		     gint count,
		     gchar *suffix)
{
  gchar *tmpf = NULL;
  gboolean exists = FALSE;

  do {
    tmpf = fs_get_filename (filename, count, suffix);
    exists = g_file_test (tmpf, G_FILE_TEST_EXISTS);
    g_free (tmpf);
  } while (exists && ++count);

  return count;
}

 
static void
fs_configure (GnomeScanPlugin *plugin,
	      GnomeScanSettings *settings)
{
  FlegitaSink *sink = FLEGITA_SINK (plugin);
  FlegitaSinkPrivate *priv = GET_PRIVATE (sink);

#if 0
  priv->action = 	gnome_scan_settings_get_enum (settings,
					       "flegita-action",
					       FLEGITA_TYPE_ACTION);
#else
  priv->action = 	FLEGITA_SAVE;
#endif
	
  switch (priv->action) {
  case FLEGITA_SAVE:
    /* enable SAVE options */
    priv->output_filename_spec->flags = G_PARAM_WRITABLE;
    gnome_scan_plugin_params_changed (plugin,
				      priv->output_filename_spec);

    priv->png_compression_spec->flags = G_PARAM_WRITABLE;
    gnome_scan_plugin_params_changed (plugin,
				      priv->png_compression_spec);

    sink->filename = gnome_scan_settings_get_string (settings, "output-filename");
    sink->suffix = NULL;
    gint i;
	
    for (i = strlen (sink->filename) ; i ; i--) {
      if (sink->filename[i] != '.') {
	sink->suffix = g_strdup_printf ("%c%s",
					sink->filename[i],
					sink->suffix);
      }
      else {
	sink->filename[i] = '\0';
	break;
      }
    }
    sink->count = fs_get_next_file_no (sink->filename, 0, sink->suffix);
    priv->png_compression = gnome_scan_settings_get_int (settings, "png-compression-level");
    break;
#ifdef	PRINT
  case FLEGITA_PRINT:
    /* disable save options */
    priv->output_filename_spec->flags = 0;
    gnome_scan_plugin_params_changed (plugin,
				      priv->output_filename_spec);

    priv->png_compression_spec->flags = 0;
    gnome_scan_plugin_params_changed (plugin,
				      priv->png_compression_spec);

    /*     sink->filename = "/tmp/flegita-print"; */
    /*     sink->suffix = "png"; */

    GtkPageSetup* page_setup = gtk_page_setup_new ();
    gtk_page_setup_set_orientation (page_setup,
				    gnome_scan_settings_get_enum (settings,
								  "page-orientation",
								  GTK_TYPE_PAGE_ORIENTATION));
    /* paper size */
    GtkPaperSize *paper_size;
    gchar *papername = gnome_scan_settings_get_string (settings,
						       "paper-size");
    if (papername && strlen(papername)) {
      paper_size = gtk_paper_size_new (papername);
    }
    else {
      GeglRectangle *area = gnome_scan_settings_get_pointer (settings,
							     "area");
      if (!area)
	break;

      paper_size = gtk_paper_size_new_custom ("custom",
					      _("Custom"),
					      area->width,
					      area->height,
					      GTK_UNIT_MM);
    }
    gtk_page_setup_set_paper_size (page_setup, paper_size);

    gtk_page_setup_set_top_margin	(page_setup, 0., GTK_UNIT_MM);
    gtk_page_setup_set_bottom_margin	(page_setup, 0., GTK_UNIT_MM);
    gtk_page_setup_set_left_margin	(page_setup, 0., GTK_UNIT_MM);
    gtk_page_setup_set_right_margin	(page_setup, 0., GTK_UNIT_MM);

    GtkPrintSettings *psettings =
      GTK_PRINT_SETTINGS (gnome_scan_settings_get_object (settings,
							  "print-settings"));
    if (!psettings)
      break;

    GtkPrinter* printer =
      GTK_PRINTER (gnome_scan_settings_get_object (settings,
						   "printer"));
    if (!printer)
      break;

    gtk_print_settings_set_printer (psettings,
				    gtk_printer_get_name (printer));

    if (priv->job) {
      g_object_unref (priv->job);
    }

    priv->job = gtk_print_job_new (_("Scan to printer"),
				   printer,
				   psettings,
				   page_setup);


    gtk_print_job_set_track_print_status (priv->job, TRUE);

    break;  
#endif
  }
}

static GList*
fs_get_child_nodes	(GnomeScanPlugin *plugin,
			 GeglNode *root)
{
  FlegitaSinkPrivate *priv = GET_PRIVATE (plugin);
  GList *list = NULL;
  priv->output = gegl_node_new_child (root,
				      "operation", "png-save",
				      NULL);

  list = g_list_append (list, priv->output);

  return list;
}

static void
fs_configure_frame (GnomeScanPlugin *plugin)
{
  FlegitaSink *sink = FLEGITA_SINK (plugin);
  FlegitaSinkPrivate *priv = GET_PRIVATE (sink);
  gchar *filename;
	
  switch (priv->action) {
  case FLEGITA_SAVE:
    filename = fs_get_filename (sink->filename, sink->count, sink->suffix);
    g_debug (G_STRLOC ": writing to %s", filename);
    gegl_node_set (priv->output,
		   "path", filename,
		   "compression", priv->png_compression,
		   NULL);
    priv->filename	 = filename;
    break;
#ifdef	PRINT
  case FLEGITA_PRINT:
    break;
#endif
  }
}

static gboolean
fs_work (GnomeScanPlugin *plugin, gdouble *progress)
{
  FlegitaSinkPrivate *priv = GET_PRIVATE (plugin);
#ifdef	PRINT
  GError *error = NULL;
  cairo_surface_t *cs;
#endif

  switch (priv->action) {
  case FLEGITA_SAVE:
    *progress = 1.;		/* all is done using gegl */
    break;
#ifdef PRINT
  case FLEGITA_PRINT:
    cs = gtk_print_job_get_surface (priv->job,
				    &error);
    cairo_t *cr = cairo_create (cs);
    GeglRectangle *extent = gegl_buffer_extent (priv->buffer);
    Babl * format = babl_format("RGB u8");
    guchar *data = g_new0 (guchar, (extent->width * extent->height * 3));
    gegl_buffer_get (plugin->buffer,  1., extent,
		     format,
		     data, GEGL_AUTO_ROWSTRIDE);
    cs = cairo_image_surface_create_for_data (data,
					      CAIRO_FORMAT_RGB24,
					      extent->width,
					      extent->height,
					      extent->width * 3);
    cairo_set_source_surface (cr, cs, 0., 0.);
    cairo_paint (cr);
    cairo_show_page (cr);
    cairo_destroy (cr);
    return FALSE;
    break;
#endif
  }

  return FALSE;
}

static void
fs_end_frame (GnomeScanPlugin *plugin)
{
  FlegitaSinkPrivate *priv = GET_PRIVATE (plugin);
  FlegitaSink *sink = FLEGITA_SINK (plugin);
  g_free (priv->filename);

  switch (priv->action) {
  case FLEGITA_SAVE:
    sink->count = fs_get_next_file_no (sink->filename,
				       sink->count+1,
				       sink->suffix);

    g_object_unref(priv->output);
    /* needs to unref one more time. Maybe a bug in GeglOp buffer ? */
    break;
#ifdef	PRINT
  case FLEGITA_PRINT:
    break;
#endif
  }
}

#ifdef	PRINT
static void complete (GtkPrintJob *job, FlegitaSink *sink, GError *error)
{
  g_debug (G_STRLOC ": job complete %i",
	   gtk_print_job_get_status (job));
  if (error) {
    g_debug (G_STRLOC ": an error occured : %s",
	     error->message);
  }
}
#endif

static void
fs_end_scan (GnomeScanPlugin *plugin)
{
  FlegitaSinkPrivate *priv = GET_PRIVATE (plugin);

  switch (priv->action) {
  case FLEGITA_SAVE:
    break;
#ifdef	PRINT
  case FLEGITA_PRINT:
    gtk_print_job_send (priv->job,
			(GtkPrintJobCompleteFunc) complete,
			sink,
			NULL);
    break;
#endif
  }
}

#if 0
static void
fs_draw_page (GtkPrintOperation *op,
	      GtkPrintContext *context,
	      gint page_no,
	      FlegitaSink *sink)
{
  gchar *filename = page_no > 0 ?
    g_strdup_printf ("/tmp/flegita-print-%i.png", page_no) :
    g_strdup ("/tmp/flegita-print.png");
  cairo_surface_t *cs = cairo_image_surface_create_from_png (filename);
  cairo_t *cr = gtk_print_context_get_cairo_context (context);
  cairo_set_source_surface (cr, cs, 0., 0.);
  cairo_paint (cr);

  g_debug (G_STRLOC ": imprimer page %i (%s)", page_no, filename);

  cairo_surface_destroy (cs);
  g_free (filename);
}
#endif
