// -*- c++ -*-
#ifndef _GLIBMM_FILEUTILS_H
#define _GLIBMM_FILEUTILS_H

/* $Id: fileutils.h,v 1.2 2002/03/16 14:28:04 daniel Exp $ */

/* fileutils.h
 *
 * Copyright (C) 2002 The gtkmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

extern "C" { typedef struct _GDir GDir; }

#include <glib/gfileutils.h>

#include <iterator>
#include <string>

#include <glibmmconfig.h>
#include <glibmm/error.h>

GTKMM_USING_STD(input_iterator_tag)
GTKMM_USING_STD(string)


namespace Glib
{

/** @defgroup FileUtils File Utilities
 * Various file-related classes and functions.
 */

/** Exception class for file-related errors.
 * @ingroup FileUtils
 */
class FileError : public Glib::Error
{
public:
  explicit FileError(GError* gobject);
  GFileError code() const;
};


class Dir;

/** The iterator type of Glib::Dir.
 * @ingroup FileUtils
 */
class DirIterator
{
public:
  typedef std::input_iterator_tag   iterator_category;
  typedef void                      value_type;
  typedef void                      difference_type;
  typedef void                      reference;
  typedef void                      pointer;

  DirIterator();

#ifndef DOXYGEN_SHOULD_SKIP_THIS
  DirIterator(GDir* gobject, const char* current);
#endif

  std::string  operator*() const;
  DirIterator& operator++();
  void         operator++(int);

  bool operator==(const DirIterator& rhs) const;
  bool operator!=(const DirIterator& rhs) const;

private:
  GDir*       gobject_;
  const char* current_;
};


/** Utility class representing an open directory.
 * @ingroup FileUtils
 * It's highly recommended to use the iterator interface.  With iterators,
 * reading an entire directory into a STL container is really easy:
 * @code
 * Glib::Dir dir (directory_path);
 * std::list<std::string> entries (dir.begin(), dir.end());
 * @endcode
 * @note The encoding of the directory entries isn't necessarily UTF-8.
 * Use Glib::filename_to_utf8() if you need to display them.
 */
class Dir
{
public:
  typedef DirIterator iterator;
  typedef DirIterator const_iterator;

  /** Opens a directory for reading. The names of the files in the
   * directory can then be retrieved using read_name().
   * @param path The path to the directory you are interested in.
   * @param flags Currently must be set to 0. Reserved for future use.
   * @throw FileError see Glib::FileError
   */
  explicit Dir(const std::string& path, unsigned int flags = 0);

#ifndef DOXYGEN_SHOULD_SKIP_THIS
  explicit Dir(GDir* gobject);
#endif

  /** Closes the directory and deallocates all related resources.
   */
  ~Dir();

  /** Retrieves the name of the next entry in the directory.
   * The '.' and '..' entries are omitted.
   * @return The entry's name or <tt>""</tt> if there are no more entries.
   * @see begin(), end()
   */
  std::string read_name();

  /** Resets the directory.  The next call to
   * read_name() will return the first entry again.
   */
  void rewind();

  /** Closes the directory and deallocates all related resources.
   * Note that close() is implicitely called by ~Dir().  Thus you don't
   * need to call close() yourself unless you want to close the directory
   * before the destructor runs.
   */
  void close();

  /** Get the begin of an input iterator sequence.
   * @return An input iterator pointing to the first directory entry.
   */
  DirIterator begin();

  /** Get the end of an input iterator sequence.
   * @return An input iterator pointing behind the last directory entry.
   */
  DirIterator end();

private:
  GDir* gobject_;

  // noncopyable
  Dir(const Dir&);
  Dir& operator=(const Dir&);
};


/** Returns @c true if any of the tests in the bitfield @a test are true.
 * @ingroup FileUtils
 * For example, <tt>(G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR)</tt> will return
 * @c true if the file exists; the check whether it's a directory doesn't
 * matter since the existence test is true. With the current set of available
 * tests, there's no point passing in more than one test at a time.
 *
 * Apart from @c G_FILE_TEST_IS_SYMLINK all tests follow symbolic links, so
 * for a symbolic link to a regular file file_test() will return @c true for
 * both @c G_FILE_TEST_IS_SYMLINK and @c G_FILE_TEST_IS_REGULAR.
 *
 * @note For a dangling symbolic link file_test() will return @c true for
 * @c G_FILE_TEST_IS_SYMLINK and @c false for all other flags.
 *
 * @param filename A filename to test.
 * @param test Bitfield of @c GFileTest flags.
 * @return Whether a test was true.
 */
bool file_test(const std::string& filename, GFileTest test);

/** Opens a temporary file.
 * @ingroup FileUtils
 * See the %mkstemp() documentation on most UNIX-like systems. This is a
 * portability wrapper, which simply calls %mkstemp() on systems that have
 * it, and implements it in GLib otherwise.
 * @param filename_template A string that should match the rules for
 *   %mkstemp(), i.e. end in <tt>"XXXXXX"</tt>. The <tt>X</tt> string
 *   will be modified to form the name of a file that didn't exist.
 * @return A file handle (as from open()) to the file opened for reading
 *   and writing. The file is opened in binary mode on platforms where there
 *   is a difference. The file handle should be closed with close(). In
 *   case of errors, <tt>-1</tt> is returned.
 */
int mkstemp(std::string& filename_template);

} // namespace Glib


/** Convenience operator.
 * @ingroup FileUtils
 * @return <tt>static_cast<GFileTest>(static_cast<int>(lhs) | static_cast<int>(rhs))</tt>
 */
inline
GFileTest operator|(GFileTest lhs, GFileTest rhs)
{
  return static_cast<GFileTest>(static_cast<int>(lhs) | static_cast<int>(rhs));
}


#endif /* _GLIBMM_FILEUTILS_H */

