// -*- c++ -*-
#ifndef _GLIBMM_MISCUTILS_H
#define _GLIBMM_MISCUTILS_H

/* $Id: miscutils.h,v 1.2 2002/04/09 13:26:04 daniel Exp $ */

/* miscutils.h
 *
 * Copyright (C) 2002 The gtkmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

extern "C" { typedef struct _GDir GDir; }

#include <string>
#include <glibmmconfig.h>
#include <glibmm/arrayhandle.h>

GTKMM_USING_STD(string)


namespace Glib
{

/** @defgroup MiscUtils Miscellaneous Utility Functions
 * Miscellaneous Utility Functions -- a selection of portable utility functions.
 * @{
 */

/** Gets the name of the program.
 * If you are using GDK or GTK+ the program name is set in gdk_init(), which
 * is called by gtk_init().  The program name is found by taking the last
 * component of <tt>argv[0]</tt>.
 * @return The name of the program.
 */
std::string get_prgname();

/** Sets the name of the program.
 * @param prgname The name of the program.
 */
void set_prgname(const std::string& prgname);

/** Returns an environment variable.
 * @param variable The environment variable to get.
 * @param found Output argument, set to @c true if the environment
 *   variable has been found, otherwise to @c false.
 * @return The value of the environment variable, or <tt>""</tt> if not found.
 */
std::string getenv(const std::string& variable, bool& found);

/** Returns an environment variable.
 * @param variable The environment variable to get.
 * @return The value of the environment variable, or <tt>""</tt> if not found.
 */
std::string getenv(const std::string& variable);

/** Gets the user name of the current user.
 * @return The name of the current user.
 */
std::string get_user_name();

/** Gets the real name of the user.
 * This usually comes from the user's entry in the @c passwd file.
 * @return The user's real name.
 */
std::string get_real_name();

/** Gets the current user's home directory.
 * @return The current user's home directory.
 */
std::string get_home_dir();

/** Gets the directory to use for temporary files.
 * This is found from inspecting the environment variables @c TMPDIR, @c TMP,
 * and @c TEMP in that order.  If none of those are defined <tt>"/tmp"</tt>
 * is returned on UNIX and <tt>"C:\"</tt> on Windows.
 * @return The directory to use for temporary files.
 */
std::string get_tmp_dir();

/** Gets the current directory.
 * @return The current directory.
 */
std::string get_current_dir();

/** Returns @c true if the given @a filename is an absolute file name,
 * i.e.\ it contains a full path from the root directory such as '/usr/local'
 * on UNIX or 'C:\\windows' on Windows systems.
 * @param filename A file name.
 * @return Whether @a filename is an absolute path.
 */
bool path_is_absolute(const std::string& filename);

/** Returns the remaining part of @a filename after the root component,
 * i.e.\ after the '/' on UNIX or 'C:\\' on Windows.  If @a filename is
 * not an absolute path, <tt>""</tt> will be returned.
 * @param filename A file name.
 * @return The file name without the root component, or <tt>""</tt>.
 */
std::string path_skip_root(const std::string& filename);

/** Gets the name of the file without any leading directory components.
 * @param filename The name of the file.
 * @return The name of the file without any leading directory components.
 */
std::string path_get_basename(const std::string& filename);

/** Gets the directory components of a file name.
 * If the file name has no directory components <tt>"."</tt> is returned.
 * @param filename The name of the file.
 * @return The directory components of the file.
 */
std::string path_get_dirname(const std::string& filename);

/** Creates a filename from a series of elements using the correct
 * separator for filenames.  This function behaves identically to
 * <tt>Glib::build_path(G_DIR_SEPARATOR_S, elements)</tt>.  No attempt is
 * made to force the resulting filename to be an absolute path.  If the first
 * element is a relative path, the result will be a relative path.
 * @param elements A container holding the elements of the path to build.
 *   Any STL compatible container type is accepted.
 * @return The resulting path.
 */
std::string build_filename(const Glib::ArrayHandle<std::string>& elements);

/** Creates a path from a series of elements using @a separator as the
 * separator between elements.  At the boundary between two elements, any
 * trailing occurrences of @a separator in the first element, or leading
 * occurrences of @a separator in the second element are removed and exactly
 * one copy of the separator is inserted.
 * @param separator A string used to separate the elements of the path.
 * @param elements A container holding the elements of the path to build.
 *   Any STL compatible container type is accepted.
 * @return The resulting path.
 */
std::string build_path(const std::string& separator,
                       const Glib::ArrayHandle<std::string>& elements);

/** Locates the first executable named @a program in the user's path, in the
 * same way that execvp() would locate it.  Returns a string with the absolute
 * path name, or <tt>""</tt> if the program is not found in the path.  If
 * @a program is already an absolute path, returns a copy of @a program if
 * @a program exists and is executable, and <tt>""</tt> otherwise.
 *
 * On Windows, if @a program does not have a file type suffix, tries to append
 * the suffixes in the @c PATHEXT environment variable (if that doesn't exist,
 * the suffixes .com, .exe, and .bat) in turn, and then look for the resulting
 * file name in the same way as CreateProcess() would.  This means first in
 * the directory where the program was loaded from, then in the current
 * directory, then in the Windows 32-bit system directory, then in the Windows
 * directory, and finally in the directories in the @c PATH environment
 * variable.  If the program is found, the return value contains the full name
 * including the type suffix.
 *
 * @param program A program name.
 * @result An absolute path, or <tt>""</tt>.
 */
std::string find_program_in_path(const std::string& program);

/** @} group MiscUtils */

} // namespace Glib


#endif /* _GLIBMM_FILEUTILS_H */

