/*
 * Copyright © 2009 Emmanuel Pacaud
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Emmanuel Pacaud <emmanuel@gnome.org>
 */

#include <lsmdebug.h>
#include <lsmattributes.h>
#include <lsmproperties.h>
#include <lsmdomdocument.h>
#include <lsmsvgelement.h>
#include <lsmsvgpatternelement.h>
#include <lsmsvggradientelement.h>
#include <lsmsvgclippathelement.h>
#include <lsmsvgmarkerelement.h>
#include <lsmsvgmaskelement.h>
#include <lsmsvgview.h>
#include <string.h>

static GObjectClass *parent_class;

/* LsmDomNode implementation */

static gboolean
lsm_svg_element_can_append_child (LsmDomNode *self, LsmDomNode *child)
{
	return (LSM_IS_SVG_ELEMENT (child));
}

static gboolean
lsm_svg_element_child_changed (LsmDomNode *parent, LsmDomNode *child)
{
	return TRUE;
}

/* LsmDomElement implementation */

static void
lsm_svg_element_set_attribute (LsmDomElement *self, const char* name, const char *value)
{
	LsmSvgElementClass *s_element_class = LSM_SVG_ELEMENT_GET_CLASS (self);
	LsmSvgElement *s_element = LSM_SVG_ELEMENT (self);

	lsm_debug ("[LsmSvgElement::set_attribute] node = %s, name = %s, value = %s",
		    lsm_dom_node_get_node_name (LSM_DOM_NODE (self)), name, value);

	/* TODO Avoid double hash table lookup */
	if (!lsm_attribute_manager_set_attribute (s_element_class->attribute_manager,
						  self, name, value))
		lsm_svg_property_bag_set_property (&s_element->property_bag, name, value);

	if (g_strcmp0 (name, "id") == 0) {
		LsmDomDocument *document;

		document = lsm_dom_node_get_owner_document (LSM_DOM_NODE (self));
		if (document != NULL)
			lsm_dom_document_register_element (document, LSM_DOM_ELEMENT (self), value);
	}
}

const char *
lsm_svg_element_get_attribute (LsmDomElement *self, const char *name)
{
	LsmSvgElementClass *s_element_class = LSM_SVG_ELEMENT_GET_CLASS(self);
	LsmSvgElement *s_element = LSM_SVG_ELEMENT (self);
	const char *value;

	/* TODO Avoid double hash table lookup */
	value = lsm_attribute_manager_get_attribute (s_element_class->attribute_manager,
						     self, name);
	if (value != NULL)
		return value;

	return lsm_svg_property_bag_get_property (&s_element->property_bag, name);
}

/* LsmSvgElement implementation */

static void
_render (LsmSvgElement *element, LsmSvgView *view)
{
	LsmDomNode *node;

	lsm_debug ("[LsmSvgElement::_render");

	lsm_svg_view_push_group_opacity (view);

	for (node = LSM_DOM_NODE (element)->first_child; node != NULL; node = node->next_sibling)
		if (LSM_IS_SVG_ELEMENT (node))
		    lsm_svg_element_render (LSM_SVG_ELEMENT (node), view);

	lsm_svg_view_pop_group_opacity (view);
}

void
lsm_svg_element_render (LsmSvgElement *element, LsmSvgView *view)
{
	LsmSvgElementClass *element_class;
	const LsmSvgStyle *parent_style;
	LsmSvgStyle *style;

	g_return_if_fail (LSM_IS_SVG_ELEMENT (element));

	parent_style = lsm_svg_view_get_current_style (view);
	style = lsm_svg_style_new_inherited (parent_style, &element->property_bag);

	if (!lsm_svg_matrix_is_identity (&element->transform.matrix))
		lsm_svg_view_push_matrix (view, &element->transform.matrix);

	lsm_svg_view_push_element (view, element);
	lsm_svg_view_push_style (view, style);

	element_class = LSM_SVG_ELEMENT_GET_CLASS (element);
	if (element_class->render != NULL) {
		lsm_debug ("[LsmSvgElement::render] Render %s (%s)",
			    lsm_dom_node_get_node_name (LSM_DOM_NODE (element)),
			    element->id.value != NULL ? element->id.value : "no id");

		element_class->render (element, view);
	}

	lsm_svg_view_pop_style (view);
	lsm_svg_view_pop_element (view);

	if (!lsm_svg_matrix_is_identity (&element->transform.matrix))
		lsm_svg_view_pop_matrix (view);

	lsm_svg_style_unref (style);
}

static void
lsm_svg_element_enable_rendering (LsmSvgElement *element)
{
	LsmSvgElementClass *element_class;

	g_return_if_fail (LSM_IS_SVG_ELEMENT (element));

	element_class = LSM_SVG_ELEMENT_GET_CLASS (element);
	g_return_if_fail (element_class->enable_rendering != NULL);

	element_class->enable_rendering (element);
}

void
lsm_svg_element_force_render (LsmSvgElement *element, LsmSvgView *view)
{
	g_return_if_fail (LSM_IS_SVG_PATTERN_ELEMENT (element) ||
			  LSM_IS_SVG_GRADIENT_ELEMENT (element) ||
			  LSM_IS_SVG_MASK_ELEMENT (element) ||
			  LSM_IS_SVG_CLIP_PATH_ELEMENT (element) ||
			  LSM_IS_SVG_MARKER_ELEMENT (element));

	lsm_svg_element_enable_rendering (element);
	lsm_svg_element_render (element, view);
}

static void
_get_extents (LsmSvgElement *self, LsmSvgView *view, LsmExtents *extents)
{
	LsmDomNode *node;
	gboolean first_child = TRUE;
	LsmExtents element_extents = {0.0, 0.0, 0.0, 0.0};

	lsm_debug ("[LsmSvgGraphic::_graphic_get_extents]");

	for (node = LSM_DOM_NODE (self)->first_child; node != NULL; node = node->next_sibling) {
		if (LSM_IS_SVG_ELEMENT (node)) {
			LsmExtents child_extents;

			lsm_svg_element_get_extents (LSM_SVG_ELEMENT (node), view, &child_extents);

			if (!lsm_svg_matrix_is_identity (&self->transform.matrix))
				lsm_svg_matrix_transform_bounding_box (&self->transform.matrix,
								       &child_extents.x1, &child_extents.y1,
								       &child_extents.x2, &child_extents.y2);
			if (first_child) {
				element_extents = child_extents;
				first_child = FALSE;
			} else {
				element_extents.x1 = MIN (element_extents.x1, child_extents.x1);
				element_extents.y1 = MIN (element_extents.y1, child_extents.y1);
				element_extents.x2 = MAX (element_extents.x2, child_extents.x2);
				element_extents.y2 = MAX (element_extents.y2, child_extents.y2);
			}
		}
	}

	*extents = element_extents;
}

void
lsm_svg_element_get_extents (LsmSvgElement *element, LsmSvgView *view, LsmExtents *extents)
{
	LsmSvgElementClass *element_class;

	g_return_if_fail (LSM_IS_SVG_ELEMENT (element));
	g_return_if_fail (LSM_IS_SVG_VIEW (view));
	g_return_if_fail (extents != NULL);

	element_class = LSM_SVG_ELEMENT_GET_CLASS (element);
	if (element_class->get_extents != NULL) {
		element_class->get_extents (element, view, extents);

		lsm_debug ("LsmSvgElement::get_extents] Exents for '%s' = %g,%g %g,%g",
			   lsm_dom_node_get_node_name (LSM_DOM_NODE (element)),
			   extents->x1, extents->y1, extents->x2, extents->y2);
	} else {
		extents->x1 = 0.0;
		extents->y1 = 0.0;
		extents->x2 = 0.0;
		extents->y2 = 0.0;
	}
}

static void
lsm_svg_element_init (LsmSvgElement *element)
{
	lsm_svg_matrix_init_identity (&element->transform.matrix);
}

static void
lsm_svg_element_finalize (GObject *object)
{
	LsmSvgElementClass *s_element_class = LSM_SVG_ELEMENT_GET_CLASS (object);
	LsmSvgElement *svg_element = LSM_SVG_ELEMENT (object);

	lsm_svg_property_bag_clean (&svg_element->property_bag);
	lsm_attribute_manager_clean_attributes (s_element_class->attribute_manager, svg_element);

	parent_class->finalize (object);
}

/* LsmSvgElement class */

static const LsmSvgMatrix matrix_default =	 { 1.0, 0.0, 0.0, 1.0, 0.0, 0.0, LSM_SVG_MATRIX_FLAGS_IDENTITY};

static const LsmAttributeInfos lsm_svg_attribute_infos[] = {
	{
		.name = "id",
		.trait_class = &lsm_null_trait_class,
		.attribute_offset = offsetof (LsmSvgElement, id)
	},
	{
		.name = "class",
		.trait_class = &lsm_null_trait_class,
		.attribute_offset = offsetof (LsmSvgElement, class_name)
	},
	{
		.name = "transform",
		.attribute_offset = offsetof (LsmSvgElement, transform),
		.trait_class = &lsm_svg_matrix_trait_class,
		.trait_default = &matrix_default
	}
};

static void
lsm_svg_element_class_init (LsmSvgElementClass *s_element_class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (s_element_class);
	LsmDomNodeClass *d_node_class = LSM_DOM_NODE_CLASS (s_element_class);
	LsmDomElementClass *d_element_class = LSM_DOM_ELEMENT_CLASS (s_element_class);

	parent_class = g_type_class_peek_parent (s_element_class);

	object_class->finalize = lsm_svg_element_finalize;

	d_node_class->can_append_child = lsm_svg_element_can_append_child;
	d_node_class->child_changed = lsm_svg_element_child_changed;

	d_element_class->get_attribute = lsm_svg_element_get_attribute;
	d_element_class->set_attribute = lsm_svg_element_set_attribute;

	s_element_class->render = _render;
	s_element_class->get_extents = _get_extents;
	s_element_class->attribute_manager = lsm_attribute_manager_new (G_N_ELEMENTS (lsm_svg_attribute_infos),
									lsm_svg_attribute_infos);
}

G_DEFINE_ABSTRACT_TYPE (LsmSvgElement, lsm_svg_element, LSM_TYPE_DOM_ELEMENT)
