/* valasemanticanalyzer.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valasemanticanalyzer.h"
#include <vala/valasemanticanalyzer.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valaattribute.h>
#include <vala/valastatement.h>
#include <vala/valaarray.h>
#include <vala/valalockable.h>
#include <vala/valatypeparameter.h>
#include <vala/valaliteral.h>
#include <vala/valaenumvalue.h>
#include <vala/valaflagsvalue.h>
#include <vala/valanamespacereference.h>
#include <vala/valapointer.h>
#include <vala/valaenum.h>
#include <vala/valainvokable.h>

struct _ValaSemanticAnalyzerPrivate {
	gboolean _memory_management;
	ValaSymbol* root_symbol;
	ValaSymbol* current_symbol;
	ValaSourceFile* current_source_file;
	ValaTypeReference* current_return_type;
	ValaClass* current_class;
	ValaStruct* current_struct;
	GList* current_using_directives;
	ValaTypeReference* bool_type;
	ValaTypeReference* string_type;
	ValaTypeReference* int_type;
	ValaTypeReference* uint_type;
	ValaTypeReference* ulong_type;
	ValaTypeReference* unichar_type;
	ValaTypeReference* type_type;
	ValaDataType* pointer_type;
	ValaDataType* initially_unowned_type;
	ValaDataType* glist_type;
	ValaDataType* gslist_type;
	ValaDataType* gerror_type;
	gint next_lambda_id;
};
#define VALA_SEMANTIC_ANALYZER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerPrivate))
enum  {
	VALA_SEMANTIC_ANALYZER_DUMMY_PROPERTY,
	VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT
};
static void vala_semantic_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file);
static void vala_semantic_analyzer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_semantic_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static GList* vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface);
static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaDataType* t);
static void vala_semantic_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_semantic_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_semantic_analyzer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb);
static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_semantic_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p);
static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_semantic_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_semantic_analyzer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_semantic_analyzer_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_semantic_analyzer_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n);
static void vala_semantic_analyzer_real_visit_begin_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_semantic_analyzer_real_visit_end_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl);
static void vala_semantic_analyzer_real_visit_begin_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_semantic_analyzer_real_visit_end_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_semantic_analyzer_real_visit_begin_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_semantic_analyzer_real_visit_end_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_semantic_analyzer_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_semantic_analyzer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_semantic_analyzer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_semantic_analyzer_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause);
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt);
static void vala_semantic_analyzer_real_visit_begin_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_end_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr);
static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr);
static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr);
static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr);
static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr);
static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr);
static void vala_semantic_analyzer_real_visit_literal_expression (ValaCodeVisitor* base, ValaLiteralExpression* expr);
static ValaTypeReference* vala_semantic_analyzer_get_static_type_for_node (ValaSemanticAnalyzer* self, ValaCodeNode* node);
static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr);
static ValaDataType* vala_semantic_analyzer_find_parent_type (ValaSemanticAnalyzer* self, ValaSymbol* sym);
static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static gboolean vala_semantic_analyzer_is_type_compatible (ValaSemanticAnalyzer* self, ValaTypeReference* expression_type, ValaTypeReference* expected_type);
static void vala_semantic_analyzer_real_visit_begin_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaSymbol* msym, GList* params, GList* args);
static void vala_semantic_analyzer_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr);
static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr);
static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr);
static void vala_semantic_analyzer_real_visit_end_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr);
static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr);
static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaTypeReference* type);
static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaTypeReference* type);
static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr);
static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr);
static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr);
static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr);
static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr);
static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr);
static ValaTypeReference* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaTypeReference* left_type, ValaTypeReference* right_type);
static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr);
static ValaTypeReference* vala_semantic_analyzer_compute_common_base_type (ValaSemanticAnalyzer* self, GList* types);
static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr);
static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self);
static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self);
static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self);
static void vala_semantic_analyzer_real_visit_begin_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l);
static void vala_semantic_analyzer_real_visit_begin_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static void vala_semantic_analyzer_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_semantic_analyzer_parent_class = NULL;
static void vala_semantic_analyzer_dispose (GObject * obj);


ValaSemanticAnalyzer* vala_semantic_analyzer_new (gboolean manage_memory)
{
	GParameter * __params;
	GParameter * __params_it;
	ValaSemanticAnalyzer * self;
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	(__params_it->name = "memory-management", g_value_init (&__params_it->value, G_TYPE_BOOLEAN), g_value_set_boolean (&__params_it->value, manage_memory), __params_it++);
	self = g_object_newv (VALA_TYPE_SEMANTIC_ANALYZER, __params_it - __params, __params);
	return self;
}


/**
 * Analyze and check code in the specified context.
 *
 * @param context a code context
 */
void vala_semantic_analyzer_analyze (ValaSemanticAnalyzer* self, ValaCodeContext* context)
{
	ValaSymbol* __temp0;
	ValaTypeReference* __temp1;
	ValaSymbol* __temp2;
	ValaTypeReference* __temp3;
	ValaSymbol* __temp4;
	ValaDataType* __temp7;
	ValaDataType* __temp6;
	ValaSymbol* __temp5;
	ValaTypeReference* __temp8;
	ValaSymbol* __temp9;
	ValaTypeReference* __temp10;
	ValaSymbol* __temp11;
	ValaTypeReference* __temp12;
	ValaSymbol* __temp13;
	ValaTypeReference* __temp14;
	ValaSymbol* __temp15;
	ValaSymbol* glib_ns;
	ValaSymbol* __temp31;
	ValaSymbol* __temp30;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	__temp0 = NULL;
	self->priv->root_symbol = (__temp0 = vala_code_context_get_root (context), (self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL))), __temp0);
	__temp1 = NULL;
	self->priv->bool_type = (__temp1 = vala_type_reference_new (), (self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL))), __temp1);
	__temp2 = NULL;
	(vala_type_reference_set_data_type (self->priv->bool_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp2 = vala_symbol_lookup (self->priv->root_symbol, "bool"))))), vala_type_reference_get_data_type (self->priv->bool_type));
	(__temp2 == NULL ? NULL : (__temp2 = (g_object_unref (__temp2), NULL)));
	__temp3 = NULL;
	self->priv->string_type = (__temp3 = vala_type_reference_new (), (self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL))), __temp3);
	__temp4 = NULL;
	(vala_type_reference_set_data_type (self->priv->string_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp4 = vala_symbol_lookup (self->priv->root_symbol, "string"))))), vala_type_reference_get_data_type (self->priv->string_type));
	(__temp4 == NULL ? NULL : (__temp4 = (g_object_unref (__temp4), NULL)));
	__temp7 = NULL;
	__temp6 = NULL;
	__temp5 = NULL;
	self->priv->pointer_type = (__temp7 = (__temp6 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp5 = vala_symbol_lookup (self->priv->root_symbol, "pointer")))), (__temp6 == NULL ? NULL : g_object_ref (__temp6))), (self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL))), __temp7);
	(__temp5 == NULL ? NULL : (__temp5 = (g_object_unref (__temp5), NULL)));
	__temp8 = NULL;
	self->priv->int_type = (__temp8 = vala_type_reference_new (), (self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL))), __temp8);
	__temp9 = NULL;
	(vala_type_reference_set_data_type (self->priv->int_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp9 = vala_symbol_lookup (self->priv->root_symbol, "int"))))), vala_type_reference_get_data_type (self->priv->int_type));
	(__temp9 == NULL ? NULL : (__temp9 = (g_object_unref (__temp9), NULL)));
	__temp10 = NULL;
	self->priv->uint_type = (__temp10 = vala_type_reference_new (), (self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL))), __temp10);
	__temp11 = NULL;
	(vala_type_reference_set_data_type (self->priv->uint_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp11 = vala_symbol_lookup (self->priv->root_symbol, "uint"))))), vala_type_reference_get_data_type (self->priv->uint_type));
	(__temp11 == NULL ? NULL : (__temp11 = (g_object_unref (__temp11), NULL)));
	__temp12 = NULL;
	self->priv->ulong_type = (__temp12 = vala_type_reference_new (), (self->priv->ulong_type == NULL ? NULL : (self->priv->ulong_type = (g_object_unref (self->priv->ulong_type), NULL))), __temp12);
	__temp13 = NULL;
	(vala_type_reference_set_data_type (self->priv->ulong_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp13 = vala_symbol_lookup (self->priv->root_symbol, "ulong"))))), vala_type_reference_get_data_type (self->priv->ulong_type));
	(__temp13 == NULL ? NULL : (__temp13 = (g_object_unref (__temp13), NULL)));
	__temp14 = NULL;
	self->priv->unichar_type = (__temp14 = vala_type_reference_new (), (self->priv->unichar_type == NULL ? NULL : (self->priv->unichar_type = (g_object_unref (self->priv->unichar_type), NULL))), __temp14);
	__temp15 = NULL;
	(vala_type_reference_set_data_type (self->priv->unichar_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp15 = vala_symbol_lookup (self->priv->root_symbol, "unichar"))))), vala_type_reference_get_data_type (self->priv->unichar_type));
	(__temp15 == NULL ? NULL : (__temp15 = (g_object_unref (__temp15), NULL)));
	/* TODO: don't require GLib namespace in semantic analyzer*/
	glib_ns = vala_symbol_lookup (self->priv->root_symbol, "GLib");
	if (glib_ns != NULL) {
		ValaDataType* __temp18;
		ValaDataType* __temp17;
		ValaSymbol* __temp16;
		ValaTypeReference* __temp19;
		ValaSymbol* __temp20;
		ValaDataType* __temp23;
		ValaDataType* __temp22;
		ValaSymbol* __temp21;
		ValaDataType* __temp26;
		ValaDataType* __temp25;
		ValaSymbol* __temp24;
		ValaDataType* __temp29;
		ValaDataType* __temp28;
		ValaSymbol* __temp27;
		__temp18 = NULL;
		__temp17 = NULL;
		__temp16 = NULL;
		self->priv->initially_unowned_type = (__temp18 = (__temp17 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp16 = vala_symbol_lookup (glib_ns, "InitiallyUnowned")))), (__temp17 == NULL ? NULL : g_object_ref (__temp17))), (self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL))), __temp18);
		(__temp16 == NULL ? NULL : (__temp16 = (g_object_unref (__temp16), NULL)));
		__temp19 = NULL;
		self->priv->type_type = (__temp19 = vala_type_reference_new (), (self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL))), __temp19);
		__temp20 = NULL;
		(vala_type_reference_set_data_type (self->priv->type_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp20 = vala_symbol_lookup (glib_ns, "Type"))))), vala_type_reference_get_data_type (self->priv->type_type));
		(__temp20 == NULL ? NULL : (__temp20 = (g_object_unref (__temp20), NULL)));
		__temp23 = NULL;
		__temp22 = NULL;
		__temp21 = NULL;
		self->priv->glist_type = (__temp23 = (__temp22 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp21 = vala_symbol_lookup (glib_ns, "List")))), (__temp22 == NULL ? NULL : g_object_ref (__temp22))), (self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL))), __temp23);
		(__temp21 == NULL ? NULL : (__temp21 = (g_object_unref (__temp21), NULL)));
		__temp26 = NULL;
		__temp25 = NULL;
		__temp24 = NULL;
		self->priv->gslist_type = (__temp26 = (__temp25 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp24 = vala_symbol_lookup (glib_ns, "SList")))), (__temp25 == NULL ? NULL : g_object_ref (__temp25))), (self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL))), __temp26);
		(__temp24 == NULL ? NULL : (__temp24 = (g_object_unref (__temp24), NULL)));
		__temp29 = NULL;
		__temp28 = NULL;
		__temp27 = NULL;
		self->priv->gerror_type = (__temp29 = (__temp28 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp27 = vala_symbol_lookup (glib_ns, "Error")))), (__temp28 == NULL ? NULL : g_object_ref (__temp28))), (self->priv->gerror_type == NULL ? NULL : (self->priv->gerror_type = (g_object_unref (self->priv->gerror_type), NULL))), __temp29);
		(__temp27 == NULL ? NULL : (__temp27 = (g_object_unref (__temp27), NULL)));
	}
	__temp31 = NULL;
	__temp30 = NULL;
	self->priv->current_symbol = (__temp31 = (__temp30 = self->priv->root_symbol, (__temp30 == NULL ? NULL : g_object_ref (__temp30))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp31);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	(glib_ns == NULL ? NULL : (glib_ns = (g_object_unref (glib_ns), NULL)));
}


static void vala_semantic_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file)
{
	ValaSemanticAnalyzer * self;
	ValaSourceFile* __temp32;
	GList* __temp33;
	GList* __temp34;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	__temp32 = NULL;
	self->priv->current_source_file = (__temp32 = g_object_ref (file), (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL))), __temp32);
	__temp33 = NULL;
	self->priv->current_using_directives = (__temp33 = vala_source_file_get_using_directives (file), (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_list_free (self->priv->current_using_directives), NULL))), __temp33);
	self->priv->next_lambda_id = 0;
	vala_source_file_accept_children (file, VALA_CODE_VISITOR (self));
	__temp34 = NULL;
	self->priv->current_using_directives = (__temp34 = NULL, (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_list_free (self->priv->current_using_directives), NULL))), __temp34);
}


static void vala_semantic_analyzer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp36;
	ValaSymbol* __temp35;
	ValaSymbol* __temp38;
	ValaSymbol* __temp37;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	__temp36 = NULL;
	__temp35 = NULL;
	self->priv->current_symbol = (__temp36 = (__temp35 = vala_code_node_get_symbol (VALA_CODE_NODE (ns)), (__temp35 == NULL ? NULL : g_object_ref (__temp35))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp36);
	vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
	__temp38 = NULL;
	__temp37 = NULL;
	self->priv->current_symbol = (__temp38 = (__temp37 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp37 == NULL ? NULL : g_object_ref (__temp37))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp38);
}


static void vala_semantic_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp40;
	ValaSymbol* __temp39;
	ValaClass* __temp41;
	GList* prerequisites;
	GList* missing_prereqs;
	ValaSymbol* __temp65;
	ValaSymbol* __temp64;
	ValaClass* __temp66;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	__temp40 = NULL;
	__temp39 = NULL;
	self->priv->current_symbol = (__temp40 = (__temp39 = vala_code_node_get_symbol (VALA_CODE_NODE (cl)), (__temp39 == NULL ? NULL : g_object_ref (__temp39))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp40);
	__temp41 = NULL;
	self->priv->current_class = (__temp41 = g_object_ref (cl), (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), __temp41);
	if (vala_class_get_base_class (cl) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_class_get_base_class (cl))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
	}
	{
		GList* __temp42;
		GList* base_type_reference_it;
		__temp42 = NULL;
		__temp42 = vala_class_get_base_types (cl);
		for (base_type_reference_it = __temp42; base_type_reference_it != NULL; base_type_reference_it = base_type_reference_it->next) {
			ValaTypeReference* base_type_reference;
			base_type_reference = base_type_reference_it->data;
			{
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type_reference))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
			}
		}
		(__temp42 == NULL ? NULL : (__temp42 = (g_list_free (__temp42), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
	/* gather all prerequisites */
	prerequisites = NULL;
	{
		GList* __temp43;
		GList* base_type_it;
		__temp43 = NULL;
		__temp43 = vala_class_get_base_types (cl);
		for (base_type_it = __temp43; base_type_it != NULL; base_type_it = base_type_it->next) {
			ValaTypeReference* base_type;
			base_type = base_type_it->data;
			{
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (base_type))) {
					prerequisites = g_list_concat (prerequisites, vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (vala_type_reference_get_data_type (base_type))));
				}
			}
		}
		(__temp43 == NULL ? NULL : (__temp43 = (g_list_free (__temp43), NULL)));
	}
	/* check whether all prerequisites are met */
	missing_prereqs = NULL;
	{
		GList* __temp44;
		GList* prereq_it;
		__temp44 = NULL;
		__temp44 = prerequisites;
		for (prereq_it = __temp44; prereq_it != NULL; prereq_it = prereq_it->next) {
			ValaDataType* prereq;
			prereq = prereq_it->data;
			{
				if (!vala_semantic_analyzer_class_is_a (self, cl, prereq)) {
					missing_prereqs = g_list_prepend (missing_prereqs, vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prereq))));
				}
			}
		}
	}
	/* report any missing prerequisites */
	if (missing_prereqs != NULL) {
		char* __temp45;
		char* __temp46;
		char* error_string;
		gboolean first;
		char* __temp50;
		(vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (cl)));
		__temp45 = NULL;
		__temp46 = NULL;
		error_string = (__temp46 = g_strdup_printf ("%s: some prerequisites (", (__temp45 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl))))), (__temp45 = (g_free (__temp45), NULL)), __temp46);
		first = TRUE;
		{
			GList* __temp49;
			GList* s_it;
			__temp49 = NULL;
			__temp49 = missing_prereqs;
			for (s_it = __temp49; s_it != NULL; s_it = s_it->next) {
				char* s;
				s = s_it->data;
				{
					if (first) {
						char* __temp47;
						__temp47 = NULL;
						error_string = (__temp47 = g_strdup_printf ("%s`%s'", error_string, s), (error_string = (g_free (error_string), NULL)), __temp47);
						first = FALSE;
					} else {
						char* __temp48;
						__temp48 = NULL;
						error_string = (__temp48 = g_strdup_printf ("%s, `%s'", error_string, s), (error_string = (g_free (error_string), NULL)), __temp48);
					}
				}
			}
		}
		__temp50 = NULL;
		error_string = (__temp50 = g_strconcat (error_string, (") are not met"), NULL), (error_string = (g_free (error_string), NULL)), __temp50);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), error_string);
		(error_string = (g_free (error_string), NULL));
	}
	{
		GList* __temp56;
		GList* base_type_it;
		__temp56 = NULL;
		__temp56 = vala_class_get_base_types (cl);
		for (base_type_it = __temp56; base_type_it != NULL; base_type_it = base_type_it->next) {
			ValaTypeReference* base_type;
			base_type = base_type_it->data;
			{
				/* all abstract symbols defined in base types have to be at least defined (or implemented) also in this type */
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (base_type))) {
					ValaInterface* __temp51;
					ValaInterface* iface;
					__temp51 = NULL;
					iface = (__temp51 = VALA_INTERFACE (vala_type_reference_get_data_type (base_type)), (__temp51 == NULL ? NULL : g_object_ref (__temp51)));
					{
						GList* __temp55;
						GList* m_it;
						__temp55 = NULL;
						__temp55 = vala_interface_get_methods (iface);
						for (m_it = __temp55; m_it != NULL; m_it = m_it->next) {
							ValaMethod* m;
							m = m_it->data;
							{
								/* We do not need to do expensive equality checking here since this is done
								 * already. We only need to guarantee the symbols are present.
								 
								 check methods */
								if (vala_method_get_is_abstract (m)) {
									ValaSymbol* sym;
									sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_method_get_name (m));
									if (sym == NULL || !(VALA_IS_METHOD (vala_symbol_get_node (sym))) || vala_method_get_base_interface_method ((VALA_METHOD (vala_symbol_get_node (sym)))) != m) {
										char* __temp54;
										char* __temp53;
										char* __temp52;
										(vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (cl)));
										__temp54 = NULL;
										__temp53 = NULL;
										__temp52 = NULL;
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), (__temp54 = g_strdup_printf ("`%s' does not implement interface method `%s'", (__temp52 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl)))), (__temp53 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
										(__temp54 = (g_free (__temp54), NULL));
										(__temp53 = (g_free (__temp53), NULL));
										(__temp52 = (g_free (__temp52), NULL));
									}
									(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								}
							}
						}
						(__temp55 == NULL ? NULL : (__temp55 = (g_list_free (__temp55), NULL)));
					}
					(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
				}
			}
		}
		(__temp56 == NULL ? NULL : (__temp56 = (g_list_free (__temp56), NULL)));
	}
	/* all abstract symbols defined in base classes have to be implemented in non-abstract classes
	 * VAPI classes don't have to specify overridden methods
	 */
	if (!vala_class_get_is_abstract (cl) && !vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cl))))) {
		ValaClass* __temp57;
		ValaClass* base_class;
		__temp57 = NULL;
		base_class = (__temp57 = vala_class_get_base_class (cl), (__temp57 == NULL ? NULL : g_object_ref (__temp57)));
		while (base_class != NULL && vala_class_get_is_abstract (base_class)) {
			ValaClass* __temp63;
			ValaClass* __temp62;
			{
				GList* __temp61;
				GList* m_it;
				__temp61 = NULL;
				__temp61 = vala_class_get_methods (base_class);
				for (m_it = __temp61; m_it != NULL; m_it = m_it->next) {
					ValaMethod* m;
					m = m_it->data;
					{
						if (vala_method_get_is_abstract (m)) {
							ValaSymbol* sym;
							sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_method_get_name (m));
							if (sym == NULL || !(VALA_IS_METHOD (vala_symbol_get_node (sym))) || vala_method_get_base_method ((VALA_METHOD (vala_symbol_get_node (sym)))) != m) {
								char* __temp60;
								char* __temp59;
								char* __temp58;
								(vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (cl)));
								__temp60 = NULL;
								__temp59 = NULL;
								__temp58 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), (__temp60 = g_strdup_printf ("`%s' does not implement abstract method `%s'", (__temp58 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl)))), (__temp59 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
								(__temp60 = (g_free (__temp60), NULL));
								(__temp59 = (g_free (__temp59), NULL));
								(__temp58 = (g_free (__temp58), NULL));
							}
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
						}
					}
				}
				(__temp61 == NULL ? NULL : (__temp61 = (g_list_free (__temp61), NULL)));
			}
			__temp63 = NULL;
			__temp62 = NULL;
			base_class = (__temp63 = (__temp62 = vala_class_get_base_class (base_class), (__temp62 == NULL ? NULL : g_object_ref (__temp62))), (base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL))), __temp63);
		}
		(base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL)));
	}
	__temp65 = NULL;
	__temp64 = NULL;
	self->priv->current_symbol = (__temp65 = (__temp64 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp64 == NULL ? NULL : g_object_ref (__temp64))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp65);
	__temp66 = NULL;
	self->priv->current_class = (__temp66 = NULL, (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), __temp66);
	(prerequisites == NULL ? NULL : (prerequisites = (g_list_foreach (prerequisites, (GFunc) g_object_unref, NULL), g_list_free (prerequisites), NULL)));
	(missing_prereqs == NULL ? NULL : (missing_prereqs = (g_list_foreach (missing_prereqs, (GFunc) g_free, NULL), g_list_free (missing_prereqs), NULL)));
}


static GList* vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface)
{
	GList* ret;
	GList* __temp70;
	GList* __temp71;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_INTERFACE (iface), NULL);
	ret = NULL;
	{
		GList* __temp69;
		GList* prereq_it;
		__temp69 = NULL;
		__temp69 = vala_interface_get_prerequisites (iface);
		for (prereq_it = __temp69; prereq_it != NULL; prereq_it = prereq_it->next) {
			ValaTypeReference* prereq;
			prereq = prereq_it->data;
			{
				ValaDataType* __temp67;
				ValaDataType* type;
				ValaDataType* __temp68;
				__temp67 = NULL;
				type = (__temp67 = vala_type_reference_get_data_type (prereq), (__temp67 == NULL ? NULL : g_object_ref (__temp67)));
				/* skip on previous errors */
				if (type == NULL) {
					continue;
				}
				__temp68 = NULL;
				ret = g_list_prepend (ret, (__temp68 = type, (__temp68 == NULL ? NULL : g_object_ref (__temp68))));
				if (VALA_IS_INTERFACE (type)) {
					ret = g_list_concat (ret, vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (type)));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(__temp69 == NULL ? NULL : (__temp69 = (g_list_free (__temp69), NULL)));
	}
	ret = g_list_reverse (ret);
	__temp70 = NULL;
	__temp71 = NULL;
	return (__temp71 = (__temp70 = ret, ret = NULL, __temp70), (ret == NULL ? NULL : (ret = (g_list_foreach (ret, (GFunc) g_object_unref, NULL), g_list_free (ret), NULL))), __temp71);
	(ret == NULL ? NULL : (ret = (g_list_foreach (ret, (GFunc) g_object_unref, NULL), g_list_free (ret), NULL)));
}


static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaDataType* t)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_CLASS (cl), FALSE);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	if (VALA_DATA_TYPE (cl) == t) {
		return TRUE;
	}
	{
		GList* __temp75;
		GList* base_type_it;
		__temp75 = NULL;
		__temp75 = vala_class_get_base_types (cl);
		for (base_type_it = __temp75; base_type_it != NULL; base_type_it = base_type_it->next) {
			ValaTypeReference* base_type;
			base_type = base_type_it->data;
			{
				if (VALA_IS_CLASS (vala_type_reference_get_data_type (base_type))) {
					if (vala_semantic_analyzer_class_is_a (self, VALA_CLASS (vala_type_reference_get_data_type (base_type)), t)) {
						return TRUE;
					}
				} else {
					if (vala_type_reference_get_data_type (base_type) == t) {
						return TRUE;
					}
				}
			}
		}
		(__temp75 == NULL ? NULL : (__temp75 = (g_list_free (__temp75), NULL)));
	}
	return FALSE;
}


static void vala_semantic_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp78;
	ValaSymbol* __temp77;
	ValaStruct* __temp79;
	ValaSymbol* __temp81;
	ValaSymbol* __temp80;
	ValaStruct* __temp82;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	__temp78 = NULL;
	__temp77 = NULL;
	self->priv->current_symbol = (__temp78 = (__temp77 = vala_code_node_get_symbol (VALA_CODE_NODE (st)), (__temp77 == NULL ? NULL : g_object_ref (__temp77))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp78);
	__temp79 = NULL;
	self->priv->current_struct = (__temp79 = g_object_ref (st), (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), __temp79);
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
	__temp81 = NULL;
	__temp80 = NULL;
	self->priv->current_symbol = (__temp81 = (__temp80 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp80 == NULL ? NULL : g_object_ref (__temp80))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp81);
	__temp82 = NULL;
	self->priv->current_struct = (__temp82 = NULL, (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), __temp82);
}


static void vala_semantic_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp84;
	ValaSymbol* __temp83;
	ValaClass* prereq_class;
	ValaSymbol* __temp95;
	ValaSymbol* __temp94;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	__temp84 = NULL;
	__temp83 = NULL;
	self->priv->current_symbol = (__temp84 = (__temp83 = vala_code_node_get_symbol (VALA_CODE_NODE (iface)), (__temp83 == NULL ? NULL : g_object_ref (__temp83))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp84);
	{
		GList* __temp85;
		GList* prerequisite_reference_it;
		__temp85 = NULL;
		__temp85 = vala_interface_get_prerequisites (iface);
		for (prerequisite_reference_it = __temp85; prerequisite_reference_it != NULL; prerequisite_reference_it = prerequisite_reference_it->next) {
			ValaTypeReference* prerequisite_reference;
			prerequisite_reference = prerequisite_reference_it->data;
			{
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (prerequisite_reference))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
			}
		}
		(__temp85 == NULL ? NULL : (__temp85 = (g_list_free (__temp85), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
	/* check prerequisites */
	prereq_class = NULL;
	{
		GList* __temp93;
		GList* prereq_it;
		__temp93 = NULL;
		__temp93 = vala_interface_get_prerequisites (iface);
		for (prereq_it = __temp93; prereq_it != NULL; prereq_it = prereq_it->next) {
			ValaTypeReference* prereq;
			prereq = prereq_it->data;
			{
				ValaDataType* __temp86;
				ValaDataType* class_or_interface;
				__temp86 = NULL;
				class_or_interface = (__temp86 = vala_type_reference_get_data_type (prereq), (__temp86 == NULL ? NULL : g_object_ref (__temp86)));
				/* skip on previous errors */
				if (class_or_interface == NULL) {
					(vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE), vala_code_node_get_error (VALA_CODE_NODE (iface)));
					continue;
				}
				/* interfaces are not allowed to have multiple instantiable prerequisites */
				if (VALA_IS_CLASS (class_or_interface)) {
					ValaClass* __temp92;
					ValaClass* __temp91;
					if (prereq_class != NULL) {
						char* __temp90;
						char* __temp89;
						char* __temp88;
						char* __temp87;
						(vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE), vala_code_node_get_error (VALA_CODE_NODE (iface)));
						__temp90 = NULL;
						__temp89 = NULL;
						__temp88 = NULL;
						__temp87 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (iface)), (__temp90 = g_strdup_printf ("%s: Interfaces cannot have multiple instantiable prerequisites (`%s' and `%s')", (__temp87 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (iface)))), (__temp88 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (class_or_interface)))), (__temp89 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prereq_class)))))));
						(__temp90 = (g_free (__temp90), NULL));
						(__temp89 = (g_free (__temp89), NULL));
						(__temp88 = (g_free (__temp88), NULL));
						(__temp87 = (g_free (__temp87), NULL));
						(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
						(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
						return;
					}
					__temp92 = NULL;
					__temp91 = NULL;
					prereq_class = (__temp92 = (__temp91 = VALA_CLASS (class_or_interface), (__temp91 == NULL ? NULL : g_object_ref (__temp91))), (prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL))), __temp92);
				}
				(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
			}
		}
		(__temp93 == NULL ? NULL : (__temp93 = (g_list_free (__temp93), NULL)));
	}
	__temp95 = NULL;
	__temp94 = NULL;
	self->priv->current_symbol = (__temp95 = (__temp94 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp94 == NULL ? NULL : g_object_ref (__temp94))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp95);
	(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
}


static void vala_semantic_analyzer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CALLBACK (cb));
	vala_code_node_accept_children (VALA_CODE_NODE (cb), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONSTANT (c));
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
	if (!vala_source_file_get_pkg (self->priv->current_source_file)) {
		if (vala_constant_get_initializer (c) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (c), TRUE), vala_code_node_get_error (VALA_CODE_NODE (c)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (c)), "A const field requires a initializer to be provided");
		}
	}
}


static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FIELD (f));
	vala_code_node_accept_children (VALA_CODE_NODE (f), VALA_CODE_VISITOR (self));
	if (f->access != VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
	} else {
		if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		}
	}
}


static void vala_semantic_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp97;
	ValaSymbol* __temp96;
	ValaTypeReference* __temp99;
	ValaTypeReference* __temp98;
	ValaAttribute* init_attr;
	ValaSymbol* __temp101;
	ValaSymbol* __temp100;
	ValaTypeReference* __temp102;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	__temp97 = NULL;
	__temp96 = NULL;
	self->priv->current_symbol = (__temp97 = (__temp96 = vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp96 == NULL ? NULL : g_object_ref (__temp96))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp97);
	__temp99 = NULL;
	__temp98 = NULL;
	self->priv->current_return_type = (__temp99 = (__temp98 = vala_method_get_return_type (m), (__temp98 == NULL ? NULL : g_object_ref (__temp98))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp99);
	init_attr = vala_code_node_get_attribute (VALA_CODE_NODE (m), "ModuleInit");
	if (init_attr != NULL) {
		(vala_code_context_set_module_init_method (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (m)))), m), vala_code_context_get_module_init_method (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (m))))));
	}
	if (vala_type_reference_get_data_type (vala_method_get_return_type (m)) != NULL) {
		/* is null if it is void or a reference to a type parameter */
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_method_get_return_type (m)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
	}
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	__temp101 = NULL;
	__temp100 = NULL;
	self->priv->current_symbol = (__temp101 = (__temp100 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp100 == NULL ? NULL : g_object_ref (__temp100))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp101);
	__temp102 = NULL;
	self->priv->current_return_type = (__temp102 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp102);
	if (vala_symbol_get_parent_symbol (self->priv->current_symbol) != NULL && VALA_IS_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (self->priv->current_symbol)))) {
		ValaMethod* __temp103;
		ValaMethod* up_method;
		ValaTypeReference* __temp105;
		ValaTypeReference* __temp104;
		/* lambda expressions produce nested methods */
		__temp103 = NULL;
		up_method = (__temp103 = VALA_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (self->priv->current_symbol))), (__temp103 == NULL ? NULL : g_object_ref (__temp103)));
		__temp105 = NULL;
		__temp104 = NULL;
		self->priv->current_return_type = (__temp105 = (__temp104 = vala_method_get_return_type (up_method), (__temp104 == NULL ? NULL : g_object_ref (__temp104))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp105);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (self->priv->current_symbol))) {
		if (!(VALA_IS_CREATION_METHOD (m))) {
			vala_semantic_analyzer_find_base_interface_method (self, m, VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)));
			if (vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				vala_semantic_analyzer_find_base_class_method (self, m, VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)));
				if (vala_method_get_base_method (m) == NULL) {
					char* __temp107;
					char* __temp106;
					__temp107 = NULL;
					__temp106 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp107 = g_strdup_printf ("%s: no suitable method found to override", (__temp106 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
					(__temp107 = (g_free (__temp107), NULL));
					(__temp106 = (g_free (__temp106), NULL));
				}
			}
		}
	} else {
		if (VALA_IS_STRUCT (vala_symbol_get_node (self->priv->current_symbol))) {
			if (vala_method_get_is_abstract (m) || vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				char* __temp109;
				char* __temp108;
				__temp109 = NULL;
				__temp108 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp109 = g_strdup_printf ("A struct member `%s' cannot be marked as override, virtual, or abstract", (__temp108 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
				(__temp109 = (g_free (__temp109), NULL));
				(__temp108 = (g_free (__temp108), NULL));
				(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
				return;
			}
		}
	}
	(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
}


static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl)
{
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (VALA_IS_CLASS (cl));
	sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_method_get_name (m));
	if (sym != NULL && VALA_IS_METHOD (vala_symbol_get_node (sym))) {
		ValaMethod* __temp110;
		ValaMethod* base_method;
		__temp110 = NULL;
		base_method = (__temp110 = VALA_METHOD (vala_symbol_get_node (sym)), (__temp110 == NULL ? NULL : g_object_ref (__temp110)));
		if (vala_method_get_is_abstract (base_method) || vala_method_get_is_virtual (base_method)) {
			if (!vala_method_equals (m, base_method)) {
				char* __temp113;
				char* __temp112;
				char* __temp111;
				(vala_code_node_set_error (VALA_CODE_NODE (m), TRUE), vala_code_node_get_error (VALA_CODE_NODE (m)));
				__temp113 = NULL;
				__temp112 = NULL;
				__temp111 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp113 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (__temp111 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp112 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_method)))))));
				(__temp113 = (g_free (__temp113), NULL));
				(__temp112 = (g_free (__temp112), NULL));
				(__temp111 = (g_free (__temp111), NULL));
				(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			(vala_method_set_base_method (m, base_method), vala_method_get_base_method (m));
			(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_method (self, m, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (VALA_IS_CLASS (cl));
	{
		GList* __temp118;
		GList* type_it;
		__temp118 = NULL;
		__temp118 = vala_class_get_base_types (cl);
		for (type_it = __temp118; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type;
			type = type_it->data;
			{
				/* FIXME report error if multiple possible base methods are found*/
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type))), vala_method_get_name (m));
					if (sym != NULL && VALA_IS_METHOD (vala_symbol_get_node (sym))) {
						ValaMethod* __temp114;
						ValaMethod* base_method;
						__temp114 = NULL;
						base_method = (__temp114 = VALA_METHOD (vala_symbol_get_node (sym)), (__temp114 == NULL ? NULL : g_object_ref (__temp114)));
						if (vala_method_get_is_abstract (base_method)) {
							if (!vala_method_equals (m, base_method)) {
								char* __temp117;
								char* __temp116;
								char* __temp115;
								(vala_code_node_set_error (VALA_CODE_NODE (m), TRUE), vala_code_node_get_error (VALA_CODE_NODE (m)));
								__temp117 = NULL;
								__temp116 = NULL;
								__temp115 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp117 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (__temp115 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp116 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_method)))))));
								(__temp117 = (g_free (__temp117), NULL));
								(__temp116 = (g_free (__temp116), NULL));
								(__temp115 = (g_free (__temp115), NULL));
								(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								return;
							}
							(vala_method_set_base_interface_method (m, base_method), vala_method_get_base_interface_method (m));
							(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							return;
						}
						(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
			}
		}
		(__temp118 == NULL ? NULL : (__temp118 = (g_list_free (__temp118), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp119;
	ValaSymbol* __temp124;
	ValaSymbol* __temp123;
	ValaTypeReference* __temp126;
	ValaTypeReference* __temp125;
	ValaSymbol* __temp128;
	ValaSymbol* __temp127;
	ValaTypeReference* __temp129;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	__temp119 = NULL;
	(vala_method_set_return_type (VALA_METHOD (m), (__temp119 = vala_type_reference_new ())), vala_method_get_return_type (VALA_METHOD (m)));
	(__temp119 == NULL ? NULL : (__temp119 = (g_object_unref (__temp119), NULL)));
	(vala_type_reference_set_data_type (vala_method_get_return_type (VALA_METHOD (m)), VALA_DATA_TYPE (vala_symbol_get_node (self->priv->current_symbol))), vala_type_reference_get_data_type (vala_method_get_return_type (VALA_METHOD (m))));
	(vala_type_reference_set_transfers_ownership (vala_method_get_return_type (VALA_METHOD (m)), TRUE), vala_type_reference_get_transfers_ownership (vala_method_get_return_type (VALA_METHOD (m))));
	if (VALA_IS_CLASS (vala_symbol_get_node (self->priv->current_symbol))) {
		ValaClass* __temp120;
		ValaClass* cl;
		/* check for floating reference*/
		__temp120 = NULL;
		cl = (__temp120 = VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)), (__temp120 == NULL ? NULL : g_object_ref (__temp120)));
		while (cl != NULL) {
			ValaClass* __temp122;
			ValaClass* __temp121;
			if (VALA_DATA_TYPE (cl) == self->priv->initially_unowned_type) {
				(vala_type_reference_set_floating_reference (vala_method_get_return_type (VALA_METHOD (m)), TRUE), vala_type_reference_get_floating_reference (vala_method_get_return_type (VALA_METHOD (m))));
				break;
			}
			__temp122 = NULL;
			__temp121 = NULL;
			cl = (__temp122 = (__temp121 = vala_class_get_base_class (cl), (__temp121 == NULL ? NULL : g_object_ref (__temp121))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp122);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
	__temp124 = NULL;
	__temp123 = NULL;
	self->priv->current_symbol = (__temp124 = (__temp123 = vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp123 == NULL ? NULL : g_object_ref (__temp123))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp124);
	__temp126 = NULL;
	__temp125 = NULL;
	self->priv->current_return_type = (__temp126 = (__temp125 = vala_method_get_return_type (VALA_METHOD (m)), (__temp125 == NULL ? NULL : g_object_ref (__temp125))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp126);
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	__temp128 = NULL;
	__temp127 = NULL;
	self->priv->current_symbol = (__temp128 = (__temp127 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp127 == NULL ? NULL : g_object_ref (__temp127))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp128);
	__temp129 = NULL;
	self->priv->current_return_type = (__temp129 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp129);
	if (vala_symbol_get_parent_symbol (self->priv->current_symbol) != NULL && VALA_IS_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (self->priv->current_symbol)))) {
		ValaMethod* __temp130;
		ValaMethod* up_method;
		ValaTypeReference* __temp132;
		ValaTypeReference* __temp131;
		/* lambda expressions produce nested methods */
		__temp130 = NULL;
		up_method = (__temp130 = VALA_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (self->priv->current_symbol))), (__temp130 == NULL ? NULL : g_object_ref (__temp130)));
		__temp132 = NULL;
		__temp131 = NULL;
		self->priv->current_return_type = (__temp132 = (__temp131 = vala_method_get_return_type (up_method), (__temp131 == NULL ? NULL : g_object_ref (__temp131))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp132);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (self->priv->current_symbol))) {
		if (!(VALA_IS_CREATION_METHOD (m))) {
			vala_semantic_analyzer_find_base_interface_method (self, VALA_METHOD (m), VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)));
			if (vala_method_get_is_virtual (VALA_METHOD (m)) || vala_method_get_overrides (VALA_METHOD (m))) {
				vala_semantic_analyzer_find_base_class_method (self, VALA_METHOD (m), VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)));
				if (vala_method_get_base_method (VALA_METHOD (m)) == NULL) {
					char* __temp134;
					char* __temp133;
					__temp134 = NULL;
					__temp133 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp134 = g_strdup_printf ("%s: no suitable method found to override", (__temp133 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
					(__temp134 = (g_free (__temp134), NULL));
					(__temp133 = (g_free (__temp133), NULL));
				}
			}
		}
	} else {
		if (VALA_IS_STRUCT (vala_symbol_get_node (self->priv->current_symbol))) {
			if (vala_method_get_is_abstract (VALA_METHOD (m)) || vala_method_get_is_virtual (VALA_METHOD (m)) || vala_method_get_overrides (VALA_METHOD (m))) {
				char* __temp136;
				char* __temp135;
				__temp136 = NULL;
				__temp135 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp136 = g_strdup_printf ("A struct member `%s' cannot be marked as override, virtual, or abstract", (__temp135 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
				(__temp136 = (g_free (__temp136), NULL));
				(__temp135 = (g_free (__temp135), NULL));
				return;
			}
		}
	}
	if (vala_method_get_body (VALA_METHOD (m)) != NULL && self->priv->current_class != NULL) {
		gint n_params;
		n_params = 0;
		{
			GList* __temp137;
			GList* stmt_it;
			__temp137 = NULL;
			__temp137 = vala_block_get_statements (vala_method_get_body (VALA_METHOD (m)));
			for (stmt_it = __temp137; stmt_it != NULL; stmt_it = stmt_it->next) {
				ValaStatement* stmt;
				stmt = stmt_it->data;
				{
					gint params;
					params = vala_statement_get_number_of_set_construction_parameters (stmt);
					if (params == -1) {
						(vala_code_node_set_error (VALA_CODE_NODE (m), TRUE), vala_code_node_get_error (VALA_CODE_NODE (m)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "class creation methods only allow property assignment statements");
						return;
					}
					n_params = n_params + (params);
				}
			}
			(__temp137 == NULL ? NULL : (__temp137 = (g_list_free (__temp137), NULL)));
		}
		(vala_creation_method_set_n_construction_params (m, n_params), vala_creation_method_get_n_construction_params (m));
	}
}


static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (p));
	vala_code_node_accept_children (VALA_CODE_NODE (p), VALA_CODE_VISITOR (self));
	if (!vala_formal_parameter_get_ellipsis (p)) {
		if (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		}
	}
	/* special treatment for construct formal parameters used in creation methods */
	if (vala_formal_parameter_get_construct_parameter (p)) {
		ValaBlock* __temp138;
		ValaBlock* method_body;
		ValaMemberAccess* left;
		ValaMemberAccess* right;
		ValaSymbol* prop_sym;
		ValaExpressionStatement* __temp141;
		if (!(VALA_IS_CREATION_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (p))))))) {
			(vala_code_node_set_error (VALA_CODE_NODE (p), TRUE), vala_code_node_get_error (VALA_CODE_NODE (p)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (p)), "construct parameters are only allowed in type creation methods");
			return;
		}
		__temp138 = NULL;
		method_body = (__temp138 = vala_method_get_body (VALA_METHOD ((VALA_CREATION_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (p)))))))), (__temp138 == NULL ? NULL : g_object_ref (__temp138)));
		left = vala_member_access_new_simple (vala_formal_parameter_get_name (p), NULL);
		right = vala_member_access_new_simple (vala_formal_parameter_get_name (p), NULL);
		/* try to lookup the requested property */
		prop_sym = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (self->priv->current_class)), vala_formal_parameter_get_name (p));
		if (!(VALA_IS_PROPERTY (vala_symbol_get_node (prop_sym)))) {
			char* __temp140;
			char* __temp139;
			(vala_code_node_set_error (VALA_CODE_NODE (p), TRUE), vala_code_node_get_error (VALA_CODE_NODE (p)));
			__temp140 = NULL;
			__temp139 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (p)), (__temp140 = g_strdup_printf ("class `%s' does not contain a property named `%s'", (__temp139 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (self->priv->current_class)))), vala_formal_parameter_get_name (p))));
			(__temp140 = (g_free (__temp140), NULL));
			(__temp139 = (g_free (__temp139), NULL));
			(method_body == NULL ? NULL : (method_body = (g_object_unref (method_body), NULL)));
			(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
			(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
			(prop_sym == NULL ? NULL : (prop_sym = (g_object_unref (prop_sym), NULL)));
			return;
		}
		(vala_expression_set_symbol_reference (VALA_EXPRESSION (left), prop_sym), vala_expression_get_symbol_reference (VALA_EXPRESSION (left)));
		(vala_expression_set_symbol_reference (VALA_EXPRESSION (right), vala_code_node_get_symbol (VALA_CODE_NODE (p))), vala_expression_get_symbol_reference (VALA_EXPRESSION (right)));
		__temp141 = NULL;
		vala_block_add_statement (method_body, VALA_STATEMENT ((__temp141 = vala_expression_statement_new (VALA_EXPRESSION (vala_assignment_new (VALA_EXPRESSION (left), VALA_EXPRESSION (right), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL)), NULL))));
		(__temp141 == NULL ? NULL : (__temp141 = (g_object_unref (__temp141), NULL)));
		(method_body == NULL ? NULL : (method_body = (g_object_unref (method_body), NULL)));
		(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
		(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
		(prop_sym == NULL ? NULL : (prop_sym = (g_object_unref (prop_sym), NULL)));
	}
}


static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl)
{
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	g_return_if_fail (VALA_IS_CLASS (cl));
	sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_property_get_name (prop));
	if (sym != NULL && VALA_IS_PROPERTY (vala_symbol_get_node (sym))) {
		ValaProperty* __temp142;
		ValaProperty* base_property;
		__temp142 = NULL;
		base_property = (__temp142 = VALA_PROPERTY (vala_symbol_get_node (sym)), (__temp142 == NULL ? NULL : g_object_ref (__temp142)));
		if (vala_property_get_is_abstract (base_property) || vala_property_get_is_virtual (base_property)) {
			if (!vala_property_equals (prop, base_property)) {
				char* __temp145;
				char* __temp144;
				char* __temp143;
				(vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE), vala_code_node_get_error (VALA_CODE_NODE (prop)));
				__temp145 = NULL;
				__temp144 = NULL;
				__temp143 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (__temp145 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (__temp143 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))), (__temp144 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_property)))))));
				(__temp145 = (g_free (__temp145), NULL));
				(__temp144 = (g_free (__temp144), NULL));
				(__temp143 = (g_free (__temp143), NULL));
				(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			(vala_property_set_base_property (prop, base_property), vala_property_get_base_property (prop));
			(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_property (self, prop, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	g_return_if_fail (VALA_IS_CLASS (cl));
	{
		GList* __temp150;
		GList* type_it;
		__temp150 = NULL;
		__temp150 = vala_class_get_base_types (cl);
		for (type_it = __temp150; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type;
			type = type_it->data;
			{
				/* FIXME report error if multiple possible base properties are found*/
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type))), vala_property_get_name (prop));
					if (sym != NULL && VALA_IS_PROPERTY (vala_symbol_get_node (sym))) {
						ValaProperty* __temp146;
						ValaProperty* base_property;
						__temp146 = NULL;
						base_property = (__temp146 = VALA_PROPERTY (vala_symbol_get_node (sym)), (__temp146 == NULL ? NULL : g_object_ref (__temp146)));
						if (vala_property_get_is_abstract (base_property)) {
							if (!vala_property_equals (prop, base_property)) {
								char* __temp149;
								char* __temp148;
								char* __temp147;
								(vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE), vala_code_node_get_error (VALA_CODE_NODE (prop)));
								__temp149 = NULL;
								__temp148 = NULL;
								__temp147 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (__temp149 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (__temp147 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))), (__temp148 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_property)))))));
								(__temp149 = (g_free (__temp149), NULL));
								(__temp148 = (g_free (__temp148), NULL));
								(__temp147 = (g_free (__temp147), NULL));
								(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								return;
							}
							(vala_property_set_base_interface_property (prop, base_property), vala_property_get_base_interface_property (prop));
							(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							return;
						}
						(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
			}
		}
		(__temp150 == NULL ? NULL : (__temp150 = (g_list_free (__temp150), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	vala_code_node_accept_children (VALA_CODE_NODE (prop), VALA_CODE_VISITOR (self));
	if (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)) != NULL) {
		/* is null if it references a type parameter */
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))))) {
		ValaClass* __temp151;
		ValaClass* cl;
		__temp151 = NULL;
		cl = (__temp151 = VALA_CLASS (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (prop))))), (__temp151 == NULL ? NULL : g_object_ref (__temp151)));
		vala_semantic_analyzer_find_base_interface_property (self, prop, cl);
		if (vala_property_get_is_virtual (prop) || vala_property_get_overrides (prop)) {
			vala_semantic_analyzer_find_base_class_property (self, prop, cl);
			if (vala_property_get_base_property (prop) == NULL) {
				char* __temp153;
				char* __temp152;
				(vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE), vala_code_node_get_error (VALA_CODE_NODE (prop)));
				__temp153 = NULL;
				__temp152 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (__temp153 = g_strdup_printf ("%s: no suitable property found to override", (__temp152 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))))));
				(__temp153 = (g_free (__temp153), NULL));
				(__temp152 = (g_free (__temp152), NULL));
			}
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc)
{
	ValaSemanticAnalyzer * self;
	ValaProperty* __temp154;
	ValaProperty* prop;
	ValaTypeReference* __temp157;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (acc));
	__temp154 = NULL;
	prop = (__temp154 = VALA_PROPERTY (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (acc))))), (__temp154 == NULL ? NULL : g_object_ref (__temp154)));
	if (vala_property_accessor_get_readable (acc)) {
		ValaTypeReference* __temp155;
		__temp155 = NULL;
		self->priv->current_return_type = (__temp155 = g_object_ref (vala_property_get_type_reference (prop)), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp155);
	} else {
		ValaTypeReference* __temp156;
		/* void*/
		__temp156 = NULL;
		self->priv->current_return_type = (__temp156 = vala_type_reference_new (), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp156);
	}
	vala_code_node_accept_children (VALA_CODE_NODE (acc), VALA_CODE_VISITOR (self));
	__temp157 = NULL;
	self->priv->current_return_type = (__temp157 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp157);
	(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
}


static void vala_semantic_analyzer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	vala_code_node_accept_children (VALA_CODE_NODE (sig), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp159;
	ValaSymbol* __temp158;
	ValaSymbol* __temp161;
	ValaSymbol* __temp160;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONSTRUCTOR (c));
	__temp159 = NULL;
	__temp158 = NULL;
	self->priv->current_symbol = (__temp159 = (__temp158 = vala_code_node_get_symbol (VALA_CODE_NODE (c)), (__temp158 == NULL ? NULL : g_object_ref (__temp158))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp159);
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
	__temp161 = NULL;
	__temp160 = NULL;
	self->priv->current_symbol = (__temp161 = (__temp160 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp160 == NULL ? NULL : g_object_ref (__temp160))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp161);
}


static void vala_semantic_analyzer_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp163;
	ValaSymbol* __temp162;
	ValaSymbol* __temp165;
	ValaSymbol* __temp164;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_DESTRUCTOR (d));
	__temp163 = NULL;
	__temp162 = NULL;
	self->priv->current_symbol = (__temp163 = (__temp162 = vala_code_node_get_symbol (VALA_CODE_NODE (d)), (__temp162 == NULL ? NULL : g_object_ref (__temp162))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp163);
	vala_code_node_accept_children (VALA_CODE_NODE (d), VALA_CODE_VISITOR (self));
	__temp165 = NULL;
	__temp164 = NULL;
	self->priv->current_symbol = (__temp165 = (__temp164 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp164 == NULL ? NULL : g_object_ref (__temp164))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp165);
}


static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NAMED_ARGUMENT (n));
}


static void vala_semantic_analyzer_real_visit_begin_block (ValaCodeVisitor* base, ValaBlock* b)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp167;
	ValaSymbol* __temp166;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BLOCK (b));
	__temp167 = NULL;
	__temp166 = NULL;
	self->priv->current_symbol = (__temp167 = (__temp166 = vala_code_node_get_symbol (VALA_CODE_NODE (b)), (__temp166 == NULL ? NULL : g_object_ref (__temp166))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp167);
}


static void vala_semantic_analyzer_real_visit_end_block (ValaCodeVisitor* base, ValaBlock* b)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp170;
	ValaSymbol* __temp169;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BLOCK (b));
	{
		GList* __temp168;
		GList* decl_it;
		__temp168 = NULL;
		__temp168 = vala_block_get_local_variables (b);
		for (decl_it = __temp168; decl_it != NULL; decl_it = decl_it->next) {
			ValaVariableDeclarator* decl;
			decl = decl_it->data;
			{
				(vala_symbol_set_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl)), FALSE), vala_symbol_get_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl))));
			}
		}
		(__temp168 == NULL ? NULL : (__temp168 = (g_list_free (__temp168), NULL)));
	}
	__temp170 = NULL;
	__temp169 = NULL;
	self->priv->current_symbol = (__temp170 = (__temp169 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp169 == NULL ? NULL : g_object_ref (__temp169))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp170);
}


static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp177;
	ValaBlock* __temp178;
	ValaBlock* block;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (decl));
	if (vala_variable_declarator_get_type_reference (decl) == NULL) {
		ValaTypeReference* __temp171;
		/* var type */
		if (vala_variable_declarator_get_initializer (decl) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "var declaration not allowed without initializer");
			return;
		}
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "var declaration not allowed with non-typed initializer");
			return;
		}
		__temp171 = NULL;
		(vala_variable_declarator_set_type_reference (decl, (__temp171 = vala_type_reference_copy (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl))))), vala_variable_declarator_get_type_reference (decl));
		(__temp171 = (g_object_unref (__temp171), NULL));
		(vala_type_reference_set_takes_ownership (vala_variable_declarator_get_type_reference (decl), (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)) == NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))))), vala_type_reference_get_takes_ownership (vala_variable_declarator_get_type_reference (decl)));
		(vala_type_reference_set_transfers_ownership (vala_variable_declarator_get_type_reference (decl), FALSE), vala_type_reference_get_transfers_ownership (vala_variable_declarator_get_type_reference (decl)));
	}
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			if (!(VALA_IS_MEMBER_ACCESS (vala_variable_declarator_get_initializer (decl)))) {
				(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "expression type not allowed as initializer");
				return;
			}
			if (VALA_IS_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl)))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))) {
				ValaMethod* __temp172;
				ValaMethod* m;
				ValaCallback* __temp173;
				ValaCallback* cb;
				__temp172 = NULL;
				m = (__temp172 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl)))), (__temp172 == NULL ? NULL : g_object_ref (__temp172)));
				__temp173 = NULL;
				cb = (__temp173 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (__temp173 == NULL ? NULL : g_object_ref (__temp173)));
				/* check whether method matches callback type */
				if (!vala_callback_matches_method (cb, m)) {
					char* __temp176;
					char* __temp175;
					char* __temp174;
					(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
					__temp176 = NULL;
					__temp175 = NULL;
					__temp174 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), (__temp176 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp174 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp175 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cb)))))));
					(__temp176 = (g_free (__temp176), NULL));
					(__temp175 = (g_free (__temp175), NULL));
					(__temp174 = (g_free (__temp174), NULL));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					return;
				}
				(vala_expression_set_static_type (vala_variable_declarator_get_initializer (decl), vala_variable_declarator_get_type_reference (decl)), vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
			} else {
				(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "expression type not allowed as initializer");
				return;
			}
		}
		if (vala_semantic_analyzer_get_memory_management (self)) {
			if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)))) {
				/* rhs transfers ownership of the expression */
				if (!vala_type_reference_get_takes_ownership (vala_variable_declarator_get_type_reference (decl))) {
					/* lhs doesn't own the value */
					(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "Invalid assignment from owned expression to unowned variable");
					return;
				}
			}
		}
	}
	if (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	__temp177 = NULL;
	(vala_code_node_set_symbol (VALA_CODE_NODE (decl), (__temp177 = vala_symbol_new (VALA_CODE_NODE (decl)))), vala_code_node_get_symbol (VALA_CODE_NODE (decl)));
	(__temp177 == NULL ? NULL : (__temp177 = (g_object_unref (__temp177), NULL)));
	vala_symbol_add (self->priv->current_symbol, vala_variable_declarator_get_name (decl), vala_code_node_get_symbol (VALA_CODE_NODE (decl)));
	__temp178 = NULL;
	block = (__temp178 = VALA_BLOCK (vala_symbol_get_node (self->priv->current_symbol)), (__temp178 == NULL ? NULL : g_object_ref (__temp178)));
	vala_block_add_local_variable (block, decl);
	(vala_symbol_set_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl)), TRUE), vala_symbol_get_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl))));
	(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_semantic_analyzer_real_visit_begin_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (list)) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))) {
		ValaArray* __temp179;
		ValaArray* edt;
		GList* inits;
		gint rank;
		ValaTypeReference* child_type;
		/* initializer is used as array initializer */
		__temp179 = NULL;
		edt = (__temp179 = VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list)))), (__temp179 == NULL ? NULL : g_object_ref (__temp179)));
		inits = vala_initializer_list_get_initializers (list);
		rank = vala_array_get_rank ((VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))));
		child_type = vala_type_reference_copy (vala_expression_get_expected_type (VALA_EXPRESSION (list)));
		if (rank > 1) {
			ValaArray* __temp180;
			__temp180 = NULL;
			(vala_type_reference_set_data_type (child_type, VALA_DATA_TYPE ((__temp180 = vala_data_type_get_array (vala_array_get_element_type (edt), rank - 1)))), vala_type_reference_get_data_type (child_type));
			(__temp180 = (g_object_unref (__temp180), NULL));
		} else {
			(vala_type_reference_set_data_type (child_type, vala_array_get_element_type (edt)), vala_type_reference_get_data_type (child_type));
		}
		{
			GList* __temp182;
			GList* e_it;
			__temp182 = NULL;
			__temp182 = inits;
			for (e_it = __temp182; e_it != NULL; e_it = e_it->next) {
				ValaExpression* e;
				e = e_it->data;
				{
					ValaTypeReference* __temp181;
					__temp181 = NULL;
					(vala_expression_set_expected_type (e, (__temp181 = vala_type_reference_copy (child_type))), vala_expression_get_expected_type (e));
					(__temp181 = (g_object_unref (__temp181), NULL));
				}
			}
		}
		(edt == NULL ? NULL : (edt = (g_object_unref (edt), NULL)));
		(inits == NULL ? NULL : (inits = (g_list_free (inits), NULL)));
		(child_type = (g_object_unref (child_type), NULL));
	}
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_semantic_analyzer_real_visit_end_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (list)) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))) {
		ValaArray* __temp183;
		ValaArray* edt;
		GList* inits;
		gint rank;
		ValaTypeReference* child_type;
		gboolean error;
		__temp183 = NULL;
		edt = (__temp183 = VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list)))), (__temp183 == NULL ? NULL : g_object_ref (__temp183)));
		inits = vala_initializer_list_get_initializers (list);
		rank = vala_array_get_rank (edt);
		child_type = vala_type_reference_copy (vala_expression_get_expected_type (VALA_EXPRESSION (list)));
		error = FALSE;
		if (rank > 1) {
			ValaArray* __temp184;
			__temp184 = NULL;
			(vala_type_reference_set_data_type (child_type, VALA_DATA_TYPE ((__temp184 = vala_data_type_get_array (vala_array_get_element_type (edt), rank - 1)))), vala_type_reference_get_data_type (child_type));
			(__temp184 = (g_object_unref (__temp184), NULL));
			{
				GList* __temp186;
				GList* e_it;
				__temp186 = NULL;
				__temp186 = inits;
				for (e_it = __temp186; e_it != NULL; e_it = e_it->next) {
					ValaExpression* e;
					e = e_it->data;
					{
						if (vala_expression_get_static_type (e) == NULL) {
							error = TRUE;
							continue;
						}
						if (!(VALA_IS_INITIALIZER_LIST (e))) {
							error = TRUE;
							(vala_code_node_set_error (VALA_CODE_NODE (e), TRUE), vala_code_node_get_error (VALA_CODE_NODE (e)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Initializer list expected");
							continue;
						}
						if (!vala_type_reference_equals (vala_expression_get_static_type (e), child_type)) {
							char* __temp185;
							error = TRUE;
							(vala_code_node_set_error (VALA_CODE_NODE (e), TRUE), vala_code_node_get_error (VALA_CODE_NODE (e)));
							__temp185 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (__temp185 = g_strdup_printf ("Expected initializer list of type `%s' but got `%s'", vala_data_type_get_name (vala_type_reference_get_data_type (child_type)), vala_data_type_get_name (vala_type_reference_get_data_type (vala_expression_get_static_type (e))))));
							(__temp185 = (g_free (__temp185), NULL));
						}
					}
				}
			}
		} else {
			(vala_type_reference_set_data_type (child_type, vala_array_get_element_type (edt)), vala_type_reference_get_data_type (child_type));
			{
				GList* __temp188;
				GList* e_it;
				__temp188 = NULL;
				__temp188 = inits;
				for (e_it = __temp188; e_it != NULL; e_it = e_it->next) {
					ValaExpression* e;
					e = e_it->data;
					{
						if (vala_expression_get_static_type (e) == NULL) {
							error = TRUE;
							continue;
						}
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (e), child_type)) {
							char* __temp187;
							error = TRUE;
							(vala_code_node_set_error (VALA_CODE_NODE (e), TRUE), vala_code_node_get_error (VALA_CODE_NODE (e)));
							__temp187 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (__temp187 = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", vala_data_type_get_name (vala_type_reference_get_data_type (child_type)), vala_data_type_get_name (vala_type_reference_get_data_type (vala_expression_get_static_type (e))))));
							(__temp187 = (g_free (__temp187), NULL));
						}
					}
				}
			}
		}
		if (!error) {
			/* everything seems to be correct */
			(vala_expression_set_static_type (VALA_EXPRESSION (list), vala_expression_get_expected_type (VALA_EXPRESSION (list))), vala_expression_get_static_type (VALA_EXPRESSION (list)));
		}
		(edt == NULL ? NULL : (edt = (g_object_unref (edt), NULL)));
		(inits == NULL ? NULL : (inits = (g_list_free (inits), NULL)));
		(child_type = (g_object_unref (child_type), NULL));
	}
}


static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_EXPRESSION_STATEMENT (stmt));
	if (vala_expression_get_static_type (vala_expression_statement_get_expression (stmt)) != NULL && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_expression_statement_get_expression (stmt)))) {
		vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Short-living reference");
	}
	(vala_code_node_set_tree_can_fail (VALA_CODE_NODE (stmt), vala_code_node_get_tree_can_fail (VALA_CODE_NODE (vala_expression_statement_get_expression (stmt)))), vala_code_node_get_tree_can_fail (VALA_CODE_NODE (stmt)));
}


static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_IF_STATEMENT (stmt));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_if_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_if_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_if_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_WHILE_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_while_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_while_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOR_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_for_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_for_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_begin_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	ValaVariableDeclarator* __temp189;
	ValaSymbol* __temp190;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_foreach_statement_get_type_reference (stmt)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_foreach_statement_get_type_reference (stmt)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	__temp189 = NULL;
	(vala_foreach_statement_set_variable_declarator (stmt, (__temp189 = vala_variable_declarator_new (vala_foreach_statement_get_variable_name (stmt), NULL, NULL))), vala_foreach_statement_get_variable_declarator (stmt));
	(__temp189 == NULL ? NULL : (__temp189 = (g_object_unref (__temp189), NULL)));
	(vala_variable_declarator_set_type_reference (vala_foreach_statement_get_variable_declarator (stmt), vala_foreach_statement_get_type_reference (stmt)), vala_variable_declarator_get_type_reference (vala_foreach_statement_get_variable_declarator (stmt)));
	__temp190 = NULL;
	(vala_code_node_set_symbol (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt)), (__temp190 = vala_symbol_new (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt))))), vala_code_node_get_symbol (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt))));
	(__temp190 == NULL ? NULL : (__temp190 = (g_object_unref (__temp190), NULL)));
	vala_symbol_add (vala_code_node_get_symbol (VALA_CODE_NODE (vala_foreach_statement_get_body (stmt))), vala_foreach_statement_get_variable_name (stmt), vala_code_node_get_symbol (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt))));
}


static void vala_semantic_analyzer_real_visit_end_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	ValaDataType* __temp191;
	ValaDataType* collection_type;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
	__temp191 = NULL;
	collection_type = (__temp191 = vala_type_reference_get_data_type (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt))), (__temp191 == NULL ? NULL : g_object_ref (__temp191)));
	if (!(VALA_IS_ARRAY (collection_type) || collection_type == self->priv->glist_type || collection_type == self->priv->gslist_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Collection not iterable");
		(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
		return;
	}
	(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_RETURN_STATEMENT (stmt));
	if (self->priv->current_return_type == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return not allowed in this context");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) == NULL && vala_type_reference_get_data_type (self->priv->current_return_type) != NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return without value in non-void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_type_reference_get_data_type (self->priv->current_return_type) == NULL && vala_type_reference_get_type_parameter (self->priv->current_return_type) == NULL) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return with value in void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)), self->priv->current_return_type)) {
		char* __temp194;
		char* __temp193;
		char* __temp192;
		__temp194 = NULL;
		__temp193 = NULL;
		__temp192 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), (__temp194 = g_strdup_printf ("Return: Cannot convert from `%s' to `%s'", (__temp192 = vala_type_reference_to_string (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)))), (__temp193 = vala_type_reference_to_string (self->priv->current_return_type)))));
		(__temp194 = (g_free (__temp194), NULL));
		(__temp193 = (g_free (__temp193), NULL));
		(__temp192 = (g_free (__temp192), NULL));
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_type_reference_get_transfers_ownership (self->priv->current_return_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return value transfers ownership but method return type hasn't been declared to transfer ownership");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt)) != NULL && VALA_IS_VARIABLE_DECLARATOR (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt)))) && vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_type_reference_get_transfers_ownership (self->priv->current_return_type)) {
		vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Local variable with strong reference used as return value and method return type hasn't been declared to transfer ownership");
	}
}


static void vala_semantic_analyzer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_THROW_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_TRY_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause)
{
	ValaSemanticAnalyzer * self;
	ValaVariableDeclarator* __temp195;
	ValaTypeReference* __temp196;
	ValaSymbol* __temp197;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (clause));
	if (vala_type_reference_get_data_type (vala_catch_clause_get_type_reference (clause)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_catch_clause_get_type_reference (clause)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	__temp195 = NULL;
	(vala_catch_clause_set_variable_declarator (clause, (__temp195 = vala_variable_declarator_new (vala_catch_clause_get_variable_name (clause), NULL, NULL))), vala_catch_clause_get_variable_declarator (clause));
	(__temp195 == NULL ? NULL : (__temp195 = (g_object_unref (__temp195), NULL)));
	__temp196 = NULL;
	(vala_variable_declarator_set_type_reference (vala_catch_clause_get_variable_declarator (clause), (__temp196 = vala_type_reference_new ())), vala_variable_declarator_get_type_reference (vala_catch_clause_get_variable_declarator (clause)));
	(__temp196 == NULL ? NULL : (__temp196 = (g_object_unref (__temp196), NULL)));
	(vala_type_reference_set_data_type (vala_variable_declarator_get_type_reference (vala_catch_clause_get_variable_declarator (clause)), self->priv->gerror_type), vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (vala_catch_clause_get_variable_declarator (clause))));
	__temp197 = NULL;
	(vala_code_node_set_symbol (VALA_CODE_NODE (vala_catch_clause_get_variable_declarator (clause)), (__temp197 = vala_symbol_new (VALA_CODE_NODE (vala_catch_clause_get_variable_declarator (clause))))), vala_code_node_get_symbol (VALA_CODE_NODE (vala_catch_clause_get_variable_declarator (clause))));
	(__temp197 == NULL ? NULL : (__temp197 = (g_object_unref (__temp197), NULL)));
	vala_symbol_add (vala_code_node_get_symbol (VALA_CODE_NODE (vala_catch_clause_get_body (clause))), vala_catch_clause_get_variable_name (clause), vala_code_node_get_symbol (VALA_CODE_NODE (vala_catch_clause_get_variable_declarator (clause))));
	vala_code_node_accept_children (VALA_CODE_NODE (clause), VALA_CODE_VISITOR (self));
}


/**
 * Visit operation called for lock statements.
 *
 * @param stmt a lock statement
 */
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LOCK_STATEMENT (stmt));
	/* resource must be a member access and denote a Lockable */
	if (!(VALA_IS_MEMBER_ACCESS (vala_lock_statement_get_resource (stmt)) && VALA_IS_LOCKABLE (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt)))))) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		(vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE), vala_code_node_get_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))), "Expression is either not a member access or does not denote a lockable member");
		return;
	}
	/* parent symbol must be the current class */
	if (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt)))) != VALA_CODE_NODE (self->priv->current_class)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		(vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE), vala_code_node_get_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))), "Only members of the current class are lockable");
	}
	vala_lockable_set_lock_used ((VALA_LOCKABLE (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))))), TRUE);
}


static void vala_semantic_analyzer_real_visit_begin_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
	if (vala_array_creation_expression_get_initializer_list (expr) != NULL) {
		ValaTypeReference* __temp198;
		ValaArray* __temp199;
		__temp198 = NULL;
		(vala_expression_set_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr)), (__temp198 = vala_type_reference_copy (vala_array_creation_expression_get_element_type (expr)))), vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))));
		(__temp198 = (g_object_unref (__temp198), NULL));
		__temp199 = NULL;
		(vala_type_reference_set_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))), VALA_DATA_TYPE ((__temp199 = vala_data_type_get_array (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr)))), vala_array_creation_expression_get_rank (expr))))), vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr)))));
		(__temp199 = (g_object_unref (__temp199), NULL));
	}
}


/* FIXME: add element type to type_argument
*
 * Visit operations called for array creation expresions.
 *
 * @param expr an array creation expression
 */
static void vala_semantic_analyzer_real_visit_end_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	gint i;
	GList* size;
	ValaTypeReference* __temp201;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
	i = 0;
	size = vala_array_creation_expression_get_sizes (expr);
	/* check for errors in the size list */
	if (size != NULL) {
		{
			GList* __temp200;
			GList* e_it;
			__temp200 = NULL;
			__temp200 = size;
			for (e_it = __temp200; e_it != NULL; e_it = e_it->next) {
				ValaExpression* e;
				e = e_it->data;
				{
					if (vala_expression_get_static_type (e) == NULL) {
						(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
						return;
					} else {
						/* return on previous error */
						if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))) {
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expression of integer type expected");
						}
					}
				}
			}
		}
		if (vala_code_node_get_error (VALA_CODE_NODE (expr))) {
			(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
			return;
		}
	}
	/* check for wrong elements inside the initializer */
	if (vala_array_creation_expression_get_initializer_list (expr) != NULL && vala_expression_get_static_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))) == NULL) {
		(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
		return;
	}
	/* try to construct the type of the array */
	if (vala_array_creation_expression_get_element_type (expr) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Cannot determine the element type of the created array");
		(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
		return;
	}
	__temp201 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp201 = vala_type_reference_copy (vala_array_creation_expression_get_element_type (expr)))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp201 = (g_object_unref (__temp201), NULL));
	if (vala_type_reference_get_data_type (vala_array_creation_expression_get_element_type (expr)) != NULL) {
		ValaArray* __temp202;
		__temp202 = NULL;
		(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((__temp202 = vala_data_type_get_array (vala_type_reference_get_data_type (vala_array_creation_expression_get_element_type (expr)), vala_array_creation_expression_get_rank (expr))))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
		(__temp202 = (g_object_unref (__temp202), NULL));
	} else {
		ValaArray* __temp203;
		__temp203 = NULL;
		(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((__temp203 = vala_type_parameter_get_array (vala_type_reference_get_type_parameter (vala_array_creation_expression_get_element_type (expr)), vala_array_creation_expression_get_rank (expr))))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
		(__temp203 = (g_object_unref (__temp203), NULL));
	}
	(vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	vala_type_reference_add_type_argument (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_array_creation_expression_get_element_type (expr));
	(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
}


static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BOOLEAN_LITERAL (expr));
	(vala_literal_set_static_type (VALA_LITERAL (expr), self->priv->bool_type), vala_literal_get_static_type (VALA_LITERAL (expr)));
}


static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp204;
	ValaSymbol* __temp205;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CHARACTER_LITERAL (expr));
	__temp204 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp204 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp204 == NULL ? NULL : (__temp204 = (g_object_unref (__temp204), NULL)));
	__temp205 = NULL;
	(vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), VALA_DATA_TYPE (vala_symbol_get_node ((__temp205 = vala_symbol_lookup (self->priv->root_symbol, "char"))))), vala_type_reference_get_data_type (vala_literal_get_static_type (VALA_LITERAL (expr))));
	(__temp205 == NULL ? NULL : (__temp205 = (g_object_unref (__temp205), NULL)));
}


static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp206;
	ValaSymbol* __temp208;
	char* __temp207;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INTEGER_LITERAL (expr));
	__temp206 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp206 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp206 == NULL ? NULL : (__temp206 = (g_object_unref (__temp206), NULL)));
	__temp208 = NULL;
	__temp207 = NULL;
	(vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), VALA_DATA_TYPE (vala_symbol_get_node ((__temp208 = vala_symbol_lookup (self->priv->root_symbol, (__temp207 = vala_integer_literal_get_type_name (expr))))))), vala_type_reference_get_data_type (vala_literal_get_static_type (VALA_LITERAL (expr))));
	(__temp208 == NULL ? NULL : (__temp208 = (g_object_unref (__temp208), NULL)));
	(__temp207 = (g_free (__temp207), NULL));
}


static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp209;
	ValaSymbol* __temp211;
	char* __temp210;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_REAL_LITERAL (expr));
	__temp209 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp209 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp209 == NULL ? NULL : (__temp209 = (g_object_unref (__temp209), NULL)));
	__temp211 = NULL;
	__temp210 = NULL;
	(vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), VALA_DATA_TYPE (vala_symbol_get_node ((__temp211 = vala_symbol_lookup (self->priv->root_symbol, (__temp210 = vala_real_literal_get_type_name (expr))))))), vala_type_reference_get_data_type (vala_literal_get_static_type (VALA_LITERAL (expr))));
	(__temp211 == NULL ? NULL : (__temp211 = (g_object_unref (__temp211), NULL)));
	(__temp210 = (g_free (__temp210), NULL));
}


static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp212;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_STRING_LITERAL (expr));
	__temp212 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp212 = vala_type_reference_copy (self->priv->string_type))), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp212 = (g_object_unref (__temp212), NULL));
	(vala_type_reference_set_non_null (vala_literal_get_static_type (VALA_LITERAL (expr)), TRUE), vala_type_reference_get_non_null (vala_literal_get_static_type (VALA_LITERAL (expr))));
}


static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp213;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NULL_LITERAL (expr));
	/* empty TypeReference represents null */
	__temp213 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp213 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp213 == NULL ? NULL : (__temp213 = (g_object_unref (__temp213), NULL)));
}


static void vala_semantic_analyzer_real_visit_literal_expression (ValaCodeVisitor* base, ValaLiteralExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LITERAL_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_literal_get_static_type (vala_literal_expression_get_literal (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static ValaTypeReference* vala_semantic_analyzer_get_static_type_for_node (ValaSemanticAnalyzer* self, ValaCodeNode* node)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_CODE_NODE (node), NULL);
	if (VALA_IS_FIELD (node)) {
		ValaField* __temp214;
		ValaField* f;
		ValaTypeReference* __temp215;
		__temp214 = NULL;
		f = (__temp214 = VALA_FIELD (node), (__temp214 == NULL ? NULL : g_object_ref (__temp214)));
		__temp215 = NULL;
		return (__temp215 = g_object_ref (vala_field_get_type_reference (f)), (f == NULL ? NULL : (f = (g_object_unref (f), NULL))), __temp215);
		(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
	} else {
		if (VALA_IS_CONSTANT (node)) {
			ValaConstant* __temp216;
			ValaConstant* c;
			ValaTypeReference* __temp217;
			__temp216 = NULL;
			c = (__temp216 = VALA_CONSTANT (node), (__temp216 == NULL ? NULL : g_object_ref (__temp216)));
			__temp217 = NULL;
			return (__temp217 = g_object_ref (vala_constant_get_type_reference (c)), (c == NULL ? NULL : (c = (g_object_unref (c), NULL))), __temp217);
			(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
		} else {
			if (VALA_IS_PROPERTY (node)) {
				ValaProperty* __temp218;
				ValaProperty* prop;
				ValaTypeReference* type;
				ValaTypeReference* __temp219;
				__temp218 = NULL;
				prop = (__temp218 = VALA_PROPERTY (node), (__temp218 == NULL ? NULL : g_object_ref (__temp218)));
				type = vala_type_reference_copy (vala_property_get_type_reference (prop));
				(vala_type_reference_set_takes_ownership (type, FALSE), vala_type_reference_get_takes_ownership (type));
				__temp219 = NULL;
				return (__temp219 = type, (prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL))), __temp219);
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
				(type = (g_object_unref (type), NULL));
			} else {
				if (VALA_IS_FORMAL_PARAMETER (node)) {
					ValaFormalParameter* __temp220;
					ValaFormalParameter* p;
					ValaTypeReference* __temp221;
					ValaTypeReference* __temp222;
					__temp220 = NULL;
					p = (__temp220 = VALA_FORMAL_PARAMETER (node), (__temp220 == NULL ? NULL : g_object_ref (__temp220)));
					__temp221 = NULL;
					__temp222 = NULL;
					return (__temp222 = (__temp221 = vala_formal_parameter_get_type_reference (p), (__temp221 == NULL ? NULL : g_object_ref (__temp221))), (p == NULL ? NULL : (p = (g_object_unref (p), NULL))), __temp222);
					(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
				} else {
					if (VALA_IS_TYPE_REFERENCE (node)) {
						ValaTypeReference* __temp223;
						__temp223 = NULL;
						return (__temp223 = VALA_TYPE_REFERENCE (node), (__temp223 == NULL ? NULL : g_object_ref (__temp223)));
					} else {
						if (VALA_IS_VARIABLE_DECLARATOR (node)) {
							ValaVariableDeclarator* __temp225;
							ValaVariableDeclarator* decl;
							ValaTypeReference* __temp226;
							ValaTypeReference* __temp227;
							__temp225 = NULL;
							decl = (__temp225 = VALA_VARIABLE_DECLARATOR (node), (__temp225 == NULL ? NULL : g_object_ref (__temp225)));
							__temp226 = NULL;
							__temp227 = NULL;
							return (__temp227 = (__temp226 = vala_variable_declarator_get_type_reference (decl), (__temp226 == NULL ? NULL : g_object_ref (__temp226))), (decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL))), __temp227);
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						} else {
							if (VALA_IS_ENUM_VALUE (node) || VALA_IS_FLAGS_VALUE (node)) {
								ValaTypeReference* type;
								type = vala_type_reference_new ();
								(vala_type_reference_set_data_type (type, VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (node))))), vala_type_reference_get_data_type (type));
								return type;
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							}
						}
					}
				}
			}
		}
	}
	return VALA_TYPE_REFERENCE (NULL);
}


ValaSymbol* vala_semantic_analyzer_symbol_lookup_inherited (ValaSymbol* sym, const char* name)
{
	ValaSymbol* result;
	gpointer __temp243;
	g_return_val_if_fail (VALA_IS_SYMBOL (sym), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	result = vala_symbol_lookup (sym, name);
	if (result != NULL) {
		return result;
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (sym))) {
		ValaClass* __temp231;
		ValaClass* cl;
		__temp231 = NULL;
		cl = (__temp231 = VALA_CLASS (vala_symbol_get_node (sym)), (__temp231 == NULL ? NULL : g_object_ref (__temp231)));
		{
			GList* __temp234;
			GList* base_type_it;
			__temp234 = NULL;
			__temp234 = vala_class_get_base_types (cl);
			for (base_type_it = __temp234; base_type_it != NULL; base_type_it = base_type_it->next) {
				ValaTypeReference* base_type;
				base_type = base_type_it->data;
				{
					ValaSymbol* __temp232;
					__temp232 = NULL;
					result = (__temp232 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp232);
					if (result != NULL) {
						ValaSymbol* __temp233;
						__temp233 = NULL;
						return (__temp233 = result, (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp233);
					}
				}
			}
			(__temp234 == NULL ? NULL : (__temp234 = (g_list_free (__temp234), NULL)));
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (vala_symbol_get_node (sym))) {
			ValaStruct* __temp235;
			ValaStruct* st;
			__temp235 = NULL;
			st = (__temp235 = VALA_STRUCT (vala_symbol_get_node (sym)), (__temp235 == NULL ? NULL : g_object_ref (__temp235)));
			{
				GList* __temp238;
				GList* base_type_it;
				__temp238 = NULL;
				__temp238 = vala_struct_get_base_types (st);
				for (base_type_it = __temp238; base_type_it != NULL; base_type_it = base_type_it->next) {
					ValaTypeReference* base_type;
					base_type = base_type_it->data;
					{
						ValaSymbol* __temp236;
						__temp236 = NULL;
						result = (__temp236 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp236);
						if (result != NULL) {
							ValaSymbol* __temp237;
							__temp237 = NULL;
							return (__temp237 = result, (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp237);
						}
					}
				}
				(__temp238 == NULL ? NULL : (__temp238 = (g_list_free (__temp238), NULL)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		} else {
			if (VALA_IS_INTERFACE (vala_symbol_get_node (sym))) {
				ValaInterface* __temp239;
				ValaInterface* iface;
				__temp239 = NULL;
				iface = (__temp239 = VALA_INTERFACE (vala_symbol_get_node (sym)), (__temp239 == NULL ? NULL : g_object_ref (__temp239)));
				{
					GList* __temp242;
					GList* prerequisite_it;
					__temp242 = NULL;
					__temp242 = vala_interface_get_prerequisites (iface);
					for (prerequisite_it = __temp242; prerequisite_it != NULL; prerequisite_it = prerequisite_it->next) {
						ValaTypeReference* prerequisite;
						prerequisite = prerequisite_it->data;
						{
							ValaSymbol* __temp240;
							__temp240 = NULL;
							result = (__temp240 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (prerequisite))), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp240);
							if (result != NULL) {
								ValaSymbol* __temp241;
								__temp241 = NULL;
								return (__temp241 = result, (iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL))), __temp241);
							}
						}
					}
					(__temp242 == NULL ? NULL : (__temp242 = (g_list_free (__temp242), NULL)));
				}
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	return VALA_SYMBOL ((__temp243 = NULL, (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp243));
	(result == NULL ? NULL : (result = (g_object_unref (result), NULL)));
}


static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp244;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PARENTHESIZED_EXPRESSION (expr));
	__temp244 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp244 = vala_type_reference_copy (vala_expression_get_static_type (vala_parenthesized_expression_get_inner (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp244 = (g_object_unref (__temp244), NULL));
	/* don't call g_object_ref_sink on inner and outer expression*/
	(vala_type_reference_set_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE), vala_type_reference_get_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
}


static ValaDataType* vala_semantic_analyzer_find_parent_type (ValaSemanticAnalyzer* self, ValaSymbol* sym)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (sym == NULL || VALA_IS_SYMBOL (sym), NULL);
	while (sym != NULL) {
		if (VALA_IS_DATA_TYPE (vala_symbol_get_node (sym))) {
			ValaDataType* __temp245;
			__temp245 = NULL;
			return (__temp245 = VALA_DATA_TYPE (vala_symbol_get_node (sym)), (__temp245 == NULL ? NULL : g_object_ref (__temp245)));
		}
		sym = vala_symbol_get_parent_symbol (sym);
	}
	return VALA_DATA_TYPE (NULL);
}


static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* base_symbol;
	ValaCodeNode* __temp266;
	ValaCodeNode* member;
	ValaMemberAccessibility access;
	ValaTypeReference* __temp270;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (expr));
	base_symbol = NULL;
	if (vala_member_access_get_inner (expr) == NULL) {
		ValaSymbol* __temp249;
		ValaSymbol* __temp248;
		ValaSymbol* __temp250;
		ValaSymbol* sym;
		__temp249 = NULL;
		__temp248 = NULL;
		base_symbol = (__temp249 = (__temp248 = self->priv->current_symbol, (__temp248 == NULL ? NULL : g_object_ref (__temp248))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp249);
		__temp250 = NULL;
		sym = (__temp250 = self->priv->current_symbol, (__temp250 == NULL ? NULL : g_object_ref (__temp250)));
		while (sym != NULL && vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			ValaSymbol* __temp251;
			ValaSymbol* __temp253;
			ValaSymbol* __temp252;
			__temp251 = NULL;
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp251 = vala_semantic_analyzer_symbol_lookup_inherited (sym, vala_member_access_get_member_name (expr)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			(__temp251 == NULL ? NULL : (__temp251 = (g_object_unref (__temp251), NULL)));
			__temp253 = NULL;
			__temp252 = NULL;
			sym = (__temp253 = (__temp252 = vala_symbol_get_parent_symbol (sym), (__temp252 == NULL ? NULL : g_object_ref (__temp252))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp253);
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			{
				GList* __temp257;
				GList* ns_it;
				__temp257 = NULL;
				__temp257 = self->priv->current_using_directives;
				for (ns_it = __temp257; ns_it != NULL; ns_it = ns_it->next) {
					ValaNamespaceReference* ns;
					ns = ns_it->data;
					{
						ValaSymbol* local_sym;
						local_sym = vala_symbol_lookup (vala_namespace_reference_get_namespace_symbol (ns), vala_member_access_get_member_name (expr));
						if (local_sym != NULL) {
							if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
								char* __temp256;
								char* __temp255;
								char* __temp254;
								(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
								__temp256 = NULL;
								__temp255 = NULL;
								__temp254 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp256 = g_strdup_printf ("`%s' is an ambiguous reference between `%s' and `%s'", vala_member_access_get_member_name (expr), (__temp254 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))), (__temp255 = vala_symbol_get_full_name (local_sym)))));
								(__temp256 = (g_free (__temp256), NULL));
								(__temp255 = (g_free (__temp255), NULL));
								(__temp254 = (g_free (__temp254), NULL));
								(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
								return;
							}
							(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), local_sym), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
						}
						(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
					}
				}
			}
		}
		(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
	} else {
		if (vala_code_node_get_error (VALA_CODE_NODE (vala_member_access_get_inner (expr)))) {
			/* if there was an error in the inner expression, skip this check */
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			return;
		}
		if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (expr)) || VALA_IS_BASE_ACCESS (vala_member_access_get_inner (expr))) {
			ValaSymbol* __temp259;
			ValaSymbol* __temp258;
			__temp259 = NULL;
			__temp258 = NULL;
			base_symbol = (__temp259 = (__temp258 = vala_expression_get_symbol_reference (vala_member_access_get_inner (expr)), (__temp258 == NULL ? NULL : g_object_ref (__temp258))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp259);
			if (VALA_IS_NAMESPACE (vala_symbol_get_node (base_symbol)) || VALA_IS_DATA_TYPE (vala_symbol_get_node (base_symbol))) {
				ValaSymbol* __temp260;
				__temp260 = NULL;
				(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp260 = vala_symbol_lookup (base_symbol, vala_member_access_get_member_name (expr)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
				(__temp260 == NULL ? NULL : (__temp260 = (g_object_unref (__temp260), NULL)));
			}
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_expression_get_static_type (vala_member_access_get_inner (expr)) != NULL) {
			ValaSymbol* __temp262;
			ValaSymbol* __temp261;
			ValaSymbol* __temp263;
			__temp262 = NULL;
			__temp261 = NULL;
			base_symbol = (__temp262 = (__temp261 = vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_member_access_get_inner (expr))))), (__temp261 == NULL ? NULL : g_object_ref (__temp261))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp262);
			__temp263 = NULL;
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp263 = vala_semantic_analyzer_symbol_lookup_inherited (base_symbol, vala_member_access_get_member_name (expr)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			(__temp263 == NULL ? NULL : (__temp263 = (g_object_unref (__temp263), NULL)));
		}
	}
	if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
		char* __temp265;
		char* __temp264;
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		__temp265 = NULL;
		__temp264 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp265 = g_strdup_printf ("The name `%s' does not exist in the context of `%s'", vala_member_access_get_member_name (expr), (__temp264 = vala_symbol_get_full_name (base_symbol)))));
		(__temp265 = (g_free (__temp265), NULL));
		(__temp264 = (g_free (__temp264), NULL));
		(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
		return;
	}
	__temp266 = NULL;
	member = (__temp266 = vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr))), (__temp266 == NULL ? NULL : g_object_ref (__temp266)));
	access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	if (VALA_IS_FIELD (member)) {
		access = (VALA_FIELD (member))->access;
	} else {
		if (VALA_IS_METHOD (member)) {
			access = (VALA_METHOD (member))->access;
		}
	}
	if (access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		ValaDataType* __temp267;
		ValaDataType* target_type;
		ValaDataType* this_type;
		__temp267 = NULL;
		target_type = (__temp267 = VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (member)))), (__temp267 == NULL ? NULL : g_object_ref (__temp267)));
		this_type = vala_semantic_analyzer_find_parent_type (self, self->priv->current_symbol);
		if (target_type != this_type) {
			char* __temp269;
			char* __temp268;
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			__temp269 = NULL;
			__temp268 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp269 = g_strdup_printf ("Access to private member `%s' denied", (__temp268 = vala_symbol_get_full_name (vala_code_node_get_symbol (member))))));
			(__temp269 = (g_free (__temp269), NULL));
			(__temp268 = (g_free (__temp268), NULL));
			(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
			(this_type == NULL ? NULL : (this_type = (g_object_unref (this_type), NULL)));
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
			return;
		}
		(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
		(this_type == NULL ? NULL : (this_type = (g_object_unref (this_type), NULL)));
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	__temp270 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp270 = vala_semantic_analyzer_get_static_type_for_node (self, vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp270 == NULL ? NULL : (__temp270 = (g_object_unref (__temp270), NULL)));
	(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
	(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
}


static gboolean vala_semantic_analyzer_is_type_compatible (ValaSemanticAnalyzer* self, ValaTypeReference* expression_type, ValaTypeReference* expected_type)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (expression_type), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (expected_type), FALSE);
	/* only null is compatible to null */
	if (vala_type_reference_get_data_type (expected_type) == NULL && vala_type_reference_get_type_parameter (expected_type) == NULL) {
		return (vala_type_reference_get_data_type (expression_type) == NULL && vala_type_reference_get_type_parameter (expected_type) == NULL);
	}
	if (vala_type_reference_get_data_type (expression_type) == NULL) {
		/* null can be cast to any reference or array type or pointer type */
		if (vala_type_reference_get_type_parameter (expected_type) != NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (expected_type)) || vala_type_reference_get_is_out (expected_type) || VALA_IS_POINTER (vala_type_reference_get_data_type (expected_type)) || VALA_IS_ARRAY (vala_type_reference_get_data_type (expected_type)) || VALA_IS_CALLBACK (vala_type_reference_get_data_type (expected_type)) || vala_type_reference_get_data_type (expected_type) == self->priv->pointer_type) {
			return TRUE;
		}
		return FALSE;
	}
	/* null is not compatible with any other type (i.e. value types) */
	if (vala_type_reference_get_data_type (expected_type) == self->priv->pointer_type) {
		/* any reference or array type or pointer type can be cast to a generic pointer */
		if (vala_type_reference_get_type_parameter (expression_type) != NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (expression_type)) || VALA_IS_POINTER (vala_type_reference_get_data_type (expression_type)) || VALA_IS_ARRAY (vala_type_reference_get_data_type (expression_type)) || VALA_IS_CALLBACK (vala_type_reference_get_data_type (expression_type)) || vala_type_reference_get_data_type (expression_type) == self->priv->pointer_type) {
			return TRUE;
		}
		return FALSE;
	}
	/* temporarily ignore type parameters */
	if (vala_type_reference_get_type_parameter (expected_type) != NULL) {
		return TRUE;
	}
	if (VALA_IS_ARRAY (vala_type_reference_get_data_type (expression_type)) != VALA_IS_ARRAY (vala_type_reference_get_data_type (expected_type))) {
		return FALSE;
	}
	if (VALA_IS_ENUM (vala_type_reference_get_data_type (expression_type)) && vala_type_reference_get_data_type (expected_type) == vala_type_reference_get_data_type (self->priv->int_type)) {
		return TRUE;
	}
	if (vala_type_reference_get_data_type (expression_type) == vala_type_reference_get_data_type (expected_type)) {
		return TRUE;
	}
	if (VALA_IS_STRUCT (vala_type_reference_get_data_type (expression_type)) && VALA_IS_STRUCT (vala_type_reference_get_data_type (expected_type))) {
		ValaStruct* __temp280;
		ValaStruct* expr_struct;
		ValaStruct* __temp281;
		ValaStruct* expect_struct;
		__temp280 = NULL;
		expr_struct = (__temp280 = VALA_STRUCT (vala_type_reference_get_data_type (expression_type)), (__temp280 == NULL ? NULL : g_object_ref (__temp280)));
		__temp281 = NULL;
		expect_struct = (__temp281 = VALA_STRUCT (vala_type_reference_get_data_type (expected_type)), (__temp281 == NULL ? NULL : g_object_ref (__temp281)));
		/* integer types may be implicitly cast to floating point types */
		if (vala_struct_is_integer_type (expr_struct) && vala_struct_is_floating_type (expect_struct)) {
			gboolean __temp282;
			return (__temp282 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), __temp282);
		}
		if ((vala_struct_is_integer_type (expr_struct) && vala_struct_is_integer_type (expect_struct)) || (vala_struct_is_floating_type (expr_struct) && vala_struct_is_floating_type (expect_struct))) {
			if (vala_struct_get_rank (expr_struct) <= vala_struct_get_rank (expect_struct)) {
				gboolean __temp283;
				return (__temp283 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), __temp283);
			}
		}
		(expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL)));
		(expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL)));
	}
	return vala_data_type_is_subtype_of (vala_type_reference_get_data_type (expression_type), vala_type_reference_get_data_type (expected_type));
}


static void vala_semantic_analyzer_real_visit_begin_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp285;
	ValaSymbol* msym;
	GList* params;
	GList* args;
	GList* arg_it;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_invocation_expression_get_call (expr)))) {
		/* if method resolving didn't succeed, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	__temp285 = NULL;
	msym = (__temp285 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (__temp285 == NULL ? NULL : g_object_ref (__temp285)));
	if (msym == NULL) {
		/* if no symbol found, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
		return;
	}
	params = NULL;
	if (VALA_IS_INVOKABLE (vala_symbol_get_node (msym))) {
		ValaInvokable* __temp286;
		ValaInvokable* m;
		__temp286 = NULL;
		m = (__temp286 = VALA_INVOKABLE (vala_symbol_get_node (msym)), (__temp286 == NULL ? NULL : g_object_ref (__temp286)));
		if (vala_invokable_is_invokable (m)) {
			GList* __temp287;
			__temp287 = NULL;
			params = (__temp287 = vala_invokable_get_parameters (m), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp287);
		} else {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation not supported in this context");
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
			(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
			return;
		}
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	} else {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation not supported in this context");
		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
		(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
		return;
	}
	args = vala_invocation_expression_get_argument_list (expr);
	arg_it = args;
	{
		GList* __temp289;
		GList* param_it;
		__temp289 = NULL;
		__temp289 = params;
		for (param_it = __temp289; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param;
			param = param_it->data;
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					break;
				}
				if (arg_it != NULL) {
					ValaExpression* __temp288;
					ValaExpression* arg;
					__temp288 = NULL;
					arg = (__temp288 = VALA_EXPRESSION (arg_it->data), (__temp288 == NULL ? NULL : g_object_ref (__temp288)));
					/* store expected type for callback parameters */
					(vala_expression_set_expected_type (arg, vala_formal_parameter_get_type_reference (param)), vala_expression_get_expected_type (arg));
					arg_it = arg_it->next;
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
			}
		}
	}
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
	(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
	(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
}


static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaSymbol* msym, GList* params, GList* args)
{
	GList* prev_arg_it;
	GList* arg_it;
	ValaAttribute* __temp290;
	gboolean __temp291;
	gboolean diag;
	gboolean ellipsis;
	gint i;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), FALSE);
	g_return_val_if_fail (VALA_IS_SYMBOL (msym), FALSE);
	prev_arg_it = NULL;
	arg_it = args;
	__temp290 = NULL;
	diag = (__temp291 = ((__temp290 = vala_code_node_get_attribute (vala_symbol_get_node (msym), "Diagnostics")) != NULL), (__temp290 == NULL ? NULL : (__temp290 = (g_object_unref (__temp290), NULL))), __temp291);
	ellipsis = FALSE;
	i = 0;
	{
		GList* __temp302;
		GList* param_it;
		__temp302 = NULL;
		__temp302 = params;
		for (param_it = __temp302; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param;
			param = param_it->data;
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					ellipsis = TRUE;
					break;
				}
				/* header file necessary if we need to cast argument */
				if (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)) != NULL) {
					vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
				}
				if (arg_it == NULL) {
					if (vala_formal_parameter_get_default_expression (param) == NULL) {
						char* __temp293;
						char* __temp292;
						(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
						__temp293 = NULL;
						__temp292 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp293 = g_strdup_printf ("Too few arguments, method `%s' does not take %d arguments", (__temp292 = vala_symbol_get_full_name (msym)), g_list_length (args))));
						(__temp293 = (g_free (__temp293), NULL));
						(__temp292 = (g_free (__temp292), NULL));
						return FALSE;
					}
				} else {
					ValaExpression* __temp295;
					ValaExpression* arg;
					__temp295 = NULL;
					arg = (__temp295 = VALA_EXPRESSION (arg_it->data), (__temp295 == NULL ? NULL : g_object_ref (__temp295)));
					if (vala_expression_get_static_type (arg) == NULL) {
						/* disallow untyped arguments except for type inference of callbacks*/
						if (!(VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param))))) {
							char* __temp296;
							gboolean __temp297;
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							__temp296 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp296 = g_strdup_printf ("Invalid type for argument %d", i + 1)));
							(__temp296 = (g_free (__temp296), NULL));
							return (__temp297 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), __temp297);
						}
					} else {
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (arg), vala_formal_parameter_get_type_reference (param))) {
							char* __temp300;
							char* __temp299;
							char* __temp298;
							gboolean __temp301;
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							__temp300 = NULL;
							__temp299 = NULL;
							__temp298 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp300 = g_strdup_printf ("Argument %d: Cannot convert from `%s' to `%s'", i + 1, (__temp298 = vala_type_reference_to_string (vala_expression_get_static_type (arg))), (__temp299 = vala_type_reference_to_string (vala_formal_parameter_get_type_reference (param))))));
							(__temp300 = (g_free (__temp300), NULL));
							(__temp299 = (g_free (__temp299), NULL));
							(__temp298 = (g_free (__temp298), NULL));
							return (__temp301 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), __temp301);
						}
					}
					prev_arg_it = arg_it;
					arg_it = arg_it->next;
					i++;
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
			}
		}
	}
	if (!ellipsis && arg_it != NULL) {
		char* __temp304;
		char* __temp303;
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		__temp304 = NULL;
		__temp303 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp304 = g_strdup_printf ("Too many arguments, method `%s' does not take %d arguments", (__temp303 = vala_symbol_get_full_name (msym)), g_list_length (args))));
		(__temp304 = (g_free (__temp304), NULL));
		(__temp303 = (g_free (__temp303), NULL));
		return FALSE;
	}
	if (diag && prev_arg_it != NULL) {
		ValaExpression* __temp306;
		ValaExpression* format_arg;
		__temp306 = NULL;
		format_arg = (__temp306 = VALA_EXPRESSION (prev_arg_it->data), (__temp306 == NULL ? NULL : g_object_ref (__temp306)));
		if (VALA_IS_LITERAL_EXPRESSION (format_arg)) {
			ValaStringLiteral* __temp307;
			ValaStringLiteral* format_lit;
			char* __temp308;
			__temp307 = NULL;
			format_lit = (__temp307 = VALA_STRING_LITERAL (vala_literal_expression_get_literal ((VALA_LITERAL_EXPRESSION (format_arg)))), (__temp307 == NULL ? NULL : g_object_ref (__temp307)));
			__temp308 = NULL;
			(vala_string_literal_set_value (format_lit, (__temp308 = g_strdup_printf ("\"%s:%d: %s", vala_source_file_get_filename (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)))), vala_source_reference_get_first_line (vala_code_node_get_source_reference (VALA_CODE_NODE (expr))), g_utf8_offset_to_pointer (vala_string_literal_get_value (format_lit), 1)))), vala_string_literal_get_value (format_lit));
			(__temp308 = (g_free (__temp308), NULL));
			(format_lit == NULL ? NULL : (format_lit = (g_object_unref (format_lit), NULL)));
		}
		(format_arg == NULL ? NULL : (format_arg = (g_object_unref (format_arg), NULL)));
	}
	return TRUE;
}


static void vala_semantic_analyzer_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaSymbol* __temp310;
	ValaSymbol* msym;
	ValaTypeReference* ret_type;
	GList* params;
	GList* __temp335;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (expr))) {
		return;
	}
	__temp310 = NULL;
	msym = (__temp310 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (__temp310 == NULL ? NULL : g_object_ref (__temp310)));
	ret_type = NULL;
	params = NULL;
	if (VALA_IS_INVOKABLE (vala_symbol_get_node (msym))) {
		ValaInvokable* __temp311;
		ValaInvokable* m;
		ValaTypeReference* __temp312;
		GList* __temp313;
		__temp311 = NULL;
		m = (__temp311 = VALA_INVOKABLE (vala_symbol_get_node (msym)), (__temp311 == NULL ? NULL : g_object_ref (__temp311)));
		__temp312 = NULL;
		ret_type = (__temp312 = vala_invokable_get_return_type (m), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), __temp312);
		__temp313 = NULL;
		params = (__temp313 = vala_invokable_get_parameters (m), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp313);
		if (vala_type_reference_get_data_type (ret_type) == NULL && vala_type_reference_get_type_parameter (ret_type) == NULL) {
			/* void return type*/
			if (!(VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node (VALA_CODE_NODE (expr))))) {
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation of void method not allowed as expression");
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
				return;
			}
		}
		/* resolve generic return values*/
		if (vala_type_reference_get_type_parameter (ret_type) != NULL) {
			ValaMemberAccess* __temp314;
			ValaMemberAccess* ma;
			if (!(VALA_IS_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)))) {
				vala_report_error (vala_code_node_get_source_reference ((VALA_CODE_NODE (m))), "internal error: unsupported generic return value");
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
				return;
			}
			__temp314 = NULL;
			ma = (__temp314 = VALA_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)), (__temp314 == NULL ? NULL : g_object_ref (__temp314)));
			if (vala_member_access_get_inner (ma) == NULL) {
				/* TODO resolve generic return values within the type hierarchy if possible*/
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: resolving generic return values within type hierarchy not supported yet");
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
				return;
			} else {
				ValaTypeReference* __temp315;
				ValaTypeReference* instance_type;
				gint param_index;
				ValaTypeReference* __temp331;
				ValaTypeReference* __temp330;
				GList* __temp329;
				__temp315 = NULL;
				instance_type = (__temp315 = vala_expression_get_static_type (vala_member_access_get_inner (ma)), (__temp315 == NULL ? NULL : g_object_ref (__temp315)));
				while (VALA_CODE_NODE (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_node (vala_symbol_get_parent_symbol (msym))) {
					GList* base_types;
					/* trace type arguments back to the datatype where the method has been declared*/
					base_types = NULL;
					if (VALA_IS_CLASS (vala_type_reference_get_data_type (instance_type))) {
						ValaClass* __temp316;
						ValaClass* cl;
						GList* __temp317;
						__temp316 = NULL;
						cl = (__temp316 = VALA_CLASS (vala_type_reference_get_data_type (instance_type)), (__temp316 == NULL ? NULL : g_object_ref (__temp316)));
						__temp317 = NULL;
						base_types = (__temp317 = vala_class_get_base_types (cl), (base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL))), __temp317);
						(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
					} else {
						if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (instance_type))) {
							ValaInterface* __temp318;
							ValaInterface* iface;
							GList* __temp319;
							__temp318 = NULL;
							iface = (__temp318 = VALA_INTERFACE (vala_type_reference_get_data_type (instance_type)), (__temp318 == NULL ? NULL : g_object_ref (__temp318)));
							__temp319 = NULL;
							base_types = (__temp319 = vala_interface_get_prerequisites (iface), (base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL))), __temp319);
							(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
						} else {
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unsupported generic type");
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
							(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
							(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
							(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
							return;
						}
					}
					{
						GList* __temp327;
						GList* base_type_it;
						__temp327 = NULL;
						__temp327 = base_types;
						for (base_type_it = __temp327; base_type_it != NULL; base_type_it = base_type_it->next) {
							ValaTypeReference* base_type;
							base_type = base_type_it->data;
							{
								ValaSymbol* __temp320;
								gboolean __temp321;
								__temp320 = NULL;
								if ((__temp321 = (__temp320 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), vala_symbol_get_name (msym))) != NULL, (__temp320 == NULL ? NULL : (__temp320 = (g_object_unref (__temp320), NULL))), __temp321)) {
									ValaTypeReference* instance_base_type;
									ValaTypeReference* __temp326;
									ValaTypeReference* __temp325;
									/* construct a new type reference for the base type with correctly linked type arguments*/
									instance_base_type = vala_type_reference_new ();
									(vala_type_reference_set_data_type (instance_base_type, vala_type_reference_get_data_type (base_type)), vala_type_reference_get_data_type (instance_base_type));
									{
										GList* __temp324;
										GList* type_arg_it;
										__temp324 = NULL;
										__temp324 = vala_type_reference_get_type_arguments (base_type);
										for (type_arg_it = __temp324; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
											ValaTypeReference* type_arg;
											type_arg = type_arg_it->data;
											{
												if (vala_type_reference_get_type_parameter (type_arg) != NULL) {
													gint param_index;
													GList* __temp323;
													/* link to type argument of derived type*/
													param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_type_parameter_get_name (vala_type_reference_get_type_parameter (type_arg)));
													if (param_index == -1) {
														char* __temp322;
														__temp322 = NULL;
														vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp322 = g_strdup_printf ("internal error: unknown type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (type_arg)))));
														(__temp322 = (g_free (__temp322), NULL));
														(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
														(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
														(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
														(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
														(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
														(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
														(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
														(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
														(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
														return;
													}
													__temp323 = NULL;
													type_arg = g_list_nth_data ((__temp323 = vala_type_reference_get_type_arguments (instance_type)), param_index);
													(__temp323 == NULL ? NULL : (__temp323 = (g_list_free (__temp323), NULL)));
												}
												vala_type_reference_add_type_argument (instance_base_type, type_arg);
											}
										}
										(__temp324 == NULL ? NULL : (__temp324 = (g_list_free (__temp324), NULL)));
									}
									__temp326 = NULL;
									__temp325 = NULL;
									instance_type = (__temp326 = (__temp325 = instance_base_type, (__temp325 == NULL ? NULL : g_object_ref (__temp325))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp326);
									(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
								}
							}
						}
					}
					(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
				}
				if (VALA_CODE_NODE (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_node (vala_symbol_get_parent_symbol (msym))) {
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: generic type parameter tracing not supported yet");
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
					return;
				}
				param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_type_parameter_get_name (vala_type_reference_get_type_parameter (ret_type)));
				if (param_index == -1) {
					char* __temp328;
					__temp328 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp328 = g_strdup_printf ("internal error: unknown type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (ret_type)))));
					(__temp328 = (g_free (__temp328), NULL));
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
					return;
				}
				__temp331 = NULL;
				__temp330 = NULL;
				__temp329 = NULL;
				ret_type = (__temp331 = (__temp330 = VALA_TYPE_REFERENCE (g_list_nth_data ((__temp329 = vala_type_reference_get_type_arguments (instance_type)), param_index)), (__temp330 == NULL ? NULL : g_object_ref (__temp330))), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), __temp331);
				(__temp329 == NULL ? NULL : (__temp329 = (g_list_free (__temp329), NULL)));
				if (ret_type == NULL) {
					char* __temp332;
					__temp332 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp332 = g_strdup_printf ("internal error: no actual argument found for type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (ret_type)))));
					(__temp332 = (g_free (__temp332), NULL));
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
					return;
				}
				(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
			}
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
		}
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	if (VALA_IS_METHOD (vala_symbol_get_node (msym))) {
		ValaMethod* __temp333;
		ValaMethod* m;
		GList* __temp334;
		__temp333 = NULL;
		m = (__temp333 = VALA_METHOD (vala_symbol_get_node (msym)), (__temp333 == NULL ? NULL : g_object_ref (__temp333)));
		__temp334 = NULL;
		(vala_code_node_set_tree_can_fail (VALA_CODE_NODE (expr), (vala_expression_set_can_fail (VALA_EXPRESSION (expr), (g_list_length ((__temp334 = vala_method_get_error_domains (m))) > 0)), vala_expression_get_can_fail (VALA_EXPRESSION (expr)))), vala_code_node_get_tree_can_fail (VALA_CODE_NODE (expr)));
		(__temp334 == NULL ? NULL : (__temp334 = (g_list_free (__temp334), NULL)));
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), ret_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	__temp335 = NULL;
	vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), msym, params, (__temp335 = vala_invocation_expression_get_argument_list (expr)));
	(__temp335 == NULL ? NULL : (__temp335 = (g_list_free (__temp335), NULL)));
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
	(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
	(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
}


static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ELEMENT_ACCESS (expr));
	if (vala_expression_get_static_type (vala_element_access_get_container (expr)) == NULL) {
		/* don't proceed if a child expression failed */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	/* assign a static_type when possible */
	if (VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_element_access_get_container (expr))))) {
		GList* args;
		args = vala_type_reference_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (expr)));
		if (g_list_length (args) != 1) {
			char* __temp336;
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			__temp336 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp336 = g_strdup_printf ("internal error: array reference with %d type arguments, expected 1", g_list_length (args))));
			(__temp336 = (g_free (__temp336), NULL));
			(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
			return;
		}
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_TYPE_REFERENCE (args->data)), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
	} else {
		if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_element_access_get_container (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
			GList* __temp337;
			gboolean __temp338;
			__temp337 = NULL;
			if ((__temp338 = g_list_length ((__temp337 = vala_element_access_get_indices (expr))) != 1, (__temp337 == NULL ? NULL : (__temp337 = (g_list_free (__temp337), NULL))), __temp338)) {
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Element access with more than one dimension is not supported for strings");
				return;
			}
			(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->unichar_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		} else {
			char* __temp340;
			char* __temp339;
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			__temp340 = NULL;
			__temp339 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp340 = g_strdup_printf ("The expression `%s' does not denote an Array", (__temp339 = vala_type_reference_to_string (vala_expression_get_static_type (vala_element_access_get_container (expr)))))));
			(__temp340 = (g_free (__temp340), NULL));
			(__temp339 = (g_free (__temp339), NULL));
		}
	}
	{
		GList* __temp341;
		GList* e_it;
		__temp341 = NULL;
		__temp341 = vala_element_access_get_indices (expr);
		for (e_it = __temp341; e_it != NULL; e_it = e_it->next) {
			ValaExpression* e;
			e = e_it->data;
			{
				/* check if the index is of type integer 
				 don't proceed if a child expression failed */
				if (vala_expression_get_static_type (e) == NULL) {
					return;
				}
				/* check if the index is of type integer */
				if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))) {
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expression of integer type expected");
				}
			}
		}
		(__temp341 == NULL ? NULL : (__temp341 = (g_list_free (__temp341), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BASE_ACCESS (expr));
	if (self->priv->current_class == NULL) {
		GList* __temp346;
		if (self->priv->current_struct == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Base access invalid outside of class and struct");
			return;
		} else {
			GList* __temp342;
			gboolean __temp343;
			__temp342 = NULL;
			if ((__temp343 = g_list_length ((__temp342 = vala_struct_get_base_types (self->priv->current_struct))) != 1, (__temp342 == NULL ? NULL : (__temp342 = (g_list_free (__temp342), NULL))), __temp343)) {
				char* __temp345;
				GList* __temp344;
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				__temp345 = NULL;
				__temp344 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp345 = g_strdup_printf ("Base access invalid without base type %d", g_list_length ((__temp344 = vala_struct_get_base_types (self->priv->current_struct))))));
				(__temp345 = (g_free (__temp345), NULL));
				(__temp344 == NULL ? NULL : (__temp344 = (g_list_free (__temp344), NULL)));
				return;
			}
		}
		__temp346 = NULL;
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), g_list_first ((__temp346 = vala_struct_get_base_types (self->priv->current_struct)))->data), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(__temp346 == NULL ? NULL : (__temp346 = (g_list_free (__temp346), NULL)));
	} else {
		ValaTypeReference* __temp347;
		__temp347 = NULL;
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp347 = vala_type_reference_new ())), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(__temp347 == NULL ? NULL : (__temp347 = (g_object_unref (__temp347), NULL)));
		(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE (vala_class_get_base_class (self->priv->current_class))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	}
	(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)))))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_POSTFIX_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_postfix_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_end_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaDataType* type;
	ValaTypeReference* __temp370;
	GList* __temp377;
	gboolean __temp378;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (expr));
	type = NULL;
	if (vala_object_creation_expression_get_type_reference (expr) == NULL) {
		ValaCodeNode* __temp348;
		ValaCodeNode* constructor_node;
		ValaCodeNode* __temp349;
		ValaCodeNode* type_node;
		GList* type_args;
		ValaTypeReference* __temp364;
		if (vala_object_creation_expression_get_member_name (expr) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Incomplete object creation expression");
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr))) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		__temp348 = NULL;
		constructor_node = (__temp348 = vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr)))), (__temp348 == NULL ? NULL : g_object_ref (__temp348)));
		__temp349 = NULL;
		type_node = (__temp349 = vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr)))), (__temp349 == NULL ? NULL : g_object_ref (__temp349)));
		type_args = vala_member_access_get_type_arguments (vala_object_creation_expression_get_member_name (expr));
		if (VALA_IS_METHOD (constructor_node)) {
			ValaCodeNode* __temp351;
			ValaCodeNode* __temp350;
			ValaMethod* __temp352;
			ValaMethod* constructor;
			GList* __temp355;
			__temp351 = NULL;
			__temp350 = NULL;
			type_node = (__temp351 = (__temp350 = vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (constructor_node))), (__temp350 == NULL ? NULL : g_object_ref (__temp350))), (type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL))), __temp351);
			__temp352 = NULL;
			constructor = (__temp352 = VALA_METHOD (constructor_node), (__temp352 == NULL ? NULL : g_object_ref (__temp352)));
			if (!(VALA_IS_CREATION_METHOD (constructor_node))) {
				char* __temp354;
				char* __temp353;
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				__temp354 = NULL;
				__temp353 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp354 = g_strdup_printf ("`%s' is not a creation method", (__temp353 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (constructor)))))));
				(__temp354 = (g_free (__temp354), NULL));
				(__temp353 = (g_free (__temp353), NULL));
				(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
				(constructor_node == NULL ? NULL : (constructor_node = (g_object_unref (constructor_node), NULL)));
				(type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL)));
				(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				return;
			}
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (constructor))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			__temp355 = NULL;
			type_args = (__temp355 = vala_member_access_get_type_arguments ((VALA_MEMBER_ACCESS (vala_member_access_get_inner (vala_object_creation_expression_get_member_name (expr))))), (type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL))), __temp355);
			(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
		} else {
			if (VALA_IS_ENUM_VALUE (constructor_node)) {
				ValaCodeNode* __temp357;
				ValaCodeNode* __temp356;
				__temp357 = NULL;
				__temp356 = NULL;
				type_node = (__temp357 = (__temp356 = vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (constructor_node))), (__temp356 == NULL ? NULL : g_object_ref (__temp356))), (type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL))), __temp357);
				(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (constructor_node)), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			}
		}
		if (VALA_IS_CLASS (type_node) || VALA_IS_STRUCT (type_node)) {
			ValaDataType* __temp359;
			ValaDataType* __temp358;
			__temp359 = NULL;
			__temp358 = NULL;
			type = (__temp359 = (__temp358 = VALA_DATA_TYPE (type_node), (__temp358 == NULL ? NULL : g_object_ref (__temp358))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp359);
		} else {
			if (VALA_IS_ENUM (type_node) && vala_enum_get_error_domain ((VALA_ENUM (type_node)))) {
				ValaDataType* __temp361;
				ValaDataType* __temp360;
				__temp361 = NULL;
				__temp360 = NULL;
				type = (__temp361 = (__temp360 = VALA_DATA_TYPE (type_node), (__temp360 == NULL ? NULL : g_object_ref (__temp360))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp361);
			} else {
				char* __temp363;
				char* __temp362;
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				__temp363 = NULL;
				__temp362 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp363 = g_strdup_printf ("`%s' is not a class, struct, or error domain", (__temp362 = vala_symbol_get_full_name (vala_code_node_get_symbol (type_node))))));
				(__temp363 = (g_free (__temp363), NULL));
				(__temp362 = (g_free (__temp362), NULL));
				(constructor_node == NULL ? NULL : (constructor_node = (g_object_unref (constructor_node), NULL)));
				(type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL)));
				(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				return;
			}
		}
		__temp364 = NULL;
		(vala_object_creation_expression_set_type_reference (expr, (__temp364 = vala_type_reference_new ())), vala_object_creation_expression_get_type_reference (expr));
		(__temp364 == NULL ? NULL : (__temp364 = (g_object_unref (__temp364), NULL)));
		(vala_type_reference_set_data_type (vala_object_creation_expression_get_type_reference (expr), type), vala_type_reference_get_data_type (vala_object_creation_expression_get_type_reference (expr)));
		{
			GList* __temp365;
			GList* type_arg_it;
			__temp365 = NULL;
			__temp365 = type_args;
			for (type_arg_it = __temp365; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
				ValaTypeReference* type_arg;
				type_arg = type_arg_it->data;
				{
					vala_type_reference_add_type_argument (vala_object_creation_expression_get_type_reference (expr), type_arg);
				}
			}
		}
		(constructor_node == NULL ? NULL : (constructor_node = (g_object_unref (constructor_node), NULL)));
		(type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL)));
		(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
	} else {
		ValaDataType* __temp367;
		ValaDataType* __temp366;
		__temp367 = NULL;
		__temp366 = NULL;
		type = (__temp367 = (__temp366 = vala_type_reference_get_data_type (vala_object_creation_expression_get_type_reference (expr)), (__temp366 == NULL ? NULL : g_object_ref (__temp366))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp367);
	}
	if (!vala_data_type_is_reference_type (type) && !(VALA_IS_ENUM (type))) {
		char* __temp369;
		char* __temp368;
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		__temp369 = NULL;
		__temp368 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp369 = g_strdup_printf ("Can't create instance of value type `%s'", (__temp368 = vala_type_reference_to_string (vala_object_creation_expression_get_type_reference (expr))))));
		(__temp369 = (g_free (__temp369), NULL));
		(__temp368 = (g_free (__temp368), NULL));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
		return;
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (type)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	__temp370 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp370 = vala_type_reference_copy (vala_object_creation_expression_get_type_reference (expr)))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp370 = (g_object_unref (__temp370), NULL));
	(vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	if (VALA_IS_CLASS (type)) {
		ValaClass* __temp371;
		ValaClass* cl;
		__temp371 = NULL;
		cl = (__temp371 = VALA_CLASS (type), (__temp371 == NULL ? NULL : g_object_ref (__temp371)));
		if (vala_class_get_is_abstract (cl)) {
			char* __temp373;
			char* __temp372;
			(vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			__temp373 = NULL;
			__temp372 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp373 = g_strdup_printf ("Can't create instance of abstract class `%s'", (__temp372 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl)))))));
			(__temp373 = (g_free (__temp373), NULL));
			(__temp372 = (g_free (__temp372), NULL));
			(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_class_get_default_construction_method (cl) != NULL) {
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (vala_class_get_default_construction_method (cl)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
		}
		while (cl != NULL) {
			ValaClass* __temp375;
			ValaClass* __temp374;
			if (VALA_DATA_TYPE (cl) == self->priv->initially_unowned_type) {
				(vala_type_reference_set_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
				break;
			}
			__temp375 = NULL;
			__temp374 = NULL;
			cl = (__temp375 = (__temp374 = vala_class_get_base_class (cl), (__temp374 == NULL ? NULL : g_object_ref (__temp374))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp375);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (type)) {
			ValaStruct* __temp376;
			ValaStruct* st;
			__temp376 = NULL;
			st = (__temp376 = VALA_STRUCT (type), (__temp376 == NULL ? NULL : g_object_ref (__temp376)));
			if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_struct_get_default_construction_method (st) != NULL) {
				(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (vala_struct_get_default_construction_method (st)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		}
	}
	__temp377 = NULL;
	if ((__temp378 = vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && g_list_length ((__temp377 = vala_object_creation_expression_get_argument_list (expr))) != 0, (__temp377 == NULL ? NULL : (__temp377 = (g_list_free (__temp377), NULL))), __temp378)) {
		char* __temp380;
		char* __temp379;
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		__temp380 = NULL;
		__temp379 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp380 = g_strdup_printf ("No arguments allowed when constructing type `%s'", (__temp379 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (type)))))));
		(__temp380 = (g_free (__temp380), NULL));
		(__temp379 = (g_free (__temp379), NULL));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
		return;
	}
	if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL && VALA_IS_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr))))) {
		ValaMethod* __temp381;
		ValaMethod* m;
		GList* __temp383;
		GList* __temp382;
		GList* __temp384;
		__temp381 = NULL;
		m = (__temp381 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))), (__temp381 == NULL ? NULL : g_object_ref (__temp381)));
		__temp383 = NULL;
		__temp382 = NULL;
		vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp382 = vala_invokable_get_parameters (VALA_INVOKABLE (m))), (__temp383 = vala_object_creation_expression_get_argument_list (expr)));
		(__temp383 == NULL ? NULL : (__temp383 = (g_list_free (__temp383), NULL)));
		(__temp382 == NULL ? NULL : (__temp382 = (g_list_free (__temp382), NULL)));
		__temp384 = NULL;
		(vala_code_node_set_tree_can_fail (VALA_CODE_NODE (expr), (vala_expression_set_can_fail (VALA_EXPRESSION (expr), (g_list_length ((__temp384 = vala_method_get_error_domains (m))) > 0)), vala_expression_get_can_fail (VALA_EXPRESSION (expr)))), vala_code_node_get_tree_can_fail (VALA_CODE_NODE (expr)));
		(__temp384 == NULL ? NULL : (__temp384 = (g_list_free (__temp384), NULL)));
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	} else {
		if (VALA_IS_ENUM (type)) {
			GList* __temp385;
			gboolean __temp386;
			__temp385 = NULL;
			if ((__temp386 = g_list_length ((__temp385 = vala_object_creation_expression_get_argument_list (expr))) == 0, (__temp385 == NULL ? NULL : (__temp385 = (g_list_free (__temp385), NULL))), __temp386)) {
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Too few arguments, errors need at least 1 argument");
			} else {
				ValaExpression* __temp388;
				GList* __temp387;
				ValaExpression* __temp389;
				ValaExpression* ex;
				__temp388 = NULL;
				__temp387 = NULL;
				__temp389 = NULL;
				ex = (__temp389 = (__temp388 = VALA_EXPRESSION ((__temp387 = vala_object_creation_expression_get_argument_list (expr))->data), (__temp388 == NULL ? NULL : g_object_ref (__temp388))), (__temp387 == NULL ? NULL : (__temp387 = (g_list_free (__temp387), NULL))), __temp389);
				if (vala_expression_get_static_type (ex) == NULL || !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (ex), self->priv->string_type)) {
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Invalid type for argument 1");
				}
				(ex == NULL ? NULL : (ex = (g_object_unref (ex), NULL)));
			}
		}
	}
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_semantic_analyzer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SIZEOF_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->ulong_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_TYPEOF_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->type_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaTypeReference* type)
{
	ValaStruct* __temp391;
	ValaStruct* st;
	gboolean __temp392;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (type)))) {
		return FALSE;
	}
	__temp391 = NULL;
	st = (__temp391 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp391 == NULL ? NULL : g_object_ref (__temp391)));
	return (__temp392 = vala_struct_is_integer_type (st) || vala_struct_is_floating_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp392);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
}


static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaTypeReference* type)
{
	ValaStruct* __temp394;
	ValaStruct* st;
	gboolean __temp395;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (type)))) {
		return FALSE;
	}
	__temp394 = NULL;
	st = (__temp394 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp394 == NULL ? NULL : g_object_ref (__temp394)));
	return (__temp395 = vala_struct_is_integer_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp395);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
}


static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_UNARY_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_unary_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_PLUS || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_MINUS) {
		/* integer or floating point type*/
		if (!vala_semantic_analyzer_is_numeric_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
			char* __temp397;
			char* __temp396;
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			__temp397 = NULL;
			__temp396 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp397 = g_strdup_printf ("Operator not supported for `%s'", (__temp396 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
			(__temp397 = (g_free (__temp397), NULL));
			(__temp396 = (g_free (__temp396), NULL));
			return;
		}
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	} else {
		if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_LOGICAL_NEGATION) {
			/* boolean type*/
			if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_unary_expression_get_inner (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
				char* __temp399;
				char* __temp398;
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				__temp399 = NULL;
				__temp398 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp399 = g_strdup_printf ("Operator not supported for `%s'", (__temp398 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
				(__temp399 = (g_free (__temp399), NULL));
				(__temp398 = (g_free (__temp398), NULL));
				return;
			}
			(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		} else {
			if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT) {
				/* integer type*/
				if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
					char* __temp401;
					char* __temp400;
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					__temp401 = NULL;
					__temp400 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp401 = g_strdup_printf ("Operator not supported for `%s'", (__temp400 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
					(__temp401 = (g_free (__temp401), NULL));
					(__temp400 = (g_free (__temp400), NULL));
					return;
				}
				(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
			} else {
				if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_DECREMENT) {
					ValaMemberAccess* ma;
					ValaMemberAccess* old_value;
					ValaBinaryExpression* bin;
					ValaAssignment* assignment;
					/* integer type*/
					if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
						char* __temp403;
						char* __temp402;
						(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
						__temp403 = NULL;
						__temp402 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp403 = g_strdup_printf ("Operator not supported for `%s'", (__temp402 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
						(__temp403 = (g_free (__temp403), NULL));
						(__temp402 = (g_free (__temp402), NULL));
						return;
					}
					ma = vala_semantic_analyzer_find_member_access (self, vala_unary_expression_get_inner (expr));
					if (ma == NULL) {
						(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Prefix operators not supported for this expression");
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
					bin = vala_binary_expression_new ((vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT ? VALA_BINARY_OPERATOR_PLUS : VALA_BINARY_OPERATOR_MINUS), VALA_EXPRESSION (old_value), VALA_EXPRESSION (vala_literal_expression_new (VALA_LITERAL (vala_integer_literal_new ("1", NULL)), NULL)), NULL);
					assignment = vala_assignment_new (VALA_EXPRESSION (ma), VALA_EXPRESSION (bin), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL);
					vala_code_node_replace (vala_code_node_get_parent_node (VALA_CODE_NODE (expr)), VALA_CODE_NODE (expr), VALA_CODE_NODE (assignment));
					vala_code_node_accept (VALA_CODE_NODE (assignment), VALA_CODE_VISITOR (self));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment == NULL ? NULL : (assignment = (g_object_unref (assignment), NULL)));
					return;
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment == NULL ? NULL : (assignment = (g_object_unref (assignment), NULL)));
				} else {
					if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_REF) {
						/* value type*/
						(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
					} else {
						if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_OUT) {
							/* reference type*/
							(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
						} else {
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unsupported unary operator");
							return;
						}
					}
				}
			}
		}
	}
}


static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), NULL);
	if (VALA_IS_PARENTHESIZED_EXPRESSION (expr)) {
		ValaParenthesizedExpression* __temp404;
		ValaParenthesizedExpression* pe;
		ValaMemberAccess* __temp405;
		__temp404 = NULL;
		pe = (__temp404 = VALA_PARENTHESIZED_EXPRESSION (expr), (__temp404 == NULL ? NULL : g_object_ref (__temp404)));
		__temp405 = NULL;
		return (__temp405 = vala_semantic_analyzer_find_member_access (self, vala_parenthesized_expression_get_inner (pe)), (pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL))), __temp405);
		(pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (expr)) {
		ValaMemberAccess* __temp406;
		__temp406 = NULL;
		return (__temp406 = VALA_MEMBER_ACCESS (expr), (__temp406 == NULL ? NULL : g_object_ref (__temp406)));
	}
	return VALA_MEMBER_ACCESS (NULL);
}


static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CAST_EXPRESSION (expr));
	if (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)) == NULL && vala_type_reference_get_type_parameter (vala_cast_expression_get_type_reference (expr)) == NULL) {
		return;
	}
	/* if type resolving didn't succeed, skip this check 
	 FIXME: check whether cast is allowed*/
	if (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_cast_expression_get_type_reference (expr)), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr)
{
	ValaSemanticAnalyzer * self;
	ValaPointer* __temp409;
	ValaPointer* pointer;
	ValaTypeReference* __temp410;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_POINTER_INDIRECTION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_pointer_indirection_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (!(VALA_IS_POINTER (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))))) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Pointer indirection not supported for this expression");
		return;
	}
	__temp409 = NULL;
	pointer = (__temp409 = VALA_POINTER (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))), (__temp409 == NULL ? NULL : g_object_ref (__temp409)));
	__temp410 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp410 = vala_type_reference_new ())), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp410 == NULL ? NULL : (__temp410 = (g_object_unref (__temp410), NULL)));
	(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_pointer_get_referent_type (pointer)), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(pointer == NULL ? NULL : (pointer = (g_object_unref (pointer), NULL)));
}


static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp411;
	ValaPointer* __temp412;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ADDRESSOF_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_addressof_expression_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Address-of operator not supported for this expression");
		return;
	}
	__temp411 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp411 = vala_type_reference_new ())), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp411 == NULL ? NULL : (__temp411 = (g_object_unref (__temp411), NULL)));
	__temp412 = NULL;
	(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((__temp412 = vala_data_type_get_pointer (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))))))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(__temp412 = (g_object_unref (__temp412), NULL));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)))), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
}


static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr)
{
	ValaSemanticAnalyzer * self;
	ValaTypeReference* __temp413;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_REFERENCE_TRANSFER_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_reference_transfer_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	if (!(VALA_IS_MEMBER_ACCESS (vala_reference_transfer_expression_get_inner (expr)) || VALA_IS_ELEMENT_ACCESS (vala_reference_transfer_expression_get_inner (expr)))) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Reference transfer not supported for this expression");
		return;
	}
	if (!vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr)))) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "No reference to be transferred");
		return;
	}
	__temp413 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp413 = vala_type_reference_copy (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp413 = (g_object_unref (__temp413), NULL));
	(vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
}


static ValaTypeReference* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaTypeReference* left_type, ValaTypeReference* right_type)
{
	ValaStruct* __temp415;
	ValaStruct* left;
	ValaStruct* __temp416;
	ValaStruct* right;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (left_type), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (right_type), NULL);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (left_type))) || !(VALA_IS_STRUCT (vala_type_reference_get_data_type (right_type)))) {
		return VALA_TYPE_REFERENCE (NULL);
	}
	/* at least one operand not struct*/
	__temp415 = NULL;
	left = (__temp415 = VALA_STRUCT (vala_type_reference_get_data_type (left_type)), (__temp415 == NULL ? NULL : g_object_ref (__temp415)));
	__temp416 = NULL;
	right = (__temp416 = VALA_STRUCT (vala_type_reference_get_data_type (right_type)), (__temp416 == NULL ? NULL : g_object_ref (__temp416)));
	if ((!vala_struct_is_floating_type (left) && !vala_struct_is_integer_type (left)) || (!vala_struct_is_floating_type (right) && !vala_struct_is_integer_type (right))) {
		gpointer __temp417;
		return VALA_TYPE_REFERENCE ((__temp417 = NULL, (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp417));
	}
	/* at least one operand not numeric*/
	if (vala_struct_is_floating_type (left) == vala_struct_is_floating_type (right)) {
		/* both operands integer or floating type*/
		if (vala_struct_get_rank (left) >= vala_struct_get_rank (right)) {
			ValaTypeReference* __temp418;
			__temp418 = NULL;
			return (__temp418 = g_object_ref (left_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp418);
		} else {
			ValaTypeReference* __temp419;
			__temp419 = NULL;
			return (__temp419 = g_object_ref (right_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp419);
		}
	} else {
		/* one integer and one floating type operand*/
		if (vala_struct_is_floating_type (left)) {
			ValaTypeReference* __temp420;
			__temp420 = NULL;
			return (__temp420 = g_object_ref (left_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp420);
		} else {
			ValaTypeReference* __temp421;
			__temp421 = NULL;
			return (__temp421 = g_object_ref (right_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp421);
		}
	}
	(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
	(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
}


static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_binary_expression_get_left (expr))) || vala_code_node_get_error (VALA_CODE_NODE (vala_binary_expression_get_right (expr)))) {
		/* if there were any errors in inner expressions, skip type check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS) {
		ValaInvocationExpression* concat_call;
		if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_type_reference_get_data_type (self->priv->string_type)) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Operands must be strings");
			return;
		}
		/* string concatenation: convert to a.concat (b) */
		concat_call = vala_invocation_expression_new (VALA_EXPRESSION (vala_member_access_new (vala_binary_expression_get_left (expr), "concat", NULL)), NULL);
		vala_invocation_expression_add_argument (concat_call, vala_binary_expression_get_right (expr));
		vala_code_node_replace (vala_code_node_get_parent_node (VALA_CODE_NODE (expr)), VALA_CODE_NODE (expr), VALA_CODE_NODE (concat_call));
		vala_code_node_accept (VALA_CODE_NODE (concat_call), VALA_CODE_VISITOR (self));
		(concat_call == NULL ? NULL : (concat_call = (g_object_unref (concat_call), NULL)));
	} else {
		if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MINUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MUL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_DIV) {
			ValaTypeReference* __temp422;
			__temp422 = NULL;
			(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp422 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
			(__temp422 == NULL ? NULL : (__temp422 = (g_object_unref (__temp422), NULL)));
			if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
				char* __temp425;
				char* __temp424;
				char* __temp423;
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				__temp425 = NULL;
				__temp424 = NULL;
				__temp423 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp425 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (__temp423 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp424 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
				(__temp425 = (g_free (__temp425), NULL));
				(__temp424 = (g_free (__temp424), NULL));
				(__temp423 = (g_free (__temp423), NULL));
				return;
			}
		} else {
			if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MOD || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_LEFT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_RIGHT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_XOR) {
				ValaTypeReference* __temp426;
				__temp426 = NULL;
				(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp426 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
				(__temp426 == NULL ? NULL : (__temp426 = (g_object_unref (__temp426), NULL)));
				if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
					char* __temp429;
					char* __temp428;
					char* __temp427;
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					__temp429 = NULL;
					__temp428 = NULL;
					__temp427 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp429 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (__temp427 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp428 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
					(__temp429 = (g_free (__temp429), NULL));
					(__temp428 = (g_free (__temp428), NULL));
					(__temp427 = (g_free (__temp427), NULL));
					return;
				}
			} else {
				if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL) {
					if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
						ValaInvocationExpression* cmp_call;
						ValaLiteralExpression* __temp430;
						/* string comparison: convert to a.collate (b) OP 0 */
						cmp_call = vala_invocation_expression_new (VALA_EXPRESSION (vala_member_access_new (vala_binary_expression_get_left (expr), "collate", NULL)), NULL);
						vala_invocation_expression_add_argument (cmp_call, vala_binary_expression_get_right (expr));
						(vala_binary_expression_set_left (expr, VALA_EXPRESSION (cmp_call)), vala_binary_expression_get_left (expr));
						__temp430 = NULL;
						(vala_binary_expression_set_right (expr, VALA_EXPRESSION ((__temp430 = vala_literal_expression_new (VALA_LITERAL (vala_integer_literal_new ("0", NULL)), NULL)))), vala_binary_expression_get_right (expr));
						(__temp430 == NULL ? NULL : (__temp430 = (g_object_unref (__temp430), NULL)));
						vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (expr)), VALA_CODE_VISITOR (self));
						(cmp_call == NULL ? NULL : (cmp_call = (g_object_unref (cmp_call), NULL)));
					} else {
						ValaTypeReference* resulting_type;
						resulting_type = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)));
						if (resulting_type == NULL) {
							char* __temp433;
							char* __temp432;
							char* __temp431;
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							__temp433 = NULL;
							__temp432 = NULL;
							__temp431 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp433 = g_strdup_printf ("Relational operation not supported for types `%s' and `%s'", (__temp431 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp432 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
							(__temp433 = (g_free (__temp433), NULL));
							(__temp432 = (g_free (__temp432), NULL));
							(__temp431 = (g_free (__temp431), NULL));
							(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
							return;
						}
						(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
					}
					(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
				} else {
					if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_EQUALITY || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_INEQUALITY) {
						/* relational operation */
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_binary_expression_get_right (expr)), vala_expression_get_static_type (vala_binary_expression_get_left (expr))) && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))) {
							char* __temp436;
							char* __temp435;
							char* __temp434;
							__temp436 = NULL;
							__temp435 = NULL;
							__temp434 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp436 = g_strdup_printf ("Equality operation: `%s' and `%s' are incompatible, comparison would always evaluate to false", (__temp434 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))), (__temp435 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))))));
							(__temp436 = (g_free (__temp436), NULL));
							(__temp435 = (g_free (__temp435), NULL));
							(__temp434 = (g_free (__temp434), NULL));
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							return;
						}
						if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
							ValaInvocationExpression* cmp_call;
							ValaLiteralExpression* __temp437;
							/* string comparison: convert to a.collate (b) OP 0 */
							cmp_call = vala_invocation_expression_new (VALA_EXPRESSION (vala_member_access_new (vala_binary_expression_get_left (expr), "collate", NULL)), NULL);
							vala_invocation_expression_add_argument (cmp_call, vala_binary_expression_get_right (expr));
							(vala_binary_expression_set_left (expr, VALA_EXPRESSION (cmp_call)), vala_binary_expression_get_left (expr));
							__temp437 = NULL;
							(vala_binary_expression_set_right (expr, VALA_EXPRESSION ((__temp437 = vala_literal_expression_new (VALA_LITERAL (vala_integer_literal_new ("0", NULL)), NULL)))), vala_binary_expression_get_right (expr));
							(__temp437 == NULL ? NULL : (__temp437 = (g_object_unref (__temp437), NULL)));
							vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (expr)), VALA_CODE_VISITOR (self));
							(cmp_call == NULL ? NULL : (cmp_call = (g_object_unref (cmp_call), NULL)));
						}
						(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
					} else {
						if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_OR) {
							/* integer type or flags type*/
							(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_binary_expression_get_left (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
						} else {
							if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_OR) {
								if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) != vala_type_reference_get_data_type (self->priv->bool_type) || vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
									(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Operands must be boolean");
								}
								(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
							} else {
								g_assert_not_reached ();
							}
						}
					}
				}
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_TYPE_CHECK (expr));
	if (vala_type_reference_get_data_type (vala_type_check_get_type_reference (expr)) == NULL) {
		/* if type resolving didn't succeed, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_type_check_get_type_reference (expr)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static ValaTypeReference* vala_semantic_analyzer_compute_common_base_type (ValaSemanticAnalyzer* self, GList* types)
{
	gboolean null_found;
	gboolean class_or_iface_found;
	gboolean type_param_found;
	gboolean ref_struct_found;
	gboolean val_struct_found;
	gboolean enum_found;
	gboolean callback_found;
	ValaTypeReference* base_type;
	ValaTypeReference* last_type;
	ValaTypeReference* __temp452;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	null_found = FALSE;
	class_or_iface_found = FALSE;
	type_param_found = FALSE;
	ref_struct_found = FALSE;
	val_struct_found = FALSE;
	enum_found = FALSE;
	callback_found = FALSE;
	base_type = NULL;
	last_type = NULL;
	{
		GList* __temp448;
		GList* type_it;
		__temp448 = NULL;
		__temp448 = types;
		for (type_it = __temp448; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type;
			type = type_it->data;
			{
				ValaTypeReference* __temp439;
				ValaTypeReference* __temp438;
				__temp439 = NULL;
				__temp438 = NULL;
				last_type = (__temp439 = (__temp438 = type, (__temp438 == NULL ? NULL : g_object_ref (__temp438))), (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp439);
				if (vala_code_node_get_error (VALA_CODE_NODE (type))) {
					ValaTypeReference* __temp440;
					ValaTypeReference* __temp441;
					__temp440 = NULL;
					base_type = (__temp440 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp440);
					(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
					__temp441 = NULL;
					return (__temp441 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp441);
				}
				if (vala_type_reference_get_data_type (type) == NULL && vala_type_reference_get_type_parameter (type) == NULL) {
					if (!null_found) {
						null_found = TRUE;
						if (val_struct_found || enum_found) {
							(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
							break;
						}
					}
				} else {
					if (VALA_IS_CLASS (vala_type_reference_get_data_type (type)) || VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
						if (!class_or_iface_found) {
							class_or_iface_found = TRUE;
							if (type_param_found || ref_struct_found || val_struct_found || enum_found || callback_found) {
								(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
								break;
							}
						}
					} else {
						if (vala_type_reference_get_type_parameter (type) != NULL) {
							if (!type_param_found) {
								type_param_found = TRUE;
								if (class_or_iface_found || ref_struct_found || val_struct_found || enum_found || callback_found) {
									(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
									break;
								}
							}
						} else {
							if (VALA_IS_STRUCT (vala_type_reference_get_data_type (type))) {
								ValaStruct* __temp442;
								ValaStruct* st;
								__temp442 = NULL;
								st = (__temp442 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp442 == NULL ? NULL : g_object_ref (__temp442)));
								if (vala_data_type_is_reference_type (VALA_DATA_TYPE (st))) {
									if (!ref_struct_found) {
										ref_struct_found = TRUE;
										if (class_or_iface_found || type_param_found || val_struct_found || enum_found || callback_found) {
											(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
											break;
										}
									}
								} else {
									if (!val_struct_found) {
										val_struct_found = TRUE;
										if (class_or_iface_found || type_param_found || ref_struct_found || enum_found || callback_found) {
											(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
											break;
										}
									}
								}
								(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
							} else {
								if (VALA_IS_ENUM (vala_type_reference_get_data_type (type))) {
									if (!enum_found) {
										enum_found = TRUE;
										if (class_or_iface_found || type_param_found || ref_struct_found || val_struct_found) {
											(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
											break;
										}
									}
								} else {
									if (VALA_IS_CALLBACK (vala_type_reference_get_data_type (type))) {
										if (!callback_found) {
											callback_found = TRUE;
											if (class_or_iface_found || type_param_found || ref_struct_found || val_struct_found || enum_found) {
												(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
												break;
											}
										}
									} else {
										ValaTypeReference* __temp443;
										char* __temp445;
										char* __temp444;
										ValaTypeReference* __temp446;
										__temp443 = NULL;
										base_type = (__temp443 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp443);
										(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
										__temp445 = NULL;
										__temp444 = NULL;
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (type)), (__temp445 = g_strdup_printf ("internal error: unsupported type `%s'", (__temp444 = vala_type_reference_to_string (type)))));
										(__temp445 = (g_free (__temp445), NULL));
										(__temp444 = (g_free (__temp444), NULL));
										__temp446 = NULL;
										return (__temp446 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp446);
									}
								}
							}
						}
					}
				}
				if (base_type == NULL) {
					ValaTypeReference* __temp447;
					__temp447 = NULL;
					base_type = (__temp447 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp447);
					(vala_type_reference_set_data_type (base_type, vala_type_reference_get_data_type (type)), vala_type_reference_get_data_type (base_type));
					(vala_type_reference_set_type_parameter (base_type, vala_type_reference_get_type_parameter (type)), vala_type_reference_get_type_parameter (base_type));
					(vala_type_reference_set_non_null (base_type, vala_type_reference_get_non_null (type)), vala_type_reference_get_non_null (base_type));
					(vala_type_reference_set_is_null (base_type, vala_type_reference_get_is_null (type)), vala_type_reference_get_is_null (base_type));
					(vala_type_reference_set_transfers_ownership (base_type, vala_type_reference_get_transfers_ownership (type)), vala_type_reference_get_transfers_ownership (base_type));
				} else {
					if (vala_type_reference_get_data_type (base_type) != vala_type_reference_get_data_type (type)) {
						if (vala_semantic_analyzer_is_type_compatible (self, type, base_type)) {
						} else {
							if (vala_semantic_analyzer_is_type_compatible (self, base_type, type)) {
								(vala_type_reference_set_data_type (base_type, vala_type_reference_get_data_type (type)), vala_type_reference_get_data_type (base_type));
							} else {
								(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
								break;
							}
						}
					}
					(vala_type_reference_set_non_null (base_type, vala_type_reference_get_non_null (base_type) && vala_type_reference_get_non_null (type)), vala_type_reference_get_non_null (base_type));
					(vala_type_reference_set_is_null (base_type, vala_type_reference_get_is_null (base_type) && vala_type_reference_get_is_null (type)), vala_type_reference_get_is_null (base_type));
					/* if one subexpression transfers ownership, all subexpressions must transfer ownership
					 FIXME add ref calls to subexpressions that don't transfer ownership*/
					(vala_type_reference_set_transfers_ownership (base_type, vala_type_reference_get_transfers_ownership (base_type) || vala_type_reference_get_transfers_ownership (type)), vala_type_reference_get_transfers_ownership (base_type));
				}
			}
		}
	}
	if (base_type != NULL && vala_code_node_get_error (VALA_CODE_NODE (base_type))) {
		char* __temp451;
		char* __temp450;
		char* __temp449;
		__temp451 = NULL;
		__temp450 = NULL;
		__temp449 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (last_type)), (__temp451 = g_strdup_printf ("`%s' is incompatible with `%s'", (__temp449 = vala_type_reference_to_string (last_type)), (__temp450 = vala_type_reference_to_string (base_type)))));
		(__temp451 = (g_free (__temp451), NULL));
		(__temp450 = (g_free (__temp450), NULL));
		(__temp449 = (g_free (__temp449), NULL));
	}
	__temp452 = NULL;
	return (__temp452 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp452);
	(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
	(last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr)
{
	ValaSemanticAnalyzer * self;
	GList* types;
	ValaTypeReference* __temp453;
	ValaTypeReference* __temp454;
	ValaTypeReference* __temp455;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONDITIONAL_EXPRESSION (expr));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_conditional_expression_get_condition (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_conditional_expression_get_condition (expr))), "Condition must be boolean");
		return;
	}
	/* FIXME: support memory management */
	types = NULL;
	__temp453 = NULL;
	types = g_list_append (types, (__temp453 = vala_expression_get_static_type (vala_conditional_expression_get_true_expression (expr)), (__temp453 == NULL ? NULL : g_object_ref (__temp453))));
	__temp454 = NULL;
	types = g_list_append (types, (__temp454 = vala_expression_get_static_type (vala_conditional_expression_get_false_expression (expr)), (__temp454 == NULL ? NULL : g_object_ref (__temp454))));
	__temp455 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp455 = vala_semantic_analyzer_compute_common_base_type (self, types))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp455 == NULL ? NULL : (__temp455 = (g_object_unref (__temp455), NULL)));
	(types == NULL ? NULL : (types = (g_list_foreach (types, (GFunc) g_object_unref, NULL), g_list_free (types), NULL)));
}


static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self)
{
	char* result;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	result = g_strdup_printf ("__lambda%d", self->priv->next_lambda_id);
	self->priv->next_lambda_id++;
	return result;
	(result = (g_free (result), NULL));
}


static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self)
{
	ValaSymbol* __temp457;
	ValaSymbol* sym;
	gpointer __temp462;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	__temp457 = NULL;
	sym = (__temp457 = self->priv->current_symbol, (__temp457 == NULL ? NULL : g_object_ref (__temp457)));
	while (sym != NULL) {
		ValaSymbol* __temp461;
		ValaSymbol* __temp460;
		if (VALA_IS_METHOD (vala_symbol_get_node (sym))) {
			ValaMethod* __temp458;
			ValaMethod* __temp459;
			__temp458 = NULL;
			__temp459 = NULL;
			return (__temp459 = (__temp458 = VALA_METHOD (vala_symbol_get_node (sym)), (__temp458 == NULL ? NULL : g_object_ref (__temp458))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp459);
		}
		__temp461 = NULL;
		__temp460 = NULL;
		sym = (__temp461 = (__temp460 = vala_symbol_get_parent_symbol (sym), (__temp460 == NULL ? NULL : g_object_ref (__temp460))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp461);
	}
	return VALA_METHOD ((__temp462 = NULL, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp462));
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self)
{
	ValaSymbol* __temp463;
	ValaSymbol* sym;
	gboolean __temp467;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	__temp463 = NULL;
	sym = (__temp463 = self->priv->current_symbol, (__temp463 == NULL ? NULL : g_object_ref (__temp463)));
	while (sym != NULL) {
		ValaSymbol* __temp466;
		ValaSymbol* __temp465;
		if (VALA_IS_CONSTRUCTOR (vala_symbol_get_node (sym))) {
			gboolean __temp464;
			return (__temp464 = TRUE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp464);
		}
		__temp466 = NULL;
		__temp465 = NULL;
		sym = (__temp466 = (__temp465 = vala_symbol_get_parent_symbol (sym), (__temp465 == NULL ? NULL : g_object_ref (__temp465))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp466);
	}
	return (__temp467 = FALSE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp467);
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_real_visit_begin_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l)
{
	ValaSemanticAnalyzer * self;
	gboolean in_instance_method;
	ValaMethod* current_method;
	ValaCallback* __temp468;
	ValaCallback* cb;
	ValaMethod* __temp469;
	ValaSymbol* __temp470;
	GList* lambda_params;
	GList* lambda_param_it;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (l));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (l)) == NULL || !(VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (l)))))) {
		(vala_code_node_set_error (VALA_CODE_NODE (l), TRUE), vala_code_node_get_error (VALA_CODE_NODE (l)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (l)), "lambda expression not allowed in this context");
		return;
	}
	in_instance_method = FALSE;
	current_method = vala_semantic_analyzer_find_current_method (self);
	if (current_method != NULL) {
		in_instance_method = vala_method_get_instance (current_method);
	} else {
		in_instance_method = vala_semantic_analyzer_is_in_constructor (self);
	}
	__temp468 = NULL;
	cb = (__temp468 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (l)))), (__temp468 == NULL ? NULL : g_object_ref (__temp468)));
	__temp469 = NULL;
	(vala_lambda_expression_set_method (l, (__temp469 = vala_method_new (vala_semantic_analyzer_get_lambda_name (self), vala_callback_get_return_type (cb), NULL))), vala_lambda_expression_get_method (l));
	(__temp469 == NULL ? NULL : (__temp469 = (g_object_unref (__temp469), NULL)));
	(vala_method_set_instance (vala_lambda_expression_get_method (l), vala_callback_get_instance (cb) && in_instance_method), vala_method_get_instance (vala_lambda_expression_get_method (l)));
	__temp470 = NULL;
	(vala_code_node_set_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)), (__temp470 = vala_symbol_new (VALA_CODE_NODE (vala_lambda_expression_get_method (l))))), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l))));
	(__temp470 == NULL ? NULL : (__temp470 = (g_object_unref (__temp470), NULL)));
	(vala_symbol_set_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l))), self->priv->current_symbol), vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))));
	lambda_params = vala_lambda_expression_get_parameters (l);
	lambda_param_it = lambda_params;
	{
		GList* __temp473;
		GList* cb_param_it;
		__temp473 = NULL;
		__temp473 = vala_callback_get_parameters (cb);
		for (cb_param_it = __temp473; cb_param_it != NULL; cb_param_it = cb_param_it->next) {
			ValaFormalParameter* cb_param;
			cb_param = cb_param_it->data;
			{
				const char* __temp471;
				char* lambda_param;
				ValaFormalParameter* param;
				ValaSymbol* __temp472;
				if (lambda_param_it == NULL) {
					break;
				}
				/* lambda expressions are allowed to have less parameters */
				__temp471 = NULL;
				lambda_param = (__temp471 = ((char*) lambda_param_it->data), (__temp471 == NULL ? NULL : g_strdup (__temp471)));
				param = vala_formal_parameter_new (lambda_param, vala_formal_parameter_get_type_reference (cb_param), NULL);
				__temp472 = NULL;
				(vala_code_node_set_symbol (VALA_CODE_NODE (param), (__temp472 = vala_symbol_new (VALA_CODE_NODE (param)))), vala_code_node_get_symbol (VALA_CODE_NODE (param)));
				(__temp472 == NULL ? NULL : (__temp472 = (g_object_unref (__temp472), NULL)));
				vala_symbol_add (vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l))), vala_formal_parameter_get_name (param), vala_code_node_get_symbol (VALA_CODE_NODE (param)));
				vala_method_add_parameter (vala_lambda_expression_get_method (l), param);
				lambda_param_it = lambda_param_it->next;
				(lambda_param = (g_free (lambda_param), NULL));
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(__temp473 == NULL ? NULL : (__temp473 = (g_list_free (__temp473), NULL)));
	}
	if (lambda_param_it != NULL) {
		/* lambda expressions may not expect more parameters */
		(vala_code_node_set_error (VALA_CODE_NODE (l), TRUE), vala_code_node_get_error (VALA_CODE_NODE (l)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (l)), "lambda expression: too many parameters");
		(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
		(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
		(lambda_params == NULL ? NULL : (lambda_params = (g_list_free (lambda_params), NULL)));
		return;
	}
	if (vala_lambda_expression_get_expression_body (l) != NULL) {
		ValaBlock* block;
		ValaSymbol* __temp474;
		block = vala_block_new (NULL);
		__temp474 = NULL;
		(vala_code_node_set_symbol (VALA_CODE_NODE (block), (__temp474 = vala_symbol_new (VALA_CODE_NODE (block)))), vala_code_node_get_symbol (VALA_CODE_NODE (block)));
		(__temp474 == NULL ? NULL : (__temp474 = (g_object_unref (__temp474), NULL)));
		(vala_symbol_set_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (block)), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))), vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (block))));
		if (vala_type_reference_get_data_type (vala_method_get_return_type (vala_lambda_expression_get_method (l))) != NULL) {
			ValaReturnStatement* __temp475;
			__temp475 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((__temp475 = vala_return_statement_new (vala_lambda_expression_get_expression_body (l), NULL))));
			(__temp475 == NULL ? NULL : (__temp475 = (g_object_unref (__temp475), NULL)));
		} else {
			ValaExpressionStatement* __temp476;
			__temp476 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((__temp476 = vala_expression_statement_new (vala_lambda_expression_get_expression_body (l), NULL))));
			(__temp476 == NULL ? NULL : (__temp476 = (g_object_unref (__temp476), NULL)));
		}
		(vala_method_set_body (vala_lambda_expression_get_method (l), block), vala_method_get_body (vala_lambda_expression_get_method (l)));
		(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
	} else {
		(vala_method_set_body (vala_lambda_expression_get_method (l), vala_lambda_expression_get_statement_body (l)), vala_method_get_body (vala_lambda_expression_get_method (l)));
		(vala_symbol_set_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_method_get_body (vala_lambda_expression_get_method (l)))), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))), vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_method_get_body (vala_lambda_expression_get_method (l))))));
	}
	/* lambda expressions should be usable like MemberAccess of a method */
	(vala_expression_set_symbol_reference (VALA_EXPRESSION (l), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (l)));
	(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
	(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
	(lambda_params == NULL ? NULL : (lambda_params = (g_list_free (lambda_params), NULL)));
}


static void vala_semantic_analyzer_real_visit_begin_assignment (ValaCodeVisitor* base, ValaAssignment* a)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp477;
		ValaMemberAccess* ma;
		__temp477 = NULL;
		ma = (__temp477 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp477 == NULL ? NULL : g_object_ref (__temp477)));
		if (vala_code_node_get_error (VALA_CODE_NODE (ma)) || vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		/* if no symbol found, skip this check */
		if (VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
			ValaSignal* __temp478;
			ValaSignal* sig;
			ValaTypeReference* __temp479;
			ValaCallback* __temp480;
			__temp478 = NULL;
			sig = (__temp478 = VALA_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp478 == NULL ? NULL : g_object_ref (__temp478)));
			__temp479 = NULL;
			(vala_expression_set_expected_type (vala_assignment_get_right (a), (__temp479 = vala_type_reference_new ())), vala_expression_get_expected_type (vala_assignment_get_right (a)));
			(__temp479 == NULL ? NULL : (__temp479 = (g_object_unref (__temp479), NULL)));
			__temp480 = NULL;
			(vala_type_reference_set_data_type (vala_expression_get_expected_type (vala_assignment_get_right (a)), VALA_DATA_TYPE ((__temp480 = vala_signal_get_callback (sig)))), vala_type_reference_get_data_type (vala_expression_get_expected_type (vala_assignment_get_right (a))));
			(__temp480 = (g_object_unref (__temp480), NULL));
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
		} else {
			/* do nothing*/
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (a))) {
			} else {
				/* do nothing*/
				(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "unsupported lvalue in assignment");
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	if (vala_code_node_get_error (VALA_CODE_NODE (a)) || vala_code_node_get_error (VALA_CODE_NODE (vala_assignment_get_left (a))) || vala_code_node_get_error (VALA_CODE_NODE (vala_assignment_get_right (a)))) {
		(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
		return;
	}
	if (vala_assignment_get_operator (a) != VALA_ASSIGNMENT_OPERATOR_SIMPLE && VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp481;
		ValaMemberAccess* ma;
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		__temp481 = NULL;
		ma = (__temp481 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp481 == NULL ? NULL : g_object_ref (__temp481)));
		if (!(VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))))) {
			ValaMemberAccess* old_value;
			ValaBinaryExpression* bin;
			old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			bin = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, VALA_EXPRESSION (old_value), VALA_EXPRESSION (vala_parenthesized_expression_new (vala_assignment_get_right (a), vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))))), NULL);
			if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR), vala_binary_expression_get_operator (bin));
			} else {
				if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND), vala_binary_expression_get_operator (bin));
				} else {
					if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR), vala_binary_expression_get_operator (bin));
					} else {
						if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_ADD) {
							(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS), vala_binary_expression_get_operator (bin));
						} else {
							if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SUB) {
								(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS), vala_binary_expression_get_operator (bin));
							} else {
								if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_MUL) {
									(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL), vala_binary_expression_get_operator (bin));
								} else {
									if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_DIV) {
										(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV), vala_binary_expression_get_operator (bin));
									} else {
										if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD), vala_binary_expression_get_operator (bin));
										} else {
											if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT), vala_binary_expression_get_operator (bin));
											} else {
												if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT), vala_binary_expression_get_operator (bin));
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			(vala_assignment_set_right (a, VALA_EXPRESSION (bin)), vala_assignment_get_right (a));
			vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_right (a)), VALA_CODE_VISITOR (self));
			(vala_assignment_set_operator (a, VALA_ASSIGNMENT_OPERATOR_SIMPLE), vala_assignment_get_operator (a));
			(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
			(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp482;
		ValaMemberAccess* ma;
		__temp482 = NULL;
		ma = (__temp482 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp482 == NULL ? NULL : g_object_ref (__temp482)));
		if (VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
			ValaSignal* __temp483;
			ValaSignal* sig;
			ValaMethod* __temp484;
			ValaMethod* m;
			__temp483 = NULL;
			sig = (__temp483 = VALA_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp483 == NULL ? NULL : g_object_ref (__temp483)));
			if (vala_expression_get_symbol_reference (vala_assignment_get_right (a)) == NULL) {
				(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))), "unsupported expression for signal handler");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			__temp484 = NULL;
			m = (__temp484 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_assignment_get_right (a)))), (__temp484 == NULL ? NULL : g_object_ref (__temp484)));
			if (vala_method_get_instance (m) && m->access != VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
				/* TODO: generate wrapper function */
				(vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE), vala_code_node_get_error (VALA_CODE_NODE (ma)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))), "public instance methods not yet supported as signal handlers");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			if (vala_method_get_instance (m)) {
				/* instance signal handlers must have the self
				 * parameter at the end
				 * do not use G_CONNECT_SWAPPED as this would
				 * rearrange the parameters for instance
				 * methods and non-instance methods
				 */
				(vala_method_set_instance_last (m, TRUE), vala_method_get_instance_last (m));
			}
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		} else {
			if (VALA_IS_PROPERTY (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
				ValaProperty* __temp485;
				ValaProperty* prop;
				__temp485 = NULL;
				prop = (__temp485 = VALA_PROPERTY (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp485 == NULL ? NULL : g_object_ref (__temp485)));
				if (vala_property_get_set_accessor (prop) == NULL) {
					char* __temp487;
					char* __temp486;
					(vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE), vala_code_node_get_error (VALA_CODE_NODE (ma)));
					__temp487 = NULL;
					__temp486 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ma)), (__temp487 = g_strdup_printf ("Property `%s' is read-only", (__temp486 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))))));
					(__temp487 = (g_free (__temp487), NULL));
					(__temp486 = (g_free (__temp486), NULL));
					(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					return;
				}
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			} else {
				if (VALA_IS_VARIABLE_DECLARATOR (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
					ValaVariableDeclarator* __temp488;
					ValaVariableDeclarator* decl;
					ValaMemberAccess* __temp489;
					ValaMemberAccess* right_ma;
					__temp488 = NULL;
					decl = (__temp488 = VALA_VARIABLE_DECLARATOR (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp488 == NULL ? NULL : g_object_ref (__temp488)));
					__temp489 = NULL;
					right_ma = (__temp489 = VALA_MEMBER_ACCESS (vala_assignment_get_right (a)), (__temp489 == NULL ? NULL : g_object_ref (__temp489)));
					if (VALA_IS_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))) {
						ValaMethod* __temp490;
						ValaMethod* m;
						ValaCallback* __temp491;
						ValaCallback* cb;
						__temp490 = NULL;
						m = (__temp490 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))), (__temp490 == NULL ? NULL : g_object_ref (__temp490)));
						__temp491 = NULL;
						cb = (__temp491 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (__temp491 == NULL ? NULL : g_object_ref (__temp491)));
						/* check whether method matches callback type */
						if (!vala_callback_matches_method (cb, m)) {
							char* __temp494;
							char* __temp493;
							char* __temp492;
							(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
							__temp494 = NULL;
							__temp493 = NULL;
							__temp492 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp494 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp492 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp493 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cb)))))));
							(__temp494 = (g_free (__temp494), NULL));
							(__temp493 = (g_free (__temp493), NULL));
							(__temp492 = (g_free (__temp492), NULL));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
							(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						(vala_expression_set_static_type (vala_assignment_get_right (a), vala_variable_declarator_get_type_reference (decl)), vala_expression_get_static_type (vala_assignment_get_right (a)));
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
						(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					} else {
						(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Assignment: Invalid callback assignment attempt");
						(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
					(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
				} else {
					if (VALA_IS_FIELD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
						ValaField* __temp495;
						ValaField* f;
						ValaMemberAccess* __temp496;
						ValaMemberAccess* right_ma;
						__temp495 = NULL;
						f = (__temp495 = VALA_FIELD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp495 == NULL ? NULL : g_object_ref (__temp495)));
						__temp496 = NULL;
						right_ma = (__temp496 = VALA_MEMBER_ACCESS (vala_assignment_get_right (a)), (__temp496 == NULL ? NULL : g_object_ref (__temp496)));
						if (VALA_IS_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))) {
							ValaMethod* __temp497;
							ValaMethod* m;
							ValaCallback* __temp498;
							ValaCallback* cb;
							__temp497 = NULL;
							m = (__temp497 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))), (__temp497 == NULL ? NULL : g_object_ref (__temp497)));
							__temp498 = NULL;
							cb = (__temp498 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), (__temp498 == NULL ? NULL : g_object_ref (__temp498)));
							/* check whether method matches callback type */
							if (!vala_callback_matches_method (cb, m)) {
								char* __temp501;
								char* __temp500;
								char* __temp499;
								(vala_code_node_set_error (VALA_CODE_NODE (f), TRUE), vala_code_node_get_error (VALA_CODE_NODE (f)));
								__temp501 = NULL;
								__temp500 = NULL;
								__temp499 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp501 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp499 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp500 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cb)))))));
								(__temp501 = (g_free (__temp501), NULL));
								(__temp500 = (g_free (__temp500), NULL));
								(__temp499 = (g_free (__temp499), NULL));
								(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
								(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
								(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							(vala_expression_set_static_type (vala_assignment_get_right (a), vala_field_get_type_reference (f)), vala_expression_get_static_type (vala_assignment_get_right (a)));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
						} else {
							(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Assignment: Invalid callback assignment attempt");
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
							(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
					} else {
						if (vala_expression_get_static_type (vala_assignment_get_left (a)) != NULL && vala_expression_get_static_type (vala_assignment_get_right (a)) != NULL) {
							if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)))) {
								char* __temp504;
								char* __temp503;
								char* __temp502;
								/* if there was an error on either side,
								 * i.e. a.{left|right}.static_type == null, skip type check */
								(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
								__temp504 = NULL;
								__temp503 = NULL;
								__temp502 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp504 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (__temp502 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_right (a)))), (__temp503 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_left (a)))))));
								(__temp504 = (g_free (__temp504), NULL));
								(__temp503 = (g_free (__temp503), NULL));
								(__temp502 = (g_free (__temp502), NULL));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							if (vala_semantic_analyzer_get_memory_management (self)) {
								if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
									/* rhs transfers ownership of the expression */
									if (!vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
										/* lhs doesn't own the value */
										(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Invalid assignment from owned expression to unowned variable");
									}
								} else {
									if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
									}
								}
							}
						}
					}
				}
			}
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		/* lhs wants to own the value
		 * rhs doesn't transfer the ownership
		 * code generator needs to add reference
		 * increment calls */
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
			ValaElementAccess* __temp505;
			ValaElementAccess* ea;
			__temp505 = NULL;
			ea = (__temp505 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (a)), (__temp505 == NULL ? NULL : g_object_ref (__temp505)));
			if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)))) {
				char* __temp508;
				char* __temp507;
				char* __temp506;
				/* if there was an error on either side,
				 * i.e. a.{left|right}.static_type == null, skip type check */
				(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
				__temp508 = NULL;
				__temp507 = NULL;
				__temp506 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp508 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (__temp506 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_right (a)))), (__temp507 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_left (a)))))));
				(__temp508 = (g_free (__temp508), NULL));
				(__temp507 = (g_free (__temp507), NULL));
				(__temp506 = (g_free (__temp506), NULL));
				(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
				return;
			}
			if (vala_semantic_analyzer_get_memory_management (self)) {
				if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
					GList* args;
					ValaTypeReference* __temp509;
					ValaTypeReference* element_type;
					/* rhs transfers ownership of the expression */
					args = vala_type_reference_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (ea)));
					if (g_list_length (args) != 1) {
						(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ea)), "internal error: array reference without type arguments");
						(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
						(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
						return;
					}
					__temp509 = NULL;
					element_type = (__temp509 = VALA_TYPE_REFERENCE (args->data), (__temp509 == NULL ? NULL : g_object_ref (__temp509)));
					if (!vala_type_reference_get_takes_ownership (element_type)) {
						/* lhs doesn't own the value */
						(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Invalid assignment from owned expression to unowned variable");
						(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
						(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
						(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
						return;
					}
					(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
					(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
				} else {
					if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
					}
				}
			}
			(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
		} else {
			return;
		}
	}
	/* lhs wants to own the value
	 * rhs doesn't transfer the ownership
	 * code generator needs to add reference
	 * increment calls */
	(vala_expression_set_static_type (VALA_EXPRESSION (a), vala_expression_get_static_type (vala_assignment_get_left (a))), vala_expression_get_static_type (VALA_EXPRESSION (a)));
}


gboolean vala_semantic_analyzer_get_memory_management (ValaSemanticAnalyzer* self)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	return self->priv->_memory_management;
}


void vala_semantic_analyzer_set_memory_management (ValaSemanticAnalyzer* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	self->priv->_memory_management = value;
}


static void vala_semantic_analyzer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (object);
	switch (property_id) {
		case VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT:
		g_value_set_boolean (value, vala_semantic_analyzer_get_memory_management (self));
		break;
	}
}


static void vala_semantic_analyzer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (object);
	switch (property_id) {
		case VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT:
		vala_semantic_analyzer_set_memory_management (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_semantic_analyzer_class_init (ValaSemanticAnalyzerClass * klass)
{
	vala_semantic_analyzer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSemanticAnalyzerPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_semantic_analyzer_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_semantic_analyzer_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_semantic_analyzer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_semantic_analyzer_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_semantic_analyzer_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_semantic_analyzer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_semantic_analyzer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_semantic_analyzer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_callback = vala_semantic_analyzer_real_visit_callback;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_semantic_analyzer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_semantic_analyzer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_semantic_analyzer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_semantic_analyzer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_semantic_analyzer_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_semantic_analyzer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_semantic_analyzer_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_semantic_analyzer_real_visit_signal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_semantic_analyzer_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_semantic_analyzer_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_named_argument = vala_semantic_analyzer_real_visit_named_argument;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_block = vala_semantic_analyzer_real_visit_begin_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_block = vala_semantic_analyzer_real_visit_end_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_variable_declarator = vala_semantic_analyzer_real_visit_variable_declarator;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_initializer_list = vala_semantic_analyzer_real_visit_begin_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_initializer_list = vala_semantic_analyzer_real_visit_end_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_semantic_analyzer_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_semantic_analyzer_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_semantic_analyzer_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_semantic_analyzer_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_foreach_statement = vala_semantic_analyzer_real_visit_begin_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_foreach_statement = vala_semantic_analyzer_real_visit_end_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_return_statement = vala_semantic_analyzer_real_visit_end_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_semantic_analyzer_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_semantic_analyzer_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_semantic_analyzer_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_semantic_analyzer_real_visit_lock_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_array_creation_expression = vala_semantic_analyzer_real_visit_begin_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_array_creation_expression = vala_semantic_analyzer_real_visit_end_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_boolean_literal = vala_semantic_analyzer_real_visit_boolean_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_character_literal = vala_semantic_analyzer_real_visit_character_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_integer_literal = vala_semantic_analyzer_real_visit_integer_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_real_literal = vala_semantic_analyzer_real_visit_real_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_string_literal = vala_semantic_analyzer_real_visit_string_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_null_literal = vala_semantic_analyzer_real_visit_null_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_literal_expression = vala_semantic_analyzer_real_visit_literal_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_semantic_analyzer_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_semantic_analyzer_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_invocation_expression = vala_semantic_analyzer_real_visit_begin_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_invocation_expression = vala_semantic_analyzer_real_visit_end_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_semantic_analyzer_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_base_access = vala_semantic_analyzer_real_visit_base_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_postfix_expression = vala_semantic_analyzer_real_visit_postfix_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_object_creation_expression = vala_semantic_analyzer_real_visit_end_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_sizeof_expression = vala_semantic_analyzer_real_visit_sizeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_typeof_expression = vala_semantic_analyzer_real_visit_typeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_semantic_analyzer_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_cast_expression = vala_semantic_analyzer_real_visit_cast_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_pointer_indirection = vala_semantic_analyzer_real_visit_pointer_indirection;
	VALA_CODE_VISITOR_CLASS (klass)->visit_addressof_expression = vala_semantic_analyzer_real_visit_addressof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_semantic_analyzer_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_semantic_analyzer_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_check = vala_semantic_analyzer_real_visit_type_check;
	VALA_CODE_VISITOR_CLASS (klass)->visit_conditional_expression = vala_semantic_analyzer_real_visit_conditional_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_lambda_expression = vala_semantic_analyzer_real_visit_begin_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_assignment = vala_semantic_analyzer_real_visit_begin_assignment;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_assignment = vala_semantic_analyzer_real_visit_end_assignment;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT, g_param_spec_boolean ("memory-management", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_semantic_analyzer_init (ValaSemanticAnalyzer * self)
{
	self->priv = VALA_SEMANTIC_ANALYZER_GET_PRIVATE (self);
	self->priv->next_lambda_id = 0;
}


static void vala_semantic_analyzer_dispose (GObject * obj)
{
	ValaSemanticAnalyzer * self;
	ValaSemanticAnalyzerClass * klass;
	GObjectClass * parent_class;
	self = VALA_SEMANTIC_ANALYZER (obj);
	(self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL)));
	(self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL)));
	(self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL)));
	(self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL)));
	(self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL)));
	(self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL)));
	(self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_list_free (self->priv->current_using_directives), NULL)));
	(self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL)));
	(self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL)));
	(self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL)));
	(self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL)));
	(self->priv->ulong_type == NULL ? NULL : (self->priv->ulong_type = (g_object_unref (self->priv->ulong_type), NULL)));
	(self->priv->unichar_type == NULL ? NULL : (self->priv->unichar_type = (g_object_unref (self->priv->unichar_type), NULL)));
	(self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL)));
	(self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL)));
	(self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL)));
	(self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL)));
	(self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL)));
	(self->priv->gerror_type == NULL ? NULL : (self->priv->gerror_type = (g_object_unref (self->priv->gerror_type), NULL)));
	klass = VALA_SEMANTIC_ANALYZER_CLASS (g_type_class_peek (VALA_TYPE_SEMANTIC_ANALYZER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_semantic_analyzer_get_type ()
{
	static GType vala_semantic_analyzer_type_id = 0;
	if (G_UNLIKELY (vala_semantic_analyzer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSemanticAnalyzerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_semantic_analyzer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSemanticAnalyzer), 0, (GInstanceInitFunc) vala_semantic_analyzer_init };
		vala_semantic_analyzer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaSemanticAnalyzer", &g_define_type_info, 0);
	}
	return vala_semantic_analyzer_type_id;
}




