/* valagidlparser.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valagidlparser.h"
#include <vala/valasourcereference.h>
#include <vala/valadatatype.h>
#include <glib/gstdio.h>
#include <vala/valareport.h>
#include <gidlparser.h>
#include <stdio.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valaformalparameter.h>
#include <vala/valaenumvalue.h>
#include <gmetadata.h>
#include <vala/valacreationmethod.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valastatement.h>
#include <vala/valaexpression.h>
#include "valagidlparser.h"

struct _ValaGIdlParserPrivate {
	ValaSourceReference* current_source_reference;
	ValaDataType* current_data_type;
	GHashTable* codenode_attributes_map;
	GHashTable* current_type_symbol_map;
};
#define VALA_GIDL_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIDL_PARSER, ValaGIdlParserPrivate))
enum  {
	VALA_GIDL_PARSER_DUMMY_PROPERTY
};
static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file);
static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, GIdlModule* module);
static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, GIdlModule* module);
static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module);
static ValaCallback* vala_gidl_parser_parse_callback (ValaGIdlParser* self, GIdlNodeFunction* f_node);
static ValaStruct* vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node);
static ValaStruct* vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node);
static ValaEnum* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node);
static ValaClass* vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node);
static ValaInterface* vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node);
static ValaTypeReference* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node);
static gboolean vala_gidl_parser_is_value_type (ValaGIdlParser* self, const char* type_name);
static void vala_gidl_parser_parse_type_string (ValaGIdlParser* self, ValaTypeReference* type, const char* n);
static ValaTypeReference* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param);
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_virtual);
static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name);
static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node);
static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node);
static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node);
static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode);
static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s);
static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node);
static gpointer vala_gidl_parser_parent_class = NULL;
static void vala_gidl_parser_dispose (GObject * obj);


/**
 * Parse all source files in the specified code context and build a
 * code tree.
 *
 * @param context a code context
 */
void vala_gidl_parser_parse (ValaGIdlParser* self, ValaCodeContext* context)
{
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
}


static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file)
{
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".gidl")) {
		vala_gidl_parser_parse_file (self, source_file);
	}
}


static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file)
{
	GError * inner_error;
	char* __temp0;
	char* __temp1;
	char* metadata_filename;
	GHashTable* __temp2;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	inner_error = NULL;
	__temp0 = NULL;
	__temp1 = NULL;
	metadata_filename = (__temp1 = g_strdup_printf ("%s.metadata", (__temp0 = g_strndup (vala_source_file_get_filename (source_file), strlen (vala_source_file_get_filename (source_file)) - strlen (".gidl")))), (__temp0 = (g_free (__temp0), NULL)), __temp1);
	__temp2 = NULL;
	self->priv->codenode_attributes_map = (__temp2 = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free), (self->priv->codenode_attributes_map == NULL ? NULL : (self->priv->codenode_attributes_map = (g_hash_table_unref (self->priv->codenode_attributes_map), NULL))), __temp2);
	if (g_file_test (metadata_filename, G_FILE_TEST_EXISTS)) {
		{
			char* metadata;
			glong metadata_len;
			metadata = NULL;
			metadata_len = 0L;
			g_file_get_contents (metadata_filename, &metadata, &metadata_len, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			{
				char** __temp6;
				char** line_it;
				__temp6 = NULL;
				__temp6 = g_strsplit (metadata, "\n", 0);
				for (line_it = __temp6; *line_it != NULL; line_it = line_it + 1) {
					char* line;
					line = *line_it;
					{
						char** __temp3;
						gint line_parts_length1;
						char** line_parts;
						const char* __temp5;
						const char* __temp4;
						__temp3 = NULL;
						line_parts = (__temp3 = g_strsplit (line, " ", 2), line_parts_length1 = -1, __temp3);
						if (line_parts[0] == NULL) {
							continue;
						}
						__temp5 = NULL;
						__temp4 = NULL;
						g_hash_table_insert (self->priv->codenode_attributes_map, (__temp4 = line_parts[0], (__temp4 == NULL ? NULL : g_strdup (__temp4))), (__temp5 = line_parts[1], (__temp5 == NULL ? NULL : g_strdup (__temp5))));
						(line_parts = (g_free (line_parts), NULL));
					}
				}
				(__temp6 = (g_free (__temp6), NULL));
			}
			(metadata = (g_free (metadata), NULL));
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* __temp7;
				__temp7 = NULL;
				vala_report_error (NULL, (__temp7 = g_strdup_printf ("Unable to read metadata file: %s", e->message)));
				(__temp7 = (g_free (__temp7), NULL));
			}
		}
		__finally0:
		;
	}
	{
		GList* modules;
		ValaSourceReference* __temp8;
		modules = g_idl_parse_file (vala_source_file_get_filename (source_file), &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_MARKUP_ERROR) {
				goto __catch1_g_markup_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		__temp8 = NULL;
		self->priv->current_source_reference = (__temp8 = vala_source_reference_new (source_file, 0, 0, 0, 0), (self->priv->current_source_reference == NULL ? NULL : (self->priv->current_source_reference = (g_object_unref (self->priv->current_source_reference), NULL))), __temp8);
		{
			GList* __temp9;
			GList* module_it;
			__temp9 = NULL;
			__temp9 = modules;
			for (module_it = __temp9; module_it != NULL; module_it = module_it->next) {
				GIdlModule* module;
				module = module_it->data;
				{
					ValaNamespace* ns;
					ns = vala_gidl_parser_parse_module (self, module);
					vala_source_file_add_namespace (source_file, ns);
					(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
				}
			}
		}
		(modules == NULL ? NULL : (modules = (g_list_foreach (modules, (GFunc) g_free, NULL), g_list_free (modules), NULL)));
	}
	goto __finally1;
	__catch1_g_markup_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			fprintf (stdout, "error parsing GIDL file: %s\n", e->message);
		}
	}
	__finally1:
	;
	(metadata_filename = (g_free (metadata_filename), NULL));
}


static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, GIdlModule* module)
{
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (type_name != NULL, NULL);
	g_return_val_if_fail (module != NULL, NULL);
	if (g_str_has_prefix (type_name, module->name)) {
		const char* __temp10;
		__temp10 = NULL;
		return (__temp10 = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (module->name, -1)), (__temp10 == NULL ? NULL : g_strdup (__temp10)));
	} else {
		if (g_utf8_collate (module->name, "GLib") == 0 && g_str_has_prefix (type_name, "G")) {
			const char* __temp12;
			__temp12 = NULL;
			return (__temp12 = g_utf8_offset_to_pointer (type_name, 1), (__temp12 == NULL ? NULL : g_strdup (__temp12)));
		}
	}
	return g_strdup (type_name);
}


static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, GIdlModule* module)
{
	char* __temp16;
	char* __temp15;
	gboolean __temp17;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (const_name != NULL, NULL);
	g_return_val_if_fail (module != NULL, NULL);
	__temp16 = NULL;
	__temp15 = NULL;
	if ((__temp17 = g_str_has_prefix (const_name, (__temp16 = g_strconcat ((__temp15 = g_utf8_strup (module->name, -1)), "_", NULL))), (__temp16 = (g_free (__temp16), NULL)), (__temp15 = (g_free (__temp15), NULL)), __temp17)) {
		const char* __temp18;
		__temp18 = NULL;
		return (__temp18 = g_utf8_offset_to_pointer (const_name, g_utf8_strlen (module->name, -1) + 1), (__temp18 == NULL ? NULL : g_strdup (__temp18)));
	} else {
		if (g_utf8_collate (module->name, "GLib") == 0 && g_str_has_prefix (const_name, "G_")) {
			const char* __temp20;
			__temp20 = NULL;
			return (__temp20 = g_utf8_offset_to_pointer (const_name, 2), (__temp20 == NULL ? NULL : g_strdup (__temp20)));
		}
	}
	return g_strdup (const_name);
}


static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module)
{
	ValaNamespace* ns;
	char** __temp23;
	gint attributes_length1;
	char** attributes;
	ValaNamespace* __temp38;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (module != NULL, NULL);
	ns = vala_namespace_new (module->name, self->priv->current_source_reference);
	__temp23 = NULL;
	attributes = (__temp23 = vala_gidl_parser_get_attributes (self, vala_namespace_get_name (ns)), attributes_length1 = -1, __temp23);
	if (attributes != NULL) {
		{
			char** __temp28;
			int attr_it;
			__temp28 = NULL;
			__temp28 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp28[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp28[attr_it];
				{
					char** __temp24;
					gint nv_length1;
					char** nv;
					__temp24 = NULL;
					nv = (__temp24 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp24);
					if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
						char* __temp25;
						__temp25 = NULL;
						vala_namespace_set_cheader_filename (ns, (__temp25 = vala_gidl_parser_eval (self, nv[1])));
						(__temp25 = (g_free (__temp25), NULL));
					} else {
						if (g_utf8_collate (nv[0], "cprefix") == 0) {
							char* __temp26;
							__temp26 = NULL;
							vala_namespace_set_cprefix (ns, (__temp26 = vala_gidl_parser_eval (self, nv[1])));
							(__temp26 = (g_free (__temp26), NULL));
						} else {
							if (g_utf8_collate (nv[0], "lower_case_cprefix") == 0) {
								char* __temp27;
								__temp27 = NULL;
								vala_namespace_set_lower_case_cprefix (ns, (__temp27 = vala_gidl_parser_eval (self, nv[1])));
								(__temp27 = (g_free (__temp27), NULL));
							}
						}
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	{
		GList* __temp37;
		GList* node_it;
		__temp37 = NULL;
		__temp37 = module->entries;
		for (node_it = __temp37; node_it != NULL; node_it = node_it->next) {
			GIdlNode* node;
			node = node_it->data;
			{
				if (node->type == G_IDL_NODE_CALLBACK) {
					ValaCallback* cb;
					char* __temp29;
					cb = vala_gidl_parser_parse_callback (self, ((GIdlNodeFunction*) node));
					if (cb == NULL) {
						continue;
					}
					__temp29 = NULL;
					(vala_data_type_set_name (VALA_DATA_TYPE (cb), (__temp29 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (cb)), module))), vala_data_type_get_name (VALA_DATA_TYPE (cb)));
					(__temp29 = (g_free (__temp29), NULL));
					vala_namespace_add_callback (ns, cb);
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
				} else {
					if (node->type == G_IDL_NODE_STRUCT) {
						ValaStruct* st;
						char* __temp30;
						st = vala_gidl_parser_parse_struct (self, ((GIdlNodeStruct*) node));
						if (st == NULL) {
							continue;
						}
						__temp30 = NULL;
						(vala_data_type_set_name (VALA_DATA_TYPE (st), (__temp30 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (st)), module))), vala_data_type_get_name (VALA_DATA_TYPE (st)));
						(__temp30 = (g_free (__temp30), NULL));
						vala_namespace_add_struct (ns, st);
						(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
					} else {
						if (node->type == G_IDL_NODE_BOXED) {
							ValaStruct* st;
							char* __temp31;
							char* __temp32;
							st = vala_gidl_parser_parse_boxed (self, ((GIdlNodeBoxed*) node));
							__temp31 = NULL;
							(vala_data_type_set_name (VALA_DATA_TYPE (st), (__temp31 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (st)), module))), vala_data_type_get_name (VALA_DATA_TYPE (st)));
							(__temp31 = (g_free (__temp31), NULL));
							vala_namespace_add_struct (ns, st);
							__temp32 = NULL;
							vala_struct_set_type_id (st, (__temp32 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (st), "TYPE_")));
							(__temp32 = (g_free (__temp32), NULL));
							(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
						} else {
							if (node->type == G_IDL_NODE_ENUM) {
								ValaEnum* en;
								char* __temp33;
								en = vala_gidl_parser_parse_enum (self, ((GIdlNodeEnum*) node));
								__temp33 = NULL;
								(vala_data_type_set_name (VALA_DATA_TYPE (en), (__temp33 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (en)), module))), vala_data_type_get_name (VALA_DATA_TYPE (en)));
								(__temp33 = (g_free (__temp33), NULL));
								vala_namespace_add_enum (ns, en);
								(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
							} else {
								if (node->type == G_IDL_NODE_OBJECT) {
									ValaClass* cl;
									char* __temp34;
									cl = vala_gidl_parser_parse_object (self, ((GIdlNodeInterface*) node));
									__temp34 = NULL;
									(vala_data_type_set_name (VALA_DATA_TYPE (cl), (__temp34 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (cl)), module))), vala_data_type_get_name (VALA_DATA_TYPE (cl)));
									(__temp34 = (g_free (__temp34), NULL));
									vala_namespace_add_class (ns, cl);
									(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
								} else {
									if (node->type == G_IDL_NODE_INTERFACE) {
										ValaInterface* iface;
										char* __temp35;
										iface = vala_gidl_parser_parse_interface (self, ((GIdlNodeInterface*) node));
										__temp35 = NULL;
										(vala_data_type_set_name (VALA_DATA_TYPE (iface), (__temp35 = vala_gidl_parser_fix_type_name (self, vala_data_type_get_name (VALA_DATA_TYPE (iface)), module))), vala_data_type_get_name (VALA_DATA_TYPE (iface)));
										(__temp35 = (g_free (__temp35), NULL));
										vala_namespace_add_interface (ns, iface);
										(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
									} else {
										if (node->type == G_IDL_NODE_CONSTANT) {
											ValaConstant* c;
											char* __temp36;
											c = vala_gidl_parser_parse_constant (self, ((GIdlNodeConstant*) node));
											__temp36 = NULL;
											(vala_constant_set_name (c, (__temp36 = vala_gidl_parser_fix_const_name (self, vala_constant_get_name (c), module))), vala_constant_get_name (c));
											(__temp36 = (g_free (__temp36), NULL));
											vala_namespace_add_constant (ns, c);
											(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
										} else {
											if (node->type == G_IDL_NODE_FUNCTION) {
												ValaMethod* m;
												m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) node), FALSE);
												if (m != NULL) {
													(vala_method_set_instance (m, FALSE), vala_method_get_instance (m));
													vala_namespace_add_method (ns, m);
												}
												(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	__temp38 = NULL;
	return (__temp38 = ns, (attributes = (g_free (attributes), NULL)), __temp38);
	(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
	(attributes = (g_free (attributes), NULL));
}


static ValaCallback* vala_gidl_parser_parse_callback (ValaGIdlParser* self, GIdlNodeFunction* f_node)
{
	GIdlNode* node;
	char** __temp39;
	gint attributes_length1;
	char** attributes;
	ValaCallback* cb;
	ValaCallback* __temp46;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (f_node != NULL, NULL);
	node = ((GIdlNode*) f_node);
	__temp39 = NULL;
	attributes = (__temp39 = vala_gidl_parser_get_attributes (self, node->name), attributes_length1 = -1, __temp39);
	if (attributes != NULL) {
		{
			char** __temp44;
			int attr_it;
			__temp44 = NULL;
			__temp44 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp44[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp44[attr_it];
				{
					char** __temp40;
					gint nv_length1;
					char** nv;
					__temp40 = NULL;
					nv = (__temp40 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp40);
					if (g_utf8_collate (nv[0], "hidden") == 0) {
						char* __temp41;
						gboolean __temp42;
						__temp41 = NULL;
						if ((__temp42 = g_utf8_collate ((__temp41 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp41 = (g_free (__temp41), NULL)), __temp42)) {
							gpointer __temp43;
							return VALA_CALLBACK ((__temp43 = NULL, (nv = (g_free (nv), NULL)), __temp43));
						}
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	cb = vala_callback_new (node->name, vala_gidl_parser_parse_param (self, f_node->result), self->priv->current_source_reference);
	VALA_DATA_TYPE (cb)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	{
		GList* __temp45;
		GList* param_it;
		__temp45 = NULL;
		__temp45 = f_node->parameters;
		for (param_it = __temp45; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaFormalParameter* p;
				param_node = ((GIdlNode*) param);
				p = vala_formal_parameter_new (param_node->name, vala_gidl_parser_parse_param (self, param), NULL);
				vala_callback_add_parameter (cb, p);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	__temp46 = NULL;
	return (__temp46 = cb, (attributes = (g_free (attributes), NULL)), __temp46);
	(attributes = (g_free (attributes), NULL));
	(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
}


static ValaStruct* vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node)
{
	GIdlNode* node;
	ValaStruct* st;
	char** __temp48;
	gint st_attributes_length1;
	char** st_attributes;
	ValaDataType* __temp54;
	ValaStruct* __temp53;
	ValaDataType* __temp56;
	ValaStruct* __temp57;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (st_node != NULL, NULL);
	node = ((GIdlNode*) st_node);
	if (st_node->deprecated) {
		return VALA_STRUCT (NULL);
	}
	st = vala_struct_new (node->name, self->priv->current_source_reference);
	VALA_DATA_TYPE (st)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	vala_struct_set_is_reference_type (st, TRUE);
	__temp48 = NULL;
	st_attributes = (__temp48 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, __temp48);
	if (st_attributes != NULL) {
		{
			char** __temp52;
			int attr_it;
			__temp52 = NULL;
			__temp52 = st_attributes;
			for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && __temp52[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp52[attr_it];
				{
					char** __temp49;
					gint nv_length1;
					char** nv;
					char* __temp50;
					gboolean __temp51;
					__temp49 = NULL;
					nv = (__temp49 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp49);
					__temp50 = NULL;
					if ((__temp51 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((__temp50 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp50 = (g_free (__temp50), NULL)), __temp51)) {
						vala_struct_set_is_reference_type (st, FALSE);
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	__temp54 = NULL;
	__temp53 = NULL;
	self->priv->current_data_type = (__temp54 = VALA_DATA_TYPE ((__temp53 = st, (__temp53 == NULL ? NULL : g_object_ref (__temp53)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp54);
	{
		GList* __temp55;
		GList* member_it;
		__temp55 = NULL;
		__temp55 = st_node->members;
		for (member_it = __temp55; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					ValaMethod* m;
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
					if (m != NULL) {
						vala_struct_add_method (st, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_FIELD) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_struct_add_field (st, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	__temp56 = NULL;
	self->priv->current_data_type = (__temp56 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp56);
	__temp57 = NULL;
	return (__temp57 = st, (st_attributes = (g_free (st_attributes), NULL)), __temp57);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	(st_attributes = (g_free (st_attributes), NULL));
}


static ValaStruct* vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node)
{
	GIdlNode* node;
	ValaStruct* st;
	char** __temp58;
	gint st_attributes_length1;
	char** st_attributes;
	ValaDataType* __temp64;
	ValaStruct* __temp63;
	ValaDataType* __temp66;
	ValaStruct* __temp67;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (boxed_node != NULL, NULL);
	node = ((GIdlNode*) boxed_node);
	st = vala_struct_new (node->name, self->priv->current_source_reference);
	VALA_DATA_TYPE (st)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	vala_struct_set_is_reference_type (st, TRUE);
	__temp58 = NULL;
	st_attributes = (__temp58 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, __temp58);
	if (st_attributes != NULL) {
		{
			char** __temp62;
			int attr_it;
			__temp62 = NULL;
			__temp62 = st_attributes;
			for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && __temp62[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp62[attr_it];
				{
					char** __temp59;
					gint nv_length1;
					char** nv;
					char* __temp60;
					gboolean __temp61;
					__temp59 = NULL;
					nv = (__temp59 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp59);
					__temp60 = NULL;
					if ((__temp61 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((__temp60 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp60 = (g_free (__temp60), NULL)), __temp61)) {
						vala_struct_set_is_reference_type (st, FALSE);
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	__temp64 = NULL;
	__temp63 = NULL;
	self->priv->current_data_type = (__temp64 = VALA_DATA_TYPE ((__temp63 = st, (__temp63 == NULL ? NULL : g_object_ref (__temp63)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp64);
	{
		GList* __temp65;
		GList* member_it;
		__temp65 = NULL;
		__temp65 = boxed_node->members;
		for (member_it = __temp65; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					ValaMethod* m;
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
					if (m != NULL) {
						vala_struct_add_method (st, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_FIELD) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_struct_add_field (st, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	__temp66 = NULL;
	self->priv->current_data_type = (__temp66 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp66);
	__temp67 = NULL;
	return (__temp67 = st, (st_attributes = (g_free (st_attributes), NULL)), __temp67);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	(st_attributes = (g_free (st_attributes), NULL));
}


static ValaEnum* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node)
{
	GIdlNode* node;
	ValaEnum* en;
	char* common_prefix;
	ValaEnum* __temp74;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (en_node != NULL, NULL);
	node = ((GIdlNode*) en_node);
	en = vala_enum_new (node->name, self->priv->current_source_reference);
	VALA_DATA_TYPE (en)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	common_prefix = NULL;
	{
		GList* __temp72;
		GList* value_it;
		__temp72 = NULL;
		__temp72 = en_node->values;
		for (value_it = __temp72; value_it != NULL; value_it = value_it->next) {
			GIdlNode* value;
			value = value_it->data;
			{
				if (common_prefix == NULL) {
					char* __temp69;
					const char* __temp68;
					__temp69 = NULL;
					__temp68 = NULL;
					common_prefix = (__temp69 = (__temp68 = value->name, (__temp68 == NULL ? NULL : g_strdup (__temp68))), (common_prefix = (g_free (common_prefix), NULL)), __temp69);
					while (g_utf8_strlen (common_prefix, -1) > 0 && !g_str_has_suffix (common_prefix, "_")) {
						char* __temp70;
						/* FIXME: could easily be made faster*/
						__temp70 = NULL;
						common_prefix = (__temp70 = g_strndup (common_prefix, strlen (common_prefix) - 1), (common_prefix = (g_free (common_prefix), NULL)), __temp70);
					}
				} else {
					while (!g_str_has_prefix (value->name, common_prefix)) {
						char* __temp71;
						__temp71 = NULL;
						common_prefix = (__temp71 = g_strndup (common_prefix, strlen (common_prefix) - 1), (common_prefix = (g_free (common_prefix), NULL)), __temp71);
					}
				}
			}
		}
	}
	vala_enum_set_cprefix (en, common_prefix);
	{
		GList* __temp73;
		GList* value2_it;
		__temp73 = NULL;
		__temp73 = en_node->values;
		for (value2_it = __temp73; value2_it != NULL; value2_it = value2_it->next) {
			GIdlNode* value2;
			value2 = value2_it->data;
			{
				ValaEnumValue* ev;
				ev = vala_enum_value_new (g_utf8_offset_to_pointer (value2->name, g_utf8_strlen (common_prefix, -1)));
				vala_enum_add_value (en, ev);
				(ev == NULL ? NULL : (ev = (g_object_unref (ev), NULL)));
			}
		}
	}
	__temp74 = NULL;
	return (__temp74 = en, (common_prefix = (g_free (common_prefix), NULL)), __temp74);
	(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
	(common_prefix = (g_free (common_prefix), NULL));
}


static ValaClass* vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node)
{
	ValaClass* cl;
	char** __temp75;
	gint attributes_length1;
	char** attributes;
	ValaDataType* __temp81;
	ValaClass* __temp80;
	GHashTable* __temp82;
	GHashTable* current_type_vfunc_map;
	ValaDataType* __temp88;
	GHashTable* __temp89;
	ValaClass* __temp90;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (node != NULL, NULL);
	cl = vala_class_new (node->gtype_name, self->priv->current_source_reference);
	VALA_DATA_TYPE (cl)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	__temp75 = NULL;
	attributes = (__temp75 = vala_gidl_parser_get_attributes (self, vala_data_type_get_name (VALA_DATA_TYPE (cl))), attributes_length1 = -1, __temp75);
	if (attributes != NULL) {
		{
			char** __temp78;
			int attr_it;
			__temp78 = NULL;
			__temp78 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp78[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp78[attr_it];
				{
					char** __temp76;
					gint nv_length1;
					char** nv;
					__temp76 = NULL;
					nv = (__temp76 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp76);
					if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
						char* __temp77;
						__temp77 = NULL;
						vala_data_type_add_cheader_filename (VALA_DATA_TYPE (cl), (__temp77 = vala_gidl_parser_eval (self, nv[1])));
						(__temp77 = (g_free (__temp77), NULL));
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	if (node->parent != NULL) {
		ValaTypeReference* parent;
		parent = vala_type_reference_new ();
		vala_gidl_parser_parse_type_string (self, parent, node->parent);
		vala_class_add_base_type (cl, parent);
		(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
	}
	{
		GList* __temp79;
		GList* iface_name_it;
		__temp79 = NULL;
		__temp79 = node->interfaces;
		for (iface_name_it = __temp79; iface_name_it != NULL; iface_name_it = iface_name_it->next) {
			char* iface_name;
			iface_name = iface_name_it->data;
			{
				ValaTypeReference* iface;
				iface = vala_type_reference_new ();
				vala_gidl_parser_parse_type_string (self, iface, iface_name);
				vala_class_add_base_type (cl, iface);
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	__temp81 = NULL;
	__temp80 = NULL;
	self->priv->current_data_type = (__temp81 = VALA_DATA_TYPE ((__temp80 = cl, (__temp80 == NULL ? NULL : g_object_ref (__temp80)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp81);
	__temp82 = NULL;
	self->priv->current_type_symbol_map = (__temp82 = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free), (self->priv->current_type_symbol_map == NULL ? NULL : (self->priv->current_type_symbol_map = (g_hash_table_unref (self->priv->current_type_symbol_map), NULL))), __temp82);
	current_type_vfunc_map = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
	{
		GList* __temp84;
		GList* member_it;
		__temp84 = NULL;
		__temp84 = node->members;
		for (member_it = __temp84; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_VFUNC) {
					const char* __temp83;
					__temp83 = NULL;
					g_hash_table_insert (current_type_vfunc_map, (__temp83 = member->name, (__temp83 == NULL ? NULL : g_strdup (__temp83))), g_strdup ("1"));
				}
			}
		}
	}
	{
		GList* __temp85;
		GList* member_it;
		__temp85 = NULL;
		__temp85 = node->members;
		for (member_it = __temp85; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					gboolean is_virtual;
					ValaMethod* m;
					is_virtual = g_hash_table_lookup (current_type_vfunc_map, member->name) != NULL;
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), is_virtual);
					if (m != NULL) {
						vala_class_add_method (cl, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_PROPERTY) {
						ValaProperty* prop;
						prop = vala_gidl_parser_parse_property (self, ((GIdlNodeProperty*) member));
						if (prop != NULL) {
							vala_class_add_property (cl, prop, FALSE);
						}
						(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					} else {
						if (member->type == G_IDL_NODE_SIGNAL) {
							ValaSignal* sig;
							sig = vala_gidl_parser_parse_signal (self, ((GIdlNodeSignal*) member));
							if (sig != NULL) {
								vala_class_add_signal (cl, sig);
							}
							(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
						}
					}
				}
			}
		}
	}
	{
		GList* __temp86;
		GList* member_it;
		__temp86 = NULL;
		__temp86 = node->members;
		for (member_it = __temp86; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FIELD) {
					if (g_hash_table_lookup (self->priv->current_type_symbol_map, member->name) == NULL) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_class_add_field (cl, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	{
		GList* __temp87;
		GList* prop_it;
		__temp87 = NULL;
		__temp87 = vala_class_get_properties (cl);
		for (prop_it = __temp87; prop_it != NULL; prop_it = prop_it->next) {
			ValaProperty* prop;
			prop = prop_it->data;
			{
				char* getter;
				char* setter;
				getter = g_strdup_printf ("get_%s", vala_property_get_name (prop));
				if (vala_property_get_get_accessor (prop) != NULL && g_hash_table_lookup (self->priv->current_type_symbol_map, getter) == NULL) {
					(vala_property_set_no_accessor_method (prop, TRUE), vala_property_get_no_accessor_method (prop));
				}
				setter = g_strdup_printf ("set_%s", vala_property_get_name (prop));
				if (vala_property_get_set_accessor (prop) != NULL && g_hash_table_lookup (self->priv->current_type_symbol_map, setter) == NULL) {
					(vala_property_set_no_accessor_method (prop, TRUE), vala_property_get_no_accessor_method (prop));
				}
				(getter = (g_free (getter), NULL));
				(setter = (g_free (setter), NULL));
			}
		}
		(__temp87 == NULL ? NULL : (__temp87 = (g_list_free (__temp87), NULL)));
	}
	__temp88 = NULL;
	self->priv->current_data_type = (__temp88 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp88);
	__temp89 = NULL;
	self->priv->current_type_symbol_map = (__temp89 = NULL, (self->priv->current_type_symbol_map == NULL ? NULL : (self->priv->current_type_symbol_map = (g_hash_table_unref (self->priv->current_type_symbol_map), NULL))), __temp89);
	__temp90 = NULL;
	return (__temp90 = cl, (attributes = (g_free (attributes), NULL)), (current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_hash_table_unref (current_type_vfunc_map), NULL))), __temp90);
	(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	(attributes = (g_free (attributes), NULL));
	(current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_hash_table_unref (current_type_vfunc_map), NULL)));
}


static ValaInterface* vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node)
{
	ValaInterface* iface;
	char** __temp91;
	gint attributes_length1;
	char** attributes;
	ValaDataType* __temp97;
	ValaInterface* __temp96;
	GHashTable* current_type_vfunc_map;
	ValaDataType* __temp101;
	ValaInterface* __temp102;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (node != NULL, NULL);
	iface = vala_interface_new (node->gtype_name, self->priv->current_source_reference);
	VALA_DATA_TYPE (iface)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	__temp91 = NULL;
	attributes = (__temp91 = vala_gidl_parser_get_attributes (self, vala_data_type_get_name (VALA_DATA_TYPE (iface))), attributes_length1 = -1, __temp91);
	if (attributes != NULL) {
		{
			char** __temp94;
			int attr_it;
			__temp94 = NULL;
			__temp94 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp94[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp94[attr_it];
				{
					char** __temp92;
					gint nv_length1;
					char** nv;
					__temp92 = NULL;
					nv = (__temp92 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp92);
					if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
						char* __temp93;
						__temp93 = NULL;
						vala_data_type_add_cheader_filename (VALA_DATA_TYPE (iface), (__temp93 = vala_gidl_parser_eval (self, nv[1])));
						(__temp93 = (g_free (__temp93), NULL));
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	{
		GList* __temp95;
		GList* prereq_name_it;
		__temp95 = NULL;
		__temp95 = node->prerequisites;
		for (prereq_name_it = __temp95; prereq_name_it != NULL; prereq_name_it = prereq_name_it->next) {
			char* prereq_name;
			prereq_name = prereq_name_it->data;
			{
				ValaTypeReference* prereq;
				prereq = vala_type_reference_new ();
				vala_gidl_parser_parse_type_string (self, prereq, prereq_name);
				vala_interface_add_prerequisite (iface, prereq);
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
			}
		}
	}
	__temp97 = NULL;
	__temp96 = NULL;
	self->priv->current_data_type = (__temp97 = VALA_DATA_TYPE ((__temp96 = iface, (__temp96 == NULL ? NULL : g_object_ref (__temp96)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp97);
	current_type_vfunc_map = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
	{
		GList* __temp99;
		GList* member_it;
		__temp99 = NULL;
		__temp99 = node->members;
		for (member_it = __temp99; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_VFUNC) {
					const char* __temp98;
					__temp98 = NULL;
					g_hash_table_insert (current_type_vfunc_map, (__temp98 = member->name, (__temp98 == NULL ? NULL : g_strdup (__temp98))), g_strdup ("1"));
				}
			}
		}
	}
	{
		GList* __temp100;
		GList* member_it;
		__temp100 = NULL;
		__temp100 = node->members;
		for (member_it = __temp100; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					gboolean is_virtual;
					ValaMethod* m;
					is_virtual = g_hash_table_lookup (current_type_vfunc_map, member->name) != NULL;
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), is_virtual);
					if (m != NULL) {
						vala_interface_add_method (iface, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_SIGNAL) {
						ValaSignal* sig;
						sig = vala_gidl_parser_parse_signal (self, ((GIdlNodeSignal*) member));
						if (sig != NULL) {
							vala_interface_add_signal (iface, sig);
						}
						(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
					}
				}
			}
		}
	}
	__temp101 = NULL;
	self->priv->current_data_type = (__temp101 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp101);
	__temp102 = NULL;
	return (__temp102 = iface, (attributes = (g_free (attributes), NULL)), (current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_hash_table_unref (current_type_vfunc_map), NULL))), __temp102);
	(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
	(attributes = (g_free (attributes), NULL));
	(current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_hash_table_unref (current_type_vfunc_map), NULL)));
}


static ValaTypeReference* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node)
{
	ValaTypeReference* type;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (type_node != NULL, NULL);
	type = vala_type_reference_new ();
	if (type_node->tag == TYPE_TAG_VOID) {
		if (type_node->is_pointer) {
			(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
		} else {
			(vala_type_reference_set_type_name (type, "void"), vala_type_reference_get_type_name (type));
		}
	} else {
		if (type_node->tag == TYPE_TAG_BOOLEAN) {
			(vala_type_reference_set_type_name (type, "bool"), vala_type_reference_get_type_name (type));
		} else {
			if (type_node->tag == TYPE_TAG_INT8) {
				(vala_type_reference_set_type_name (type, "char"), vala_type_reference_get_type_name (type));
			} else {
				if (type_node->tag == TYPE_TAG_UINT8) {
					(vala_type_reference_set_type_name (type, "uchar"), vala_type_reference_get_type_name (type));
				} else {
					if (type_node->tag == TYPE_TAG_INT16) {
						(vala_type_reference_set_type_name (type, "short"), vala_type_reference_get_type_name (type));
					} else {
						if (type_node->tag == TYPE_TAG_UINT16) {
							(vala_type_reference_set_type_name (type, "ushort"), vala_type_reference_get_type_name (type));
						} else {
							if (type_node->tag == TYPE_TAG_INT32) {
								(vala_type_reference_set_type_name (type, "int"), vala_type_reference_get_type_name (type));
							} else {
								if (type_node->tag == TYPE_TAG_UINT32) {
									(vala_type_reference_set_type_name (type, "uint"), vala_type_reference_get_type_name (type));
								} else {
									if (type_node->tag == TYPE_TAG_INT64) {
										(vala_type_reference_set_type_name (type, "int64"), vala_type_reference_get_type_name (type));
									} else {
										if (type_node->tag == TYPE_TAG_UINT64) {
											(vala_type_reference_set_type_name (type, "uint64"), vala_type_reference_get_type_name (type));
										} else {
											if (type_node->tag == TYPE_TAG_INT) {
												(vala_type_reference_set_type_name (type, "int"), vala_type_reference_get_type_name (type));
											} else {
												if (type_node->tag == TYPE_TAG_UINT) {
													(vala_type_reference_set_type_name (type, "uint"), vala_type_reference_get_type_name (type));
												} else {
													if (type_node->tag == TYPE_TAG_LONG) {
														(vala_type_reference_set_type_name (type, "long"), vala_type_reference_get_type_name (type));
													} else {
														if (type_node->tag == TYPE_TAG_ULONG) {
															(vala_type_reference_set_type_name (type, "ulong"), vala_type_reference_get_type_name (type));
														} else {
															if (type_node->tag == TYPE_TAG_SSIZE) {
																(vala_type_reference_set_type_name (type, "long"), vala_type_reference_get_type_name (type));
															} else {
																if (type_node->tag == TYPE_TAG_SIZE) {
																	(vala_type_reference_set_type_name (type, "ulong"), vala_type_reference_get_type_name (type));
																} else {
																	if (type_node->tag == TYPE_TAG_FLOAT) {
																		(vala_type_reference_set_type_name (type, "float"), vala_type_reference_get_type_name (type));
																	} else {
																		if (type_node->tag == TYPE_TAG_DOUBLE) {
																			(vala_type_reference_set_type_name (type, "double"), vala_type_reference_get_type_name (type));
																		} else {
																			if (type_node->tag == TYPE_TAG_UTF8) {
																				(vala_type_reference_set_type_name (type, "string"), vala_type_reference_get_type_name (type));
																			} else {
																				if (type_node->tag == TYPE_TAG_FILENAME) {
																					(vala_type_reference_set_type_name (type, "string"), vala_type_reference_get_type_name (type));
																				} else {
																					if (type_node->tag == TYPE_TAG_ARRAY) {
																						ValaTypeReference* __temp103;
																						__temp103 = NULL;
																						type = (__temp103 = vala_gidl_parser_parse_type (self, type_node->parameter_type1), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp103);
																						(vala_type_reference_set_array_rank (type, 1), vala_type_reference_get_array_rank (type));
																					} else {
																						if (type_node->tag == TYPE_TAG_LIST) {
																							(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																							(vala_type_reference_set_type_name (type, "List"), vala_type_reference_get_type_name (type));
																						} else {
																							if (type_node->tag == TYPE_TAG_SLIST) {
																								(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																								(vala_type_reference_set_type_name (type, "SList"), vala_type_reference_get_type_name (type));
																							} else {
																								if (type_node->tag == TYPE_TAG_HASH) {
																									(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																									(vala_type_reference_set_type_name (type, "HashTable"), vala_type_reference_get_type_name (type));
																								} else {
																									if (type_node->tag == TYPE_TAG_ERROR) {
																										(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																										(vala_type_reference_set_type_name (type, "Error"), vala_type_reference_get_type_name (type));
																									} else {
																										if (type_node->is_interface) {
																											const char* __temp104;
																											char* n;
																											__temp104 = NULL;
																											n = (__temp104 = type_node->interface, (__temp104 == NULL ? NULL : g_strdup (__temp104)));
																											if (g_utf8_collate (n, "") == 0) {
																												gpointer __temp105;
																												return VALA_TYPE_REFERENCE ((__temp105 = NULL, (n = (g_free (n), NULL)), __temp105));
																											}
																											if (g_str_has_prefix (n, "const-")) {
																												char* __temp107;
																												const char* __temp106;
																												__temp107 = NULL;
																												__temp106 = NULL;
																												n = (__temp107 = (__temp106 = g_utf8_offset_to_pointer (n, g_utf8_strlen ("const-", -1)), (__temp106 == NULL ? NULL : g_strdup (__temp106))), (n = (g_free (n), NULL)), __temp107);
																											}
																											if (type_node->is_pointer && (g_utf8_collate (n, "gchar") == 0 || g_utf8_collate (n, "char") == 0)) {
																												(vala_type_reference_set_type_name (type, "string"), vala_type_reference_get_type_name (type));
																											} else {
																												if (g_utf8_collate (n, "gunichar") == 0) {
																													(vala_type_reference_set_type_name (type, "unichar"), vala_type_reference_get_type_name (type));
																												} else {
																													if (g_utf8_collate (n, "gchar") == 0) {
																														(vala_type_reference_set_type_name (type, "char"), vala_type_reference_get_type_name (type));
																													} else {
																														if (g_utf8_collate (n, "guchar") == 0 || g_utf8_collate (n, "guint8") == 0) {
																															(vala_type_reference_set_type_name (type, "uchar"), vala_type_reference_get_type_name (type));
																															if (type_node->is_pointer) {
																																(vala_type_reference_set_array_rank (type, 1), vala_type_reference_get_array_rank (type));
																															}
																														} else {
																															if (g_utf8_collate (n, "gushort") == 0) {
																																(vala_type_reference_set_type_name (type, "ushort"), vala_type_reference_get_type_name (type));
																															} else {
																																if (g_utf8_collate (n, "gshort") == 0) {
																																	(vala_type_reference_set_type_name (type, "short"), vala_type_reference_get_type_name (type));
																																} else {
																																	if (g_utf8_collate (n, "gconstpointer") == 0 || g_utf8_collate (n, "void") == 0) {
																																		(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
																																	} else {
																																		if (g_utf8_collate (n, "goffset") == 0) {
																																			(vala_type_reference_set_type_name (type, "int64"), vala_type_reference_get_type_name (type));
																																		} else {
																																			if (g_utf8_collate (n, "value_array") == 0) {
																																				(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																																				(vala_type_reference_set_type_name (type, "ValueArray"), vala_type_reference_get_type_name (type));
																																			} else {
																																				if (g_utf8_collate (n, "time_t") == 0) {
																																					(vala_type_reference_set_type_name (type, "ulong"), vala_type_reference_get_type_name (type));
																																				} else {
																																					if (g_utf8_collate (n, "pid_t") == 0) {
																																						(vala_type_reference_set_type_name (type, "int"), vala_type_reference_get_type_name (type));
																																					} else {
																																						if (g_utf8_collate (n, "FILE") == 0) {
																																							(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
																																							(vala_type_reference_set_type_name (type, "FileStream"), vala_type_reference_get_type_name (type));
																																						} else {
																																							vala_gidl_parser_parse_type_string (self, type, n);
																																							if (type_node->is_pointer && vala_gidl_parser_is_value_type (self, n)) {
																																								(vala_type_reference_set_is_out (type, TRUE), vala_type_reference_get_is_out (type));
																																							}
																																						}
																																					}
																																				}
																																			}
																																		}
																																	}
																																}
																															}
																														}
																													}
																												}
																											}
																											(n = (g_free (n), NULL));
																										} else {
																											fprintf (stdout, "%d\n", type_node->tag);
																										}
																									}
																								}
																							}
																						}
																					}
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	return type;
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static gboolean vala_gidl_parser_is_value_type (ValaGIdlParser* self, const char* type_name)
{
	char** __temp109;
	gint type_attributes_length1;
	char** type_attributes;
	gboolean __temp115;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), FALSE);
	g_return_val_if_fail (type_name != NULL, FALSE);
	/* FIXME only works if both types are in current package, e.g. doesn't work when Gtk uses GdkRectangle*/
	__temp109 = NULL;
	type_attributes = (__temp109 = vala_gidl_parser_get_attributes (self, type_name), type_attributes_length1 = -1, __temp109);
	if (type_attributes != NULL) {
		{
			char** __temp114;
			int attr_it;
			__temp114 = NULL;
			__temp114 = type_attributes;
			for (attr_it = 0; (type_attributes_length1 != -1 && attr_it < type_attributes_length1) || (type_attributes_length1 == -1 && __temp114[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp114[attr_it];
				{
					char** __temp110;
					gint nv_length1;
					char** nv;
					char* __temp111;
					gboolean __temp112;
					__temp110 = NULL;
					nv = (__temp110 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp110);
					__temp111 = NULL;
					if ((__temp112 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((__temp111 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp111 = (g_free (__temp111), NULL)), __temp112)) {
						gboolean __temp113;
						return (__temp113 = TRUE, (nv = (g_free (nv), NULL)), __temp113);
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	return (__temp115 = FALSE, (type_attributes = (g_free (type_attributes), NULL)), __temp115);
	(type_attributes = (g_free (type_attributes), NULL));
}


static void vala_gidl_parser_parse_type_string (ValaGIdlParser* self, ValaTypeReference* type, const char* n)
{
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	g_return_if_fail (n != NULL);
	/* Generated GIDL misses explicit namespace specifier,
	 so try to guess namespace*/
	if (g_str_has_prefix (n, "H") || g_str_has_suffix (n, "Class") || g_utf8_collate (n, "va_list") == 0 || g_str_has_prefix (n, "LOGFONT")) {
		/* unsupported*/
		(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
	} else {
		if (g_str_has_prefix (n, "cairo")) {
			(vala_type_reference_set_namespace_name (type, "Cairo"), vala_type_reference_get_namespace_name (type));
			if (g_utf8_collate (n, "cairo_t") == 0) {
				(vala_type_reference_set_type_name (type, "Context"), vala_type_reference_get_type_name (type));
			} else {
				if (g_utf8_collate (n, "cairo_surface_t") == 0) {
					(vala_type_reference_set_type_name (type, "Surface"), vala_type_reference_get_type_name (type));
				} else {
					(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
					(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
				}
			}
		} else {
			if (g_str_has_prefix (n, "Atk")) {
				(vala_type_reference_set_namespace_name (type, "Atk"), vala_type_reference_get_namespace_name (type));
				(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Atk", -1))), vala_type_reference_get_type_name (type));
				if (g_utf8_collate (vala_type_reference_get_type_name (type), "AttributeSet") == 0) {
					(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
					(vala_type_reference_set_type_name (type, "SList"), vala_type_reference_get_type_name (type));
				}
			} else {
				if (g_str_has_prefix (n, "Gtk")) {
					(vala_type_reference_set_namespace_name (type, "Gtk"), vala_type_reference_get_namespace_name (type));
					(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Gtk", -1))), vala_type_reference_get_type_name (type));
					if (g_utf8_collate (vala_type_reference_get_type_name (type), "TextLayout") == 0) {
						(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
						(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
					}
				} else {
					if (g_str_has_prefix (n, "Gdk")) {
						(vala_type_reference_set_namespace_name (type, "Gdk"), vala_type_reference_get_namespace_name (type));
						(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Gdk", -1))), vala_type_reference_get_type_name (type));
						if (g_utf8_collate (vala_type_reference_get_type_name (type), "NativeWindow") == 0 || g_utf8_collate (vala_type_reference_get_type_name (type), "XEvent") == 0) {
							(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
							(vala_type_reference_set_type_name (type, "pointer"), vala_type_reference_get_type_name (type));
						}
					} else {
						if (g_str_has_prefix (n, "Pango")) {
							(vala_type_reference_set_namespace_name (type, "Pango"), vala_type_reference_get_namespace_name (type));
							(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Pango", -1))), vala_type_reference_get_type_name (type));
							if (g_utf8_collate (vala_type_reference_get_type_name (type), "Glyph") == 0) {
								(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
								(vala_type_reference_set_type_name (type, "uint"), vala_type_reference_get_type_name (type));
							} else {
								if (g_utf8_collate (vala_type_reference_get_type_name (type), "GlyphUnit") == 0) {
									(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
									(vala_type_reference_set_type_name (type, "int"), vala_type_reference_get_type_name (type));
								}
							}
						} else {
							if (g_str_has_prefix (n, "Vte")) {
								(vala_type_reference_set_namespace_name (type, "Vte"), vala_type_reference_get_namespace_name (type));
								(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Vte", -1))), vala_type_reference_get_type_name (type));
							} else {
								if (g_str_has_prefix (n, "Goo")) {
									(vala_type_reference_set_namespace_name (type, "Goo"), vala_type_reference_get_namespace_name (type));
									(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Goo", -1))), vala_type_reference_get_type_name (type));
								} else {
									if (g_str_has_prefix (n, "G")) {
										(vala_type_reference_set_namespace_name (type, "GLib"), vala_type_reference_get_namespace_name (type));
										(vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("G", -1))), vala_type_reference_get_type_name (type));
										if (g_utf8_collate (vala_type_reference_get_type_name (type), "Strv") == 0) {
											(vala_type_reference_set_namespace_name (type, NULL), vala_type_reference_get_namespace_name (type));
											(vala_type_reference_set_type_name (type, "string"), vala_type_reference_get_type_name (type));
											(vala_type_reference_set_array_rank (type, 1), vala_type_reference_get_array_rank (type));
										}
									} else {
										char** __temp116;
										gint name_parts_length1;
										char** name_parts;
										__temp116 = NULL;
										name_parts = (__temp116 = g_strsplit (n, ".", 2), name_parts_length1 = -1, __temp116);
										if (name_parts[1] == NULL) {
											(vala_type_reference_set_type_name (type, name_parts[0]), vala_type_reference_get_type_name (type));
										} else {
											(vala_type_reference_set_namespace_name (type, name_parts[0]), vala_type_reference_get_namespace_name (type));
											(vala_type_reference_set_type_name (type, name_parts[1]), vala_type_reference_get_type_name (type));
										}
										(name_parts = (g_free (name_parts), NULL));
									}
								}
							}
						}
					}
				}
			}
		}
	}
}


static ValaTypeReference* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param)
{
	ValaTypeReference* type;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (param != NULL, NULL);
	type = vala_gidl_parser_parse_type (self, param->type);
	return type;
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


/* disable for now as null_ok not yet correctly set
 type.non_null = !param.null_ok;*/
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_virtual)
{
	GIdlNode* node;
	ValaTypeReference* return_type;
	ValaMethod* m;
	gboolean add_ellipsis;
	char** __temp131;
	gint attributes_length1;
	char** attributes;
	gboolean first;
	ValaFormalParameter* last_param;
	ValaMethod* __temp145;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (f != NULL, NULL);
	node = ((GIdlNode*) f);
	if (f->deprecated) {
		return VALA_METHOD (NULL);
	}
	return_type = NULL;
	if (f->result != NULL) {
		ValaTypeReference* __temp119;
		__temp119 = NULL;
		return_type = (__temp119 = vala_gidl_parser_parse_param (self, f->result), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), __temp119);
	}
	m = NULL;
	if (f->is_constructor || g_str_has_prefix (node->name, "new")) {
		ValaMethod* __temp120;
		__temp120 = NULL;
		m = (__temp120 = VALA_METHOD (vala_creation_method_new (node->name, self->priv->current_source_reference)), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), __temp120);
		if (g_utf8_collate (vala_method_get_name (m), "new") == 0) {
			(vala_method_set_name (m, NULL), vala_method_get_name (m));
		} else {
			if (g_str_has_prefix (vala_method_get_name (m), "new_")) {
				(vala_method_set_name (m, g_utf8_offset_to_pointer (vala_method_get_name (m), g_utf8_strlen ("new_", -1))), vala_method_get_name (m));
			}
		}
	} else {
		ValaMethod* __temp121;
		__temp121 = NULL;
		m = (__temp121 = vala_method_new (node->name, return_type, self->priv->current_source_reference), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), __temp121);
	}
	m->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	(vala_method_set_is_virtual (m, is_virtual), vala_method_get_is_virtual (m));
	/* GIDL generator can't provide array parameter information yet*/
	(vala_method_set_no_array_length (m, TRUE), vala_method_get_no_array_length (m));
	if (self->priv->current_type_symbol_map != NULL) {
		const char* __temp122;
		__temp122 = NULL;
		g_hash_table_insert (self->priv->current_type_symbol_map, (__temp122 = node->name, (__temp122 == NULL ? NULL : g_strdup (__temp122))), g_strdup ("1"));
	}
	if (self->priv->current_data_type != NULL) {
		char** __temp125;
		gint sig_attributes_length1;
		char* __temp123;
		char** __temp124;
		char** sig_attributes;
		__temp125 = NULL;
		__temp123 = NULL;
		__temp124 = NULL;
		sig_attributes = (__temp125 = (__temp124 = vala_gidl_parser_get_attributes (self, (__temp123 = g_strdup_printf ("%s::%s", vala_data_type_get_name (self->priv->current_data_type), node->name))), (__temp123 = (g_free (__temp123), NULL)), __temp124), sig_attributes_length1 = -1, __temp125);
		if (sig_attributes != NULL) {
			{
				char** __temp130;
				int attr_it;
				__temp130 = NULL;
				__temp130 = sig_attributes;
				for (attr_it = 0; (sig_attributes_length1 != -1 && attr_it < sig_attributes_length1) || (sig_attributes_length1 == -1 && __temp130[attr_it] != NULL); attr_it = attr_it + 1) {
					char* attr;
					attr = __temp130[attr_it];
					{
						char** __temp126;
						gint nv_length1;
						char** nv;
						char* __temp127;
						gboolean __temp128;
						__temp126 = NULL;
						nv = (__temp126 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp126);
						__temp127 = NULL;
						if ((__temp128 = g_utf8_collate (nv[0], "has_emitter") == 0 && g_utf8_collate ((__temp127 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp127 = (g_free (__temp127), NULL)), __temp128)) {
							gpointer __temp129;
							return VALA_METHOD ((__temp129 = NULL, (nv = (g_free (nv), NULL)), __temp129));
						}
						(nv = (g_free (nv), NULL));
					}
				}
			}
		}
		(sig_attributes = (g_free (sig_attributes), NULL));
	}
	add_ellipsis = FALSE;
	__temp131 = NULL;
	attributes = (__temp131 = vala_gidl_parser_get_attributes (self, f->symbol), attributes_length1 = -1, __temp131);
	if (attributes != NULL) {
		{
			char** __temp138;
			int attr_it;
			__temp138 = NULL;
			__temp138 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp138[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp138[attr_it];
				{
					char** __temp132;
					gint nv_length1;
					char** nv;
					__temp132 = NULL;
					nv = (__temp132 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp132);
					if (g_utf8_collate (nv[0], "hidden") == 0) {
						char* __temp133;
						gboolean __temp134;
						__temp133 = NULL;
						if ((__temp134 = g_utf8_collate ((__temp133 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp133 = (g_free (__temp133), NULL)), __temp134)) {
							gpointer __temp135;
							return VALA_METHOD ((__temp135 = NULL, (nv = (g_free (nv), NULL)), __temp135));
						}
					} else {
						if (g_utf8_collate (nv[0], "ellipsis") == 0) {
							char* __temp136;
							gboolean __temp137;
							__temp136 = NULL;
							if ((__temp137 = g_utf8_collate ((__temp136 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp136 = (g_free (__temp136), NULL)), __temp137)) {
								add_ellipsis = TRUE;
							}
						}
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	vala_method_set_cname (m, f->symbol);
	first = TRUE;
	last_param = NULL;
	{
		GList* __temp143;
		GList* param_it;
		__temp143 = NULL;
		__temp143 = f->parameters;
		for (param_it = __temp143; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaFormalParameter* p;
				char* __temp139;
				gboolean __temp140;
				ValaFormalParameter* __temp142;
				ValaFormalParameter* __temp141;
				param_node = ((GIdlNode*) param);
				if (first) {
					first = FALSE;
					if (self->priv->current_data_type != NULL && param->type->is_interface && (g_utf8_collate (param_node->name, "self") == 0 || g_str_has_suffix (param->type->interface, vala_data_type_get_name (self->priv->current_data_type)))) {
						/* instance method*/
						if (!vala_data_type_is_reference_type (self->priv->current_data_type) && param->type->is_pointer) {
							(vala_method_set_instance_by_reference (m, TRUE), vala_method_get_instance_by_reference (m));
						}
						continue;
					} else {
						/* static method*/
						(vala_method_set_instance (m, FALSE), vala_method_get_instance (m));
					}
				}
				p = vala_formal_parameter_new (param_node->name, vala_gidl_parser_parse_param (self, param), NULL);
				vala_method_add_parameter (m, p);
				__temp139 = NULL;
				if ((__temp140 = last_param != NULL && g_utf8_collate (vala_formal_parameter_get_name (p), (__temp139 = g_strconcat ("n_", vala_formal_parameter_get_name (last_param), NULL))) == 0, (__temp139 = (g_free (__temp139), NULL)), __temp140)) {
					/* last_param is array, p is array length*/
					(vala_type_reference_set_array_rank (vala_formal_parameter_get_type_reference (last_param), 1), vala_type_reference_get_array_rank (vala_formal_parameter_get_type_reference (last_param)));
					(vala_type_reference_set_is_out (vala_formal_parameter_get_type_reference (last_param), FALSE), vala_type_reference_get_is_out (vala_formal_parameter_get_type_reference (last_param)));
				}
				__temp142 = NULL;
				__temp141 = NULL;
				last_param = (__temp142 = (__temp141 = p, (__temp141 == NULL ? NULL : g_object_ref (__temp141))), (last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL))), __temp142);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	if (first) {
		/* no parameters => static method*/
		(vala_method_set_instance (m, FALSE), vala_method_get_instance (m));
	}
	if (add_ellipsis) {
		ValaFormalParameter* __temp144;
		__temp144 = NULL;
		vala_method_add_parameter (m, (__temp144 = vala_formal_parameter_new_with_ellipsis (NULL)));
		(__temp144 == NULL ? NULL : (__temp144 = (g_object_unref (__temp144), NULL)));
	}
	__temp145 = NULL;
	return (__temp145 = m, (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (attributes = (g_free (attributes), NULL)), (last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL))), __temp145);
	(return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL)));
	(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	(attributes = (g_free (attributes), NULL));
	(last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL)));
}


static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name)
{
	GString* str;
	const char* __temp146;
	char* i;
	const char* __temp149;
	char* __temp150;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	str = g_string_new ("");
	__temp146 = NULL;
	i = (__temp146 = name, (__temp146 == NULL ? NULL : g_strdup (__temp146)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* __temp148;
		const char* __temp147;
		c = g_utf8_get_char (i);
		if (c == '-') {
			g_string_append_c (str, '_');
		} else {
			g_string_append_unichar (str, c);
		}
		__temp148 = NULL;
		__temp147 = NULL;
		i = (__temp148 = (__temp147 = g_utf8_next_char (i), (__temp147 == NULL ? NULL : g_strdup (__temp147))), (i = (g_free (i), NULL)), __temp148);
	}
	__temp149 = NULL;
	__temp150 = NULL;
	return (__temp150 = (__temp149 = str->str, (__temp149 == NULL ? NULL : g_strdup (__temp149))), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), __temp150);
	(str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL)));
	(i = (g_free (i), NULL));
}


static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node)
{
	GIdlNode* node;
	ValaPropertyAccessor* get_acc;
	ValaPropertyAccessor* set_acc;
	ValaProperty* prop;
	ValaProperty* __temp155;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (prop_node != NULL, NULL);
	node = ((GIdlNode*) prop_node);
	if (prop_node->deprecated) {
		return VALA_PROPERTY (NULL);
	}
	if (!prop_node->readable && !prop_node->writable) {
		/* buggy GIDL definition*/
		prop_node->readable = TRUE;
		prop_node->writable = TRUE;
	}
	get_acc = NULL;
	set_acc = NULL;
	if (prop_node->readable) {
		ValaPropertyAccessor* __temp152;
		__temp152 = NULL;
		get_acc = (__temp152 = vala_property_accessor_new (TRUE, FALSE, FALSE, NULL, NULL), (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), __temp152);
	}
	if (prop_node->writable) {
		ValaPropertyAccessor* __temp153;
		__temp153 = NULL;
		set_acc = (__temp153 = vala_property_accessor_new (FALSE, FALSE, FALSE, NULL, NULL), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), __temp153);
		if (prop_node->construct_only) {
			(vala_property_accessor_set_construction (set_acc, TRUE), vala_property_accessor_get_construction (set_acc));
		} else {
			(vala_property_accessor_set_writable (set_acc, TRUE), vala_property_accessor_get_writable (set_acc));
			(vala_property_accessor_set_construction (set_acc, prop_node->construct), vala_property_accessor_get_construction (set_acc));
		}
	}
	prop = vala_property_new (vala_gidl_parser_fix_prop_name (self, node->name), vala_gidl_parser_parse_type (self, prop_node->type), get_acc, set_acc, self->priv->current_source_reference);
	(vala_property_set_access (prop, VALA_MEMBER_ACCESSIBILITY_PUBLIC), vala_property_get_access (prop));
	(vala_property_set_interface_only (prop, TRUE), vala_property_get_interface_only (prop));
	if (self->priv->current_type_symbol_map != NULL) {
		const char* __temp154;
		__temp154 = NULL;
		g_hash_table_insert (self->priv->current_type_symbol_map, (__temp154 = vala_property_get_name (prop), (__temp154 == NULL ? NULL : g_strdup (__temp154))), g_strdup ("1"));
	}
	__temp155 = NULL;
	return (__temp155 = prop, (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), __temp155);
	(get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL)));
	(set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL)));
	(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
}


static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node)
{
	GIdlNode* node;
	ValaTypeReference* type;
	ValaConstant* c;
	ValaConstant* __temp157;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (const_node != NULL, NULL);
	node = ((GIdlNode*) const_node);
	type = vala_gidl_parser_parse_type (self, const_node->type);
	if (type == NULL) {
		gpointer __temp156;
		return VALA_CONSTANT ((__temp156 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp156));
	}
	c = vala_constant_new (node->name, type, NULL, self->priv->current_source_reference);
	__temp157 = NULL;
	return (__temp157 = c, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp157);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
	(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
}


static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node)
{
	GIdlNode* node;
	ValaTypeReference* type;
	ValaField* field;
	ValaField* __temp161;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (field_node != NULL, NULL);
	node = ((GIdlNode*) field_node);
	type = vala_gidl_parser_parse_type (self, field_node->type);
	if (type == NULL) {
		gpointer __temp158;
		return VALA_FIELD ((__temp158 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp158));
	}
	if (!field_node->readable) {
		gpointer __temp159;
		return VALA_FIELD ((__temp159 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp159));
	}
	if (self->priv->current_type_symbol_map != NULL) {
		const char* __temp160;
		__temp160 = NULL;
		g_hash_table_insert (self->priv->current_type_symbol_map, (__temp160 = node->name, (__temp160 == NULL ? NULL : g_strdup (__temp160))), g_strdup ("1"));
	}
	field = vala_field_new (node->name, type, NULL, self->priv->current_source_reference);
	field->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	__temp161 = NULL;
	return (__temp161 = field, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp161);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
	(field == NULL ? NULL : (field = (g_object_unref (field), NULL)));
}


static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode)
{
	const char* __temp162;
	char* attributes;
	char** __temp164;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (codenode != NULL, NULL);
	__temp162 = NULL;
	attributes = (__temp162 = g_hash_table_lookup (self->priv->codenode_attributes_map, codenode), (__temp162 == NULL ? NULL : g_strdup (__temp162)));
	if (attributes == NULL) {
		gpointer __temp163;
		return (__temp163 = NULL, (attributes = (g_free (attributes), NULL)), __temp163);
	}
	__temp164 = NULL;
	return (__temp164 = g_strsplit (attributes, " ", 0), (attributes = (g_free (attributes), NULL)), __temp164);
	(attributes = (g_free (attributes), NULL));
}


static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s)
{
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (s != NULL, NULL);
	return g_strndup (g_utf8_offset_to_pointer (s, 1), strlen (s) - 2);
}


static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node)
{
	GIdlNode* node;
	ValaSignal* sig;
	char** __temp169;
	gint attributes_length1;
	char* __temp167;
	char** __temp168;
	char** attributes;
	gboolean first;
	ValaSignal* __temp179;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (sig_node != NULL, NULL);
	node = ((GIdlNode*) sig_node);
	if (sig_node->deprecated || sig_node->result == NULL) {
		return VALA_SIGNAL (NULL);
	}
	sig = vala_signal_new (vala_gidl_parser_fix_prop_name (self, node->name), vala_gidl_parser_parse_param (self, sig_node->result), self->priv->current_source_reference);
	(vala_signal_set_access (sig, VALA_MEMBER_ACCESSIBILITY_PUBLIC), vala_signal_get_access (sig));
	__temp169 = NULL;
	__temp167 = NULL;
	__temp168 = NULL;
	attributes = (__temp169 = (__temp168 = vala_gidl_parser_get_attributes (self, (__temp167 = g_strdup_printf ("%s::%s", vala_data_type_get_name (self->priv->current_data_type), vala_signal_get_name (sig)))), (__temp167 = (g_free (__temp167), NULL)), __temp168), attributes_length1 = -1, __temp169);
	if (attributes != NULL) {
		{
			char** __temp177;
			int attr_it;
			__temp177 = NULL;
			__temp177 = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && __temp177[attr_it] != NULL); attr_it = attr_it + 1) {
				char* attr;
				attr = __temp177[attr_it];
				{
					char** __temp170;
					gint nv_length1;
					char** nv;
					__temp170 = NULL;
					nv = (__temp170 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp170);
					if (g_utf8_collate (nv[0], "name") == 0) {
						char* __temp171;
						vala_signal_set_cname (sig, vala_signal_get_name (sig));
						__temp171 = NULL;
						(vala_signal_set_name (sig, (__temp171 = vala_gidl_parser_eval (self, nv[1]))), vala_signal_get_name (sig));
						(__temp171 = (g_free (__temp171), NULL));
					} else {
						char* __temp172;
						gboolean __temp173;
						__temp172 = NULL;
						if ((__temp173 = g_utf8_collate (nv[0], "has_emitter") == 0 && g_utf8_collate ((__temp172 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp172 = (g_free (__temp172), NULL)), __temp173)) {
							(vala_signal_set_has_emitter (sig, TRUE), vala_signal_get_has_emitter (sig));
						} else {
							if (g_utf8_collate (nv[0], "hidden") == 0) {
								char* __temp174;
								gboolean __temp175;
								__temp174 = NULL;
								if ((__temp175 = g_utf8_collate ((__temp174 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp174 = (g_free (__temp174), NULL)), __temp175)) {
									gpointer __temp176;
									return VALA_SIGNAL ((__temp176 = NULL, (nv = (g_free (nv), NULL)), __temp176));
								}
							}
						}
					}
					(nv = (g_free (nv), NULL));
				}
			}
		}
	}
	first = TRUE;
	{
		GList* __temp178;
		GList* param_it;
		__temp178 = NULL;
		__temp178 = sig_node->parameters;
		for (param_it = __temp178; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaFormalParameter* p;
				if (first) {
					/* ignore implicit first signal parameter (sender)*/
					first = FALSE;
					continue;
				}
				param_node = ((GIdlNode*) param);
				p = vala_formal_parameter_new (param_node->name, vala_gidl_parser_parse_param (self, param), NULL);
				vala_signal_add_parameter (sig, p);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	__temp179 = NULL;
	return (__temp179 = sig, (attributes = (g_free (attributes), NULL)), __temp179);
	(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
	(attributes = (g_free (attributes), NULL));
}


static void vala_gidl_parser_class_init (ValaGIdlParserClass * klass)
{
	vala_gidl_parser_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaGIdlParserPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_gidl_parser_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_gidl_parser_real_visit_source_file;
}


static void vala_gidl_parser_init (ValaGIdlParser * self)
{
	self->priv = VALA_GIDL_PARSER_GET_PRIVATE (self);
}


static void vala_gidl_parser_dispose (GObject * obj)
{
	ValaGIdlParser * self;
	ValaGIdlParserClass * klass;
	GObjectClass * parent_class;
	self = VALA_GIDL_PARSER (obj);
	(self->priv->current_source_reference == NULL ? NULL : (self->priv->current_source_reference = (g_object_unref (self->priv->current_source_reference), NULL)));
	(self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL)));
	(self->priv->codenode_attributes_map == NULL ? NULL : (self->priv->codenode_attributes_map = (g_hash_table_unref (self->priv->codenode_attributes_map), NULL)));
	(self->priv->current_type_symbol_map == NULL ? NULL : (self->priv->current_type_symbol_map = (g_hash_table_unref (self->priv->current_type_symbol_map), NULL)));
	klass = VALA_GIDL_PARSER_CLASS (g_type_class_peek (VALA_TYPE_GIDL_PARSER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_gidl_parser_get_type ()
{
	static GType vala_gidl_parser_type_id = 0;
	if (G_UNLIKELY (vala_gidl_parser_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGIdlParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gidl_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGIdlParser), 0, (GInstanceInitFunc) vala_gidl_parser_init };
		vala_gidl_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGIdlParser", &g_define_type_info, 0);
	}
	return vala_gidl_parser_type_id;
}




