/* FIXME check whether the corresponding data type of one of the
       parameters is a base type of the corresponding data
       type of the other parameter and check along the path
       whether one parameter maps to the other
 valatypereference.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */

#include <vala/valatypereference.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valadatatype.h>
#include <vala/valatypeparameter.h>
#include <vala/valasourcereference.h>
#include <vala/valaexpression.h>
#include <vala/valamemberaccess.h>
#include <vala/valareport.h>
#include <vala/valacodevisitor.h>
#include <vala/valaarray.h>
#include <vala/valasymbol.h>

struct _ValaTypeReferencePrivate {
	gboolean _transfers_ownership;
	gboolean _takes_ownership;
	gboolean _is_out;
	gboolean _non_null;
	gboolean _is_null;
	ValaDataType* _data_type;
	ValaTypeParameter* _type_parameter;
	gboolean _floating_reference;
	char* _namespace_name;
	char* _type_name;
	gint _array_rank;
	gint _pointer_level;
	gboolean _is_ref;
	gboolean _is_weak;
	GeeArrayList* type_argument_list;
};
#define VALA_TYPE_REFERENCE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_TYPE_REFERENCE, ValaTypeReferencePrivate))
enum  {
	VALA_TYPE_REFERENCE_DUMMY_PROPERTY,
	VALA_TYPE_REFERENCE_TRANSFERS_OWNERSHIP,
	VALA_TYPE_REFERENCE_TAKES_OWNERSHIP,
	VALA_TYPE_REFERENCE_IS_OUT,
	VALA_TYPE_REFERENCE_NON_NULL,
	VALA_TYPE_REFERENCE_IS_NULL,
	VALA_TYPE_REFERENCE_DATA_TYPE,
	VALA_TYPE_REFERENCE_TYPE_PARAMETER,
	VALA_TYPE_REFERENCE_FLOATING_REFERENCE,
	VALA_TYPE_REFERENCE_NAMESPACE_NAME,
	VALA_TYPE_REFERENCE_TYPE_NAME,
	VALA_TYPE_REFERENCE_ARRAY_RANK,
	VALA_TYPE_REFERENCE_POINTER_LEVEL,
	VALA_TYPE_REFERENCE_IS_REF,
	VALA_TYPE_REFERENCE_IS_WEAK
};
static void vala_type_reference_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_type_reference_parent_class = NULL;
static void vala_type_reference_dispose (GObject * obj);


ValaTypeReference* vala_type_reference_new (void) {
	ValaTypeReference * self;
	self = g_object_newv (VALA_TYPE_TYPE_REFERENCE, 0, NULL);
	return self;
}


/**
 * Creates a new type reference.
 *
 * @param ns        optional namespace name
 * @param type_name type symbol name
 * @param source    reference to source code
 * @return          newly created type reference
 */
ValaTypeReference* vala_type_reference_new_from_name (const char* ns, const char* type, ValaSourceReference* source) {
	ValaTypeReference * self;
	g_return_val_if_fail (type != NULL, NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = g_object_newv (VALA_TYPE_TYPE_REFERENCE, 0, NULL);
	vala_type_reference_set_namespace_name (self, ns);
	vala_type_reference_set_type_name (self, type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	return self;
}


/**
 * Creates a new type reference from a code expression.
 *
 * @param expr   member access expression
 * @param source reference to source code
 * @return       newly created type reference
 */
ValaTypeReference* vala_type_reference_new_from_expression (ValaExpression* expr) {
	char* ns;
	char* type_name;
	gpointer _tmp5;
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), NULL);
	ns = NULL;
	type_name = NULL;
	if (VALA_IS_MEMBER_ACCESS (expr)) {
		ValaTypeReference* type_ref;
		ValaMemberAccess* _tmp0;
		ValaMemberAccess* ma;
		type_ref = NULL;
		_tmp0 = NULL;
		ma = (_tmp0 = VALA_MEMBER_ACCESS (expr), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		if (vala_member_access_get_inner (ma) != NULL) {
			if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (ma))) {
				ValaMemberAccess* _tmp1;
				ValaMemberAccess* simple;
				ValaTypeReference* _tmp2;
				_tmp1 = NULL;
				simple = (_tmp1 = VALA_MEMBER_ACCESS (vala_member_access_get_inner (ma)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
				_tmp2 = NULL;
				type_ref = (_tmp2 = vala_type_reference_new_from_name (vala_member_access_get_member_name (simple), vala_member_access_get_member_name (ma), vala_code_node_get_source_reference (VALA_CODE_NODE (ma))), (type_ref == NULL ? NULL : (type_ref = (g_object_unref (type_ref), NULL))), _tmp2);
				(simple == NULL ? NULL : (simple = (g_object_unref (simple), NULL)));
			}
		} else {
			ValaTypeReference* _tmp3;
			_tmp3 = NULL;
			type_ref = (_tmp3 = vala_type_reference_new_from_name (NULL, vala_member_access_get_member_name (ma), vala_code_node_get_source_reference (VALA_CODE_NODE (ma))), (type_ref == NULL ? NULL : (type_ref = (g_object_unref (type_ref), NULL))), _tmp3);
		}
		if (type_ref != NULL) {
			GeeCollection* type_args;
			ValaTypeReference* _tmp4;
			type_args = vala_member_access_get_type_arguments (ma);
			{
				GeeCollection* arg_collection;
				GeeIterator* arg_it;
				arg_collection = type_args;
				arg_it = gee_iterable_iterator (GEE_ITERABLE (arg_collection));
				while (gee_iterator_next (arg_it)) {
					ValaTypeReference* arg;
					arg = gee_iterator_get (arg_it);
					{
						vala_type_reference_add_type_argument (type_ref, arg);
						(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
					}
				}
				(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
			}
			_tmp4 = NULL;
			return (_tmp4 = type_ref, (type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL))), (ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL))), (ns = (g_free (ns), NULL)), (type_name = (g_free (type_name), NULL)), _tmp4);
			(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
		}
		(type_ref == NULL ? NULL : (type_ref = (g_object_unref (type_ref), NULL)));
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	}
	vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Type reference must be simple name or member access expression");
	return (_tmp5 = NULL, (ns = (g_free (ns), NULL)), (type_name = (g_free (type_name), NULL)), _tmp5);
	(ns = (g_free (ns), NULL));
	(type_name = (g_free (type_name), NULL));
}


/**
 * Appends the specified type as generic type argument.
 *
 * @param arg a type reference
 */
void vala_type_reference_add_type_argument (ValaTypeReference* self, ValaTypeReference* arg) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (arg));
	gee_collection_add (GEE_COLLECTION (self->priv->type_argument_list), arg);
}


/**
 * Returns a copy of the list of generic type arguments.
 *
 * @return type argument list
 */
GeeList* vala_type_reference_get_type_arguments (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), NULL);
	return GEE_LIST (gee_read_only_list_new (g_object_ref, g_object_unref, GEE_LIST (self->priv->type_argument_list)));
}


/**
 * Removes all generic type arguments.
 */
void vala_type_reference_remove_all_type_arguments (ValaTypeReference* self) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	gee_collection_clear (GEE_COLLECTION (self->priv->type_argument_list));
}


static void vala_type_reference_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaTypeReference * self;
	self = VALA_TYPE_REFERENCE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (gee_collection_get_size (GEE_COLLECTION ((GEE_LIST (self->priv->type_argument_list)))) > 0) {
		{
			GeeArrayList* type_arg_collection;
			GeeIterator* type_arg_it;
			type_arg_collection = self->priv->type_argument_list;
			type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
			while (gee_iterator_next (type_arg_it)) {
				ValaTypeReference* type_arg;
				type_arg = gee_iterator_get (type_arg_it);
				{
					vala_code_node_accept (VALA_CODE_NODE (type_arg), visitor);
					(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
				}
			}
			(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
		}
	}
	vala_code_visitor_visit_type_reference (visitor, self);
}


/**
 * Returns the name and qualifiers of this type as it is used in C code.
 *
 * @return the type string to be used in C code
 */
char* vala_type_reference_get_cname (ValaTypeReference* self, gboolean var_type, gboolean const_type) {
	char* ptr;
	char* arr;
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), NULL);
	if (vala_type_reference_get_data_type (self) == NULL && vala_type_reference_get_type_parameter (self) == NULL) {
		if (var_type) {
			return g_strdup ("gpointer");
		} else {
			return g_strdup ("void");
		}
	}
	ptr = NULL;
	arr = NULL;
	if (vala_type_reference_get_type_parameter (self) != NULL || (!vala_data_type_is_reference_type (vala_type_reference_get_data_type (self)) && !vala_type_reference_get_is_ref (self) && !vala_type_reference_get_is_out (self))) {
		char* _tmp2;
		_tmp2 = NULL;
		ptr = (_tmp2 = g_strdup (""), (ptr = (g_free (ptr), NULL)), _tmp2);
	} else {
		if ((vala_data_type_is_reference_type (vala_type_reference_get_data_type (self)) && !vala_type_reference_get_is_ref (self) && !vala_type_reference_get_is_out (self)) || (!vala_data_type_is_reference_type (vala_type_reference_get_data_type (self)) && (vala_type_reference_get_is_ref (self) || vala_type_reference_get_is_out (self)))) {
			char* _tmp3;
			_tmp3 = NULL;
			ptr = (_tmp3 = g_strdup ("*"), (ptr = (g_free (ptr), NULL)), _tmp3);
		} else {
			char* _tmp4;
			_tmp4 = NULL;
			ptr = (_tmp4 = g_strdup ("**"), (ptr = (g_free (ptr), NULL)), _tmp4);
		}
	}
	if (vala_type_reference_get_data_type (self) != NULL) {
		char* _tmp5;
		char* _tmp6;
		char* _tmp7;
		_tmp5 = NULL;
		_tmp6 = NULL;
		_tmp7 = NULL;
		return (_tmp7 = (_tmp6 = g_strconcat ((_tmp5 = vala_data_type_get_cname (vala_type_reference_get_data_type (self), const_type)), ptr, arr, NULL, NULL), (_tmp5 = (g_free (_tmp5), NULL)), _tmp6), (ptr = (g_free (ptr), NULL)), (arr = (g_free (arr), NULL)), _tmp7);
	} else {
		if (vala_type_reference_get_type_parameter (self) != NULL) {
			char* _tmp8;
			_tmp8 = NULL;
			return (_tmp8 = g_strconcat ("gpointer", ptr, arr, NULL, NULL), (ptr = (g_free (ptr), NULL)), (arr = (g_free (arr), NULL)), _tmp8);
		} else {
			gpointer _tmp9;
			/* raise error */
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), "unresolved type reference");
			return (_tmp9 = NULL, (ptr = (g_free (ptr), NULL)), (arr = (g_free (arr), NULL)), _tmp9);
		}
	}
	(ptr = (g_free (ptr), NULL));
	(arr = (g_free (arr), NULL));
}


/**
 * Returns the name and qualifiers of this type as it is used in C code
 * in a const declaration.
 *
 * @return the type string to be used in C code const declarations
 */
char* vala_type_reference_get_const_cname (ValaTypeReference* self) {
	char* ptr;
	ValaDataType* t;
	char* _tmp6;
	char* _tmp7;
	char* _tmp8;
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), NULL);
	ptr = NULL;
	t = NULL;
	/* FIXME: dirty hack to make constant arrays possible */
	if (VALA_IS_ARRAY (vala_type_reference_get_data_type (self))) {
		ValaDataType* _tmp1;
		ValaDataType* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		t = (_tmp1 = (_tmp0 = vala_array_get_element_type ((VALA_ARRAY (vala_type_reference_get_data_type (self)))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), _tmp1);
	} else {
		ValaDataType* _tmp3;
		ValaDataType* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		t = (_tmp3 = (_tmp2 = vala_type_reference_get_data_type (self), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), _tmp3);
	}
	if (!vala_data_type_is_reference_type (t)) {
		char* _tmp4;
		_tmp4 = NULL;
		ptr = (_tmp4 = g_strdup (""), (ptr = (g_free (ptr), NULL)), _tmp4);
	} else {
		char* _tmp5;
		_tmp5 = NULL;
		ptr = (_tmp5 = g_strdup ("*"), (ptr = (g_free (ptr), NULL)), _tmp5);
	}
	_tmp6 = NULL;
	_tmp7 = NULL;
	_tmp8 = NULL;
	return (_tmp8 = (_tmp7 = g_strdup_printf ("const %s%s", (_tmp6 = vala_data_type_get_cname (t, FALSE)), ptr), (_tmp6 = (g_free (_tmp6), NULL)), _tmp7), (ptr = (g_free (ptr), NULL)), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), _tmp8);
	(ptr = (g_free (ptr), NULL));
	(t == NULL ? NULL : (t = (g_object_unref (t), NULL)));
}


/**
 * Returns a user-readable name of the type corresponding to this type
 * reference.
 *
 * @return display name
 */
char* vala_type_reference_to_string (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), NULL);
	if (vala_type_reference_get_data_type (self) != NULL) {
		return vala_symbol_get_full_name (VALA_SYMBOL (vala_type_reference_get_data_type (self)));
	} else {
		if (vala_type_reference_get_type_parameter (self) != NULL) {
			const char* _tmp1;
			_tmp1 = NULL;
			return (_tmp1 = vala_symbol_get_name (VALA_SYMBOL (vala_type_reference_get_type_parameter (self))), (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
		} else {
			return g_strdup ("null");
		}
	}
}


/**
 * Creates a shallow copy of this type reference.
 *
 * @return copy of this type reference
 */
ValaTypeReference* vala_type_reference_copy (ValaTypeReference* self) {
	ValaTypeReference* result;
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), NULL);
	result = vala_type_reference_new ();
	vala_code_node_set_source_reference (VALA_CODE_NODE (result), vala_code_node_get_source_reference (VALA_CODE_NODE (self)));
	vala_type_reference_set_transfers_ownership (result, vala_type_reference_get_transfers_ownership (self));
	vala_type_reference_set_takes_ownership (result, vala_type_reference_get_takes_ownership (self));
	vala_type_reference_set_is_out (result, vala_type_reference_get_is_out (self));
	vala_type_reference_set_non_null (result, vala_type_reference_get_non_null (self));
	vala_type_reference_set_data_type (result, vala_type_reference_get_data_type (self));
	vala_type_reference_set_type_parameter (result, vala_type_reference_get_type_parameter (self));
	vala_type_reference_set_floating_reference (result, vala_type_reference_get_floating_reference (self));
	vala_type_reference_set_namespace_name (result, vala_type_reference_get_namespace_name (self));
	vala_type_reference_set_type_name (result, vala_type_reference_get_type_name (self));
	vala_type_reference_set_array_rank (result, vala_type_reference_get_array_rank (self));
	vala_type_reference_set_pointer_level (result, vala_type_reference_get_pointer_level (self));
	vala_type_reference_set_is_ref (result, vala_type_reference_get_is_ref (self));
	vala_type_reference_set_is_weak (result, vala_type_reference_get_is_weak (self));
	{
		GeeArrayList* arg_collection;
		GeeIterator* arg_it;
		arg_collection = self->priv->type_argument_list;
		arg_it = gee_iterable_iterator (GEE_ITERABLE (arg_collection));
		while (gee_iterator_next (arg_it)) {
			ValaTypeReference* arg;
			arg = gee_iterator_get (arg_it);
			{
				ValaTypeReference* _tmp0;
				_tmp0 = NULL;
				gee_collection_add (GEE_COLLECTION (result->priv->type_argument_list), (_tmp0 = vala_type_reference_copy (arg)));
				(_tmp0 = (g_object_unref (_tmp0), NULL));
				(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
			}
		}
		(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
	}
	return result;
	(result == NULL ? NULL : (result = (g_object_unref (result), NULL)));
}


/**
 * Checks two type references for equality. May only be used with
 * resolved type references.
 *
 * @param type2 a type reference
 * @return      true if this type reference is equal to type2, false
 *              otherwise
 */
gboolean vala_type_reference_equals (ValaTypeReference* self, ValaTypeReference* type2) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type2), FALSE);
	if (vala_type_reference_get_transfers_ownership (type2) != vala_type_reference_get_transfers_ownership (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_takes_ownership (type2) != vala_type_reference_get_takes_ownership (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_is_ref (type2) != vala_type_reference_get_is_ref (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_is_out (type2) != vala_type_reference_get_is_out (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_non_null (type2) != vala_type_reference_get_non_null (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_data_type (type2) != vala_type_reference_get_data_type (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_type_parameter (type2) != NULL || vala_type_reference_get_type_parameter (self) != NULL) {
		if (vala_type_reference_get_type_parameter (type2) == NULL || vala_type_reference_get_type_parameter (self) == NULL) {
			return FALSE;
		}
		if (!vala_type_parameter_equals (vala_type_reference_get_type_parameter (type2), vala_type_reference_get_type_parameter (self))) {
			return FALSE;
		}
	}
	if (vala_type_reference_get_floating_reference (type2) != vala_type_reference_get_floating_reference (self)) {
		return FALSE;
	}
	return TRUE;
}


/**
 * Checks whether this type reference is at least as strict as the
 * specified type reference type2.
 *
 * @param type2 a type reference
 * @return      true if this type reference is stricter or equal
 */
gboolean vala_type_reference_stricter (ValaTypeReference* self, ValaTypeReference* type2) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type2), FALSE);
	if (vala_type_reference_get_transfers_ownership (type2) != vala_type_reference_get_transfers_ownership (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_takes_ownership (type2) != vala_type_reference_get_takes_ownership (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_is_ref (type2) != vala_type_reference_get_is_ref (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_is_out (type2) != vala_type_reference_get_is_out (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_non_null (type2) && !vala_type_reference_get_non_null (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_data_type (type2) != vala_type_reference_get_data_type (self)) {
		return FALSE;
	}
	/* FIXME: allow this type reference to refer to a
	        subtype of the type type2 is referring to*/
	if (vala_type_reference_get_type_parameter (type2) != vala_type_reference_get_type_parameter (self)) {
		return FALSE;
	}
	if (vala_type_reference_get_floating_reference (type2) != vala_type_reference_get_floating_reference (self)) {
		return FALSE;
	}
	return TRUE;
}


gboolean vala_type_reference_get_transfers_ownership (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), FALSE);
	return self->priv->_transfers_ownership;
}


void vala_type_reference_set_transfers_ownership (ValaTypeReference* self, gboolean value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_transfers_ownership = value;
}


gboolean vala_type_reference_get_takes_ownership (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), FALSE);
	return self->priv->_takes_ownership;
}


void vala_type_reference_set_takes_ownership (ValaTypeReference* self, gboolean value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_takes_ownership = value;
}


gboolean vala_type_reference_get_is_out (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), FALSE);
	return self->priv->_is_out;
}


void vala_type_reference_set_is_out (ValaTypeReference* self, gboolean value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_is_out = value;
}


gboolean vala_type_reference_get_non_null (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), FALSE);
	return self->priv->_non_null;
}


void vala_type_reference_set_non_null (ValaTypeReference* self, gboolean value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_non_null = value;
}


gboolean vala_type_reference_get_is_null (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), FALSE);
	return self->priv->_is_null;
}


void vala_type_reference_set_is_null (ValaTypeReference* self, gboolean value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_is_null = value;
}


ValaDataType* vala_type_reference_get_data_type (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), NULL);
	return self->priv->_data_type;
}


void vala_type_reference_set_data_type (ValaTypeReference* self, ValaDataType* value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_data_type = value;
}


ValaTypeParameter* vala_type_reference_get_type_parameter (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), NULL);
	return self->priv->_type_parameter;
}


void vala_type_reference_set_type_parameter (ValaTypeReference* self, ValaTypeParameter* value) {
	ValaTypeParameter* _tmp2;
	ValaTypeParameter* _tmp1;
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_type_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_type_parameter == NULL ? NULL : (self->priv->_type_parameter = (g_object_unref (self->priv->_type_parameter), NULL))), _tmp2);
}


gboolean vala_type_reference_get_floating_reference (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), FALSE);
	return self->priv->_floating_reference;
}


void vala_type_reference_set_floating_reference (ValaTypeReference* self, gboolean value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_floating_reference = value;
}


char* vala_type_reference_get_namespace_name (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), NULL);
	return self->priv->_namespace_name;
}


void vala_type_reference_set_namespace_name (ValaTypeReference* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_namespace_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_namespace_name = (g_free (self->priv->_namespace_name), NULL)), _tmp2);
}


char* vala_type_reference_get_type_name (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), NULL);
	return self->priv->_type_name;
}


void vala_type_reference_set_type_name (ValaTypeReference* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_type_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_type_name = (g_free (self->priv->_type_name), NULL)), _tmp2);
}


gint vala_type_reference_get_array_rank (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), 0);
	return self->priv->_array_rank;
}


void vala_type_reference_set_array_rank (ValaTypeReference* self, gint value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_array_rank = value;
}


gint vala_type_reference_get_pointer_level (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), 0);
	return self->priv->_pointer_level;
}


void vala_type_reference_set_pointer_level (ValaTypeReference* self, gint value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_pointer_level = value;
}


gboolean vala_type_reference_get_is_ref (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), FALSE);
	return self->priv->_is_ref;
}


void vala_type_reference_set_is_ref (ValaTypeReference* self, gboolean value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_is_ref = value;
}


gboolean vala_type_reference_get_is_weak (ValaTypeReference* self) {
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (self), FALSE);
	return self->priv->_is_weak;
}


void vala_type_reference_set_is_weak (ValaTypeReference* self, gboolean value) {
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (self));
	self->priv->_is_weak = value;
}


static void vala_type_reference_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaTypeReference * self;
	self = VALA_TYPE_REFERENCE (object);
	switch (property_id) {
		case VALA_TYPE_REFERENCE_TRANSFERS_OWNERSHIP:
		g_value_set_boolean (value, vala_type_reference_get_transfers_ownership (self));
		break;
		case VALA_TYPE_REFERENCE_TAKES_OWNERSHIP:
		g_value_set_boolean (value, vala_type_reference_get_takes_ownership (self));
		break;
		case VALA_TYPE_REFERENCE_IS_OUT:
		g_value_set_boolean (value, vala_type_reference_get_is_out (self));
		break;
		case VALA_TYPE_REFERENCE_NON_NULL:
		g_value_set_boolean (value, vala_type_reference_get_non_null (self));
		break;
		case VALA_TYPE_REFERENCE_IS_NULL:
		g_value_set_boolean (value, vala_type_reference_get_is_null (self));
		break;
		case VALA_TYPE_REFERENCE_DATA_TYPE:
		g_value_set_object (value, vala_type_reference_get_data_type (self));
		break;
		case VALA_TYPE_REFERENCE_TYPE_PARAMETER:
		g_value_set_object (value, vala_type_reference_get_type_parameter (self));
		break;
		case VALA_TYPE_REFERENCE_FLOATING_REFERENCE:
		g_value_set_boolean (value, vala_type_reference_get_floating_reference (self));
		break;
		case VALA_TYPE_REFERENCE_NAMESPACE_NAME:
		g_value_set_string (value, vala_type_reference_get_namespace_name (self));
		break;
		case VALA_TYPE_REFERENCE_TYPE_NAME:
		g_value_set_string (value, vala_type_reference_get_type_name (self));
		break;
		case VALA_TYPE_REFERENCE_ARRAY_RANK:
		g_value_set_int (value, vala_type_reference_get_array_rank (self));
		break;
		case VALA_TYPE_REFERENCE_POINTER_LEVEL:
		g_value_set_int (value, vala_type_reference_get_pointer_level (self));
		break;
		case VALA_TYPE_REFERENCE_IS_REF:
		g_value_set_boolean (value, vala_type_reference_get_is_ref (self));
		break;
		case VALA_TYPE_REFERENCE_IS_WEAK:
		g_value_set_boolean (value, vala_type_reference_get_is_weak (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_type_reference_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaTypeReference * self;
	self = VALA_TYPE_REFERENCE (object);
	switch (property_id) {
		case VALA_TYPE_REFERENCE_TRANSFERS_OWNERSHIP:
		vala_type_reference_set_transfers_ownership (self, g_value_get_boolean (value));
		break;
		case VALA_TYPE_REFERENCE_TAKES_OWNERSHIP:
		vala_type_reference_set_takes_ownership (self, g_value_get_boolean (value));
		break;
		case VALA_TYPE_REFERENCE_IS_OUT:
		vala_type_reference_set_is_out (self, g_value_get_boolean (value));
		break;
		case VALA_TYPE_REFERENCE_NON_NULL:
		vala_type_reference_set_non_null (self, g_value_get_boolean (value));
		break;
		case VALA_TYPE_REFERENCE_IS_NULL:
		vala_type_reference_set_is_null (self, g_value_get_boolean (value));
		break;
		case VALA_TYPE_REFERENCE_DATA_TYPE:
		vala_type_reference_set_data_type (self, g_value_get_object (value));
		break;
		case VALA_TYPE_REFERENCE_TYPE_PARAMETER:
		vala_type_reference_set_type_parameter (self, g_value_get_object (value));
		break;
		case VALA_TYPE_REFERENCE_FLOATING_REFERENCE:
		vala_type_reference_set_floating_reference (self, g_value_get_boolean (value));
		break;
		case VALA_TYPE_REFERENCE_NAMESPACE_NAME:
		vala_type_reference_set_namespace_name (self, g_value_get_string (value));
		break;
		case VALA_TYPE_REFERENCE_TYPE_NAME:
		vala_type_reference_set_type_name (self, g_value_get_string (value));
		break;
		case VALA_TYPE_REFERENCE_ARRAY_RANK:
		vala_type_reference_set_array_rank (self, g_value_get_int (value));
		break;
		case VALA_TYPE_REFERENCE_POINTER_LEVEL:
		vala_type_reference_set_pointer_level (self, g_value_get_int (value));
		break;
		case VALA_TYPE_REFERENCE_IS_REF:
		vala_type_reference_set_is_ref (self, g_value_get_boolean (value));
		break;
		case VALA_TYPE_REFERENCE_IS_WEAK:
		vala_type_reference_set_is_weak (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_type_reference_class_init (ValaTypeReferenceClass * klass) {
	vala_type_reference_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaTypeReferencePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_type_reference_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_type_reference_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_type_reference_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_type_reference_real_accept;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_TRANSFERS_OWNERSHIP, g_param_spec_boolean ("transfers-ownership", "transfers-ownership", "transfers-ownership", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_TAKES_OWNERSHIP, g_param_spec_boolean ("takes-ownership", "takes-ownership", "takes-ownership", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_IS_OUT, g_param_spec_boolean ("is-out", "is-out", "is-out", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_NON_NULL, g_param_spec_boolean ("non-null", "non-null", "non-null", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_IS_NULL, g_param_spec_boolean ("is-null", "is-null", "is-null", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_DATA_TYPE, g_param_spec_object ("data-type", "data-type", "data-type", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_TYPE_PARAMETER, g_param_spec_object ("type-parameter", "type-parameter", "type-parameter", VALA_TYPE_TYPE_PARAMETER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_FLOATING_REFERENCE, g_param_spec_boolean ("floating-reference", "floating-reference", "floating-reference", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_NAMESPACE_NAME, g_param_spec_string ("namespace-name", "namespace-name", "namespace-name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_TYPE_NAME, g_param_spec_string ("type-name", "type-name", "type-name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_ARRAY_RANK, g_param_spec_int ("array-rank", "array-rank", "array-rank", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_POINTER_LEVEL, g_param_spec_int ("pointer-level", "pointer-level", "pointer-level", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_IS_REF, g_param_spec_boolean ("is-ref", "is-ref", "is-ref", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_TYPE_REFERENCE_IS_WEAK, g_param_spec_boolean ("is-weak", "is-weak", "is-weak", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_type_reference_init (ValaTypeReference * self) {
	self->priv = VALA_TYPE_REFERENCE_GET_PRIVATE (self);
	self->priv->type_argument_list = gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal);
}


static void vala_type_reference_dispose (GObject * obj) {
	ValaTypeReference * self;
	self = VALA_TYPE_REFERENCE (obj);
	(self->priv->_type_parameter == NULL ? NULL : (self->priv->_type_parameter = (g_object_unref (self->priv->_type_parameter), NULL)));
	(self->priv->_namespace_name = (g_free (self->priv->_namespace_name), NULL));
	(self->priv->_type_name = (g_free (self->priv->_type_name), NULL));
	(self->priv->type_argument_list == NULL ? NULL : (self->priv->type_argument_list = (g_object_unref (self->priv->type_argument_list), NULL)));
	G_OBJECT_CLASS (vala_type_reference_parent_class)->dispose (obj);
}


GType vala_type_reference_get_type (void) {
	static GType vala_type_reference_type_id = 0;
	if (G_UNLIKELY (vala_type_reference_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaTypeReferenceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_type_reference_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaTypeReference), 0, (GInstanceInitFunc) vala_type_reference_init };
		vala_type_reference_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaTypeReference", &g_define_type_info, 0);
	}
	return vala_type_reference_type_id;
}




