/* valadestructor.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valadestructor.h>
#include <vala/valablock.h>
#include <vala/valaformalparameter.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaDestructorPrivate {
	ValaBlock* _body;
	ValaFormalParameter* _this_parameter;
	MemberBinding _binding;
};

#define VALA_DESTRUCTOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_DESTRUCTOR, ValaDestructorPrivate))
enum  {
	VALA_DESTRUCTOR_DUMMY_PROPERTY
};
static void vala_destructor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_destructor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_destructor_parent_class = NULL;
static void vala_destructor_finalize (ValaCodeNode* obj);



/**
 * Creates a new destructor.
 *
 * @param source_reference reference to source code
 * @return                 newly created destructor
 */
ValaDestructor* vala_destructor_construct (GType object_type, ValaSourceReference* source_reference) {
	ValaDestructor* self;
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = ((ValaDestructor*) (g_type_create_instance (object_type)));
	vala_symbol_init (VALA_SYMBOL (self));
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


ValaDestructor* vala_destructor_new (ValaSourceReference* source_reference) {
	return vala_destructor_construct (VALA_TYPE_DESTRUCTOR, source_reference);
}


static void vala_destructor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDestructor * self;
	self = VALA_DESTRUCTOR (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_destructor (visitor, self);
}


static void vala_destructor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDestructor * self;
	self = VALA_DESTRUCTOR (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	if (self->priv->_body != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (self->priv->_body), visitor);
	}
}


ValaBlock* vala_destructor_get_body (ValaDestructor* self) {
	g_return_val_if_fail (VALA_IS_DESTRUCTOR (self), NULL);
	return self->priv->_body;
}


void vala_destructor_set_body (ValaDestructor* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (VALA_IS_DESTRUCTOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL))), _tmp2);
}


ValaFormalParameter* vala_destructor_get_this_parameter (ValaDestructor* self) {
	g_return_val_if_fail (VALA_IS_DESTRUCTOR (self), NULL);
	return self->priv->_this_parameter;
}


void vala_destructor_set_this_parameter (ValaDestructor* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (VALA_IS_DESTRUCTOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_this_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL))), _tmp2);
}


MemberBinding vala_destructor_get_binding (ValaDestructor* self) {
	g_return_val_if_fail (VALA_IS_DESTRUCTOR (self), 0);
	return self->priv->_binding;
}


void vala_destructor_set_binding (ValaDestructor* self, MemberBinding value) {
	g_return_if_fail (VALA_IS_DESTRUCTOR (self));
	self->priv->_binding = value;
}


static void vala_destructor_class_init (ValaDestructorClass * klass) {
	vala_destructor_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_destructor_finalize;
	g_type_class_add_private (klass, sizeof (ValaDestructorPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_destructor_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_destructor_real_accept_children;
}


static void vala_destructor_instance_init (ValaDestructor * self) {
	self->priv = VALA_DESTRUCTOR_GET_PRIVATE (self);
	self->priv->_binding = MEMBER_BINDING_INSTANCE;
}


static void vala_destructor_finalize (ValaCodeNode* obj) {
	ValaDestructor * self;
	self = VALA_DESTRUCTOR (obj);
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)));
	(self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL)));
	VALA_CODE_NODE_CLASS (vala_destructor_parent_class)->finalize (obj);
}


GType vala_destructor_get_type (void) {
	static GType vala_destructor_type_id = 0;
	if (vala_destructor_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDestructorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_destructor_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDestructor), 0, (GInstanceInitFunc) vala_destructor_instance_init, NULL };
		vala_destructor_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaDestructor", &g_define_type_info, 0);
	}
	return vala_destructor_type_id;
}




