/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * Wayne Conrad, 3 Apr 1999: Color-coded DCC file transfer status windows
 */

#include "style.h"
#include "xchat.h"
#include "util.h"
#include <sys/stat.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include "dcc.h"
#include "draw.h"
#include "gtkutil.h"
#include "plugin.h"
#include <time.h>

/* update the window every 4k */
#define DCC_UPDATE_WINDOW 4096

extern GdkColor colors[];

static char *dcctypes[]= { "SEND", "RECV", "CHAT", "CHAT", "DRAW", "DRAW"};


/* The name and color of each dcc status. */

struct dccstat_info
{
  char * name;  /* Display name */
  int color;    /* Display color (index into colors[] ) */
};

static struct dccstat_info dccstat[] =
{
  {"Queued" , 1 /*black*/ },
  {"Active" , 12 /*cyan*/ },
  {"Failed" , 4 /*red*/ },
  {"Done"   , 3 /*green*/ },
  {"Connect", 1 /*black*/ },
  {"Aborted", 4 /*red*/ },
};


struct dccwindow
{
   GtkWidget *window;
   GtkWidget *list;
};

struct dcc_send
{
   struct session *sess;
   char nick[64];
   GtkWidget *freq;
};

struct dcc_chat
{
   char linebuf[1024];
   int pos;
};

static struct dccwindow dccrwin; /* recv */
static struct dccwindow dccswin; /* send */
static struct dccwindow dcccwin; /* chat */

extern GSList *sess_list;
extern GSList *serv_list;
extern GSList *dcc_list;

extern struct xchatprefs prefs;

static void update_dcc_recv_window(void);
static void update_dcc_send_window(void);
static void update_dcc_chat_window(void);
static void update_dcc_send(struct DCC *dcc);
static void update_dcc_recv(struct DCC *dcc);

extern int tcp_send_len(struct server *serv, char *buf, int len);
extern int tcp_send(struct server *serv, char *buf);
extern char *errorstring(int err);
extern void private_msg(struct server *serv, char *tbuf, char *from, char *ip, char *text);
extern void PrintText(struct session *sess, char *text);
extern int waitline(int sok, char *buf, int bufsize);
extern char *file_part(char *file);
extern void check_homedir(char *file);
extern void path_part(char *file, char *path);
extern void url_checkurl(char *buf);
extern void channel_action(struct session *sess, char *tbuf, char *chan, char *from, char *text, char fromme);
extern struct session *find_session_from_channel(char *chan, struct server *serv);



void dcc_check_timeouts(void)
{
   time_t tim = 0;
   struct DCC *dcc;
   GSList *list = dcc_list;

   while(list)
   {
      if(!tim) tim = time(0);
      dcc = (struct DCC *) list -> data;
      switch(dcc->stat)
      {
       case STAT_ACTIVE:
	 if(dcc->type == TYPE_SEND || dcc->type == TYPE_RECV)
	 {
	    if(prefs.dccstalltimeout > 0)
	    {
	       if(tim - dcc->lasttime > prefs.dccstalltimeout)
	       {
		  /*  char tbuf[1024]; */
		  /*  snprintf(tbuf, sizeof tbuf, */
/*  			   STARTON" DCC %s \00311%s \003 to \00311%s \003 stalled - aborting.\n", */
/*  			   dcctypes[(int)dcc->type], file_part(dcc->file), dcc->nick); */
/*  		  PrintText(dcc->serv->front_session, tbuf); */
		  EMIT_SIGNAL(XP_TE_DCCSTALL, dcc->serv->front_session,
			      dcctypes[(int)dcc->type], file_part(dcc->file),
			      dcc->nick, NULL, 0);
		  dcc_close(dcc, 0, TRUE);
		  dcc_check_timeouts();
		  return;
	       }
	    }
	 }
	 break;
       case STAT_QUEUED:
	 if(dcc->type == TYPE_SEND || dcc->type == TYPE_CHATSEND)
	 {
	    if(tim - dcc->offertime > prefs.dcctimeout)
	    {
	       if(prefs.dcctimeout > 0)
	       {
		 /*   char tbuf[1024]; */
/*  		  snprintf(tbuf, sizeof tbuf, */
/*  			   STARTON" DCC %s \00311%s \003 to \00311%s \003 timed out - aborted.\n", */
/*  			   dcctypes[(int)dcc->type], file_part(dcc->file), dcc->nick); */
/*  		  PrintText(dcc->serv->front_session, tbuf); */
		  EMIT_SIGNAL(XP_TE_DCCTOUT, dcc->serv->front_session,
			      dcctypes[(int)dcc->type], file_part(dcc->file),
			      dcc->nick, NULL, 0);
		  dcc_close(dcc, 0, TRUE);
		  dcc_check_timeouts();
		  return;
	       }
	    }
	 }
	 break;
      }
      list = list -> next;
   }
}

static void dcc_send_filereq_done(struct session *sess, char *nick, char *file)
{
   char tbuf[1024];

   if(file)
   {
      dcc_send(sess, tbuf, nick, file);
      free(file);
   }
   free(nick);
}

void dcc_send_filereq(struct session *sess, char *nick)
{
   char tbuf[128];

   nick = strdup(nick);
   snprintf(tbuf, sizeof tbuf, "Send file to %s", nick);
   gtkutil_file_req(tbuf, dcc_send_filereq_done, sess, nick, FALSE);
}

static int dcc_connect_sok(struct DCC *dcc)
{
   int sok;

   sok = socket(AF_INET, SOCK_STREAM, 0);
   if(sok == -1) return -1;

   dcc->SAddr.sin_port = htons(dcc->port);
   dcc->SAddr.sin_family = AF_INET;
   dcc->SAddr.sin_addr.s_addr = dcc->addr;

   fcntl(sok, F_SETFL, O_NONBLOCK);
   connect(sok, (struct sockaddr *)&dcc->SAddr, sizeof(struct sockaddr_in));

   return sok;
}

static void update_dcc_window(int type)
{
   switch(type)
   {
    case TYPE_SEND: update_dcc_send_window(); break;
    case TYPE_RECV: update_dcc_recv_window(); break;
    case TYPE_CHATRECV:
    case TYPE_CHATSEND: update_dcc_chat_window(); break;
   }
}

void dcc_close(struct DCC *dcc, int stat, int destroy)
{
   char type = dcc->type;
   if(dcc->sok != -1)
   {
      gdk_input_remove(dcc->iotag);
      close(dcc->sok);
      dcc->sok = -1;
   }
   if(dcc->fp != -1 )
   {
      close(dcc->fp);
      dcc->fp = -1;
   }
   dcc->stat = stat;
   if(type == TYPE_DRAWSEND || type == TYPE_DRAWRECV)
   {
      if(dcc->dccdraw) dcc_close_draw_window(dcc);
   }
   if(dcc->dccchat)
   {
      free(dcc->dccchat);
      dcc->dccchat = 0;
   }
   if(destroy)
   {
      dcc_list = g_slist_remove(dcc_list, dcc);
      free(dcc);
      update_dcc_window(type);
      return;
   }
   switch(type)
   {
    case TYPE_SEND:
      update_dcc_send(dcc);
      break;
    case TYPE_RECV:
      update_dcc_recv(dcc);
      break;
    default:
      update_dcc_window(type);
   }
}

void dcc_notify_kill(struct server *serv)
{
   struct server *replaceserv = 0;
   struct DCC *dcc;
   GSList *list = dcc_list;
   if(serv_list) replaceserv = (struct server *)serv_list->data;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(dcc->serv == serv) dcc->serv = replaceserv;
      list = list->next;
   }
}

struct sockaddr_in *dcc_write_chat(char *nick, char *text)
{
   struct DCC *dcc;
   dcc = find_dcc(nick, "", TYPE_CHATRECV);
   if(!dcc) dcc = find_dcc(nick, "", TYPE_CHATSEND);
   if(dcc && dcc->stat == STAT_ACTIVE)
   {
      int len = strlen(text);
      dcc->size += len;
      send(dcc->sok, text, len, 0);
      send(dcc->sok, "\n", 1, 0);
      update_dcc_chat_window();
      return &dcc->SAddr;
   }  
   return 0;
}

static void dcc_read_chat(struct DCC *dcc, gint sok)
{
   int i;
   long len;
   char tbuf[1226];
   char lbuf[1026];

   while(1)
   {
      len = recv(sok, lbuf, sizeof lbuf-2, 0);
      if(len < 1)
      {
	 if(len < 0)
	 {
	    if(errno == EAGAIN || errno == EWOULDBLOCK) return;
	 }
	 /*  snprintf(tbuf, sizeof tbuf, STARTON" DCC CHAT failed. Connection to %s lost.\n", dcc->nick); */
/*  	 PrintText(dcc->serv->front_session, tbuf); */
	 EMIT_SIGNAL(XP_TE_DCCCHATF, dcc->serv->front_session, dcc->nick, NULL,
		     NULL, NULL, 0);
	 dcc_close(dcc, STAT_FAILED, FALSE);
	 return;
      }

      i = 0;
      lbuf[len] = 0;
      while(i < len)
      {
	 switch(lbuf[i])
	 {
	  case '\r':
	    break;
	  case '\n':
	    dcc->dccchat->linebuf[dcc->dccchat->pos] = 0;
	    url_checkurl(dcc->dccchat->linebuf);

	    if(dcc->dccchat->linebuf[0] == 1 && !strncasecmp(dcc->dccchat->linebuf+1, "ACTION", 6))
            {
               char *po = strchr(dcc->dccchat->linebuf+8, '\001');
	       if (po) po[0] = 0;
               channel_action(find_session_from_channel(dcc->nick, dcc->serv), tbuf, dcc->serv->nick, dcc->nick, dcc->dccchat->linebuf+8, FALSE);
            }
            else
	    {
	       private_msg(dcc->serv, tbuf, dcc->nick, "", dcc->dccchat->linebuf);
	    }
	    dcc->pos += dcc->dccchat->pos;
	    dcc->dccchat->pos = 0;
	    update_dcc_chat_window();
	    break;
	  default:
	    dcc->dccchat->linebuf[dcc->dccchat->pos] = lbuf[i];
	    dcc->dccchat->pos++;
	    if(dcc->dccchat->pos == 1023) dcc->dccchat->pos = 1022;
	 }
	 i++;
      }
   }
}

static void dcc_read(struct DCC *dcc, gint sok)
{
   long n;
   char buf[4098];
   struct stat st;
   
   if(dcc->fp == -1)
   {
      if(dcc->resumable)
      {
#ifdef __EMX__ /* OS/2 needs O_BINARY */
	 dcc->fp = open(dcc->destfile, O_BINARY | O_WRONLY | O_APPEND);
#else
	 dcc->fp = open(dcc->destfile, O_WRONLY | O_APPEND);
#endif
	 dcc->pos = dcc->resumable;
	 dcc->ack = dcc->resumable;
      } else {
	 if (stat(dcc->destfile, &st) != -1) {
	    time_t current_time;
	    strcpy(buf, prefs.dccdir);
	    if(prefs.dccdir[strlen(prefs.dccdir) - 1] != '/')
	       strcat(buf, "/");
	    if (prefs.dccwithnick) {
	       strcat(buf, dcc->serv->nick);
	       strcat(buf, ".");
	    }
	    strcat(buf, dcc->file);
	    current_time = time(NULL);
	    strcat(buf, "-");
	    strcat(buf, ctime(&current_time));
	    buf[strlen(buf) - 1] = 0;
	    rename(dcc->destfile, buf);
	    EMIT_SIGNAL(XP_TE_DCCRENAME, dcc->serv->front_session,
			dcc->destfile, buf, NULL, NULL, 0);
	 }
	    
#ifdef __EMX__
	dcc->fp = open(dcc->destfile, O_BINARY | O_TRUNC | O_WRONLY | O_CREAT, prefs.dccpermissions);
#else
        dcc->fp = open(dcc->destfile, O_TRUNC | O_WRONLY | O_CREAT, prefs.dccpermissions);
#endif
      }
   }

   if(dcc->fp == -1)
   {
      /*  snprintf(buf, sizeof buf, STARTON" DCC RECV: Cannot open %s for writing - aborting.\n", dcc->destfile); */
/*        PrintText(dcc->serv->front_session, buf); */
      EMIT_SIGNAL(XP_TE_DCCFILEERR, dcc->serv->front_session, dcc->destfile,
		  NULL, NULL, NULL, 0);
      dcc_close(dcc, STAT_FAILED, FALSE);
      return;
   }

   while(1)
   {
      n = recv(dcc->sok, buf, sizeof buf-2, 0);
      if(n < 1)
      {
	 if(n < 0)
	 {
	    if(errno == EAGAIN || errno == EWOULDBLOCK) return;
	 }
	 /*  snprintf(buf, sizeof buf, STARTON" DCC RECV %s (%s) failed. Connection to %s lost.\n", dcc->file, dcc->destfile, dcc->nick); */
/*  	 PrintText(dcc->serv->front_session, buf); */
	 EMIT_SIGNAL(XP_TE_DCCRECVERR, dcc->serv->front_session, dcc->file,
		     dcc->destfile, dcc->nick, NULL, 0);
	 dcc_close(dcc, STAT_FAILED, FALSE);
	 return;
      }
      if(n > 0)
      {
	 guint32 pos;
	 time_t sec;

	 write(dcc->fp, buf, n);
	 dcc->pos += n;
	 pos = htonl(dcc->pos);
	 send(dcc->sok, (char *)&pos, 4, 0);

	 dcc->lasttime = time(0);
	 sec = dcc->lasttime - dcc->starttime;
	 if(sec < 1) sec = 1;
	 dcc->cps = dcc->pos / sec;

	 if(dcc->pos >= dcc->size)
         {
	    dcc_close(dcc, STAT_DONE, FALSE);
	    snprintf(buf, sizeof(buf), "%ld", dcc->cps);
	   /*   snprintf(buf, sizeof buf, STARTON" DCC RECV %s (%s) from %s complete (%ld cps).\n", dcc->file, dcc->destfile, dcc->nick, dcc->cps); */
/*  	    PrintText(dcc->serv->front_session, buf); */
	    EMIT_SIGNAL(XP_TE_DCCRECVCOMP, dcc->serv->front_session, dcc->file,
			dcc->destfile, dcc->nick, buf, 0);
	    return;
	 } else {
	    if(dcc->pos >= (dcc->oldpos + DCC_UPDATE_WINDOW))
	    {
	       update_dcc_recv(dcc);
	       dcc->oldpos = dcc->pos;
	    }
	 }
      }
   }
}

static void dcc_connect_finished(struct DCC *dcc, gint sok)
{
   int er;
   /*  char outbuf[300]; */

   gdk_input_remove(dcc->iotag);

   if(connect(sok, (struct sockaddr *)&dcc->SAddr, sizeof(struct sockaddr_in)) < 0)
   {
      er = errno;
      if(er != EISCONN)
      {
	 /*  snprintf(outbuf, sizeof outbuf, STARTON" DCC %s connect attempt to %s failed (err=%s).\n", */
/*  	      dcctypes[(int)dcc->type], dcc->nick, errorstring(er)); */
/*  	 PrintText(dcc->serv->front_session, outbuf); */
	 EMIT_SIGNAL(XP_TE_DCCCONFAIL, dcc->serv->front_session,
		     dcctypes[(int)dcc->type], dcc->nick, errorstring(er),
		     NULL, 0);
	 dcc->stat = STAT_FAILED;
	 update_dcc_window(dcc->type);
	 return;
      }
   }
   fcntl(sok, F_SETFL, O_NONBLOCK);
   dcc->stat = STAT_ACTIVE;
   switch(dcc->type)
   {
    case TYPE_RECV: 
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ | GDK_INPUT_EXCEPTION,
				 (GdkInputFunction)dcc_read, dcc);
      break;

    case TYPE_CHATRECV:
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ | GDK_INPUT_EXCEPTION,
				 (GdkInputFunction)dcc_read_chat, dcc);
      dcc->dccchat = malloc(sizeof(struct dcc_chat));
      dcc->dccchat->pos = 0;
      break;

    case TYPE_DRAWRECV: 
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ | GDK_INPUT_EXCEPTION,
				 (GdkInputFunction)dcc_read_draw, dcc);
      dcc_open_draw_window(dcc);
      break;
   }
   update_dcc_window(dcc->type);
   dcc->starttime = time(0);
   dcc->lasttime = dcc->starttime;

   /*  snprintf(outbuf, sizeof outbuf, STARTON" DCC %s connection established to %s (%s)\n", */
/*  	   dcctypes[(int)dcc->type], dcc->nick, inet_ntoa(dcc->SAddr.sin_addr)); */
/*     PrintText(dcc->serv->front_session, outbuf); */
   EMIT_SIGNAL(XP_TE_DCCCON, dcc->serv->front_session, dcctypes[(int)dcc->type]
	       , dcc->nick, inet_ntoa(dcc->SAddr.sin_addr), "to", 0);
}

void dcc_connect(struct session *sess, struct DCC *dcc)
{
   if(dcc->stat == STAT_CONNECTING) return;
   dcc->stat = STAT_CONNECTING;
   dcc->sok = dcc_connect_sok(dcc);
   if(dcc->sok == -1)
   {
      dcc->stat = STAT_FAILED;
      /*if(sess) PrintText(sess, STARTON" Connection failed.\n");*/
      update_dcc_window(dcc->type);
      return;
   }
   dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_WRITE | GDK_INPUT_EXCEPTION,
			      (GdkInputFunction)dcc_connect_finished, dcc);
   update_dcc_recv(dcc);
}

static void update_dcc_recv(struct DCC *dcc)
{
   char pos[14], cps[14];
   gint row;
#ifdef USE_GNOME
   char file[512];
#endif

   if(!dccrwin.window) return;

   row = gtk_clist_find_row_from_data(GTK_CLIST(dccrwin.list), (gpointer)dcc);

   sprintf(pos, "%lu", dcc->pos);
   sprintf(cps, "%lu", dcc->cps);

   gtk_clist_freeze(GTK_CLIST(dccrwin.list));
   gtk_clist_set_text
     (GTK_CLIST(dccrwin.list), row, 0, dccstat[(int)dcc->stat].name);
   gtk_clist_set_text(GTK_CLIST(dccrwin.list), row, 3, pos);
   gtk_clist_set_text(GTK_CLIST(dccrwin.list), row, 4, cps);
   gtk_clist_set_foreground
     (GTK_CLIST(dccrwin.list), row, colors+dccstat[(int)dcc->stat].color);
#ifdef USE_GNOME
   if (dcc->stat == STAT_DONE) {
      snprintf(file, sizeof(file), "%s/%s", prefs.dccdir, dcc->file);
      gtk_clist_set_text(GTK_CLIST(dccrwin.list), row, 6, (char *) gnome_mime_type_of_file (file));
   }
#endif 
   gtk_clist_thaw(GTK_CLIST(dccrwin.list));
}

static void update_dcc_send(struct DCC *dcc)
{
   char pos[14], cps[14], ack[14];
   gint row;

   if(!dccswin.window) return;

   row = gtk_clist_find_row_from_data(GTK_CLIST(dccswin.list), (gpointer)dcc);

   sprintf(pos, "%lu", dcc->pos);
   sprintf(cps, "%lu", dcc->cps);
   sprintf(ack, "%lu", dcc->ack);

   gtk_clist_freeze(GTK_CLIST(dccswin.list));
   gtk_clist_set_text
     (GTK_CLIST(dccswin.list), row, 0, dccstat[(int)dcc->stat].name);
   gtk_clist_set_foreground
     (GTK_CLIST(dccswin.list), row, colors+dccstat[(int)dcc->stat].color);
   gtk_clist_set_text(GTK_CLIST(dccswin.list), row, 3, pos);
   gtk_clist_set_text(GTK_CLIST(dccswin.list), row, 4, ack);
   gtk_clist_set_text(GTK_CLIST(dccswin.list), row, 5, cps);
   gtk_clist_thaw(GTK_CLIST(dccswin.list));
}

static int dcc_read_ack(struct DCC *dcc, char *buf)
{
   int len;
   guint32 ack;

   len = recv(dcc->sok, (char *)&ack, 4, MSG_PEEK);
   if(len < 1)
   {
      if(len < 0)
      {
	 if(errno == EWOULDBLOCK) return 1;
      }
      /*  snprintf(buf, 2048, STARTON" DCC SEND %s failed. Connection to %s lost.\n", file_part(dcc->file), dcc->nick); */
/*        PrintText(dcc->serv->front_session, buf); */
      EMIT_SIGNAL(XP_TE_DCCSENDFAIL, dcc->serv->front_session,
		  file_part(dcc->file), dcc->nick, NULL, NULL, 0);
      dcc_close(dcc, STAT_FAILED, FALSE);
      return 1;
   }
   if(len < 4) return 1;
   recv(dcc->sok, (char *)&ack, 4, 0);
   dcc->ack = ntohl(ack);
   return 0;
}

static void dcc_send_data(struct DCC *dcc, gint sok)
{
   char buf[2048];
   long len, sec, sent;

   while(1)
   {
      if(sok != -1)
      {
	 if(dcc_read_ack(dcc, buf)) return;
      } else {
	 sok = dcc->sok;
      }

      if(dcc->pos < dcc->size)
      {

	 if(!prefs.fastdccsend)
	 {
	    if((dcc->ack + 4096) < dcc->pos) return;
	 }

	 lseek(dcc->fp, dcc->pos, SEEK_SET);
	 len = read(dcc->fp, buf, sizeof buf);

	 sent = send(dcc->sok, buf, len, 0);

	 if(sent < 0 && errno != EWOULDBLOCK)
	 {
	   /*   snprintf(buf, sizeof buf, STARTON" DCC SEND %s failed. Connection to %s lost.\n", file_part(dcc->file), dcc->nick); */
/*  	    PrintText(dcc->serv->front_session, buf); */
	    EMIT_SIGNAL(XP_TE_DCCSENDFAIL, dcc->serv->front_session,
			file_part(dcc->file), dcc->nick, NULL, NULL, 0);
	    dcc_close(dcc, STAT_FAILED, FALSE);
	    return;
	 }

	 dcc->lasttime = time(0);
	 sec = dcc->lasttime - dcc->starttime;
	 if(sec < 1) sec = 1;
	 dcc->cps = dcc->pos / sec;
	 if(sent > 0) dcc->pos += sent;

      }
      
      if(dcc->pos >= dcc->size  &&  dcc->ack >= dcc->size)
      {
	 dcc_close(dcc, STAT_DONE, FALSE);
	 snprintf(buf, sizeof(buf), "%ld", dcc->cps);
	/*   snprintf(buf, sizeof buf, STARTON" DCC SEND %s to %s complete (%ld cps).\n", file_part(dcc->file), dcc->nick, dcc->cps); */
/*  	 PrintText(dcc->serv->front_session, buf); */
	 EMIT_SIGNAL(XP_TE_DCCSENDCOMP, dcc->serv->front_session,
		     file_part(dcc->file), dcc->nick, buf, NULL, 0);
	 return;
      } else {
	 if(dcc->pos >= (dcc->oldpos + DCC_UPDATE_WINDOW) ||
	    dcc->ack >= (dcc->oldack + DCC_UPDATE_WINDOW))
	 {
	    dcc->oldack = dcc->ack;
	    dcc->oldpos = dcc->pos;
	    update_dcc_send(dcc);
	 }
      }
   }
}

static void dcc_accept(struct DCC *dcc, gint sokk)
{
   struct sockaddr_in CAddr;
   int len, sok;
   /*  char tbuf[256]; */
   
   len = sizeof(CAddr);
   sok = accept(dcc->sok, (struct sockaddr *)&CAddr, &len);
   gtk_input_remove(dcc->iotag);
   close(dcc->sok);
   if(sok < 0)
   {
      dcc_close(dcc, STAT_FAILED, FALSE);
      return;
   }
   fcntl(sok, F_SETFL, O_NONBLOCK);
   dcc->sok = sok;
   dcc->addr = ntohl(CAddr.sin_addr.s_addr);
   memcpy((char *)&dcc->SAddr.sin_addr, (char *)&CAddr.sin_addr, sizeof(struct in_addr));
   dcc->stat = STAT_ACTIVE;
   dcc->starttime = time(0);
   dcc->lasttime = dcc->starttime;
   switch(dcc->type)
   {
    case TYPE_SEND:
      dcc->iotag = gdk_input_add(sok, GDK_INPUT_READ | GDK_INPUT_EXCEPTION,
				 (GdkInputFunction)dcc_send_data, dcc);
      dcc_send_data(dcc, -1);
      break;

    case TYPE_CHATSEND:
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ | GDK_INPUT_EXCEPTION,
				 (GdkInputFunction)dcc_read_chat, dcc);
      dcc->dccchat = malloc(sizeof(struct dcc_chat));
      dcc->dccchat->pos = 0;
      break;

    case TYPE_DRAWSEND:
      dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ | GDK_INPUT_EXCEPTION,
				 (GdkInputFunction)dcc_read_draw, dcc);
      dcc_open_draw_window(dcc);
      break;
   }

   update_dcc_window(dcc->type);

   /*  snprintf(tbuf, sizeof tbuf, STARTON" DCC %s connection established from %s (%s)\n", */
/*  	    dcctypes[(int)dcc->type], dcc->nick, inet_ntoa(CAddr.sin_addr)); */
/*     PrintText(dcc->serv->front_session, tbuf); */
   EMIT_SIGNAL(XP_TE_DCCCON, dcc->serv->front_session,dcctypes[(int)dcc->type],
	       dcc->nick, inet_ntoa(CAddr.sin_addr), "from", 0);
}

int dcc_listen_init(struct DCC *dcc)
{
   int len;
   struct sockaddr_in SAddr;

   dcc->sok = socket(AF_INET, SOCK_STREAM, 0);
   if(dcc->sok == -1) return 0;

   memset(&SAddr, 0, sizeof(struct sockaddr_in));

   len = sizeof(SAddr);
   getsockname(dcc->serv->sok, (struct sockaddr *)&SAddr, &len); 

   SAddr.sin_family = AF_INET;
   SAddr.sin_port = 0;

   bind(dcc->sok, (struct sockaddr *)&SAddr, sizeof(SAddr));

   len = sizeof(SAddr);
   getsockname(dcc->sok, (struct sockaddr *) &SAddr, &len);

   memcpy((char *)&dcc->SAddr, (char *)&SAddr, sizeof(struct sockaddr_in));
   dcc->port = ntohs(SAddr.sin_port);
   if(prefs.my_ip != 0)
     dcc->addr = ntohl(prefs.my_ip);
   else
     dcc->addr = ntohl(SAddr.sin_addr.s_addr);

   fcntl(dcc->sok, F_SETFL, O_NONBLOCK);
   listen(dcc->sok, 1);
   fcntl(dcc->sok, F_SETFL, 0);

   dcc->iotag = gdk_input_add(dcc->sok, GDK_INPUT_READ | GDK_INPUT_EXCEPTION,
			      (GdkInputFunction)dcc_accept, dcc);

   return TRUE;
}

void dcc_send(struct session *sess, char *tbuf, char *to, char *file)
{
   struct stat st;
   struct DCC *dcc = new_dcc();
   if(!dcc) return;

   strcpy(dcc->file, file);
   check_homedir(dcc->file);

   if(stat(dcc->file, &st) != -1)
   {
      if(*file_part(dcc->file))
      {
	 if(st.st_size > 0)
	 {
	    dcc->offertime = time(0);
	    dcc->serv = sess->server;
	    dcc->stat = STAT_QUEUED;
	    dcc->size = st.st_size;
	    dcc->type = TYPE_SEND;
#ifdef __EMX__
	    dcc->fp = open(dcc->file, O_BINARY | O_RDONLY);
#else
	    dcc->fp = open(dcc->file, O_RDONLY);
#endif
	    if(dcc->fp != -1)
	    {
	       if(dcc_listen_init(dcc))
	       {
		  file = dcc->file;
		  while(*file)
		  {
		     if(*file == ' ') *file = '_';
		     file++;
		  }
		  strcpy(dcc->nick, to);
		  if(!prefs.noautoopendccsendwindow)
		    open_dcc_send_window();
		  else
		    update_dcc_send_window();
		  snprintf(tbuf, 255, "PRIVMSG %s :\001DCC SEND %s %lu %ld %ld\001\r\n",
			  to, file_part(dcc->file), dcc->addr, dcc->port, dcc->size);
		  tcp_send(sess->server, tbuf);
		  /*  snprintf(tbuf, 255, STARTON" Offering \00311%s \003 to \00311%s\003 \n", file_part(dcc->file), to); */
/*  		  PrintText(sess, tbuf); */
		  EMIT_SIGNAL(XP_TE_DCCOFFER, sess, file_part(dcc->file), to,
			      NULL, NULL, 0);
	       } else {
		  dcc_close(dcc, 0, TRUE);
	       }
	       return;
	    }
	 }
      }
   }
   snprintf(tbuf, 255, "Cannot access %s\n", dcc->file);
   PrintText(sess, tbuf);
   dcc_close(dcc, 0, TRUE);
}

static struct DCC *find_dcc_from_port(int port, int type)
{
   struct DCC *dcc;
   GSList *list = dcc_list;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(dcc->port == port &&
	 dcc->stat == STAT_QUEUED &&
	 dcc->type == type)
	return dcc;
      list = list->next;
   }
   return 0;  
}

struct DCC *find_dcc(char *nick, char *file, int type)
{
   GSList *list = dcc_list;
   struct DCC *dcc;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(!strcasecmp(nick, dcc->nick))
      {
	 if(type == -1 || dcc->type == type)
	 {
	    if(!file[0]) return dcc;
	    if(!strcasecmp(file, file_part(dcc->file))) return dcc;
	    if(!strcasecmp(file, dcc->file)) return dcc;
	 }
      }
      list = list->next;
   }
   return 0;
}

static void dcc_abort(struct DCC *dcc)
{
   /*  char tbuf[256]; */
   if(dcc)
   {
      switch(dcc->stat)
      {
       case STAT_QUEUED:
       case STAT_CONNECTING:
       case STAT_ACTIVE:
	  dcc_close(dcc, STAT_ABORTED, FALSE);
	  /*  snprintf(tbuf, sizeof tbuf, STARTON" DCC %s \00311%s \003 to \00311%s \003 aborted.\n", */
/*  		  dcctypes[(int)dcc->type], file_part(dcc->file), dcc->nick); */
/*  	  PrintText(dcc->serv->front_session, tbuf); */
	  EMIT_SIGNAL(XP_TE_DCCABORT, dcc->serv->front_session,
		      dcctypes[(int)dcc->type], file_part(dcc->file),
		      dcc->nick, NULL, 0);
       	  break;
       default:
	  dcc_close(dcc, 0, TRUE);
      }
   }
}

static void dcc_resume(struct DCC *dcc)
{
   if(dcc)
   {
     if(dcc->stat == STAT_QUEUED && dcc->resumable)
     {
	char tbuf[256];
	snprintf(tbuf, sizeof tbuf, "PRIVMSG %s :\001DCC RESUME %s %ld %lu\001\r\n",
	      dcc->nick, dcc->file, dcc->port, dcc->resumable);
	tcp_send(dcc->serv, tbuf);
     }
   }
}

static void dcc_get(struct DCC *dcc)
{
   if(dcc)
   {
      switch(dcc->stat)
      {
	 case STAT_QUEUED:
	   dcc->resumable = 0;
	   dcc->pos = 0;
	   dcc_connect(0, dcc);
	   break;
	 case STAT_DONE:
	 case STAT_FAILED:
	 case STAT_ABORTED:
	   dcc_close(dcc, 0, TRUE);
	   break;
      }
   }
}

void dcc_get_nick(struct session *sess, char *nick)
{
   struct DCC *dcc;
   GSList *list = dcc_list;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(!strcasecmp(nick, dcc->nick))
      {
	   if(dcc->stat == STAT_QUEUED && dcc->type == TYPE_RECV)
	   {
	      dcc->resumable = 0;
	      dcc->pos = 0;
	      dcc_connect(sess, dcc);
	      return;
	   }
      }
      list = list->next;
   }
   if(sess) /*  PrintText(sess, STARTON" No such DCC offer.\n"); */
      EMIT_SIGNAL(XP_TE_DCCIVAL, sess, NULL, NULL, NULL, NULL, 0);
}

struct DCC *new_dcc(void)
{
   struct DCC *dcc = malloc(sizeof(struct DCC));
   if(!dcc) return 0;
   memset(dcc, 0, sizeof(struct DCC));
   dcc->sok = -1;
   dcc->fp = -1;
   dcc_list = g_slist_append(dcc_list, dcc);
   return(dcc);
}

void dcc_chat(struct session *sess, char *nick)
{
   char outbuf[256];
   struct DCC *dcc;

   dcc = find_dcc(nick, "", TYPE_CHATSEND);
   if(dcc)
   {
      switch(dcc->stat)
      {
       case STAT_ACTIVE:
       case STAT_QUEUED:
       case STAT_CONNECTING:
	 /*  snprintf(outbuf, sizeof outbuf, STARTON" Already offering CHAT to %s\n", nick); */
/*  	 PrintText(sess, outbuf); */
	 EMIT_SIGNAL(XP_TE_DCCCHATREOFFER, sess, nick, NULL, NULL, NULL, 0);
	 return;
       case STAT_ABORTED:
       case STAT_FAILED:
	 dcc_close(dcc, 0, TRUE);
      }
   }

   dcc = find_dcc(nick, "", TYPE_CHATRECV);
   if(dcc)
   {
      switch(dcc->stat)
      {
	 case STAT_QUEUED:
	   dcc_connect(0, dcc);
	   break;
	 case STAT_FAILED:
	 case STAT_ABORTED:
	   dcc_close(dcc, 0, TRUE);
      }
      return;
   }

   /* offer DCC CHAT */

   dcc = new_dcc();
   if(!dcc) return;
   dcc->offertime = time(0);
   dcc->serv = sess->server;
   dcc->stat = STAT_QUEUED;
   dcc->type = TYPE_CHATSEND;
   time(&dcc->starttime);
   strcpy(dcc->nick, nick);
   if(dcc_listen_init(dcc))
   {
      snprintf(outbuf, sizeof outbuf, "PRIVMSG %s :\001DCC CHAT chat %lu %ld\001\r\n",
		 nick, dcc->addr, dcc->port);
      tcp_send(dcc->serv, outbuf);
      /*  sprintf(outbuf, STARTON" Offering DCC CHAT to %s\n", nick); */
/*        PrintText(sess, outbuf); */
      EMIT_SIGNAL(XP_TE_DCCCHATOFFERING, sess, nick, NULL, NULL, NULL, 0);
   } else
      dcc_close(dcc, 0, TRUE);
}

static void dcc_malformed(struct session *sess, char *nick, char *data)
{
   char tbuf[1024];
   snprintf(tbuf, sizeof tbuf,
	    STARTON" Received a malformed DCC request from %s\n"
	    STARTON" Exact contents: \"%s\"\n",
	    nick, data);
   PrintText(sess, tbuf);
}

void handle_dcc(struct session *sess, char *outbuf, char *nick, char *word[], char *word_eol[])
{
   struct DCC *dcc;
   char *type = word[5];
   int port;

   if(!strcasecmp(type, "DRAW"))
   {
      unsigned long addr;
      long port = atol(word[9]);

      sscanf(word[8], "%lu", &addr);
      addr = ntohl(addr);
      if(!addr || port < 1024)
      {
	 dcc_malformed(sess, nick, word_eol[4]+2);
	 return;
      }

      dcc = new_dcc();
      if(dcc)
      {
	 dcc->pos = atoi(word[6]); /* x */
	 dcc->size = atoi(word[7]); /* y */
	 dcc->serv = sess->server;
	 dcc->type = TYPE_DRAWRECV;
	 dcc->stat = STAT_QUEUED;
	 dcc->addr = addr;
	 dcc->port = port;
	 strcpy(dcc->nick, nick);
	 /*  snprintf(outbuf, 255, */
/*  		 STARTON" Received a DCC DRAW offer from %s\n" */
/*  		 STARTON" Type \00311/DCC DRAW %s \003 to accept\n", */
/*  		 nick, nick); */
/*  	 PrintText(sess->server->front_session, outbuf); */
	 EMIT_SIGNAL(XP_TE_DCCDRAWOFFER, sess->server->front_session, nick,
		     NULL, NULL, NULL, 0);
	 return;
      }  
   }

   if(!strcasecmp(type, "CHAT"))
   {
      unsigned long addr;
      long port = atol(word[8]);

      sscanf(word[7], "%lu", &addr);
      addr = ntohl(addr);

      if(!addr || port < 1024)
      {
	 dcc_malformed(sess, nick, word_eol[4]+2);
	 return;
      }

      dcc = find_dcc(nick, "", TYPE_CHATSEND);
      if(dcc) dcc_close(dcc, 0, TRUE);

      dcc = new_dcc();
      if(dcc)
      {
	 dcc->serv = sess->server;
	 dcc->type = TYPE_CHATRECV;
	 dcc->stat = STAT_QUEUED;
	 dcc->addr = addr;
	 dcc->port = port;
	 strcpy(dcc->nick, nick);
	 /*  snprintf(outbuf, 255, STARTON" Received a DCC CHAT offer from %s\n", nick); */
/*  	 PrintText(sess->server->front_session, outbuf); */
	 EMIT_SIGNAL(XP_TE_DCCCHATOFFER, sess->server->front_session, nick,
		     NULL, NULL, NULL, 0);
	 if(prefs.autodccchat) dcc_connect(0, dcc);
	 return;
      }
   }

   if(!strcasecmp(type, "RESUME"))
   {
      port = atol(word[7]);
      dcc = find_dcc_from_port(port, TYPE_SEND);
      if(dcc && dcc->stat == STAT_QUEUED)
      {
	 sscanf(word[8], "%lu", &dcc->resumable);
	 dcc->pos = dcc->resumable;
	 dcc->ack = dcc->resumable;
	 lseek(dcc->fp, dcc->pos, SEEK_SET);
	 snprintf(outbuf, 255, "PRIVMSG %s :\001DCC ACCEPT file.ext %ld %lu\001\r\n",
		 dcc->nick, dcc->port, dcc->resumable);
	 tcp_send(dcc->serv, outbuf);
	 sprintf(outbuf, "%lu", dcc->pos);
	 /*  sprintf(outbuf, */
/*  		 STARTON" \00311%s \003 has requested to resume \00311%s \003 from \00311%lu\003 .\n", */
/*  		 nick, file_part(dcc->file), dcc->pos); */
/*  	 PrintText(sess, outbuf); */
	 EMIT_SIGNAL(XP_TE_DCCRESUMEREQUEST, sess, nick, file_part(dcc->file),
		     outbuf, NULL, 0);
	 return;
      }
   }

   if(!strcasecmp(type, "ACCEPT"))
   {
      port = atol(word[7]);
      dcc = find_dcc_from_port(port, TYPE_RECV);
      if(dcc && dcc->stat == STAT_QUEUED)
      {
	 dcc_connect(0, dcc);
	 return;
      }
   }

   if(!strcasecmp(type, "SEND"))
   {
      unsigned long addr;
      char *file= file_part(word[6]);
      long port = atol(word[8]);
      long size = atol(word[9]);

      sscanf(word[7], "%lu", &addr);
      addr = ntohl(addr);

      if(!addr || !size || port < 1024)
      {
	 dcc_malformed(sess, nick, word_eol[4]+2);
	 return;
      }

      dcc = new_dcc();
      if(dcc)
      {
	 struct stat st;

         strcpy(dcc->destfile, prefs.dccdir);	 
	 if(prefs.dccdir[strlen(prefs.dccdir)-1] != '/')
	    strcat(dcc->destfile, "/");
	 if (prefs.dccwithnick) {
	    strcat(dcc->destfile, nick);
	    strcat(dcc->destfile, ".");
	 }
	 strcat(dcc->destfile, file);
	 strcpy(dcc->file, file);

	 if(stat(dcc->destfile, &st) != -1)
	 {
	   /*if(st.st_size < 1)
	   {
	      snprintf(outbuf, 255, STARTON" Cannot accept %s from %s, filesize is zero.\n", file, nick);
	      PrintText(sess->server->front_session, outbuf);
	      dcc_close(dcc, 0, TRUE);
	      return;
	   }*/
	   dcc->resumable = st.st_size;
	 } else
	   dcc->resumable = 0;
	 if(st.st_size == size) dcc->resumable = 0;
	 dcc->serv = sess->server;
	 dcc->type = TYPE_RECV;
	 dcc->stat = STAT_QUEUED;
	 dcc->addr = addr;
	 dcc->SAddr.sin_addr.s_addr = addr;
	 dcc->port = port;
	 dcc->size = size;
	 strcpy(dcc->nick, nick);
	 if(!prefs.noautoopendccrecvwindow)
	   open_dcc_recv_window();
	 else
	   update_dcc_recv_window();
	 if(prefs.autodccsend) dcc_connect(sess, dcc);
      }
      /*  snprintf(outbuf, 255, STARTON" \00311%s \003 has offered \00311%s \003 (\00311%ld \003 bytes)\n", */
/*  	      nick, file, size); */
      sprintf(outbuf, "%ld", size);
      EMIT_SIGNAL(XP_TE_DCCSENDOFFER, sess->server->front_session, nick, file,
		  outbuf, NULL, 0);
   } else
     /*  sprintf(outbuf, STARTON" Received '%s' from %s\n",  */
/*  	    word_eol[4]+2, nick); */
/*     PrintText(sess->server->front_session, outbuf); */
     EMIT_SIGNAL(XP_TE_DCCGENERICOFFER, sess->server->front_session,
		  word_eol[4]+2, nick, NULL, NULL, 0);
}

void dcc_show_list(struct session *sess, char *outbuf)
{
   int i = 0;
   struct DCC *dcc;
   GSList *list = dcc_list;

   PrintText(sess,
	     "\00308,02 Type  To/From    Status  Size    Pos     File   \003 \n"
	     "\002\00303-------------------------------------------------\002\003 \n"
	     );
   while(list)
   {
      dcc = (struct DCC *)list->data;
      i++;
      snprintf(outbuf, 255, " %-5.5s %-10.10s %-7.7s %-7ld %-7ld %s\n",
	      dcctypes[(int)dcc->type], dcc->nick, 
              dccstat[(int)dcc->stat].name, dcc->size, dcc->pos, dcc->file);
      PrintText(sess, outbuf);
      list = list->next;
   }
   if(!i) PrintText(sess, "No active DCCs\n");
}

static int close_dcc_recv_window(void)
{
   dccrwin.window = 0;
   return 0;
}

static void update_dcc_recv_window(void)
{
   struct DCC *dcc;
   GSList *list = dcc_list;
   gchar *new[1][7];
   char size[16];
   char pos[16];
   char cps[16];
#ifdef USE_GNOME
   char	file[512];
#endif
   gint row;
   int selrow;

   if(!dccrwin.window) return;

   selrow = gtkutil_clist_selection(dccrwin.list);

   gtk_clist_clear((GtkCList*)dccrwin.list);
   new[0][2] = size;
   new[0][3] = pos;
   new[0][4] = cps;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(dcc->type == TYPE_RECV)
      {
	 new[0][0] = dccstat[(int)dcc->stat].name;
	 new[0][1] = dcc->file;
	 new[0][5] = dcc->nick;
#ifdef USE_GNOME
	 if (dcc->stat == STAT_DONE) {
	    snprintf(file, sizeof(file), "%s/%s", prefs.dccdir, dcc->file);
	    new[0][6] = (char *) gnome_mime_type_of_file (file);
	 }
	 else
	    new[0][6] = "";
#endif 
	 sprintf(size, "%lu", dcc->size);
	 if(dcc->stat == STAT_QUEUED)
	   sprintf(pos, "%lu", dcc->resumable);
	 else
	   sprintf(pos, "%lu", dcc->pos);
	 sprintf(cps, "%lu", dcc->cps);
	 row = gtk_clist_append(GTK_CLIST(dccrwin.list), new[0]);
	 gtk_clist_set_row_data(GTK_CLIST(dccrwin.list), row, (gpointer)dcc);
	 gtk_clist_set_foreground
	   (
	   GTK_CLIST(dccrwin.list), row, 
	   colors+dccstat[(int)dcc->stat].color
	   );
      }
      list = list->next;
   }
   if(selrow != -1)
     gtk_clist_select_row((GtkCList*)dccrwin.list, selrow, 0);
}

static void dcc_info(struct DCC *dcc)
{
   char tbuf[256];
   snprintf(tbuf, 255, "      File: %s\n"
	         "   To/From: %s\n"
	         "      Size: %ld\n"
	         "      Port: %ld\n"
	         " IP Number: %s\n"
	         "Start Time: %s",
	   dcc->file,
	   dcc->nick,
	   dcc->size,
	   dcc->port,
	   inet_ntoa(dcc->SAddr.sin_addr),
	   ctime(&dcc->starttime));
   gtkutil_simpledialog(tbuf);
}

static void resume_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dccrwin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccrwin.list), row);
      gtk_clist_unselect_row (GTK_CLIST(dccrwin.list), row, 0); 
      dcc_resume(dcc);
   }
}

static void abort_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dccrwin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccrwin.list), row);
      /*gtk_clist_unselect_row (GTK_CLIST(dccrwin.list), row, 0);*/
      dcc_abort(dcc);
   }
}

static void accept_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dccrwin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccrwin.list), row);
      gtk_clist_unselect_row (GTK_CLIST(dccrwin.list), row, 0);
      dcc_get(dcc);
   }
}

static void info_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dccrwin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccrwin.list), row);
      /*gtk_clist_unselect_row (GTK_CLIST(dccrwin.list), row, 0); */
      if(dcc) dcc_info(dcc);
   }
}

static void recv_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even)
{
     if(even && even->type == GDK_2BUTTON_PRESS) accept_clicked(0, 0);
}

void open_dcc_recv_window(void)
{  
   GtkWidget *vbox, *bbox;
#ifdef USE_GNOME
   static gchar *titles[] = { "Status","File","Size","Position","CPS","From","MIME Type"};
#else
   static gchar *titles[] = { "Status","File","Size","Position","CPS","From" };
#endif

   if(dccrwin.window)
   {
      update_dcc_recv_window();
      return;
   }

   dccrwin.window = gtkutil_window_new("X-Chat: DCC Receive List", "dccrecv", 600, 0,
				       close_dcc_recv_window, 0);

   vbox = gtk_vbox_new(FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(dccrwin.window), vbox);
   gtk_widget_show(vbox);

#ifdef USE_GNOME
   dccrwin.list = gtkutil_clist_new(7, titles, vbox, GTK_POLICY_ALWAYS,
				    recv_row_selected, 0,
				    0, 0, GTK_SELECTION_SINGLE);
#else
   dccrwin.list = gtkutil_clist_new(6, titles, vbox, GTK_POLICY_ALWAYS,
				    recv_row_selected, 0,
				    0, 0, GTK_SELECTION_SINGLE);
#endif
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 0, 65);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 1, 100);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 2, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 3, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 4, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccrwin.list), 5, 60);

   bbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
   gtk_widget_show(bbox);

   gtkutil_button_new("Accept", 0, 0, accept_clicked, 0, bbox);

   gtkutil_button_new("Resume", 0, 0, resume_clicked, 0, bbox);

   gtkutil_button_new("Abort", 0, 0, abort_clicked, 0, bbox);

   gtkutil_button_new("Info", 0, 0, info_clicked, 0, bbox);
   
   gtk_widget_show(dccrwin.window);
   update_dcc_recv_window();
}

static int close_dcc_send_window(void)
{
   dccswin.window = 0;
   return 0;
}

static void update_dcc_send_window(void)
{
   struct DCC *dcc;
   GSList *list = dcc_list;
   gchar *new[1][7];
   char size[14];
   char pos[14];
   char cps[14];
   char ack[14];
   gint row;
   int selrow;

   if(!dccswin.window) return;

   selrow = gtkutil_clist_selection(dccswin.list);

   gtk_clist_clear((GtkCList*)dccswin.list);
   new[0][2] = size;
   new[0][3] = pos;
   new[0][4] = ack;
   new[0][5] = cps;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if(dcc->type == TYPE_SEND)
      {
	 new[0][0] = dccstat[(int)dcc->stat].name;
	 new[0][1] = file_part(dcc->file);
	 new[0][6] = dcc->nick;
	 sprintf(size, "%lu", dcc->size);
	 sprintf(pos, "%lu", dcc->pos);
	 sprintf(cps, "%lu", dcc->cps);
	 sprintf(ack, "%lu", dcc->ack);
	 row = gtk_clist_append(GTK_CLIST(dccswin.list), new[0]);
	 gtk_clist_set_row_data(GTK_CLIST(dccswin.list), row, (gpointer)dcc);
	 gtk_clist_set_foreground
	   (
	   GTK_CLIST(dccswin.list), row, 
	   colors+dccstat[(int)dcc->stat].color
	   );
      }
      list = list->next;
   }
   if(selrow != -1)
     gtk_clist_select_row((GtkCList*)dccswin.list, selrow, 0);
}

static void send_row_selected(GtkWidget *clist, gint row, gint column,
		       GdkEventButton *even)
{
   if(even && even->type == GDK_2BUTTON_PRESS)
   {
      struct DCC *dcc;

      dcc = gtk_clist_get_row_data(GTK_CLIST(clist), row);
      if(dcc)
      {
	 switch(dcc->stat)
	 {
	  case STAT_FAILED:
	  case STAT_ABORTED:
	  case STAT_DONE:
	    dcc_abort(dcc);
	 }
      }
   }
}

static void info_send_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dccswin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccswin.list), row);
      if(dcc) dcc_info(dcc);
   }
}

static void abort_send_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;
   
   row = gtkutil_clist_selection(dccswin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dccswin.list), row);
      /*gtk_clist_unselect_row (GTK_CLIST(dccswin.list), row, 0);*/
      dcc_abort(dcc);
   }
}

void open_dcc_send_window(void)
{  
   GtkWidget *vbox, *bbox;
   static gchar *titles[] = { "Status","File","Size","Position","Ack","CPS","To" };

   if(dccswin.window)
   {
      update_dcc_send_window();
      return;
   }

   dccswin.window = gtkutil_window_new("X-Chat: DCC Send List", "dccsend", 550, 0,
				       close_dcc_send_window, 0);

   vbox = gtk_vbox_new(FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(dccswin.window), vbox);
   gtk_widget_show(vbox);

   dccswin.list = gtkutil_clist_new(7, titles, vbox, GTK_POLICY_ALWAYS,
				    send_row_selected, 0,
   				    0, 0, GTK_SELECTION_SINGLE);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 0, 65);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 1, 100);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 2, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 3, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 4, 50);
   gtk_clist_set_column_width(GTK_CLIST(dccswin.list), 5, 50);

   bbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
   gtk_widget_show(bbox);

   gtkutil_button_new("Abort", 0, 0, abort_send_clicked, 0, bbox);

   gtkutil_button_new("Info", 0, 0, info_send_clicked, 0, bbox);

   gtk_widget_show(dccswin.window);
   update_dcc_send_window();
}


/* DCC CHAT GUIs BELOW */


static void abort_chat_clicked(GtkWidget *wid, gpointer none)
{
   int row;
   struct DCC *dcc;

   row = gtkutil_clist_selection(dcccwin.list);
   if(row != -1)
   {
      dcc = gtk_clist_get_row_data(GTK_CLIST(dcccwin.list), row);
      dcc_abort(dcc);
   }
}

static void update_dcc_chat_window(void)
{
   struct DCC *dcc;
   GSList *list = dcc_list;
   gchar *new[1][5];
   char pos[14];
   char siz[14];
   gint row;
   int selrow;

   if(!dcccwin.window) return;

   selrow = gtkutil_clist_selection(dcccwin.list);

   gtk_clist_clear((GtkCList*)dcccwin.list);
   new[0][2] = pos;
   new[0][3] = siz;
   while(list)
   {
      dcc = (struct DCC *)list->data;
      if((dcc->type == TYPE_CHATSEND || dcc->type == TYPE_CHATRECV))
      {
	 new[0][0] = dccstat[(int)dcc->stat].name;
	 new[0][1] = dcc->nick;
	 sprintf(pos, "%lu", dcc->pos);
	 sprintf(siz, "%lu", dcc->size);
	 new[0][4] = ctime(&dcc->starttime);
	 row = gtk_clist_append(GTK_CLIST(dcccwin.list), new[0]);
	 gtk_clist_set_row_data(GTK_CLIST(dcccwin.list), row, (gpointer)dcc);
      }
      list = list->next;
   }
   if(selrow != -1)
     gtk_clist_select_row((GtkCList*)dcccwin.list, selrow, 0);
}

static int close_dcc_chat_window(void)
{
   dcccwin.window = 0;
   return 0;
}

void open_dcc_chat_window(void)
{  
   GtkWidget *vbox, *bbox;
   static gchar *titles[] = { "Status","To/From","Recv","Sent","StartTime" };

   if(dcccwin.window)
   {
      update_dcc_chat_window();
      return;
   }

   dcccwin.window = gtkutil_window_new("X-Chat: DCC Chat List", "dccchat", 550, 0,
				       close_dcc_chat_window, 0);

   vbox = gtk_vbox_new(FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
   gtk_container_add(GTK_CONTAINER(dcccwin.window), vbox);
   gtk_widget_show(vbox);

   dcccwin.list = gtkutil_clist_new(5, titles, vbox, GTK_POLICY_ALWAYS,
				    0, 0,
				    0, 0, GTK_SELECTION_BROWSE);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 0, 65);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 1, 100);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 2, 65);
   gtk_clist_set_column_width(GTK_CLIST(dcccwin.list), 3, 65);

   bbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
   gtk_widget_show(bbox);

   gtkutil_button_new("Abort", 0, 0, abort_chat_clicked, 0, bbox);

   gtk_widget_show(dcccwin.window);
   update_dcc_chat_window();
}
