/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "xchat.h"
#include "util.h"
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/utsname.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include "cfgfiles.h"
#include "dcc.h"
#include "gtkutil.h"
#ifndef USE_GNOME
#ifdef USE_IMLIB
#include <gdk_imlib.h>
#endif
#endif
#include <sys/types.h>
#include <sys/wait.h>
#ifdef USE_PANEL
#include <applet-widget.h>
GtkWidget *panel_applet = NULL, *panel_box;
int nopanel;
#endif

GdkFont *font_normal;
GdkFont *font_bold;

GdkFont *dialog_font_normal;
GdkFont *dialog_font_bold;

GSList *sess_list = 0;
GSList *serv_list = 0;
GSList *dcc_list = 0;
GSList *url_list = 0;
GSList *fkey_list = 0;
GSList *away_list = 0;

GtkStyle *channelwin_style;
GtkStyle *dialogwin_style;

gint notify_tag = -1;
gint xchat_is_quitting = 0;

extern GSList *ctcp_list;
extern GSList *popup_list;
extern GSList *button_list;
extern GSList *command_list;
extern GSList *replace_list;

#ifdef USE_PERL
extern struct session *perl_sess;
#endif
extern struct session *current_tab;
extern GtkWidget *main_window;
extern struct session *lastlog_last_sess;
extern struct session *menu_sess;
extern GdkColor colors[];
extern GtkWidget *main_book;
struct xchatprefs prefs;

void xchat_cleanup(void);
void kill_session(struct session *sess);
struct session *new_session(struct server *serv, char *from);
void sigchild_handler (int sig);

/* fkeys.c */

extern void fkeys_load(void);

/* maingui.c */

extern void make_non_channel_window(struct session *sess, int state);
extern void create_window(struct session *sess);
extern GtkStyle *my_widget_get_style(char *bg_pic, int bg_color);


/* rawlog.c */

extern void add_rawlog(struct server *serv, char *text, int outbound);
extern void close_rawlog(GtkWidget *wid, struct server *serv);

/* outbound.c */

extern void handle_inputgad(GtkWidget *igad, struct session *sess);

/* inbound.c */

extern void new_topic(char *buf);
extern void names_list(char *buf);
extern void you_joined(struct session *sess, char *chan);
extern void user_joined(char *chan, char *user, char *ip);
extern void new_nick(struct session *sess, char *newnick);
extern int process_line(struct session *sess, struct server *serv, char *buf);

/* plugin.c */

extern void module_setup();
extern void signal_setup();

/* dialog.c */

extern struct dialog *new_dialog(char *nick);
extern void dialog_notify_kill(struct server *serv);
extern int close_dialog(GtkWidget *wid, struct session *sess);

/* server.c */

extern void connecting_fin(struct session *);
extern void connect_server(struct session *sess, char *server, int port, int quiet);
extern void disconnect_server(struct session *sess, int sendquit);

/* serverlist.c */

extern void serverlist_autoconnect(struct session *sess, int aut);
extern void open_server_list(GtkWidget *widd, struct session *sess);

/* userlist.c */

extern void init_userlist_xpm(struct session *sess);
extern struct user *find_name(struct session *sess, char *name);

/* notify.c */

extern void notify_load(void);
extern void notify_save(void);
extern void notify_checklist(void);

/* lastlog.c */

extern void hide_lastlog(GtkWidget *wid, gpointer *data);

#ifdef USE_PERL
/* perl.c */

void perl_init(struct session *, int);
void perl_end(void);
int perl_inbound(struct session *sess, struct server *serv, char *buf);
#endif

/* editlist.c */

extern void list_loadconf(char *, GSList **, char *);
extern void list_free(GSList **);

/* text.c */

extern void PrintText(struct session *sess, unsigned char *text);
extern unsigned char *strip_color(unsigned char *text);
extern void end_logging (int fd);
extern void load_text_events();  
extern void pevent_dialog_save (char *fn);

/* util.c */

extern int is_data_avail(int sok);
extern int get_mhz(void);

extern void menu_about(GtkWidget *wid, gpointer sess);


int tcp_send_len(struct server *serv, char *buf, int len)
{
   add_rawlog(serv, buf, TRUE);
   return send(serv->sok, buf, len, 0);
}

int tcp_send(struct server *serv, char *buf)
{
   return tcp_send_len(serv, buf, strlen(buf));
}

struct session *find_session_from_channel(char *chan, struct server *serv)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(!strcasecmp(chan, sess->channel))
      {
	 if(!serv) return sess;
	 if(serv == sess->server) return sess;
      }
      list = list->next;
   }
   return 0;
}

struct session *find_session_from_nick(char *nick, struct server *serv)
{
   struct session *sess;
   GSList *list = sess_list;

   if(find_name(serv->front_session, nick)) return serv->front_session;

   sess = find_session_from_channel(nick, serv);
   if(sess) return sess;

   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->server == serv)
      {
	 if(find_name(sess, nick)) return sess;
      }
      list = list->next;
   }
   return 0;
}

struct session *find_session_from_waitchannel(char *chan, struct server *serv)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->server == serv)
      {
	 if(!strcasecmp(chan, sess->waitchannel)) return sess;
      }
      list = list->next;
   }
   return 0;
}

void show_generic_channel(struct server *serv, char *chan, char *msg)
{
   struct session *sess = find_session_from_channel(chan, serv);
   if(sess) PrintText(sess, msg);
}

void auto_reconnect(struct server *serv)
{
   struct session *s;
   GSList *list = sess_list; /* make sure auto rejoin can work */

   if (serv->front_session == NULL)
     return;

   while(list)
   {
      s = (struct session *)list->data;
      if(s->channel[0] == '#' || s->channel[0] == '&')
      {
	 strcpy(s->waitchannel, s->channel);
	 strcpy(s->willjoinchannel, s->channel);
      }
      list = list->next;
   }
   disconnect_server(serv->front_session, FALSE);
   connect_server(serv->front_session, serv->hostname, serv->port, FALSE);
}

void read_data(struct server *serv, gint sok)
{
   int len;
   char lbuf[2050];

   while(1)
   {
      len = recv(sok, lbuf, sizeof lbuf-2, 0);
      if(len < 1)
      {
	 if(len < 0)
	 {
	    if(errno == EAGAIN || errno == EWOULDBLOCK) return;
	 }
	 if(prefs.autoreconnect)
	   auto_reconnect(serv);
	 else
	   disconnect_server(serv->front_session, FALSE);
 	 return;
      } else {
	 int j, i = 0;
	 lbuf[len] = 0;

	 while(i < len)
	 {
	    switch(lbuf[i])
	    {
	     case '\r':
	       break;
	     case '\n':
	       serv->linebuf[serv->pos] = 0;
	       if(prefs.stripcolor)
	       {
		  char *temp = strip_color(serv->linebuf);
		  j = process_line(serv->front_session, serv, temp);
		  free(temp);
	       } else
		 j = process_line(serv->front_session, serv, serv->linebuf);
	       serv->pos = 0;
	       /*if(j == -1)
	       {
		  if(prefs.autoreconnect) auto_reconnect(serv);
		  return;
	       }*/
	       break;
	     default:
	       serv->linebuf[serv->pos] = lbuf[i];
	       serv->pos++;
	       if(serv->pos == 2047) serv->pos = 2046;
	    }
	    i++; 
	 }
      }
   }
}

gint kill_session_callback(GtkWidget *win, struct session *sess)
{
   char tbuf[512];
   int willquit = TRUE;
   struct server *serv = sess->server;
   struct session *killsess = sess;
   struct session *s;
   struct nbexec *re;
   GSList *list = sess_list;

#ifdef USE_GNOME /* remove the configure_event for zvt */
   if(sess->zvt && sess->is_tab && main_window)
     gtk_signal_disconnect_by_data(GTK_OBJECT(main_window), sess->textgad);
#endif

#ifdef USE_PANEL   
   if (sess->panel_button)
      gtk_widget_destroy(sess->panel_button);
#endif
   if (sess->running_exec != NULL) {
      re = sess->running_exec;
      sess->running_exec = NULL;
      kill(re->childpid, SIGKILL);
      gdk_input_remove(re->iotag);
      close(re->myfd);
      close(re->childfd);
      free(re);
   }

#ifdef USE_PERL
   if(sess->channel[0] == '#' || sess->channel[0] == '&')
   {
      snprintf(tbuf, sizeof tbuf, ":%s KILLSESSION %s %s",
	       serv->nick,
	       sess->channel,
	       serv->servername);
      perl_inbound(sess, serv, tbuf);
   }

   if(perl_sess == sess) /* need to find a new perl_sess, this one's closing */
   {
      struct session *s;
      GSList *list = sess_list;
      while(list)
      {
	 s = (struct session *)list -> data;
	 if(s->server == perl_sess->server && s != perl_sess)
	 {
	    perl_sess = s;
	    break;
	 }
	 list = list -> next;
      }
      if(perl_sess == sess) perl_sess = 0;
   }
#endif

   if(!sess->quitreason) sess->quitreason = prefs.quitreason;

   while(list)
   {
      s = (struct session *)list->data;
      if(s->server == killsess->server && s != killsess)
      {
	 willquit = FALSE;
	 list = 0;
      } else
	 list = list->next;
   }

   if(current_tab == sess) current_tab = 0;
   if(menu_sess == sess) menu_sess = (struct session *)sess_list->data;
   if(sess->server->front_session == sess) sess->server->front_session = 0;
   if(sess->bar) connecting_fin(sess);

   if(sess->server && sess->server->connected && willquit)
   {
      snprintf(tbuf, sizeof tbuf, "QUIT :%s\r\n", sess->quitreason);
      tcp_send(sess->server, tbuf);
   } else {
      if(sess->server && sess->server->connected && sess->channel[0] && 
	((sess->channel[0] == '#') || (sess->channel[0] == '&')))
      {
	 snprintf(tbuf, sizeof tbuf, "PART %s\r\n", sess->channel);
	 tcp_send(sess->server, tbuf);
      }
   }

   if(lastlog_last_sess == sess) hide_lastlog(0, 0);

   if (sess->is_tab) {
      if (main_book) {
         if (gtk_notebook_get_nth_page(GTK_NOTEBOOK(main_book), 0) == NULL)
	    gtk_widget_destroy(main_book);
      }
   }
   if (sess->logfd != -1)
   	end_logging(sess->logfd);
   sess_list = g_slist_remove(sess_list, sess);
   free(sess);

   if(sess_list)
   {
         list = sess_list;
	 while(list)
	 {
	    sess = (struct session *)list->data;
	    if(sess->server == serv) return TRUE;
	    list = list->next;
	 }
	 if(serv->connected)
	 {
	    gdk_input_remove(serv->iotag);
	    close(serv->sok);
	 }
         if(serv->connecting)
	 {
	    gdk_input_remove(serv->iotag);
	    close(serv->childread);
	    close(serv->sok);
	 }
         dialog_notify_kill(serv);
	 if(serv->rawlog_window) close_rawlog(0, serv);
         serv_list = g_slist_remove(serv_list, serv);
         dcc_notify_kill(serv);
         free(serv);
	 return TRUE;
   }
   if(prefs.tabchannels && main_window) gtk_widget_destroy(main_window);
   
   xchat_cleanup();
   return TRUE;
}

struct server *new_server(void)
{
   struct server *serv = malloc(sizeof(struct server));
   if(!serv) return 0;
   memset(serv, 0, sizeof(struct server));
   serv->sok = -1;
   strcpy(serv->nick, prefs.nick1);
   serv_list = g_slist_append(serv_list, serv);
   return serv;
}

GdkFont *my_font_load(char *fontname)
{
   GdkFont *font;

   if(!*fontname) fontname = "fixed";
   font = gdk_font_load(fontname);
   if(!font)
   {
      char temp[256];
      sprintf(temp, "Cannot open font:\n\n%s", fontname);
      gtkutil_simpledialog(temp);
      /*strcpy(fontname, "fixed");*/
      font = gdk_font_load("fixed");
      if(!font)
      {
	 g_error("gdk_font_load failed");
	 gtk_exit(0);
      }
   }
   return font;
}

struct session *new_session(struct server *serv, char *from)
{
   struct session *sess;

   sess = malloc(sizeof(struct session));
   if(!sess) return 0;

   memset(sess, 0, sizeof(struct session));
   sess->server = serv;
   sess->logfd = -1;
   sess->running_exec = NULL;
   if (from) {
      sess->is_dialog = TRUE;
      strcpy(sess->channel, from);
   } else
      sess->is_dialog = FALSE;

   create_window(sess);
   if (!sess->is_dialog)
      make_non_channel_window(sess, FALSE);

   sess_list = g_slist_append(sess_list, sess);

   return(sess);
}

void free_sessions(void)
{
   struct session *sess;
   GSList *list = sess_list;

   while(list)
   {
      sess = (struct session *)list->data;
      if (sess->logfd != -1)
	end_logging(sess->logfd);
      history_free(&sess->history);
      kill_session_callback(0, sess);
      list = sess_list;
   }
}

struct away_msg *find_away_message( struct server *serv, char *nick )
{
   struct away_msg *away;
   GSList *list = away_list;
   while(list)
   {
      away = (struct away_msg *)list->data;
      if( away->server == serv && !strcasecmp(nick, away->nick))
	  return away;
      list = list->next;
   }
   return 0;
}

void save_away_message( struct server *serv, char *nick, char *msg )
{
    struct away_msg *away = find_away_message( serv, nick );

    if( away )			/* Change message for known user */
    {
	if( away->message )
	    free( away->message );
	away->message = strdup( msg );
    }
    else			/* Create brand new entry */
    {
	away = malloc( sizeof( struct away_msg ));
	if( away )
	{
	    away->server = serv;
	    strcpy( away->nick, nick );
	    away->message = strdup( msg );
	    away_list = g_slist_append( away_list, away );
	}
    }
}

void show_init_msg(struct session *sess)
{
   char buf[256];
   struct utsname un;
   time_t tim = time(0);
   int mhz;

   uname(&un);

   mhz = get_mhz();
   if(mhz)
     snprintf(buf, sizeof buf,
	      "X-Chat \00311"VERSION" \003 started at %s"
	      "Running on %s \00311%s \003 \0032[\00311%s\0032/\00311%dMHz\0032]\003 \n",
	      ctime(&tim), un.sysname, un.release, un.machine, mhz);
   else
     snprintf(buf, sizeof buf,
	      "X-Chat \00311"VERSION" \003 started at %s"
	      "Running on %s \00311%s \003 \0032[\00311%s\0032]\003 \n",
	      ctime(&tim), un.sysname, un.release, un.machine);
   PrintText(sess, buf);
}

#define defaultconf_ctcp "NAME TIME\nCMD /nctcp %s TIME %t\n\nNAME PING\nCMD /nctcp %s PING %d\n\nNAME USERINFO\nCMD /notice %s What are you doing Dave?\n"
#define defaultconf_popup "NAME SUB\nCMD CTCP\n\nNAME CTCP Version\nCMD /ctcp %s VERSION\n\nNAME CTCP Userinfo\nCMD /ctcp %s USERINFO\n\nNAME CTCP Ping\nCMD /ping %s\n\nNAME CTCP Time\nCMD /ctcp %s TIME\n\nNAME ENDSUB\nCMD \n\nNAME SEP\nCMD \n\nNAME Give Voice\nCMD /voice %s\n\nNAME Take Voice\nCMD /devoice %s\n"
#define defaultconf_buttons "NAME Op\nCMD /op %s\n\nNAME DeOp\nCMD /deop %s\n\nNAME Ban\nCMD /ban %s\n\nNAME Kick\nCMD /kick %s\n\nNAME Send\nCMD /dcc send %s\n\nNAME Chat\nCMD /chat %s\n\nNAME Dialog\nCMD /query %s\n\nNAME Whois\nCMD /whois %s\n"
#define defaultconf_replace "NAME teh\nCMD the\n\nNAME r\nCMD are\n\nNAME u\nCMD you\n\n"
char *defaultconf_commands =
"NAME BANLIST\nCMD /quote MODE %c +b\n\n"
"NAME CHAT\nCMD /dcc chat %2\n\n"
"NAME CONNECT\nCMD /quote CONNECT &2\n\n"
"NAME DIALOG\nCMD /query %2\n\n"
"NAME DRAW\nCMD /dcc draw %2\n\n"
"NAME EXIT\nCMD /quit\n\n"
"NAME INFO\nCMD /quote INFO\n\n"
"NAME J\nCMD /join &2\n\n"
"NAME KILL\nCMD /quote KILL %2 :&3\n\n"
"NAME LEAVE\nCMD /part &2\n\n"
"NAME LINKS\nCMD /quote LINKS\n\n"
"NAME LIST\nCMD /quote LIST\n\n"
"NAME LUSERS\nCMD /quote LUSERS\n\n"
"NAME M\nCMD /msg &2\n\n"
"NAME MAP\nCMD /quote MAP\n\n"
"NAME MOTD\nCMD /quote MOTD\n\n"
"NAME OPER\nCMD /quote OPER %2 %3\n\n"
"NAME RAW\nCMD /quote &2\n\n"
"NAME REHASH\nCMD /quote REHASH\n\n"
"NAME SERVHELP\nCMD /quote HELP\n\n"
"NAME TIME\nCMD /quote TIME\n\n"
"NAME SQUIT\nCMD /quote SQUIT &2\n\n"
"NAME STATS\nCMD /quote STATS &2\n\n"
"NAME UPTIME\nCMD /quote STATS u\n\n"
"NAME VER\nCMD /ctcp %2 VERSION\n\n"
"NAME VERSION\nCMD /ctcp %2 VERSION\n\n"
"NAME WII\nCMD /quote WHOIS %2 %2\n\n"
"NAME WHO\nCMD /quote WHO &2\n\n"
"NAME WHOIS\nCMD /quote WHOIS &2\n\n"
"NAME WHOWAS\nCMD /quote WHOWAS &2\n\n"
;

void xchat_init(int autoconnect)
{
   struct session *sess;
   struct server *serv;

#ifndef USE_GNOME
#ifdef USE_IMLIB
   gdk_imlib_init();
#endif
#endif

   fkeys_load();
   signal_setup();
   load_text_events();
   list_loadconf("popup.conf", &popup_list, defaultconf_popup);
   list_loadconf("ctcpreply.conf", &ctcp_list, defaultconf_ctcp);
   list_loadconf("buttons.conf", &button_list, defaultconf_buttons);
   list_loadconf("commands.conf", &command_list, defaultconf_commands);
   list_loadconf("replace.conf", &replace_list, defaultconf_replace);
   notify_load();

   channelwin_style = my_widget_get_style(prefs.background, prefs.bg_color);
   dialogwin_style = my_widget_get_style(prefs.background_dialog, prefs.dialog_bg_color);

   font_normal = my_font_load(prefs.font_normal);
   font_bold = my_font_load(prefs.font_bold);
   dialog_font_normal = my_font_load(prefs.dialog_font_normal);
   dialog_font_bold = my_font_load(prefs.dialog_font_bold);

   serv = new_server();
   if(serv)
   {
      sess = new_session(serv, 0);
      init_userlist_xpm(sess);
      show_init_msg(sess);
#ifdef USE_PLUGIN
      module_setup();
#endif
#ifdef USE_PERL
      perl_init(sess, TRUE);
#endif

      if(!prefs.skipserverlist || autoconnect) {
	 open_server_list(0, sess);
      }

      if(autoconnect) serverlist_autoconnect(sess, autoconnect);
      if(prefs.notify_timeout)
	notify_tag = gtk_timeout_add(prefs.notify_timeout, (GtkFunction) notify_checklist, 0);

      gtk_timeout_add(5000, (GtkFunction) dcc_check_timeouts, 0);
   }
}

void xchat_cleanup(void)
{
   xchat_is_quitting = TRUE;
   if(prefs.autosave)
   {
      save_config();
      pevent_dialog_save(NULL);
   }
   notify_save();
   free_sessions();
   list_free(&serv_list);
   list_free(&dcc_list);
   list_free(&ctcp_list);
   list_free(&popup_list);
   list_free(&button_list);
   list_free(&command_list);
   list_free(&replace_list);
#ifdef USE_PERL
   perl_end();
#endif
   gtk_exit(0);
}

void sigchild_handler (int sig)
{
   /* We get a SIGCHLD everytime one of our children dies, this could be
   a /exec child and if it is we need to free it's struct and set
   sess->running_exec = NULL */

   struct session *sess;
   GSList *list = sess_list;
   
   while(list)
   {
      sess = (struct session *)list->data;

      /* Scan for killed connections */
      if (sess->server->childpid)
	 waitpid(sess->server->childpid, NULL, WNOHANG);
      if(sess->running_exec != NULL)
      {
         if (waitpid(sess->running_exec->childpid, NULL, WNOHANG) > 0)
	    close(sess->running_exec->childfd);
      }
      list = list->next;
   }
   return;
}

#ifdef USE_PANEL
static void
panel_invalidate (GtkWidget *wid, gpointer *arg2)
{
   panel_applet = NULL;
}

void create_panel_widget()
{
   panel_applet = applet_widget_new("xchat_applet");
   gtk_widget_realize(panel_applet);      

   if (prefs.panel_vbox)
      panel_box = gtk_vbox_new(0, 0);
   else
      panel_box = gtk_hbox_new(0, 0);
   applet_widget_add(APPLET_WIDGET(panel_applet), panel_box);
   gtk_widget_show(panel_box);

   gtkutil_label_new("X-Chat:", panel_box);

   applet_widget_register_stock_callback (APPLET_WIDGET(panel_applet),
					  "about",
					  GNOME_STOCK_MENU_ABOUT,
					  _("About..."),
					  GTK_SIGNAL_FUNC(menu_about),
					  NULL);

   gtk_signal_connect(GTK_OBJECT(panel_applet), "destroy", panel_invalidate, NULL);
   gtk_widget_show (panel_applet);
}
#endif

int main(int argc, char *argv[])
{
   int autoconnect = 0;
   struct sigaction act;
   sigset_t sigs;
#ifdef USE_GNOME
   struct poptOption options[] = {
        { "connect", 'c', POPT_ARG_INT,  0, 0, "Auto connect", "Entry number"},
#ifdef USE_PANEL
        { "no-panel",'n', POPT_ARG_NONE, 0, 0, "Don't use GNOME Panel", 0},
#endif
        POPT_AUTOHELP
        { 0, '\0', 0, 0 }
    };
#endif

   if(argc > 1)
   {
#ifdef USE_PANEL
      if(!strcasecmp(argv[1], "-n") || !strcasecmp(argv[1], "--no-panel"))
      {
	 nopanel = TRUE;
      }
#endif
      if(!strcasecmp(argv[1], "-v") || !strcasecmp(argv[1], "--version"))
      {
	 puts("X-Chat "VERSION"");
         return 0;
      }
#ifndef USE_GNOME
      if(!strcasecmp(argv[1], "-h") || !strcasecmp(argv[1], "--help"))
      {
	 puts("X-Chat "VERSION" Options:\n\n"
	      "   --connect      -c <n>    : auto connect to server entry <n>\n"
	      "   --version      -v        : show version information\n"
	      );
         return 0;
      }
#endif
   }

   if(argc > 2)
   {
      if(!strcasecmp(argv[1], "-c") || !strcasecmp(argv[1], "--connect")) autoconnect = atoi(argv[2]);
   }

#ifdef SOCKS
   SOCKSinit(argv[0]);
#endif

#ifdef USE_PANEL
   if(nopanel)
     gnome_init_with_popt_table(argv[0], VERSION, argc, argv, options, 0, 0);
   else
     applet_widget_init(argv[0], VERSION, argc, argv, options, 0, 0);
#else
#ifdef USE_GNOME
   gnome_init_with_popt_table(argv[0], VERSION, argc, argv, options, 0, 0);
#else
   gtk_init(&argc, &argv);
#endif
#endif
   gtk_set_locale();

   load_config();

   act.sa_handler = SIG_IGN;
   sigemptyset(&sigs);
   act.sa_mask = sigs;
   act.sa_flags = 0;
   sigaction(SIGPIPE, &act, NULL);

   act.sa_handler = sigchild_handler;
   sigemptyset(&sigs);
   act.sa_mask = sigs;
   act.sa_flags = 0;
   sigaction(SIGCHLD, &act, NULL);

   xchat_init(autoconnect);

#ifdef USE_PANEL
   if(nopanel)
     gtk_main();
   else
     applet_widget_gtk_main(); 
#else
   gtk_main();
#endif

   xchat_cleanup();

   return 0;
}
