// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of APE.
// 
// The exception is that, if you link the APE library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the APE library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name APE.  If you copy code from other releases into a copy of
// APE, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for APE, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include "config.h"
#include "macros.h"
#include "thread.h"
#include "file.h"
#include <stdio.h>
#include <fcntl.h>
#include <sys/stat.h>

#ifndef	F_LOCK

enum
{
	F_ULOCK = 1,
	F_LOCK,
	F_TLOCK,
	F_TEST
};

static	int lockf(int fd, int cmd, long len)
{
	struct	flock lck;

	lck.l_start = 0l;
	lck.l_whence = SEEK_CUR;
	lck.l_len = len;

	switch(cmd)
	{
	case F_ULOCK:
		lck.l_type = F_UNLCK;
		return fcntl(fd, F_SETLK, &lck);
	case F_LOCK:
		lck.l_type = F_WRLCK;
		return fcntl(fd, F_SETLKW, &lck);
	case F_TLOCK:
		lck.l_type = F_WRLCK;
		return fcntl(fd, F_SETLK, &lck);
	case F_TEST:
		lck.l_type = F_WRLCK;
		fcntl(fd, F_GETLK, &lck);
		if(lck.l_type == F_UNLCK)
			return 0;
		return -1;
	}
}
		
#endif

static const char *clearfile(const char *pathname)
{
	remove(pathname);
	return pathname;
}

static const char *clearfifo(const char *pathname, int mode)
{
	remove(pathname);
	mkfifo(pathname, mode);
	return pathname;
}


File::File(const char *fname, int access)
{
	_fd = open(fname, access);
	if(_fd < 0)
		throw(this);
}

File::File(const char *fname, int access, int mode)
{
	_fd = open(fname, access | O_CREAT, mode);
	if(_fd < 0)
		throw(this);
}

File::File(int fd)
{
	if(fd < 0)
		throw(this);
	_fd = fd;
}

File::File(const File &f)
{
	_fd = dup(f._fd);
	if(_fd < 0)
		throw(this);
}

File &File::operator=(const File &f)
{
	close(_fd);
	_fd = dup(f._fd);
	if(_fd < 0)
		throw(this);

	return *this;
}

NewFile::NewFile(const char *fname, int access, int mode) :
File(open(clearfile(fname), access | O_CREAT, mode)){};

NamedPipe::NamedPipe(const char *fname, int mode) :
File(open(clearfifo(fname, mode), FILE_OPEN_READWRITE | O_CREAT, mode)) {};

LockedFile::LockedFile(const char *fname, int access) :
File(fname, access) {};

LockedFile::LockedFile(const LockedFile &f) :
File((const File &)f) {};

int LockedFile::Append(void *buf, size_t len)
{
	int rc;

	pos_t pos = (pos_t)lseek(_fd, 0l, SEEK_END);
	lockf(_fd, F_LOCK, -1);
	lseek(_fd, 0l, SEEK_END);
	rc = write(_fd, (char *)buf, len);
	lseek(_fd, pos, SEEK_SET);
	lockf(_fd, F_ULOCK, -1);
	return rc;
}

int LockedFile::Request(pos_t pos, void *buf, size_t len)
{
	pos_t npos = (pos_t)lseek(_fd, pos, SEEK_SET);
	if(npos != pos)
		return -1;

	lockf(_fd, F_LOCK, len);
	return read(_fd, (char *)buf, len);
}

int LockedFile::Update(pos_t pos, void *buf, size_t len)
{
	pos_t npos = (pos_t)lseek(_fd, pos, SEEK_SET);
	if(npos != pos)
		return -1;

	int rc = write(_fd, (char *)buf, len);
	lockf(_fd, F_ULOCK, len);
	return rc;
}

int LockedFile::Clear(pos_t pos, size_t len)
{
	pos_t npos = (pos_t)lseek(_fd, pos, SEEK_SET);
	if(pos != pos)
		return -1;

	lockf(_fd, F_ULOCK, len);
	return 0;
}

SharedFile::SharedFile(const char *fname, int access, int mode) :
File(fname, access, mode), Mutex() {};

SharedFile::SharedFile(const char *fname, int access) :
File(fname, access), Mutex() {};

SharedFile::SharedFile(const SharedFile &f) :
File((const File &)f), Mutex() {};

SharedFile::SharedFile(int fd) :
File(fd), Mutex() {};

int SharedFile::Append(void *buf, size_t len)
{
	int rc;
	
	ENTER_CRITICAL
	lseek(_fd, (off_t)0l, SEEK_END);
	rc = write(_fd, (char *)buf, len);
	LEAVE_CRITICAL
	return rc;
}

int SharedFile::Read(pos_t pos, void *buf, size_t len)
{
	int rc;
	pos_t npos;	

	ENTER_CRITICAL
	npos = (pos_t)lseek(_fd, (off_t)pos, SEEK_SET);
	if(npos == pos)
		rc = read(_fd, (char *)buf, len);
	else
		rc = -1;
	LEAVE_CRITICAL
	return rc;
}

SharedFile &SharedFile::operator=(SharedFile &f)
{
	ENTER_CRITICAL
	close(_fd);
	_fd = dup(f._fd);
	LEAVE_CRITICAL
	if(_fd < 0)
		throw(this);

	return *this;
}

MappedFile::MappedFile(const char *fname, pos_t pos, size_t len, int mode) :
File(fname, mode)
{
	unsigned prot = 0, fmode = 0;

	switch(mode)
	{
	case O_RDONLY:
		prot = PROT_READ;
		break;
	case O_WRONLY:
		prot = PROT_WRITE;
		break;
	case O_RDWR:
		prot = PROT_READ | PROT_WRITE;
	}

	lseek(_fd, pos + len, SEEK_SET);
	_mem = (char *)mmap(NULL, len, prot, MAP_SHARED, _fd, pos);
	_len = len;
	if(_mem == MAP_FAILED)
	{
		close(_fd);
		_fd = -1;
		throw(this);
	}
} 
	
MappedFile::~MappedFile()
{
	msync(_mem, _len, MS_SYNC);
	munmap(_mem, _len);
	close(_fd);
}
	
