// Copyright (C) 1999 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "misc.h"
#include "audio.h"
#include "thread.h"
#include "rtp.h"
#include <sys/uio.h>

RTPSocket::RTPSocket(InetAddress &bind, short port) :
UDPSocket(bind, port)
{
	rtpsend.flags[0] = 0x80;        // version 2 RTP, no conf ids
	rtpsend.flags[1] = 0x08;        // u-law
	setFrameSampleSize(160);        // 20 milliseconds
	setRTShort(rtpsend.sequence, 0);
	setRTLong(rtpsend.timestamp, 0);
	setRTLong(rtpsend.source, 0);
};

RTPSocket::RTPSocket(BroadcastAddress &bind, short port) :
UDPSocket(bind, port)
{
	rtpsend.flags[0] = 0x80;        // version 2 RTP, no conf ids
	rtpsend.flags[1] = 0x08;        // u-law
	setFrameSampleSize(160);        // 20 milliseconds
	setRTShort(rtpsend.sequence, 0);
	setRTLong(rtpsend.timestamp, 0);
	setRTLong(rtpsend.source, 0);
};

void RTPSocket::setPayload(unsigned payload, timeout_t timer)
{
	payload &= 0x7f;
	rtpsend.flags[1] = rtpsend.flags[1] & 0x80 | payload;
	if(!timer)
		return;

	switch(payload)
	{
	default: // 8khz u-law   
		setFrameSampleSize(8 * timer);
		break;
	}
}

void RTPSocket::addConference(unsigned long source)
{
	unsigned cc;
	
	if(rtpsend.flags[0] & 0x0f > 15)
		return;
	
	cc = (++rtpsend.flags[0]) & 0x0f;
	setRTLong(rtpsend.conf[cc], source);
}
		
void RTPSocket::Skip(int count)
{
	setRTLong(rtpsend.timestamp, getRTLong(rtpsend.timestamp) + 
		getFrameSampleSize() * count);
}

int RTPSocket::Write(void *buf, size_t len)
{
	int rtn;
	struct iovec iov[2];
	size_t hdrlen = 12 + (rtpsend.flags[0] & 0x0f * 4);
	
	iov[0].iov_base = &rtpsend;
	iov[0].iov_len = hdrlen;
	iov[1].iov_base = buf;
	iov[1].iov_len = len;
	rtn = writev(so, iov, 2);
	Skip();
	setRTShort(rtpsend.sequence, getRTShort(rtpsend.sequence) + 1);
	if(rtn >= hdrlen)
		rtn -= hdrlen;
	return rtn;
}

int RTPSocket::Read(void *buf, size_t len)
{
	int rtn;
	struct iovec iov[2];
	size_t hdrlen;
	
#ifndef MSG_WAITALL
#define MSG_WAITALL 0
#endif
	
	rtn = recv(so, &rtprecv, 12, MSG_PEEK | MSG_WAITALL);
	if(rtn < 0)
		return rtn;
	if(rtn < 12)
		return 0;

	hdrlen = 12 + (rtprecv.flags[0] & 0x0f * 4);
	iov[0].iov_base = &rtprecv;
	iov[0].iov_len = hdrlen;
	iov[1].iov_base = buf;
	iov[1].iov_len = len;
	rtn = readv(so, iov, 2);
	if(rtn >= hdrlen)
		rtn -= hdrlen;
	return rtn;
}

unsigned RTPSocket::getHostHeader(unsigned short *seq, unsigned long *stamp)
{
	if(seq)
		*seq = getRTShort(rtpsend.sequence);

	if(stamp)
		*stamp = getRTShort(rtpsend.timestamp);

	return rtpsend.flags[1] & 0x7f;
}

unsigned RTPSocket::getPeerHeader(unsigned short *seq, unsigned long *stamp)
{
	if(seq)
		*seq = getRTShort(rtprecv.sequence);

	if(stamp)
		*stamp = getRTShort(rtprecv.timestamp);

	return rtprecv.flags[1] & 0x7f;
}

int RTPSocket::Wait(timeout_t timeout)
{
	fd_set inp;
	struct timeval tv;
	int rts;
	
	tv.tv_sec = timeout / 1000;
	tv.tv_usec = timeout % 1000 * 1000;
	
	FD_ZERO(&inp);
	FD_SET(so, &inp);
	if(!select(so + 1, &inp, NULL, NULL, &tv))
		return -1;
	
	rts = Peek(&rtprecv, 12);
	if(rts < 0)
		return rts;
	if(rts < 12)
		return -1;
	return 0;
}
