/* sh_send_rcv.cc

   Author    : Marko Meyer
   Date      : 95/10/21  (derived from neuron_serv_tcp_3.cc)
   Time-stamp: <96/01/12 14:45:24 mme>

   Contains functions for sending and receiveing the different structs.
   Used both by Server and Client.

   This file belongs to BACKNET, thus restrictions stated there apply.
   
	BACKNET - A library for simulating neural BACKPROPAGATION-Networks
    Copyright (C) 1995	Marko Meyer

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


	If you have any suggestions, ideas or comments regarding this library
	feel free to contact me:

	Email:	mme@pub.th-zwickau.de
	Ordinary Mail:	Marko Meyer
					Teichstrasse 27
					D-08289 Schneeberg
					Germany

	Changing History:

	95/12/03 -- Marko Meyer: Included send_(int,char*,unsigned,sockaddr_in*,
	                                        int)
											[and recv_(...) as well]
							 For being able to handle UDP and TCP connections
							 with the same method.
							 Changed the headers of the other send_/recv_-
							 methods as well.

	95/12/27 -- Marko Meyer: Something entirely new for us: a buffered 
	                         receiver [for UDP], that is connected to
							 somewhere through a pipe. 
							 buffered_receive(...).

	96/01/06 -- Marko Meyer: Inserted functions for ntoh- and hton'ing 
	                         our structs. This should be independent 
							 from sending them.

	96/01/11 -- Marko Meyer: SO_RCVTIMEO is not longer used; using 
	                         select(2) instead.

    96/01/12 -- Marko Meyer: in recv_sel(...) select was used suboptimally.
*/

#include <netbacknet.h>

#include <netinet/in.h>

int 
recv_sel(int sockd, char *buf, int count, unsigned int flags,
		 sockaddr_in *addr = NULL, int *len = NULL, timeval *timeout = NULL)
{
	fd_set sdvar; 
	int sel_ret = 0;
	
	if(timeout)
	{
		DBG(cerr<<"recv_sel: Timeout set to "<<timeout->tv_sec<<":"
			<<timeout->tv_usec<<"."<<endl);
		FD_ZERO(&sdvar);
		FD_SET(sockd, &sdvar); 
		while( (sel_ret=select(sockd + 1, &sdvar, NULL, NULL, timeout)) < 0)
		{
			/* Maybe we got a signal; then we continue waiting. */
			if(errno == EINTR)
			{ 
				DBG(cerr<<"recv_sel: Got Interrupt."<<endl);
				continue;
			}
			else
			{
				perror("recv_sel: Error at select");
				exit(-1);
			}
		}

		/* Otherwise the connection is timed out or a fd is ready. 
		DBG(cerr<<"neuron_serv: Select returned, timeout state is:"
			<<endl;
			cerr<<"\t\t"<<timeout->tv_sec<<" s\t"<<timeout->tv_usec
			<<" s."<<endl);*/
		
		if((sel_ret == 0) || ((timeout->tv_sec == 0) && (timeout->tv_usec == 0)))
		{
			DBG(cerr<<"neuron_serv: Select timed out."<<endl);
			errno = EWOULDBLOCK;
			return -1;
		}
	}
	return recv_(sockd, buf, count, flags, addr, len);
}

int 
recv_(int sd, char *rcvbuf, int buflen, unsigned int flags, 
	  sockaddr_in *addr = NULL, int *len = NULL)
{
	int socktype = 0;
	int optlen = sizeof(socktype);
	int ret_val = -1;
	
	if(getsockopt(sd,SOL_SOCKET,SO_TYPE,(char *)&socktype,&optlen) == -1)
		perror("recv_: Error at getsockopt ");
	else
	{
		if(socktype == SOCK_STREAM)
			ret_val = recv(sd, rcvbuf, buflen, flags); /* TCP */
		else
			if(socktype == SOCK_DGRAM)
			{
				if((addr != NULL) && (len != NULL))
					ret_val = recvfrom(sd, rcvbuf, buflen, flags, 
									(sockaddr *)addr, len);
				else
				{
					DBG(cerr<<"recv_(...void...): addr/len - null pointer."
						<<endl);
					errno = EFAULT;
					ret_val = -1;
				}
			}
			else
			{
				DBG(cerr<<"recv_(... void ...): Wrong socktype!"<<endl);
				errno = EPROTONOSUPPORT;
				ret_val = -1;
			}
	}
	return ret_val;
}

int 
send_(int sd, char *sndbuf, int buflen, unsigned int flags, 
	  sockaddr_in *addr = NULL, int addrlen = 0)
{
	/* Here is decided whether send(2) or sendto(2) is used, thus whether
	   an UDP or TCP connection is bound to sd. --mme 95/12/03*/

	int socktype = 0;
	int optlen = sizeof(socktype);
	int ret_val = -1;

	DBG(cerr<<"send_: Entering ..."<<endl);
	if(getsockopt(sd,SOL_SOCKET,SO_TYPE, (char *)&socktype,&optlen) == -1)
		perror("send_: Error at getsockopt ");
	else
	{
		if(socktype == SOCK_STREAM)
		{
			ret_val = send(sd, sndbuf, buflen, flags); /* TCP */
			if (ret_val < 0) perror("send_: Error at send");
		}			
		else
			if(socktype == SOCK_DGRAM)
			{
				if((addr == NULL) || (addrlen == 0))
				{
					cerr<<"send_: No address given at UDP send!"<<endl;
					errno = EINVAL;
					ret_val = -1;
				}
				else
				{
					DBG(cerr<<"send_: reaching sendto:"<<endl
						<<"\tsocket: "<<sd<<"\tsendbuf: "<<sndbuf
						<<"\tbuflen: "<<buflen<<"\tflags: "<<flags<<endl
						<<"\taddr: "<<addr<<"\t"<<addr->sin_addr.s_addr<<endl
						<<"\taddrlen: "<<addrlen<<"\t("<<sizeof(sockaddr_in)
						<<")"<<endl);
					ret_val = sendto(sd, sndbuf, buflen, flags, 
								  (sockaddr *) addr,addrlen);
					DBG(cerr<<"send_: Buf sent. Retval: "<<ret_val<<endl);
					if(ret_val<0) perror("send_: Error at sendto");
				}
			}
			else
			{
				cerr<<"send_: Sockettype "<<socktype<<" not supported."<<endl;
				errno = EPROTONOSUPPORT;
				ret_val = -1;
			}
	}
	return ret_val;
}

int
back_hton(struct NISt *NI)
{
	if(NI)
	{
		NI->Magic       = htonl(NI->Magic);
		NI->i_NeuronID  = htonl(NI->i_NeuronID);
		NI->i_NeuronLID = htonl(NI->i_NeuronLID);
		NI->i_NIStList  = htonl(NI->i_NIStList);
		NI->i_WeightID  = htonl(NI->i_WeightID);
		NI->i_WeightCnt = htonl(NI->i_WeightCnt);
		NI->i_InputCnt  = htonl(NI->i_InputCnt);
		NI->i_PrErrLID  = htonl(NI->i_PrErrLID);
		NI->i_InitMode  = htonl(NI->i_InitMode);
		NI->i_LearnMode = htonl(NI->i_LearnMode);
		NI->i_FuncMode  = htonl(NI->i_FuncMode);
		NI->i_UseBias   = htonl(NI->i_UseBias);

		return NO_ERROR;
	}

	errno = EINVAL;
	return -1;
}

int 
back_hton(struct RS *R)
{
	if(R)
	{
		R->Magic       = htonl(R->Magic);
		R->ReqID       = htonl(R->ReqID);
		R->RSID        = htonl(R->RSID);
		R->NeuronID    = htonl(R->NeuronID);
		R->AttNIStList = htonl(R->AttNIStList);
		R->RSFollow    = htonl(R->RSFollow);
		R->NIStFollow  = htonl(R->NIStFollow);

		return NO_ERROR;
	}
	errno = EINVAL;
	return -1;
}

int 
back_hton(struct RDSt *RD)
{
	if(RD)
	{
		RD->Magic    = htonl(RD->Magic);
		RD->RSID     = htonl(RD->RSID);
		RD->RSSID    = htonl(RD->RSSID);
		RD->RSFollow = htonl(RD->RSFollow);
		RD->DVSize   = htonl(RD->DVSize);

		return NO_ERROR;
	}
	errno = EINVAL;
	return -1;
}

int
back_hton(struct AS *A)
{
	if(A)
	{
		A->Magic    = htonl(A->Magic);
		A->ReqID    = htonl(A->ReqID);
		A->RSID     = htonl(A->RSID);
		A->Error    = htonl(A->Error);
		A->RSFollow = htonl(A->RSFollow);
		A->NIStFollow = htonl(A->NIStFollow);

		return NO_ERROR;
	}
	errno = EINVAL;
	return -1;
}

int
send_(int sd, struct NISt *NI, unsigned int flags, 
	  sockaddr_in *addr = NULL, int len = 0)
{
	if(NI)
	{
		if(back_hton(NI) < 0) 
		{
			perror("Error at back_hton(NI)");
			errno = EINVAL;
			return -1;
		}
		
		return send_(sd, (char *) NI, RCV_NISt, flags, addr, len);
	}
	else 
	{
		DBG(cerr<<"send_(NISt): Invalid Argument"<<endl);
		errno = EINVAL;
		return -1;
	}
}

int
send_(int sd, struct RS *R, unsigned int flags, 
	  sockaddr_in *addr = NULL, int len = 0)
{
	if(R)
	{
		if(back_hton(R) < 0) 
		{
			perror("Error at back_hton(R)");
			errno = EINVAL;
			return -1;
		}

		return send_(sd, (char *) R, RCV_RS, flags, addr, len);
	}
	else
	{
		DBG(cerr<<"send_(RS): Invalid Argument"<<endl);
		errno = EINVAL;
		return -1;
	}
}

int
send_(int sd, struct RDSt *RD, unsigned int flags, sockaddr_in *addr = NULL,
	  int len = 0)
{
	if(RD)
	{
		if(back_hton(RD) < 0) 
		{
			perror("Error at back_hton(RD)");
			errno = EINVAL;
			return -1;
		}

		return send_(sd, (char *) RD, RCV_RDSt, flags, addr, len);
	}
	else
	{
		DBG(cerr<<"send_(RDSt): Invalid Argument"<<endl);
		errno = EINVAL;
		return -1;
	}
}

int 
send_(int sd, struct AS *A, unsigned int flags, sockaddr_in *addr = NULL,
	  int len = 0)
{
	if(A)
	{
		if(back_hton(A) < 0) 
		{
			perror("Error at back_hton(A)");
			errno = EINVAL;
			return -1;
		}

		return send_(sd, (char *) A, RCV_AS, flags, addr, len);
	}
	else
	{
		DBG(cerr<<"send_(AS): Invalid Argument"<<endl);
		errno = EINVAL;
		return -1;
	}
}

int 
back_ntoh(struct NISt *NI)
{
	if(NI)
	{
		NI->Magic         = ntohl(NI->Magic);
		NI->i_NeuronID    = ntohl(NI->i_NeuronID);
		NI->i_NeuronLID   = ntohl(NI->i_NeuronLID);
		NI->i_NIStList    = ntohl(NI->i_NIStList);
		NI->i_WeightID    = ntohl(NI->i_WeightID);
		NI->i_WeightCnt   = ntohl(NI->i_WeightCnt);
		NI->i_InputCnt    = ntohl(NI->i_InputCnt);
		NI->i_PrErrLID    = ntohl(NI->i_PrErrLID);
		NI->i_InitMode    = ntohl(NI->i_InitMode);
		NI->i_LearnMode   = ntohl(NI->i_LearnMode);
		NI->i_FuncMode    = ntohl(NI->i_FuncMode);
		NI->i_UseBias     = ntohl(NI->i_UseBias);

		return NO_ERROR;
	}
	errno = EINVAL;
	return -1;
}

int
recv_(int sd, struct NISt *NI, unsigned int flags, sockaddr_in *addr = NULL,
	  int *len = NULL, struct timeval *timeout = NULL)
{
	int rec;

	DBG(cerr<<"neuron_serv: recv_ for NISt."<<endl);
	if(NI)
	{
		if((rec = recv_sel(sd, (char *) NI, RCV_NISt, flags, addr, len, 
						   timeout)) > 0) 
		{
			if(back_ntoh(NI) < 0)
			{
				perror("Error at back_ntoh(NI)");
				errno = EINVAL;
				return -1;
			}
			
		}
		
		return rec;
	}
	else
	{
		errno = EINVAL;
		return -1;
	}
}		


int
back_ntoh(struct RS *R)
{
	if(R)
	{
		R->Magic     = ntohl(R->Magic);
		R->ReqID     = ntohl(R->ReqID);
		R->RSID      = ntohl(R->RSID);
		R->NeuronID  = ntohl(R->NeuronID);
		R->AttNIStList = ntohl(R->AttNIStList);
		R->RSFollow  = ntohl(R->RSFollow);
		R->NIStFollow = ntohl(R->NIStFollow);
		
		return NO_ERROR;
	}
	errno = EINVAL;
	return -1;
}

int
recv_(int sd, struct RS *R, unsigned int flags, sockaddr_in *addr = NULL,
	  int *len = NULL, struct timeval *timeout = NULL)
{
	int rec;
	
	DBG(cerr<<"neuron_serv: recv_ for RS."<<endl);
	if(R)
	{
		if((rec = recv_sel(sd, (char *) R, RCV_RS, flags, addr, len, timeout)) 
		   > 0)
		{
			if(back_ntoh(R) < 0)
			{
				perror("Error at back_ntoh(R)");
				errno = EINVAL;
				return -1;
			}
			
		}
		return rec;
	}
	else
	{
		DBG(cerr<<"recv_(...RS...): RS is NULL pointer."<<endl);
		errno = EINVAL;
		return -1;
	}
}

int
back_ntoh(struct RDSt *RD)
{
	if(RD)
	{
		RD->Magic    = ntohl(RD->Magic);
		RD->RSID     = ntohl(RD->RSID);
		RD->RSSID    = ntohl(RD->RSSID);
		RD->RSFollow = ntohl(RD->RSFollow);
		RD->DVSize   = ntohl(RD->DVSize);
		
		return NO_ERROR;
	}
	errno = EINVAL;
	return -1;
}


int
recv_(int sd, struct RDSt *RD, unsigned int flags, sockaddr_in *addr = NULL,
	  int *len = NULL, struct timeval *timeout = NULL)
{
	int rec;
	
	DBG(cerr<<"neuron_serv: recv_ for RDSt."<<endl);
	if(RD)
	{
		if((rec = recv_sel(sd, (char *) RD, RCV_RDSt, flags, addr, len, 
						   timeout)) > 0)
		{
			if(back_ntoh(RD) < 0)
			{
				perror("Error at back_ntoh(RD)");
				errno = EINVAL;
				return -1;
			}
		}
		
		return rec;
	}
	else
	{
		errno = EINVAL;
		return -1;
	}
}

int
back_ntoh(struct AS *A)
{
	if(A)
	{
		A->Magic      = ntohl(A->Magic);
		A->ReqID      = ntohl(A->ReqID);
		A->RSID       = ntohl(A->RSID);
		A->Error      = ntohl(A->Error);
		A->RSFollow   = ntohl(A->RSFollow);
		A->NIStFollow = ntohl(A->NIStFollow);
		
		return NO_ERROR;
			
	}
	errno = EINVAL;
	return -1;
}


int 
recv_(int sd, struct AS *A, unsigned int flags, sockaddr_in *addr = NULL,
	  int *len = NULL, struct timeval *timeout = NULL)
{
	int rec;
	
	DBG(cerr<<"neuron_serv: recv_ for AS."<<endl);
	if(A)
	{
		if((rec = recv_sel(sd, (char *) A, RCV_AS, flags, addr, len, timeout)) 
		   > 0)
		{
			if(back_ntoh(A) < 0)
			{
				perror("Error at back_ntoh(A)");
				errno = EINVAL;
				return -1;
			}
			
		}
		
		return rec;
	}
	else
	{
		errno = EINVAL;
		return -1;
		
	}
}

/** This is actually in planning stage. **/

/*
struct Recv_Buf
{
	int RSID_Cnt;
	int *RSID;
	int *RSSID_Cnt;
	int *RDWait_Cnt;
	int **RSSID;
	int *Com_Size;
	char **Complete;
}

int
is_member(int *list, int len, int elem)
{
	int i;
	
	for(i = 0; i < len; i++)
		if(list[len] == elem) return i;

	return -1;
}	

int
is_member(unsigned long int *list, int len, unsigned long int elem)
{
	int i;
	
	for(i = 0; i < len; i++)
		if(list[len] == elem) return i;

	return -1;
}	

void
apply_new_RS(Recv_Buf *RB, RS *RStruct)
{
	RB->RSID_Cnt++;
	if((RB->RSID = (int *) realloc(RB->RSID, RSID_Cnt * sizeof(int))) == NULL)
	{
		perror("Error at realloc");
		exit(-1);
	}
	if((RB->RSSID = (int **) realloc(RB->RSSID,RSID_Cnt *sizeof(int*)))==NULL)
	{
		perror("Error at realloc");
		exit(-1);
	}
	if((RB->Complete = (char **) realloc(RB->Complete, 
										 RSID_Cnt * sizeof(char **))) == NULL)
	{
		perror("Error at realloc");
		exit(-1);
	}
	
	if((RB->Com_Size = (int *) realloc(RB->Com_Size, RSID_Cnt * sizeof(int)))
	   == NULL)
	{
		perror("Error at realloc");
		exit(-1);
	}

	if((RB->RDWait_Cnt = (int *) realloc(RB->RDWait_Cnt,RSID_Cnt*sizeof(int)))
	   == NULL)
	{
		perror("Error at realloc");
		exit(-1);
	}
	
	
	RB->RSID[RB->RSID_Cnt - 1] = RStruct->RSID;
	RB->RDWait_Cnt[RB->RSID_Cnt - 1] = RStruct->RSFollow;
	
	if(RStruct->RSFollow != FLW_NO)
		if((RB->RSSID[RB->RSID_Cnt - 1] = (int *) calloc(RStruct->RSFollow,
														 sizeof(int))) == NULL)
		{
			perror("Error at calloc");
			exit(-1);
		}
	
	if((RB->Complete[RB->RSID_Cnt - 1] = (char *) calloc(1,sizeof(RS)))== NULL)
	{
		perror("Error at calloc");
		exit(-1);
	}
	
	RB->Com_Size += sizeof(RS);
	
	memcpy(RB->Complete[RB->RSID_Cnt - 1], RStruct, sizeof(RS));
}

void
apply_new_RDSt(Recv_Buf *RB, int pos, RDSt *RDStruct)
{
	RB->RSSID_Cnt[pos]++;
	RB->RDWait_Cnt[pos]--;

	RB->RSSID[pos][RB->RSSIC_Cnt[pos] - 1] = RDStruct->RSSID;
	
	if((RB->Complete[pos] = (char *) realloc(1, RB->Com_Size[pos] + 
											 sizeof(RDSt))) 
	   == NULL)
	{
		perror("Error at realloc");
		exit(-1);
	}
	
	memcpy(RB->Complete[pos] + RB->Com_Size[pos], RDStruct, sizeof(RDSt));
	
	RB->Com_Size += sizeof(RDSt);
}

int
buffered_receive(int sock, int pipewfd)
{
	# Buffered receive that writes complete structures into a 
	   pipe. #

	unsigned long int *address = NULL;
	Recv_Buf *RBuf = NULL;
	int i_Cnt = 0;
	int Mag_Test;
	sockaddr_in FromAddr;
	int len;

	int i,j,k;

	RS RS_Dummy;
	RDSt RDSt_Dummy;
	

	if(recvfrom(sock, &Mag_Test, sizeof(int), MSG_PEEK, &FromAddr, &len)<0)
	{
		perror("Error at recvfrom");
	}
	else
	{
		switch(Mag_Test)
		{
		  case MAG_RS:
			recvfrom(sock, &RS_Dummy, sizeof(RS), 0, &FromAddr, &len);
			
			if((i = is_member(address, i_Cnt, FromAddr.sin_addr.s_addr))
			   >= 0)
			{
				if((j = is_member(RBuf[i].RSID, RBuf[i].RSID_Cnt,
								  RS_Dummy.RSID)) >= 0)
					{
						# We found our RS at RBuf[i].RSID[j]. #
						break;
					}
				else
				{
					# We found the address of our RS, but not itself. #
					
					apply_new_RS(&(RBuf[i]), &RS_Dummy);
				}
			}
			else
			{
				# We found no address like the one we received from. #
				
				i_Cnt++;
				if((RBuf = (Recv_Buf *) realloc(RBuf, 
												i_Cnt * sizeof(Recv_Buf)))
				   == NULL)
				{
					perror("Error at realloc");
					exit(-1);
				}
				memset(RBuf[i_Cnt - 1], 0, sizeof(Recv_Buf));
				
				apply_new_RS(&(RBuf[i_Cnt - 1]), &RS_Dummy);
			}
			break;
		  case MAG_RDSt:
			recvfrom(sock, &RDStDummy, sizeof(RDSt), 0, &FromAddr, &len);
			
			if((i = is_member(address, i_Cnt, FromAddr.sin_addr.s_addr)) >=0)
			{
				if((j = is_member(RBuf[i].RSID, RBuf[i].RSID_Cnt, 
								  RDStDummy.RSID)) >= 0)
				{
					if((k = is_member(RBuf[i].RSSID[j], RBuf[i].RSSID_Cnt,
									  RDStDummy.RSSID)) >= 0)
					{
						# Okay, we found the RDSt. #
						break;
					}
					else
					{
						# We found address and RSID, but not itself. #
						apply_new_RDSt(&(RBuf[i]), j, &RDStDummy);
					}
					
						
			
		}
		

}
*/
