/*nnet.cc
 
  Neural Network as class.

  Author: Marko Meyer
  Date:   13.05.1995 (creation)
  Time-stamp: <96/01/29 10:58:02 mme>

	BACKNET - A library for simulating neural BACKPROPAGATION-Networks
    Copyright (C) 1995	Marko Meyer

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


	If you have any suggestions, ideas or comments regarding this library
	feel free to contact me:

	Email:	mme@pub.th-zwickau.de
	Ordinary Mail:	Marko Meyer
					Teichstrasse 27
					D-08289 Schneeberg
					Germany

	Changes:

	95/12/08 -- Marko Meyer: Fixed a serious problem with Take_Layer.

	95/12/09 -- Marko Meyer: Included int save()-implementation.
*/

#include <nnet.h>

C_NNet::C_NNet(int i_LayerCnt  /*Amount of layers.*/,
			   float f_LearnR  /*Learning Rate.*/,
			   int i_BiasQuest /*Should BIAS be used?*/,
			   char *s_GWIName  /*Filename for GWI-file.*/)
{
	/* This is the constructor. It has to do a lot of very difficult jobs:
	   
	   - Initialization of the ListManager for the NetWeights.
	   - Initialization of the ListManager for the Layers.
	   - Initialization of some lists: LastErrList, OutList, ...
	   - Opening of the weightfiles.
	   - Initialization of all the Layers.

	   We are doing this with a lot of concentration, be aware!*/

	int i_IntError;
	
	DBG(cerr<<"NNet: Entering constructor ..."<<endl);
	i_IntError=NO_ERROR;
	if(i_LayerCnt<1) 
	{
		cerr<<"NNet: i_LayerCnt less than 1 not supported. Exiting ..."<<endl;
		exit(255);
	}
	if((i_BiasQuest!=NO_BIAS)&&(i_BiasQuest!=USE_BIAS))
	{
		cerr<<"NNet: i_Bias is out of range. Exiting ..."<<endl;
		exit(255);
	}
	i_LastLayer=i_LayerCnt;
	s_Net_GWI=s_GWIName;
	f_LearnRate=f_LearnR;
	i_Bias=i_BiasQuest;
	
	
	/* Initialization of the ListManager for the NetWeights.*/

	C_NetLists=(C_List<ELEMTYPE>*)new C_List<ELEMTYPE>();
	
	
	if(i_IntError==NO_ERROR)
	{
		
		/* Creating the ListManager for the layers.*/
		DBG(cerr<<"NNet: Now we have to build up the layers."<<endl);
		
		C_LayStor=(C_List<C_Layer*>*)new C_List<C_Layer*>();
		if(C_LayStor)
			DBG(cerr<<"NNet: C_LayStor is up!"<<endl);
		else i_IntError=ERR_NOLSTR_NET;
	}
	DBG(cerr<<"NNet: Leaving constructor with Error: "<<i_IntError<<endl);
}

int
C_NNet::Init(int *i_ArNeuCnt  /*Array with amount of neurons/layer.*/,
			 S_Fermi_Param *S_FermiData /*Struct with FermiData.*/,
			 int i_InitMode  /*InitMode for neurons.*/)
{
	int i_Error=NO_ERROR;
	C_Layer *C_LastLay;
    S_Layer_Param *S_LastLayData;
	
	i_InputNeu=i_ArNeuCnt[0];
	i_OutputNeu=i_ArNeuCnt[i_LastLayer-1];
	S_FermiD=S_FermiData;

	/* First we count the Neurons ...*/

	if(i_ArNeuCnt)
	{
		i_NeuronCnt=0;
		/* The array is no fake, it's really there, that's fine! */
		for(int i=0;i<i_LastLayer;i++) i_NeuronCnt+=i_ArNeuCnt[i];
	}
	else return ERR_ETYNAR_NET;
	DBG(cerr<<"NNet: Counting Neurons gave: "<<i_NeuronCnt<<endl);
/**/
	/* Creating the lists: */
	i_Error=Create_Lists();
	DBG(cerr<<"NNet: Back in the constructor. Error is: "<<i_Error<<endl);

	if(i_InitMode==INIT_FROM_GWI) i_Error=Open_GWI(s_Net_GWI);
	else i_Error=NO_ERROR;
	if(i_Error==NO_ERROR)
	{
		/*GWI-file has been correctly opened!*/
		/* Now it's time to initialise all the layers.*/
		
		DBG(cerr<<"NNet: We initialize the layers!"<<endl);
		
		i_Error=Init_Layers(C_LayStor,i_ArNeuCnt,f_LearnRate,
							i_InitMode,i_Bias);
				
		if(i_InitMode==INIT_FROM_GWI) I_GWI_File.close();
		/*Now we need some data from the last Layer for further use
		  in the NNet. */
		if(i_Error==NO_ERROR)
		{
			i_Error=C_LayStor->read(i_LayList,i_LastLayer,&C_LastLay);
			if((i_Error==NO_ERROR)&&(C_LastLay))
			{
				S_LastLayData=C_LastLay->hand_lay_data();
				if(S_LastLayData)
				{
					i_OutList=S_LastLayData->i_Output_List;
					i_LastErrList=S_LastLayData->i_Error_List;
				}
				else i_Error=ERR_LDREAD_NET;
			}
		}
		/* Work has been done. */
	}
	return i_Error;
}

int C_NNet::Create_Lists()
{
	/* This one creates all the lists needed by nnet.
	   These are:
	      - Global Inputlist
		  - Global LernOutList
		  - Global LastErrList
	   It uses the netwide accessible ListManager. */

	int i_IntError /*Internal Errorstore.*/;
	
	DBG(cerr<<"NNet: Entering Create_Lists ..."<<endl);
	if(C_NetLists)
	{
		/* The ListMan for the NetWeights is up and running. */
		/* Creating InputList ...*/
		
		i_InList=C_NetLists->create(0,&i_IntError);
		if(i_IntError==NO_ERROR)
		{
			/*InputList is available!*/
			/*Appending lots of elements.*/
			for(int i=1;(i<i_InputNeu)&&(i_IntError==NO_ERROR);i++)
				i_IntError=C_NetLists->append(i_InList,0);
			/*Enough elements appended.*/

			if(i_IntError==NO_ERROR)
			{
				/* Creating the LernOutList ...*/
				i_LernOutList=C_NetLists->create(0,&i_IntError);
				if(i_IntError==NO_ERROR)
				{
					/* LernOutList is available!*/
					/* Appending lots of elements.*/
					for(int i=1;
						(i<i_OutputNeu)&&(i_IntError==NO_ERROR);
						i++)
						
						i_IntError=C_NetLists->append(i_LernOutList,0);
					/* Enough elements appended! */
					if(i_IntError==NO_ERROR)
					{
						/*Creating the LastErrList ...*/
						i_LastErrList=C_NetLists->create(0,&i_IntError);
						if(i_IntError==NO_ERROR)
						{
							/*LastErrList is  available. */
							/*Appending lots of elements.*/
							for(int i=1;(i<i_OutputNeu)&&\
								(i_IntError==NO_ERROR);i++)
								
								i_IntError=C_NetLists->append(i_LastErrList,0);
							/* If no error appeared everything is fine; lets
							   go home! */
							DBG(cerr<<"NNet: Error at the end of Create_Lists:"
								<<i_IntError<<endl);
							
							if(i_IntError==NO_ERROR) return NO_ERROR;
							else return ERR_NOLERR_NET;
						}
						else return ERR_NOLERR_NET;
					}
					else return ERR_NOLROU_NET;
				}
				else return ERR_NOLROU_NET;
			}
			else return ERR_NOILST_NET;
		}
		else return ERR_NOILST_NET;
	}
	else return ERR_NOLMAN_NET;
}


		

int C_NNet::Open_GWI(char *s_Name)
{
	/* This one openes the GWI-file. It uses functions from the C++-stream-
	   library. For information about the format of the GWI-file please
	   refer to the file doc/FILEFORMATS! */

	char *s_GWI_Name;
	char *s_GWI_Ext;
   
	char c_FirstLine[LINELENGTH];
	char cc_EndPoint[LINELENGTH];
	
	int i_GWI_amount=0;
	
	/*Assigning memory.*/
	DBG(cerr<<"NNet: Entering Open_GWI ... "<<endl);
	s_GWI_Name=(char*)new char[strlen(s_Name)+5];
		
	if(s_GWI_Name)
	{
		/*s_GWI_Name exists - building complete GWI-File-Name.*/
		s_GWI_Name=strcpy(s_GWI_Name,s_Name);
		
		/* We check the extension of the file. mme - 09/10 95 */
		s_GWI_Ext=s_GWI_Name + strlen(s_GWI_Name) - strlen(GWI_EXTENSION);
		if(strcmp(s_GWI_Ext,GWI_EXTENSION) != 0)
			s_GWI_Name=strcat(s_GWI_Name,GWI_EXTENSION);

		/*Opening the file.*/

		I_GWI_File.open(s_GWI_Name,ios::in);

		if(I_GWI_File.good()&&I_GWI_File.rdbuf()->is_open!=0) 
		{ 
			I_GWI_File.getline(c_FirstLine,LINELENGTH);
			/* There may be comment-lines as first lines, we have to 
			   detect this. */
			while(c_FirstLine[0]=='#') 
				I_GWI_File.getline(c_FirstLine,LINELENGTH);
			i_GWI_amount=(int)strtoul(c_FirstLine,(char**)&cc_EndPoint,0);
			/*if((*c_FirstLine!='\0')&&(**cc_EndPoint=='\0')) \
				return ERR_READWG_NET;*/
			if(i_GWI_amount!=i_NeuronCnt) return ERR_GWICOL_NET;
			DBG(cerr<<"NNet: Just fine: No Error!"<<endl);
			return NO_ERROR;
		}
		else return ERR_GWINOP_NET;
	}
	else return ERR_MEMORY_NET;
}


int C_NNet::Init_Layers(C_List<C_Layer*> *C_LStore,int *i_ArNeuCnt,
						float f_LearnRate,int i_InitMode,int i_Bias)
{
	/* This one initializes all the layers. */

	C_Layer *C_NewLayer,*C_PrevLayer;
	int i_IntError;

	DBG(cerr<<"NNet: Entering Init_Layers ..."<<endl);
	if(C_LStore)
	{

		/*First we create a list that contains all the layers.*/
		
		i_LayList=C_LStore->create(NULL,&i_IntError);
		
		if(i_IntError==NO_ERROR)
		{
			/* The amount of layers to be created is given  in the parameters
			   to the constructor. */
			C_PrevLayer=NULL;
			for(int i=1;i<=i_LastLayer;i++)
			{
				C_NewLayer=(C_Layer*)new C_Layer(C_NetLists,
												 i_ArNeuCnt[i-1],
												 C_PrevLayer,
												 (i==1)?i_InList:0,
												 f_LearnRate,
												 i_Bias);
				if(!C_NewLayer) return ERR_MEMORY_NET;
				i_IntError=C_NewLayer->Init(S_FermiD, i_InitMode, &I_GWI_File);
				if(i_IntError==NO_ERROR)
				{
					if(i==1)i_IntError=C_LStore->write(i_LayList,i,C_NewLayer);

					else i_IntError=C_LStore->append(i_LayList,C_NewLayer);
					if(i_IntError==NO_ERROR) C_PrevLayer=C_NewLayer;
					else return i_IntError;
				}
				else return i_IntError;
			}
			DBG(cerr<<"NNet: Leaving Init_Layers without errors."<<endl);
			/* Now we've got it! */
			return NO_ERROR;
		}
		else return i_IntError;
	}
	else return ERR_NOLSTR_NET;
}

int 
C_NNet::Action(int i_Action /*Determines, what action.*/,
	   char *s_Name /*Prename of files.*/,
	   int *i_Steps /*If i_Action==ACTN_LEARN number of learning steps,
					  otherwise unused.*/)
{
	/* This is the really active part of the program.
	   i_Action determines, whether the Net is learned or recalled.
	   The prename of the MST-file is given. It is used similar as prename
	   of the LRN-file and of the OUT-file. 
	   
	   So this one has to do some things:
	    - at first - open the MST-file (it is needed in both cases of i_Action)
		- decide by i_Action - Recall or Learn
		- in case of Learn: 
		    - read some pattern from the MST-file
			- start recalling by calling 'Action' of C_Layer
			- read the appropriate pattern from the LRN-file
			- calculate difference between the resulting pattern and the
			  appropriate pattern read from the LRN-file
			- start learning by calling 'Action' of C_Layer
			- read a new pattern from the MST-file
			  and so on ...
		- in case of Recall:
		    - read a pattern from the MST-file
			- start recalling by calling 'Action' of C_Layer
			- write the result to the OUT-file
			- read  a new pattern from the MST-file
			  and so on ...

	   This is really a lot, so we've to concentrate on that once again ...*/

	char *s_MST_Name;
	int i_IntError=NO_ERROR;
	int i_MST_Cnt;

	DBG(cerr<<"NNet: Entering Action"<<endl);
	DBG(cerr<<"NNet: Given name is: "<<s_Name<<endl);
	/* --- rsc: 95/12/17 --- */
	i_Kind_Of_Action=i_Action;
	
	if(s_Name)
	{
		
		/* Opening of MST-file. */
		/* Preparing the name ...*/
		
		s_MST_Name=(char*)new char[strlen(s_Name)+5];
		s_MST_Name=strcpy(s_MST_Name,s_Name);
		s_MST_Name=strcat(s_MST_Name,MST_EXTENSION);

		DBG(cerr<<"NNet: MST-Name is: "<<s_MST_Name<<endl);
		/* Calling of Open_MST(...) ... */

		i_IntError=Open_MST(s_MST_Name,&i_MST_Cnt);
		
		if(i_IntError==NO_ERROR)
		{
			/* File was correctly opened and fits with the net. */
			/* Now it's time to decide, what to do ... */
			
			switch(i_Action)
			{
			  case ACTN_LEARN:
				i_IntError=Learn(s_Name,i_Steps,i_MST_Cnt);
				break;
			  case ACTN_RECAL:
				i_IntError=Recall(s_Name,i_MST_Cnt);
				break;
			  default:
				i_IntError=ERR_WRACTN_NET;
			}
			I_MST_File.close();
			DBG(cerr<<"NNet: Leaving Action with Error: "<<i_IntError<<endl);
			return i_IntError;
		}
		else return i_IntError;
	}
	else return ERR_NONAME_NET;
}

int 
C_NNet::Open_MST(char *s_Name, int *i_MST)
{
	char c_I_Neurons[LINELENGTH],c_MST_Cnt[LINELENGTH],c_Endptr[1];
	int i_I_Neurons;

	DBG(cerr<<"NNet: Entering Open_MST."<<endl);
	/*This one opens the MST-file and does some tests on it. */
	if(s_Name&&i_MST)
	{
		/* A name is given, just open the file ...*/

		I_MST_File.open(s_Name,ios::in);
		
		if(I_MST_File.good()&&I_MST_File.rdbuf()->is_open!=0)
		{
			/*The file is really open! */
			/*Checking collisions between NNet and MST-file ... */
			
			I_MST_File.getline(c_I_Neurons,LINELENGTH);
			if(I_MST_File.good())
			{
				/*There may be comment-lines at the beginning, we have to
				  detect this.*/

				while(c_I_Neurons[0]=='#') 
					I_MST_File.getline(c_I_Neurons,LINELENGTH);

				c_Endptr[0]='\0';
				i_I_Neurons=(int)strtoul(c_I_Neurons,NULL,0);
				DBG(cerr<<"NNet: i_I_Neurons is "<<i_I_Neurons<<endl);
				if(i_I_Neurons==i_InputNeu)
				{
					I_MST_File.getline(c_MST_Cnt,LINELENGTH);
					if(I_MST_File.good())
					{
						c_Endptr[0]='\0';
						*i_MST=(int)strtoul(c_MST_Cnt,NULL,0);
						DBG(cerr<<"NNet: Leaving Open_MST without errors."
							<<endl);
						return NO_ERROR;
					}
					else return ERR_READMS_NET;
				}
				else return ERR_MSTCOL_NET;
			}
			else return ERR_READMS_NET;
		}
		else return ERR_NOIMST_NET;
	}
	else return ERR_NONAME_NET;
}

int
C_NNet::Take_Layer(int i_Num)
{
	/* This one bums the Layer number i_Num into C_AktLayer and returns 
	   an ErrorValue. --mme*/

	/*  THIS IS BROKEN. FIXME. --mme*/

	
	return C_LayStor->read(i_LayList,i_Num,&C_AktLayer);
}

int
C_NNet::learn()
{
	int i_IntError=NO_ERROR;
	
	/* Do a learning cycle. (Exactly one.) */
	/*Do a learn over all the layers. Remember the different count! */
	for(int k=i_LastLayer;k>=1;k--)
	{
		if(C_LayStor)   /* Take_Layer is broken. FIXME  --mme*/
		{
			C_LayStor->read(i_LayList,k,&C_AktLayer);
			i_IntError=C_AktLayer->\
				Action(ACTN_LEARN);
			if(i_IntError!=NO_ERROR)
				return i_IntError;
		}
		else return ERR_TAKLAY_NET;
	}
	return i_IntError;
}


int C_NNet::Learn(char *s_Name,int *i_Steps,int i_MST)
{
	/* This one does the learning:
	     - open the LRN-file - using s_MSTName as prename - for reading
		 - open the ERR-file - using s_MSTName as prename - for writing
		 - prepare a random order for the patterns to be learned
		 - and then, for all layers, for all learning steps:
		    - call Action of all the layers with 'Recall'.
			- write Error to ERR-file.
			- call Action of all the layers with 'Learn'. */

	int i_IntError=NO_ERROR;
	int i_LRN_Cnt;
	int *i_PatternArray;
	int i_AktStep;
	

	DBG(cerr<<"NNet: Entering learn ..."<<endl);
	if(s_Name&&i_Steps)
	{
		/* Name and Steps exist, so let's begin. */
		/*Preparing the names...*/
		
		/* Opening LRN-File. */
		i_IntError=Open_LRN(s_Name,&i_LRN_Cnt);
		
		if(i_IntError==NO_ERROR)
		{
			if(i_LRN_Cnt==i_MST)
			{
				/*Opening the ERR-file. */
				i_IntError=Open_ERR(s_Name);
				if(i_IntError==NO_ERROR)
				{
					/* Both files correctly opened. */
					/* Just now we have to find a random order for the 
					   patterns. This time I see the patterns as numbered
					   from 1 to i_MST (it should be the same as i_LRN_Cnt,
					   otherwise the program would never come to this point).*/

					i_PatternArray=(int*)new int[i_MST];
					if(i_PatternArray==NULL) return ERR_MEMORY_NET;
					for (int i=0;i<i_MST;i++) i_PatternArray[i]=i+1;

					/* Now we have to do the main-loop. 
					   It takes all patterns, runs over all learning steps or
					   until an error occures. */
					
					i_AktStep=*i_Steps;
					
					while((i_AktStep>0)&&(i_IntError==NO_ERROR))
					{
						/* At first: randomizing the patterns. */
						i_IntError=Randomize_Patterns(i_PatternArray,i_MST);
						if(i_IntError!=NO_ERROR) return i_IntError;
						
						/* Over all the patterns ...*/
						for(int i=0;(i<i_MST)&&(i_IntError==NO_ERROR);i++)
						{
							/*Load the MST-pattern ... */
							i_IntError=Load_Pattern\
								(LDPAT_MST,i_PatternArray[i]);
							
							if(i_IntError==NO_ERROR)
							{
								/* Do a recall over all layers. */
								recal();
								
								/* Load the LRN-DATA */
								i_IntError=Load_Pattern\
									(LDPAT_LRN,i_PatternArray[i]);
								if(i_IntError==NO_ERROR)
								{
									/* Calculate difference between datas.*/
									i_IntError=Detect_Error();
									if(i_IntError!=NO_ERROR) return i_IntError;
								}
								else return i_IntError;
								
								/* Do a learning step. */
								learn();
							}
							else return i_IntError;
						}
						/* At last: decrementing i_AktStep. */
						i_AktStep--;
					}
					/* The ERR-file can be closed now. */
					O_ERR_File.close();
				}
				else return ERR_NOOERR_NET;
			}
			else return ERR_BCOLIS_NET;
		
			/* The LRN-file can be closed now. */
			I_LRN_File.close();
		}
		else return ERR_NOILRN_NET;
	}
	else return ERR_NONAME_NET;

	DBG(cerr<<"NNet: Leaving learn without errors!"<<endl);
	return NO_ERROR;
	/* This is the final point! */
}

int C_NNet::Open_LRN(char *s_Name,int *i_LRN)
{
	char c_I_Neurons[LINELENGTH],c_LRN_Cnt[LINELENGTH],c_EndPtr[1];
	int i_I_Neurons,i_LRN_Cnt;
	char *c_LRN_Name=NULL;

	DBG(cerr<<"NNet: Entering Open_LRN ..."<<endl);
	if(s_Name&&i_LRN)
	{
		/* The name is given, now open the File. */

		c_LRN_Name=(char*)new char[strlen(s_Name)+5];
		c_LRN_Name=strcpy(c_LRN_Name,s_Name);
		c_LRN_Name=strcat(c_LRN_Name,LRN_EXTENSION);
		DBG(cerr<<"NNet: LRN-Name is: "<<c_LRN_Name<<endl);
		
		I_LRN_File.open(c_LRN_Name,ios::in);
		if(I_LRN_File.good()&&I_LRN_File.rdbuf()->is_open!=0)
		{
			/* The file is opened. */
			/* Reading first entry. */
			I_LRN_File.getline(c_I_Neurons,LINELENGTH);
			if(I_LRN_File.good())
			{
				/*There may be comment-lines at the beginning, we have
				  to detect this.*/

				while(c_I_Neurons[0]=='#')
					I_LRN_File.getline(c_I_Neurons,LINELENGTH);
				
				c_EndPtr[0]='\0';
				/*Expecting int-value. */
				i_I_Neurons=(int)strtoul(c_I_Neurons,NULL,0);

				if(i_I_Neurons==i_OutputNeu)
				{
					/* NNet and LRN-file fits, so inquire the learn-amount.*/
					I_LRN_File.getline(c_LRN_Cnt,LINELENGTH);
					if(I_LRN_File.good())
					{
						c_EndPtr[0]='\0';
						i_LRN_Cnt=(int)strtoul(c_LRN_Cnt,NULL,0);
						*i_LRN=i_LRN_Cnt;
						DBG(cerr<<"NNet: Leaving Open_LRN without errors."
							<<endl);
						
						return NO_ERROR;
					}
					else return ERR_READLR_NET;
				}
				else return ERR_LRNCOL_NET;
			}
			else return ERR_READLR_NET;
		}
		else return ERR_NOILRN_NET;
	}
	else return ERR_NONAME_NET;
}

int C_NNet::Open_ERR(char *s_Name)
{
	
	/* This one opens the ERR-File for writing. */

	char *s_ReName;
	char *c_ERR_Name=NULL;
	char *c_ERR_Ext;
	
	DBG(cerr<<"NNet: Entering Open_ERR ..."<<endl);
	if(s_Name)
	{
		/* Maybe there's an old ERR-File left. Let's check that and save it
		   for better times. */

		c_ERR_Name=(char*)new char[strlen(s_Name)+5];
		c_ERR_Name=strcpy(c_ERR_Name,s_Name);

		/* We check the extension of the file. mme - 09/10 95 */
		c_ERR_Ext = c_ERR_Name + strlen(c_ERR_Name) - strlen(ERR_EXTENSION);
		if(strcmp(c_ERR_Ext,ERR_EXTENSION) != 0)
			c_ERR_Name=strcat(c_ERR_Name, ERR_EXTENSION);

		DBG(cerr<<"NNet: ERR-Name is: "<<c_ERR_Name<<endl);

		O_ERR_File.open(c_ERR_Name,ios::app);
		if(O_ERR_File.tellp()!=0)
		{
			/* There's really an old ERR-File. */
			/* We close the stream and move the old one to a new name. */
			O_ERR_File.close();
			s_ReName=(char*)new char[strlen(c_ERR_Name)+5];
			if(s_ReName)
			{
				s_ReName=strcpy(s_ReName,c_ERR_Name);
				s_ReName=strcat(s_ReName,".bak");
			    rename(c_ERR_Name,s_ReName);
				delete s_ReName;
			}
		}
		O_ERR_File.close();

		/* The name is given, now open the file. */
		O_ERR_File.open(c_ERR_Name,ios::out);
		if(O_ERR_File.good()&&O_ERR_File.rdbuf()->is_open!=0)
		{
			/*The file is open, now writing the 'header'.*/
			
			O_ERR_File<<"# "<<c_ERR_Name<<endl;
			O_ERR_File<<"# Errorfile written by "<<NAME_ID<<" "<<VERSION_NR<<
				endl;
			O_ERR_File<<"# "<<endl;
			if(O_ERR_File.good()) return NO_ERROR;
			return ERR_WRTEER_NET;
		}
		else return ERR_NOOERR_NET;
	}
	else return ERR_NONAME_NET;
}

 int C_NNet::Randomize_Patterns(int *i_PatArr,int i_MST)
{
	/* This one randomizes the given pattern array.
	   Here is the algorithm:
	   1. Invite an array of int's [i_MST] long, that stores the pattern-index
          [e.g.: array[0]=1 array[1]=2 ...]
	   2. Set i_MaxRand to i_MST-1.
	   3. Draw a random int (=i_AktMST) between 0 and i_MaxRand.
	   4. Exchange the entry nr. i_MaxRand and the entry nr. i_AktMST.
	   5. Decrement i_MaxRand.
	   6. If i_MaxRand > 0 goto 3, otherwise you're through. */
	
	/* Step 1 is done outside */
	


	int i_AktMST,i_Exchange;
	

	DBG(cerr<<"NNet: Entering Randomize_Patterns"<<endl);
	/* Step 2,6,5 */
	for(int i_MaxRand=i_MST-1;i_MaxRand>0;i_MaxRand--)
	{
		/* Step 3 */
		i_AktMST=rand()*i_MaxRand/RAND_MAX;
		
		/* Step 4 */
		if(i_AktMST!=i_MaxRand)
		{
			i_Exchange=i_PatArr[i_MaxRand];
			i_PatArr[i_MaxRand]=i_PatArr[i_AktMST];
			i_PatArr[i_AktMST]=i_Exchange;
		}
	}
	/* Now we're through. i_PatternArray contains now  a
	   random sequence of Pattern-indizes. */	   
	return NO_ERROR;
}


 int C_NNet::Load_Pattern(int i_Where,int i_Num)
{
	/*This one loads the by i_Num specified pattern from the MST-stream
	  into the appropriate list. It does not check for correct values. It
	  thinks that the MST-file contains x patterns with i_InputNeu values
	  each. It takes them from I_MST_File and doesn't check it to be open.
	  (Although it checks for I_MST_File.good(), but that's another thing.)*/
	
	int i_DataSetPos;
	char c_TestLine[LINELENGTH],c_Inp_Line[LINELENGTH],c_EndPtr[1];
	ELEMTYPE Inp_Val;
	ifstream *I_Appr_File;
	int i_Appr_List;
	int i_Appr_Cnt;
	int i_Appr_ERR;
	int i_IntError;
	int i;
	
	
	
	DBG(cerr<<"NNet: Entering Load_Pattern"<<endl);
	/*Determining what stream, list and how much. */

	switch(i_Where)
	{
	  case LDPAT_MST:
		DBG(cerr<<"NNet: Decided for loading from MST-File."<<endl);
		I_Appr_File=&I_MST_File;
		i_Appr_List=i_InList;
		i_Appr_Cnt=i_InputNeu;
		i_Appr_ERR=ERR_READMS_NET;
		break;
	  case LDPAT_LRN:
		DBG(cerr<<"NNet: Decided for loading from LRN-File."<<endl);
		I_Appr_File=&I_LRN_File;
		i_Appr_List=i_LernOutList;
		i_Appr_Cnt=i_OutputNeu;
		i_Appr_ERR=ERR_READLR_NET;
		break;
	  default:
		return ERR_WRWHER_NET;
	}
	
	/* Now we have to find the position of the appropriate data-set.
	   Because every data set is separated by a line with at least one '#',
	   we have to search for that for so much times we have to load the 
	   pattern. */
	
	I_Appr_File->clear();
	I_Appr_File->seekg(0,ios::beg);
	
	/* This has been the first bug I found after international release:
	   there can be comment lines at the begin of the file, therefore we
	   have to wind forward. */
	I_Appr_File->getline(c_TestLine,LINELENGTH);
	while((c_TestLine[0]=='#')&&(I_Appr_File->good())) 
		I_Appr_File->getline(c_TestLine,LINELENGTH);
	
	i_DataSetPos=0;
	DBG(cerr<<"NNet: We have to search for: "<<i_Num<<endl);
	
	while(i_DataSetPos<i_Num)
	{
		I_Appr_File->getline(c_TestLine,LINELENGTH);
		DBG(cerr<<"NNet: c_TestChar is: "<<c_TestLine<<endl);
		if(!I_Appr_File->good()) return i_Appr_ERR;
		if(strchr(c_TestLine,'#')!=NULL) i_DataSetPos++;
	}

	/* The appropriate position is reached. */
	/* Now reading i_InputNeu values into the InputList. */
	DBG(cerr<<"NNet: DataSetPos is: "<<i_DataSetPos<<endl);
	
	i=1;
	while(i<=i_Appr_Cnt)
	{
		if(!I_Appr_File->good()) return i_Appr_ERR;
		I_Appr_File->getline(c_Inp_Line,LINELENGTH);
		if(c_Inp_Line[0]!='#')
		{
			c_EndPtr[0]='\0';
			Inp_Val=strtod(c_Inp_Line,NULL);
			DBG(cerr<<"NNet: Read value: "<<Inp_Val<<endl);
			i_IntError=C_NetLists->write(i_Appr_List,i,Inp_Val);
			if(i_IntError!=NO_ERROR) return i_IntError;
			i++;
		}
		else DBG(cerr<<"NNet: Found '#' !"<<endl);
	}
	DBG(cerr<<"NNet: Leaving Load_Pattern without Errors."<<endl);
	/* We are ready now. */
	return NO_ERROR;
}

 int C_NNet::Detect_Error()
{
	/* This one detects the error between the LernOutList's values and
	   the values of the OutList - the real OutList-values, produced by the
	   very last Layer of the NNet.
	   Then it calculates the error, writes it to the LastErrList and makes
	   a remark in the I_ERR_File. */

	ELEMTYPE Lern_val,Pres_val,Det_Err,Cum_Err /*,Rel_Err*/;
	int i_IntError;

	/* First: detecting and writing Error. */

	DBG(cerr<<"NNet: Entering Detect_Error ..."<<endl);
	Cum_Err=0;
	
	for(int i=1;i<=i_OutputNeu;i++)
	{
		i_IntError=C_NetLists->read(i_LernOutList,i,&Lern_val);
		if(i_IntError==NO_ERROR)
		{
			i_IntError=C_NetLists->read(i_OutList,i,&Pres_val);
			if(i_IntError==NO_ERROR)
			{
				Det_Err=Lern_val-Pres_val;
				Cum_Err+=Det_Err;
				i_IntError=C_NetLists->write(i_LastErrList,i,Det_Err);
				if(i_IntError!=NO_ERROR) return i_IntError;
			}
			else return i_IntError;
		}
		else return i_IntError;
	}

	/* Then making the remark in the I_ERR_File. 
	Rel_Err=fabs(Cum_Err/(ELEMTYPE)i_OutputNeu);
	O_ERR_File<<Rel_Err<<endl;
	if(O_ERR_File.good()) return NO_ERROR;
	else return ERR_WRTEER_NET; */
	return NO_ERROR;
}

int 
C_NNet::Recall(char *s_Name,int i_MST)
{
	/* This one does the Recall:
	     - open the OUT-file - using s_MSTName as prename - for writing
		 - for all patterns:
		   - do a recall for all the layers
		   - write result of last layer to OUT-file. */

	int i_IntError;

	DBG(cerr<<"NNet: Entering recall ..."<<endl);
	if(s_Name)
	{
		/*Opening the OUT-file for writing.*/
		
		i_IntError=Open_OUT(s_Name,i_MST);
		
		if(i_IntError==NO_ERROR)
		{
			/* Doing the main loop. */
			
			for(int i=1;i<=i_MST;i++)
			{
				/* Loading the MST. */
				i_IntError=Load_Pattern(LDPAT_MST,i);
				
				/* Do the recall. */
				recal();

				/*Okay, the result should be in the OutList.*/
				i_IntError=Write_OUT();
				if(i_IntError!=NO_ERROR) return i_IntError;
			}
			O_OUT_File.close();
		}
		else return i_IntError;
	}
	else return ERR_NONAME_NET;
	DBG(cerr<<"NNet: Leaving Recall"<<endl);
	
	return NO_ERROR;
}

int
C_NNet::recal()
{
	/* This one does exactly one recall, from first to last layer. The
	   result can be found in the OutputList.*/

	int i_IntError=NO_ERROR;

	/* Doing the recall. */
	for(int j=1;j<=i_LastLayer;j++)
	{
		/*Catching the layer.*/
		if(C_LayStor)  /* new Take_Layer --mme */
		{
			C_LayStor->read(i_LayList,j,&C_AktLayer);
			i_IntError=C_AktLayer->Action(ACTN_RECAL);
			if(i_IntError!=NO_ERROR) return i_IntError;
		}
		else return ERR_TAKLAY_NET;
	}
	return i_IntError;
}


int C_NNet::Open_OUT(char *s_Name,int i_MST)
{
	/* This one opens the OUT-File. The conditions are very hard: Any old
	   OUT-File with the same name is overwritten. The problem is: A new
	   OUT-File of the same name can be a sign for a new attempt to solve
	   the problem. Therefore, the USER must have had attention on the old
	   file and hopefully has saved it.
	   Maybe further versions will bring some news to that.
	   The format of the OUT-File is not that hard than the format of the
	   MST or LRN file. We've got some rows in the beginning for advertising
	   (officially let's call it 'user information' :-)) as comments, then
	   two important lines: the number of Neurons at OutputLayer and the 
	   number of patterns being represented here.
	   Then there's the first separator for the patterns: A '#'. But this 
	   one is written by Write_OUT, we don't care. */

	/*Okay, we open it now ... */
	
	char *s_OUT_Name;
	char *s_OUT_Ext;

	DBG(cerr<<"NNet: Entering Open_OUT ..."<<endl);
	if(s_Name)
	{
		/* Preparing the name ...*/
		s_OUT_Name=(char*)new char[strlen(s_Name)+5];
		if(s_OUT_Name)
		{
			s_OUT_Name=strcpy(s_OUT_Name,s_Name);

			/* We check the extension of the file. mme - 09/10 95 */
			s_OUT_Ext = s_OUT_Name + strlen(s_OUT_Name) - 
				strlen(OUT_EXTENSION);
			if(strcmp(s_OUT_Ext,OUT_EXTENSION) != 0)
				s_OUT_Name=strcat(s_OUT_Name,OUT_EXTENSION);
			
			O_OUT_File.open(s_OUT_Name,ios::out);
			if(O_OUT_File.good()&&(O_OUT_File.rdbuf()->is_open!=0))
			{
				/* The file is now open. We write the 'user information'.*/
				O_OUT_File<<"# "<<s_OUT_Name<<endl;
				O_OUT_File<<"# Output file. Written by: "<<NAME_ID<<" "\
					<<VERSION_NR<<endl;
				O_OUT_File<<"#"<<endl;
				/* Then we write the Header-Data: */
				O_OUT_File<<i_OutputNeu<<endl;
				O_OUT_File<<i_MST<<endl;
				
				/*Now the first pattern will follow sometimes ...*/
				if(O_OUT_File.good()) return NO_ERROR;
				else return ERR_WRTOUT_NET;
			}
			else return ERR_NOOOUT_NET;
		}
		else return ERR_NONAME_NET;
	}
	else return ERR_NONAME_NET;
}

int C_NNet::Write_OUT()
{
	/* This one writes the OutList to the O_OUT_File. */

	/* The format of the OUT-File is much better than the format of the
	   MST- or LRN- File. It's clear: We only write into the OUT-File,
	   the USER has to do the reading himself, so all the problems, that
	   accompanying strange comment-lines and so forth are on his side.
	   We only give some small comments: each Output-Pattern is separated by
	   a '#'. This character must be present on the first column of a row and
	   may be followed by any text 'til the end of the row (marked by '\n').
	*/
	
	ELEMTYPE Out_Val;
	int i_IntError;

	DBG(cerr<<"NNet: Entering Write_OUT ..."<<endl);
	if(O_OUT_File.good())
	{
		/* Writing the separator: */
		O_OUT_File<<"#"<<endl;
		
		/* Doing a loop for writing the OutList. */
		for(int i=1;i<=i_OutputNeu;i++)
		{
			/*Taking the value.*/
			i_IntError=C_NetLists->read(i_OutList,i,&Out_Val);
			if(i_IntError==NO_ERROR)
			{
				/* Writing the value.*/
				O_OUT_File<<Out_Val<<endl;
				if(!O_OUT_File.good()) return ERR_WRTOUT_NET;
			}
			else return i_IntError;
		}
		/* Values are written. Going home. */
		return NO_ERROR;
	}
	else return ERR_WRTOUT_NET;
}

int
C_NNet::save()
{
	/* This method is for being replaced by methods of the inheriting 
	   classes! */

	int i_Error=NO_ERROR;

	/*Now calling Save of all the Layers.*/
	
	if(C_LayStor)
	{
		for(int i=1;(i<=i_LastLayer)&&(i_Error==NO_ERROR);i++)
		{
			/*---rsc:95/12/13 I would say that it should be C_AktLayer
			  and not C_AktLay---*/
			i_Error=C_LayStor->read(i_LayList,i,&C_AktLayer);
			
			DBG(cerr<<"NNet: Error at reading ActLayer for Save is:"
				<<i_Error<<endl);
			/*---rsc:95/12/13 And here it should be C_AktLayer too*/
			if(i_Error==NO_ERROR) i_Error=C_AktLayer->Save(&O_GWI_Save);
		}
	}
	else i_Error=ERR_NOLSTR_NET;
	return i_Error;
}



int C_NNet::Save()
{
	/* This one saves everything according to the Net. */
	/* This is the class-independent method. It should be used from all
	   classes without any doubt. --mme*/

	char *s_Save_GWI, *s_Real_GWI;
	char *s_Save_Ext, *s_Real_Ext;
	
	/* C_Layer *C_ActLay; -- changed by rsc 95/12/02 
	LAYERTYPE *C_ActLay; */
	
	
	int i_Error;

	s_Real_GWI=new char[strlen(s_Net_GWI)+5];
	
	s_Real_GWI=strcpy(s_Real_GWI,s_Net_GWI);
	
	/* We check the extension of the file.  mme - 09/10 1995 */
	s_Real_Ext = s_Real_GWI + strlen(s_Real_GWI) - strlen(GWI_EXTENSION);
	if(strcmp(s_Real_Ext, GWI_EXTENSION) != 0)
		s_Real_GWI=strcat(s_Real_GWI,GWI_EXTENSION);

	/*Testing, whether GWI-File exists or not. */
	
	O_GWI_Save.open(s_Real_GWI,ios::app);
	if(O_GWI_Save.good()&&O_GWI_Save.rdbuf()->is_open)
	{
		/*GWI File already exists, so save it. */
		O_GWI_Save.close();
		
		s_Save_GWI=new char[strlen(s_Net_GWI)+5];
		s_Save_GWI=strcpy(s_Save_GWI,s_Net_GWI);
		
		/* We check the extension of the file. mme - 09/10 1995 */
		s_Save_Ext = s_Save_GWI + strlen(s_Save_GWI) - strlen(GWI_EXTENSION);
		if(strcmp(s_Save_Ext, GWI_EXTENSION))
		{
			s_Save_Ext = 0;
			s_Save_GWI=strcat(s_Save_GWI,GWB_EXTENSION);
		}
		rename(s_Real_GWI,s_Save_GWI);
	}
	O_GWI_Save.open(s_Real_GWI,ios::out);
	if(O_GWI_Save.good()&&O_GWI_Save.rdbuf()->is_open)
	{
		/* Write some useful information ;-] */
		O_GWI_Save<<"# Weight-file written by "<<NAME_ID<<" "<<VERSION_NR
			<<endl;
		/* Write the amount of neurons. */
		O_GWI_Save<<i_NeuronCnt<<endl;
		i_Error=NO_ERROR;
	}
	else i_Error=ERR_CRTSAV_NET;
	
	if(i_Error==NO_ERROR) i_Error=save();

	return i_Error;
}


C_NNet::~C_NNet()
{
	C_Layer *C_ActLayer;
	int i_RetVal;

	/*The destructor of the whole net saves the weights. */
	Save();
	if(C_LayStor)
	{
		for(int i=1;i<=i_LastLayer;i++)
		{
			i_RetVal=C_LayStor->read(i_LayList,i,&C_ActLayer);
			DBG(cerr<<"NNet: Error at reading ActLayer for Destroying:"
				<<i_RetVal<<endl);
			if((C_ActLayer)&&(i_RetVal==NO_ERROR)) delete C_ActLayer;
			/* --- rsc changed on 95/12/02 --- */
		}
		delete C_LayStor;
	}
	
}

