/* nnet_fast.cc
 
   Author: Marko Meyer
   Date:   10.08.1995 (creation)
   Time-stamp: <95/12/17 13:00:02 rsc>

   Fast Network class.
   In difference to C_NNet C_FNNet loads the patterns from the patternfile
   into the memory when opening the patternfiles. It does not provide an
   errorfile at all. 

	BACKNET - A library for simulating neural BACKPROPAGATION-Networks
    Copyright (C) 1995	Marko Meyer

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


	If you have any suggestions, ideas or comments regarding this library
	feel free to contact me:

	Email:	mme@pub.th-zwickau.de
	Ordinary Mail:	Marko Meyer
					Teichstrasse 27
					D-08289 Schneeberg
					Germany

	Changes:
	
	95/12/08 -- Marko Meyer: Fixed Take_Layer in parent class, only 
	            commented here; fixed #include - problem for D*S
*/

#include <dosbug.h>
#include __NN_FAST__

C_FNNet::C_FNNet(int i_LayerCnt  /*Amount of layers.*/,
				 float f_LearnRate  /*Learning Rate.*/,
				 int i_Bias /*Should BIAS be used?*/,
				 char *s_GWIName  /*Filename for GWI-file.*/)
:C_NNet(i_LayerCnt, f_LearnRate, i_Bias, s_GWIName)
{
	/*The constructor is the same as in C_NNet, so we don't have to
	  do additional things.*/
}


/* The changes are not very big: we only change five methods:

	virtual int Open_MST(char *s_Name, int *i_MST);
	virtual int Open_LRN(char *s_Name, int *i_LRN);
	virtual int Open_ERR(char *s_Name);
	virtual int Load_Pattern(int i_Where, int i_Num);
	virtual int Detect_Error();
*/

int
C_FNNet::Open_MST(char *s_Name, int *i_MST)
{
   /* To avoid reinventions of any kind we first call the method 
	  Open_MST(...) of C_NNet. It will open the file and read in some
	  important values, as the Number of Patterns being contained within
	  the patternfiles. And it will do some important tests: checking the
	  amount of pattern-data with the amount of input neurons. */

	int i_MSTCnt, i_IntError,i,j;
	char c_FileLine[LINELENGTH];
	ELEMTYPE value;
	
	DBG(cerr<<"FNNet: First calling NNet!"<<endl);
	
	i_IntError = C_NNet::Open_MST(s_Name,&i_MSTCnt);

	DBG(cerr<<"FNNet: NNet returned: "<<i_IntError<<endl);
	
	if(i_IntError == NO_ERROR)
	{
		/* The pattern file is now open and some values are already read
		   in. We begin with the loops. */
		if((Patterns = (ELEMTYPE**) new ELEMTYPE*[i_MSTCnt])!=NULL)
		{
			/* We allocated the memory for the rows with the 
			   pointers to the Pattern-data-fields.
			   Now we can do a loop over amount of patterns stored
			   in the file.*/
			DBG(cerr<<"FNNet: Allocated Patternrows."<<endl);
			
			
			for(i=0; i < i_MSTCnt; i++)
			{
				DBG(cerr<<"FNNet: i-Loop. i is: "<<i<<endl);
				if(I_MST_File.good() && (I_MST_File.rdbuf()->is_open != 0))
				{
					/* The file is open and good. Now we must check, where the 
					   filepointer stands. Theoretically there must be a # 
					   within the next line. */
					
					DBG(cerr<<"FNNet: I_MST_File is open and good!"<<endl);
					
					I_MST_File.getline(c_FileLine,LINELENGTH);
					
					if(c_FileLine[0] == '#')
					{
						/* Okay, the # is present, we are on the right 
						   position. Now we can allocate the memory for the 
						   pattern array. */
						DBG(cerr<<"FNNet: Found a #. Good."<<endl);
						
						
						/* First we have to allocate memory for the data
						   of the i-th pattern. */
						
						if((Patterns[i] = (ELEMTYPE*) new ELEMTYPE[i_InputNeu])
						   != NULL)
						{
							/* The memory is allocated. Now we can start to
							   read the patterns. */
							DBG(cerr<<"FNNet: Allocated memory for Patterns."
								<<endl);
							DBG(cerr<<"FNNet: j-loop. Counting j."<<endl
								<<"FNNet:\t");
							
							for(j=0; j < i_InputNeu; j++)
							{
								DBG(cerr<<". ");
								if(!I_MST_File.good()) return ERR_READMS_NET;
								I_MST_File.getline(c_FileLine,LINELENGTH);
								if(c_FileLine[0] == '#') return ERR_READMS_NET;
								value=strtod(c_FileLine,NULL);
								Patterns[i][j] = value;
							}
							DBG(cerr<<endl);
							
						}
						else return ERR_MEMORY_NET;
					}
					else return ERR_READMS_NET;
				}
				else return ERR_NOIMST_NET;
			}
		}
		else return ERR_MEMORY_NET;
	}
	else return i_IntError;
	*i_MST=i_MSTCnt;
	return NO_ERROR;
}

int
C_FNNet::Open_LRN(char *s_Name, int *i_LRN)
{
	/* Like in Open_MST(...) we call the inherited method first.
	   Then we load the LearnPatterns into the LearnPattern array. */
	
	int i_LRNCnt, i_IntError, i, j;
	char c_FileLine[LINELENGTH];
	ELEMTYPE value;
	
	DBG(cerr<<"FNNet: Entering Open_LRN; Calling NNet first."<<endl);

	i_IntError = C_NNet::Open_LRN(s_Name,&i_LRNCnt); 

	DBG(cerr<<"FNNet: NNet's method returned: "<<i_IntError<<endl);
	
	/* If there was no error, we continue by testing the filestatus.*/
	if(i_IntError==NO_ERROR)
	{
		if((LearnPatterns=(ELEMTYPE**)new ELEMTYPE*[i_LRNCnt])!=NULL)
		{
			/* Memory allocated going on to allocate the memory 
			   for the columns. */
			
			DBG(cerr<<"FNNet: Allocated pattern memory stage 1"<<endl);
			
			for(i=0; i < i_LRNCnt; i++)
			{
				DBG(cerr<<"FNNet: i-loop. i is: "<<i<<endl);
				
				if(I_LRN_File.good() && (I_LRN_File.rdbuf()->is_open != 0))
				{
					/* The status is OK, we look for the correct position. */
					DBG(cerr<<"FNNet: LRN File is open and good."<<endl);
					
					I_LRN_File.getline(c_FileLine,LINELENGTH);
					if(c_FileLine[0] == '#')
					{
						/*The position is okay, too. Now we can allocate the 
						  memory. */
						DBG(cerr<<"FNNet: Found a #. Good."<<endl);
						
						if((LearnPatterns[i]=
							(ELEMTYPE*)new ELEMTYPE[i_OutputNeu]) != NULL)
						{
							DBG(cerr<<"FNNet: Allocated pattern memory stage 2"
								<<endl);
							
							for(j=0; j < i_OutputNeu; j++)
							{
								DBG(cerr<<"FNNet: j-loop. j is: "<<j<<endl);
								if(!I_LRN_File.good()) return ERR_READLR_NET;
								I_LRN_File.getline(c_FileLine, LINELENGTH);
								if(c_FileLine[0] == '#') return ERR_READLR_NET;
								value=strtod(c_FileLine, NULL);
								LearnPatterns[i][j]=value;
							}
							
						}
						else return ERR_MEMORY_NET;
					}
					else return ERR_READLR_NET;
				}
				else return ERR_NOILRN_NET;
			}
		}
		else return ERR_MEMORY_NET;
	}
	else return i_IntError;
	*i_LRN = i_LRNCnt;
	return NO_ERROR;
}

int 
C_FNNet::Open_ERR(char *s_Name)
{
	/* This one once was used to provide a file with the errors occured
	   during learning. Therefore much time is needed and a lot of space
	   as well. I'm working on a solution with other kinds of storage
	   (i.e. named pipes) but this is architecture dependend and has to
	   be discussed before doing so. */
	
	/* For this time we leave this empty. */

	s_Name=NULL;
	return NO_ERROR;
}


int
C_FNNet::Load_Pattern(int i_Where, int i_Num)
{
	/* This substitutes the old Load_Pattern(...) method completely.
	   The reason is obvious: the old method worked on a file, that
	   was specified by the flag i_Where. The new one works on one
	   of the two PatternArrays, again specified by i_Where. The new
	   one is faster; this is to be proven. */

	ELEMTYPE **Appr_Store;
	int i_Appr_List;
	int i_Appr_Amnt;
	int i;
	
	DBG(cerr<<"FNNet: Entered Load_Pattern"<<endl);

	switch(i_Where)
	{
	  case LDPAT_LRN:
		Appr_Store = LearnPatterns;
		i_Appr_List = i_LernOutList;
		i_Appr_Amnt = i_OutputNeu;
		break;
	  case LDPAT_MST:
		Appr_Store = Patterns;
		i_Appr_List = i_InList;
		i_Appr_Amnt = i_InputNeu;
		break;
	  default:
		return ERR_WRWHER_NET;
	}
	
	/* Now we can start. */
	DBG(cerr<<"FNNet: Loading Pattern "<<i_Num<<endl);
	DBG(cerr<<"FNNet: Loading fields: "<<endl<<"FNNet:\t");
	for(i=1; i <= i_Appr_Amnt; i++)
	{
		DBG(cerr<<". ");
		C_NetLists->write(i_Appr_List,i,Appr_Store[i_Num-1][i-1]);
	}
	DBG(cerr<<endl<<"FNNet: Finished loading"<<endl);
	return NO_ERROR;
}

int 
C_FNNet::Detect_Error()
{
	/* This one has to be done completely new because we have to erase the
	   writing to the error file. */

	ELEMTYPE Lern_val,Pres_val,Det_Err,Cum_Err;
	int i_IntError;

	/* First: detecting and writing Error. */

	DBG(cerr<<"NNet: Entering Detect_Error ..."<<endl);
	Cum_Err=0;
	
	for(int i=1;i<=i_OutputNeu;i++)
	{
		i_IntError=C_NetLists->read(i_LernOutList,i,&Lern_val);
		if(i_IntError==NO_ERROR)
		{
			i_IntError=C_NetLists->read(i_OutList,i,&Pres_val);
			if(i_IntError==NO_ERROR)
			{
				Det_Err=Lern_val-Pres_val;
				Cum_Err+=Det_Err;
				i_IntError=C_NetLists->write(i_LastErrList,i,Det_Err);
				if(i_IntError!=NO_ERROR) return i_IntError;
			}
			else return i_IntError;
		}
		else return i_IntError;
	}

	return NO_ERROR;
}

/* The other methods just call the parent methods. */

int 
C_FNNet::Learn(char *s_Name, int *i_Steps, int i_MST)
{
	return C_NNet::Learn(s_Name, i_Steps, i_MST);
}

int 
C_FNNet::learn()
{
	return C_NNet::learn();
}


int 
C_FNNet::Recall(char *s_Name, int i_MST)
{
	return C_NNet::Recall(s_Name, i_MST);
}

int
C_FNNet::recal()
{
	return C_NNet::recal();
}

int 
C_FNNet::Create_Lists()
{
	return C_NNet::Create_Lists();
}

int 
C_FNNet::Open_GWI(char *s_Name)
{
	return 	C_NNet::Open_GWI(s_Name);
}

int 
C_FNNet::Init_Layers(C_List<C_Layer*> *C_LStore, int *i_ArNeuCnt,
					 float f_LearnRate, int i_InitMode, int i_Bias)
{
	return C_NNet::Init_Layers(C_LStore, i_ArNeuCnt, f_LearnRate, i_InitMode, 
							   i_Bias); 
}

int 
C_FNNet::Open_OUT(char *s_Name, int i_MST)
{
	return C_NNet::Open_OUT(s_Name, i_MST);
}

int 
C_FNNet::Randomize_Patterns(int *i_PatArr, int i_MST)
{
	return C_NNet::Randomize_Patterns(i_PatArr, i_MST);
}

int 
C_FNNet::Write_OUT()
{
	return C_NNet::Write_OUT();
}

int 
C_FNNet::Action(int i_Action /*Determines, what action.*/,
				char *s_Name /*Prename of files.*/,
				int *i_Steps /*If i_Action==ACTN_LEARN number of learning
							   steps, otherwise unused.*/)
{
	return C_NNet::Action(i_Action, s_Name,	i_Steps);
}

int
C_FNNet::Take_Layer(int i_Num)
{
	/* new prototype ! --mme */
	return C_NNet::Take_Layer(i_Num);
}

int
C_FNNet::Save()
{
	return C_NNet::Save();
}

int
C_FNNet::save()
{
	return C_NNet::save();
}

C_FNNet::~C_FNNet()
{
}





