//:ts=3  -*- C++ -*-

// Simulation of an FDDI-Tokenring
// mz, Stephan Mehrholz (cg)

// CNCL Includes
#include <iostream.h>
#include <CNCL/QueueFIFO.h>
#include <CNCL/sim.h>
#include <CNCL/FiboG.h>
#include <CNCL/NegExp.h>
#include <CNCL/Moments.h>
#include <CNCL/Batches.h>
#include <CNCL/DiscUniform.h>
#include <CNCL/String.h>

#ifdef EZDDISP
#include <CNCL/EZDQueue.h>
#include <CNCL/EZDWindow.h>
#include <CNCL/EZDPushButton.h>
#include <CNCL/EZDText.h>
#include <CNCL/EZDTimer.h>
EZDWindow *win;        // main window for animation
EZDDrawing *draw;      // drawing -> win
EZDWindow *info_win;   // window for information
EZDDrawing *info_draw; // drawing -> info_win
EZDText *ezdst;        // simulation time in [ms]
EZDText *ezdpc;        // packet counter
long ezd_PL=10;        // default pixel width of a packet!
#endif//EZDDISP

// C++ Includes
#include <stdlib.h>
#include <GetOpt.h>

// Local Include
//#include "Job.h"
#include <CNCL/Job.h>

bool ezd = TRUE;       // TRUE if animation is recommended
bool thton = FALSE;    // TRUE if Token Holding Timer is recommended
bool trton = FALSE;    // TRUE if Token Rotation Timer is recommended

void usage(void)
{
 cerr << "This is the CNCL FDDI Simulation\n"
      << "Usage: fddi -d -h [-l] [-s] [-L] [-Q] [-T]\n\n"
      << "option \tparameters\tcorresponding action\n"
      << "\t(mandatory)\n"
      << "------ \t----------\t--------------------\n"
      << "  -d  \t\t\tchoose animation with ezd on the X display\n"
      << "\t\t\t\tdefault: no animation\n"
      << "  -h  \t\t\tdisplay this message\n"
      << "  -l \t<double>  \tchange the load (load > 0)\n"
      << "\t\t\t\tdefault: RHO = 0.3\n"
      << "  -s \t<integer> \t# of stations in the ring, (1..30)\n"
      << "\t\t\t\tdefault: NSTATION = 10\n"
      << "  -t \t<double>  \tchange the ttrt\n"
      << "\t\t\t\tdefault: TTRT = 0.004 s\n"
      << "  -L \t<double>  \tchange the avarage packet length\n"
      << "\t\t\t\tdefault: PL = 5000 Bit\n"
      << "  -Q \t<integer> \tchange the length of all queues, (1..100)\n"
      << "\t\t\t\tdefault: QL = 10\n"
      << "  -T \t<string>  \tactivate Timer \tTHT with <THT>, TRT with <TRT>\n"
      << "\t\t\t\t\tTHT and TRT with <ALL>\n"
      << "\t\t\t\tdefault: no Timer display\n"
      << endl;
 exit (0);
}

// variables for FDDI, ring parameter
// ----------------------------------
long    NSTATION = 10;          // default # of stations
long          QL = 10;          // default Queue Length
CNSimTime   TTRT = 0.004;       // default Target Token Rotation Timer [s]
double        CM = 200000000.0; // propagation speed of the Media [m/s]
double        RV = 100000000.0; // Ring Velocity (Transferrate) [Bit/s]
double        RL = 100000.0;    // Ring Length [m]
double        TL = 88.0;        // Token Length [Bit]
double        PL = 5000.0;      // default Paket Length [Bit]
double      SLAT = 60.0;        // Station LATency [Bit]
double        CP = 1.0;         // coefficient of variation
double      RLAT = 0.0;         // RingLATency
double TOKENTIME = 0.0;         // TIME for TOKEN from Station [i] to [i+1]
double    LAMBDA;               // packet generating rate
double       RHO = 0.3;         // default load


// variables for Simulation
// ------------------------
enum { EV_JOB, EV_INIT, EV_TOKEN, EV_SEND }; // CNEvent types for simulation
int aktstation = 0;             // # of active station
long global_packet_counter = 0; // counts # of all packets

CNRNG *rng;                     // base random generator
CNBatches *bat_waiting_time;    // batch means evaluation waiting time
ofstream bat_outfile;           // file for Batch Means data
double theo_waiting_time=0.0;   // theoretical waiting time

// Array of <names> for EZD Objects, (hack)
CNString names[] = {
 "q01", "q02", "q03", "q04", "q05", "q06", "q07", "q08", "q09", "q10",
 "q11", "q12", "q13", "q14", "q15", "q16", "q17", "q18", "q19", "q20",
 "q21", "q22", "q23", "q24", "q25", "q26", "q27", "q28", "q29", "q30"
};


// CNClasses
// --------------
class Station : public CNEventHandler
{
private:
  int       no;                    // station #
  CNJob       *job;                  // served job
  CNQueueFIFO queue;               // CNQueue
  CNRandom  &rnd_packet_length;    // distribution of service time b
  CNRandom  &rnd_packet_rate;      // distribution of packet arrival
  CNMoments mom_waiting_time;      // evaluation  waiting time
  CNMoments mom_service_time;      //             service time
  CNMoments mom_rotation_time;     //            rotation time
  CNMoments mom_packet_rate;       //            packet_rate
  enum {ST_WAITING, ST_SERVING };
  CNSimTime ttrt;                  // Target Token Rotation Timer
  CNSimTime trt;                   // Token Rotation Timer
  CNSimTime tht;                   // Token Holding Timer
  CNSimTime ta;                    // packet arrival time
  long      late_count;             
  long      packet_counter; 
  long      lost_packet_counter;
  CNSimTime rotation_time;         // time one trip around the ring
  CNSimTime ring_time;             // ring_time-last_ring_time yields
  CNSimTime last_ring_time;        //                rotation_time
#ifdef EZDDISP
  EZDQueue  *ezdq;
  EZDText   *ezdl;                 // # of packets in queue printed 
  EZDText   *ezdlpc;               // # of lost packets 
  EZDText   *ezdt;                 // show token at the station
  EZDText   *ezdlc;                // show late_count
  EZDTimer  *ezdtimtht;            // THTimer
  EZDTimer  *ezdtimtrt;            // TRTimer
#endif//EZDDISP
public:
#ifdef EZDDISP
  EZDText   *ezdtor;               // show token on the ring
#endif//EZDDISP
  inline void rotation_time_check();
  inline void trt_check(CNSimTime delay);
  virtual void event_handler(const CNEvent *ev);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
  void eval_job(CNJob *job);
  void print_results();
  Station(CNRandom &rnd1, CNRandom &rnd2, int ii) : 
          job(NIL),
          rnd_packet_length(rnd1), 
          rnd_packet_rate(rnd2), 
          mom_waiting_time("waiting time T_w"), 
          mom_service_time("service time T_s"),
          mom_rotation_time("rotation time T_c"),
          mom_packet_rate("packet rate T_A")
  {
    state(ST_WAITING);             // default state() of all stations
    no = ii;                       // identifikation # of the station
    // initialize all trt, as if the token had been send through an empty ring
    // e.g. #0 has NSTATION*TOKENTIME, #9 has 1*TOKENTIME
    trt  = (NSTATION-ii)*TOKENTIME;
    ttrt = TTRT;                   // global Value TTRT

    trt=0.0;            
    tht=0.0;            
    late_count=0;
    packet_counter=0;
    lost_packet_counter = 0;
    rotation_time=0.0;  
    ring_time=0.0;      
    last_ring_time=0.0; 
    ta = 0.0;
#ifdef EZDDISP
    if(ezd)
    {
      ezdq   = new EZDQueue(names[ii],      100, 150+ii*36, 100, 10, 30);
      ezdt   = new EZDText(names[ii]+"t",   233, 144+ii*36,      "");
      ezdl   = new EZDText(names[ii]+"l",   210, 138+ii*36,      "");
      ezdlpc = new EZDText(names[ii]+"lpc",  50, 144+ii*36,      "");
      ezdtor = new EZDText(names[ii]+"tor", 260, 144+36/2+ii*36, "");
      ezdlc  = new EZDText(names[ii]+"lc",  270, 144+ii*36,      "");
      if(thton) 
        ezdtimtht = new EZDTimer(names[ii]+"timtht",295, 150+ii*36, 8, 8);
      if(trton)
        ezdtimtrt = new EZDTimer(names[ii]+"timtrt",325, 150+ii*36, 8, 8);
      // draw the queues in the EZDWindow
      ezdq->redraw();
      ezdq->length(0);
      ezdq->color("DarkOrchid1");
      ezdl->set_text_val ("", 0);
      ezdst->set_text_val ("", 0);
      ezdt->set_text("| ");
      if(thton)
      {
        ezdtimtht->active_color("red");
        ezdtimtht->redraw();
      }
      if(trton)
        ezdtimtrt->redraw();
      // draw conection to the ring
      EZD::draw_line(242, 150+ii*36, 262, 150+ii*36, "black");
    }
#endif//EZDDISP
  }
};

// array of pointer of Station 
Station **stations;

#ifdef EZDDISP
bool ezd_step_flag = TRUE;

void create_info_win()
{
  info_win = new EZDWindow("FDDI-INFO", 300, 300);
  info_draw = new EZDDrawing("INFO-DRAW");
  info_win->overlay(info_draw);
  info_draw->set(); // info_draw is active drawing
  EZD::draw_text(0, 0, 300, 30, "center center", "FDDI Information",
                 "", "-*-times-bold-r-*-*-20-*-*-*-*-*-*-*");
  new EZDPushButton ("CLOSE-INFO", 249, 279, 50, 20, "Close");

  EZDText t01("t01",05, 30, "");
  t01.set_text_val("                    Stationsanzahl:", int(NSTATION));
  EZDText t02("t02",05, 45, "");
  t02.set_text_val("        Uebertragungsrate [Mbit/s]:",RV);
  EZDText t03("t03",05, 60, "");
  t03.set_text_val("Ausbreitungsgeschwindigkeit [km/s]:",CM/1000);
  EZDText t04("t04",05, 75, "");
  t04.set_text_val("                  Kabellaenge [km]:",RL/1000);
  EZDText t05("t05",05, 90, "");
  t05.set_text_val("              Stationslatenz [Bit]:",SLAT);
  EZDText t06("t06",05,105, "");
  t06.set_text_val("                  Ringlatenz [Bit]:",RLAT);
  EZDText t07("t07",05,120, "");
  t07.set_text_val("                 Tokenlaenge [Bit]:",TL);
  EZDText t08("t08",05,135, "");
  t08.set_text_val("  Mittelwert der Paketlaenge [Bit]:",PL);
  EZDText t09("t09",05,150, "");
  t09.set_text_val("             Variationskoeffizient:",CP);
  EZDText t10("t10",05,165, "");
  t10.set_text_val("             Warteschlangenplaetze:",int(QL));
  EZDText t11("t11",05,180, "");
  t11.set_text_val("   Target Token Rotation Time [ms]:",TTRT);
  EZDText t12("t12",05,195, "");
  t12.set_text_val("                                  :",LAMBDA);
  EZDText t13("t13",05,210, "");
  t13.set_text_val("                                  :",RHO);
  EZD::draw_text( 145,192, 20, 20, "right center", "l","",
		  "-adobe-symbol-*-*-*-*-12-*-*-*-*-*-*-*");
  EZD::draw_text( 145,207, 20, 20, "right center", "r","",
		  "-adobe-symbol-*-*-*-*-12-*-*-*-*-*-*-*");

  EZDText t14("t14",05,240, "");  
  t14.set_text_val("              T_w mean (theo) [ms]:",
		   theo_waiting_time * 1000);

  draw->set(); // main draw is active drawing
}

void delete_info_win()
{
  delete info_win;
  delete info_draw;
  info_win=0;
  info_draw=0;
}

void ezd_input()
{
  if(ezd)
  {
    if(ezd_step_flag || EZD::test_event())
    {
      CNString in;
      in = EZD::event();
      if(strstr(in, " EXIT "))
        exit(0);
      if(strstr(in, " STEP "))
        ezd_step_flag = TRUE;
      if(strstr(in, " RUN "))
        ezd_step_flag = FALSE;
      if(strstr(in, " INFO "))
        create_info_win();
      if(strstr(in, " CLOSE-INFO "))
        delete_info_win();
    }
  }
}
#endif//EZDDISP

// member function for Station
// ---------------------------
void Station::rotation_time_check()
{
  last_ring_time = ring_time;
  ring_time = now();
  rotation_time = ring_time - last_ring_time;
  mom_rotation_time.put(rotation_time);
}

void Station::trt_check(CNSimTime delay)
{
  for(int z=0;z<NSTATION;z++)
  {
    stations[z]->trt += delay;
    if(stations[z]->trt > stations[z]->ttrt)
    {
      stations[z]->late_count++;
      stations[z]->trt = 0.0;
#ifdef EZDDISP
      if((ezd) && (stations[z]->late_count!=0))
      {
        stations[z]->ezdlc->set_text("L");
      }
#endif//EZDDISP
    } // if(trt>ttrt)..
#ifdef EZDDISP
    if((ezd)&&(trton))
    {
     stations[z]->ezdtimtrt->set_angle(int(stations[z]->trt/stations[z]->ttrt*360));
    }
#endif//EZDDISP
  } // for ..
}

void Station::event_handler(const CNEvent *ev)
{
  CNSimTime ts;  // service time


#ifdef EZDDISP
  // simulation control 
  if(ezd) ezd_input();
#endif//EZDDISP

  switch(state())
  {
    case ST_WAITING:
    {
      // station is waiting for the TOKEN
      switch(ev->type())
      {
        case EV_INIT:
        {
          // send EV_JOB TO SELF
          ta = rnd_packet_rate();
          mom_packet_rate.put(ta);
          send_delay(new CNEvent(EV_JOB, stations[no], new CNJob), ta);
        }
        break; // EV_INIT

        case EV_JOB:
        // Incoming Job, put into queue
        {
          packet_counter++;
          if(queue.length() < QL)
          {
            CNJob *job;
            job = (CNJob *)ev->object();
            job->in = now();
            queue.put(job);
#ifdef EZDDISP
            if(ezd)
            {
              ezdq->length( queue.length()*ezd_PL );
              ezdl->set_text_val("", queue.length());
              ezdst->set_text_val("", now());
            }
#endif//EZDDISP
          }
          else // queue is full!
          {
            lost_packet_counter++;
#ifdef EZDDISP
            if(ezd)
            {
              ezdlpc->set_text_val("%",
                          int(1.0*lost_packet_counter/packet_counter*100.0));
              ezdst->set_text_val("", now());
            }
#endif//EZDDISP
          }
          // generate packets, until Batch Means Error is satisfied
            global_packet_counter++;
#ifdef EZDDISP
            if(ezd)
              ezdpc->set_text_val("",int(global_packet_counter));
#endif//EZDDISP
            // send EV_JOB TO SELF
            ta = rnd_packet_rate();
            mom_packet_rate.put(ta);
            send_delay(new CNEvent(EV_JOB, stations[no], new CNJob), ta);
        }
        break; // EV_JOB

        case EV_TOKEN:
          // Station gets TOKEN, has right to send!
          {
            // check the rotation time of the token
            rotation_time_check();
#ifdef EZDDISP
            if(ezd)
            {
              ezdst->set_text_val("", now());
              // (no+NSTATTION-1)%NSTATION returns always Station (n-1) !!
              stations[(no+NSTATION-1)%NSTATION]->ezdtor->set_text(" ");    
              ezdt->set_text("--");    // 020993, mz: oder "[]"
            }
#endif//EZDDISP
            // set trt=0.0 if(late_count==0)
            if(late_count==0)
            {
              tht=trt; // Token holding timer = token rotation timer
              trt=0.0; // token rotation timer = NULL
            }
            // Send first paket!
            // packets in queue ??
            if((!queue.empty()) && (late_count==0))
            {
              job = (CNJob *)queue.get();
              job->start = now();
              job->length = rnd_packet_length();
              // CNRandom service time
              ts = (1.0 * job->length)/RV;          // [bit] / [bit/s] = [s]
              // tht + service time
              tht +=ts;
              // CNMoments for service time
              mom_service_time.put(ts);
              trt_check(ts);
              send_delay(new CNEvent(EV_SEND), ts);
#ifdef EZDDISP
              if(ezd)
              {
                ezdq->length( queue.length()*ezd_PL );
                ezdl->set_text_val("", queue.length());
                ezdst->set_text_val("", now());
                if(thton)
                {
                  ezdtimtht->set_angle(int(tht/ttrt*360));
                  ezdtimtht->activate();
                }
              }
#endif//EZDDISP

              state(ST_SERVING);
            }
            else
            {
              // if late_count != 0 or !queue.empty()
              // even if the late_count is zero and is set to zero again!
              late_count = 0;

              // queue empty, send TOKEN to stations[aktstation+1]
              // TOKENTIME is the time for the TOKEN to reach next station!
              aktstation = (aktstation+1) % NSTATION;
              trt_check(TOKENTIME);
              send_delay(new CNEvent(EV_TOKEN, stations[aktstation],0.0), TOKENTIME);
#ifdef EZDDISP
              if(ezd)
              {
                ezdlc->set_text(" ");
                ezdst->set_text_val("", now());
                ezdt->set_text("| ");
                ezdtor->set_text("*");
                if(thton)
                  ezdtimtht->set_angle(int(tht/ttrt*360));
              }
#endif//EZDDISP
              // change state()
              state(ST_WAITING);
            }
          }
          break; // EV_TOKEN

        case EV_SEND:
          {
            error("Station: ", "Event 'EV_SEND' in state(ST_WAITING) NOT allowed!");
          }
          break; // EV_SEND

        default:
          error("Station: ", "illegal event in event_handler(), state ST_WAITING");
          ev->print();
          break;
        } // switch(ev->type() in state ST_WAITING
    } // ST_WAITING
    break;


    case ST_SERVING:
    {
    // station has TOKEN, sending until ( (tht>ttrt) !! (queue.empty()) )
      switch(ev->type())
      {
        case EV_INIT:
        {
            error("Station: ","Event 'EV_INIT' in state(ST_SERVING) NOT allowed!");
        }
        break; // EV_INIT

        case EV_JOB:
        {
          // Incoming Job, if(queue.length() < QL) put into queue
          packet_counter++;
          if(queue.length() < QL)
          {
            CNJob *job;
            job = (CNJob *)ev->object();
            job->in = now();
            queue.put(job);
#ifdef EZDDISP
            if(ezd)
            {
              ezdq->length( queue.length()*ezd_PL );
              ezdl->set_text_val("", queue.length());
              ezdst->set_text_val("", now());
            }
#endif//EZDDISP
          }
          else // queue is full! packet get's lost ..
          {
            lost_packet_counter++;
#ifdef EZDDISP
            if(ezd)
            {
              ezdlpc->set_text_val("%",
                         int(1.0*lost_packet_counter/packet_counter*100.0));
              ezdst->set_text_val("", now());
            }
#endif//EZDDISP
          }
            global_packet_counter++;
#ifdef EZDDISP
            if(ezd)
              ezdpc->set_text_val("",int(global_packet_counter));
#endif//EZDDISP
            // send EV_JOB TO SELF
            ta = rnd_packet_rate();
            mom_packet_rate.put(ta);
            send_delay(new CNEvent(EV_JOB, stations[no], new CNJob), ta);
        }
        break; // EV_JOB

        case EV_TOKEN:
          {
            error("Station: ","Event 'EV_TOKEN' in state(ST_SERVING) NOT allowed!");
          }
          break; // EV_TOKEN

        case EV_SEND:
          // station is sending until queue empty   [ OR tht>ttrt ]
          {
            // transmitted job
            job->out = now();
            // Evaluate job
            eval_job(job);
            delete job;
            job = NIL;
        
            if((!queue.empty()) && (tht < ttrt))
            {
              job = (CNJob *)queue.get();
              job->start = now();
              job->length = rnd_packet_length();
              // CNRandom service time
              ts = (1.0 * job->length)/RV;         // [bit] / [bit/s] = [s]
              // tht + service_time
              tht += ts;
              // CNMoments for service time
              mom_service_time.put(ts);
              trt_check(ts);
              send_delay(new CNEvent(EV_SEND), ts);
#ifdef EZDDISP
              if(ezd)
              {
                ezdq->length( queue.length()*ezd_PL );
                ezdl->set_text_val("", queue.length());
                ezdst->set_text_val("", now());
                if(thton)
                  ezdtimtht->set_angle(int(tht/ttrt*360));
              }
#endif//EZDDISP
            }
            else
            {
              // queue empty, send TOKEN to stations[aktstation+1]
              // TOKENTIME is the time for the TOKEN to reach next station!
              aktstation = (aktstation+1) % NSTATION;
              trt_check(TOKENTIME);
              send_delay(new CNEvent(EV_TOKEN, stations[aktstation],0.0), TOKENTIME);
#ifdef EZDDISP
              if(ezd)
              {
                ezdst->set_text_val("", now());
                ezdt->set_text("| ");
                ezdtor->set_text("*");
                if(thton)
                {
                  ezdtimtht->set_angle(int(tht/ttrt*360));
                  ezdtimtht->deactivate();
                }
              }
#endif//EZDDISP
              // change state() 
              state(ST_WAITING);
            }
          }
          break; // EV_SEND

        default:
          error("Station: ", "illegal event in state ST_SERVING");
          break;
      } // switch(ev->type() in state ST_SERVING
    } // ST_SERVING
    break;

    default:
    {
      error("Station: ", "illegal state in event_handler(const CNEvent *ev) ");
      cout << get_state();
    }
    break;
  } // switch(state())
}

void Station::print(ostream &) const
{
}

void Station::dump(ostream &strm) const
{
  strm << "Station { state=" << state() << " no=" << no << " }" << endl;
}

void Station::eval_job(CNJob *job)
{
    bat_waiting_time->put(job->start - job->in);
    mom_waiting_time.put(job->start - job->in);
}

void Station::print_results()
{
    cout << "--- SERVER --- " << no << "\n" 
         << "lost packets: "  << lost_packet_counter << "\n"
         << mom_rotation_time << "\n"
         << mom_waiting_time  << "\n"
         << mom_service_time  << endl;
}


main(int argc, char **argv)
{
  // parse commandline arguments
  GetOpt getopt(argc, argv, "dhl:s:t:L:Q:T:");
  int    opt;
	
  while ((opt = getopt()) != EOF)
    switch(opt)
      {
      case 'd':
        ezd=TRUE;
        break;
	   case 'h':
        usage();
        break;
      case 'l':
        RHO = atof(getopt.optarg);
        if(RHO==0.0) usage();
        break;
      case 's':
        NSTATION = atoi(getopt.optarg);
        if((NSTATION==0)||(NSTATION>30)) usage();
        break;
      case 't':
        TTRT = atof(getopt.optarg);
        break;
      case 'L':
        PL = atof(getopt.optarg);
        break;
      case 'Q':
        QL = atoi(getopt.optarg);
#ifdef EZDDISP
        ezd_PL = 100/QL;
#endif//EZDDISP
        if((QL==0)||((ezd)&&(QL>100))) usage();
        break;
	   case 'T':
        if((CNString(getopt.optarg)=="THT")||(CNString(getopt.optarg)=="tht"))
          thton = TRUE;
        if((CNString(getopt.optarg)=="TRT")||(CNString(getopt.optarg)=="trt"))
          trton = TRUE;
        if((CNString(getopt.optarg)=="ALL")||(CNString(getopt.optarg)=="all"))
        {
          thton = TRUE;
          trton = TRUE;
        }
        break;
	   }

  cout << "Type fddi -h <RETURN> for help on fddi\n\n"
       << "Parameter\n~~~~~~~~~\n"
       << "     ezd=" << (  ezd==TRUE ? "True" : "False")<<"\n"
       << "   thton=" << (thton==TRUE ? "True" : "False")<<"\n"
       << "   trton=" << (trton==TRUE ? "True" : "False")<<"\n"
       << "     RHO=" << RHO << "\n"
       << "NSTATION=" << NSTATION << "\n" 
       << "    TTRT=" << TTRT << "\n"
       << "      PL=" << PL << "\n"
       << "      QL=" << QL << "\n"
#ifdef EZDDISP
       << " ezd_PL=" << ezd_PL << "\n"
#endif//EZDDISP
       << "\nstarting simulation! ...\n"
       << endl;

  // compute values depending on the parameters of the Ring
  LAMBDA    = RHO / (PL/RV * NSTATION);
  RLAT      = RL/CM * RV  +  NSTATION * SLAT;
  TOKENTIME = RL / (NSTATION * CM)  +  TL / RV;
  // compute theoretical waiting time
  theo_waiting_time = ((RLAT+NSTATION*TL)/RV*(1-RHO/NSTATION))/(2*(1-RHO))+
      (RHO*(CP*CP+1)*PL/RV)/(2*(1-RHO)); // theoretical waiting time in [s]
  cout << "   LAMBDA=" << LAMBDA << "\n"
       << "     RLAT=" << RLAT << "\n"
       << "TOKENTIME=" << TOKENTIME << "\n"
       << "      T_w=" << theo_waiting_time << "\n"
       << endl;
       

  rng   = new CNFiboG;
  CNNegExp rnd_packet_rate(1.0/LAMBDA, rng);        // RANDOM # of packet arrival
  CNNegExp rnd_packet_length(1.0 * PL, rng);        // RANDOM # of packet length
  if(ezd)
   bat_waiting_time=new CNBatches(0.0,0.004,0.5,10,100,20,95,"waiting time");
  else
   bat_waiting_time=new CNBatches(0.0,0.005,0.2,10,10000,20,95,"waiting time");
// bat_waiting_time=new CNBatches(0.0,0.0025,0.02,10,300000,20,95,"waiting time");
                    // (BOTTOM, TOP, ERROR, NOG, SOG, NOI, CONF, NAME)	
  ofstream bat_outfile("fddi.btm", ios::out);


#ifdef EZDDISP
  if(ezd)
  {
    win = new EZDWindow("FDDI", 800, 600);
    draw = new EZDDrawing("MAIN-DRAW");
    win->overlay(draw);

    EZD::draw_text(0, 0, 800, 50, "center center", "FDDI Simulation",
	      "", "-*-times-bold-r-*-*-40-*-*-*-*-*-*-*");
    EZD::draw_text(  0,  75, 560, 20, "right center", "Press button:", "", "");
    EZD::draw_text( 25,  75, 100, 20, "left  center", "Simulation time:" , "", ""); 
    EZD::draw_text( 25,  90, 100, 20, "left  center", "   # of packets:" , "", ""); 
    if(thton)
      EZD::draw_text(287, 110, 100, 20, "left  center", "THT" , "", ""); 
    if(trton)
      EZD::draw_text(317, 110, 100, 20, "left  center", "TRT" , "", ""); 
    ezdst = new EZDText("st", 125, 78, "");       // Value simulation time in [ms]
    ezdpc = new EZDText("pc", 125, 93, "");       // value # of packet
    ezdpc->set_text_val ("", 0);

    // Ring
    EZD::draw_line(262, 150-36, 262, 150+NSTATION*36, "black");
    EZD::draw_line(412, 150-36, 412, 150+NSTATION*36, "black");
    EZD::draw_arc( 262,              150-75,  150,  75, 180, 181, "black");
    EZD::draw_arc( 262, 148+(NSTATION-1)*36,  150,  75,   0, 180, "black");

    new EZDPushButton ("INFO", 569, 75, 50, 20, "Info");
    new EZDPushButton ("STEP", 629, 75, 50, 20, "Step");
    new EZDPushButton ("RUN",  689, 75, 50, 20, "Run");
    new EZDPushButton ("EXIT", 749, 75, 50, 20, "Exit");
  }
#endif//EZDDISP

  // create Stations
  stations = new Station *[NSTATION];
  for(int i=0; i<NSTATION; i++)
    stations[i] = new Station(rnd_packet_length, rnd_packet_rate, i);

  // scheduler
  CNEventScheduler scheduler;

  // send EV_JOB with delay to ALL Stations!
  for(int i=0; i<NSTATION; i++)
    scheduler.send_event(new CNEvent(EV_INIT, stations[i], 0.0));

  // start simulation with EV_TOKEN
  scheduler.start(new CNEvent(EV_TOKEN, stations[aktstation], 0.0));

  // print results
  for(int i=0; i<NSTATION; i++)
    stations[i]->print_results();

  // print batch means results
  //bat_outfile << bat_waiting_time << endl;
  bat_waiting_time->print(bat_outfile);
  
  // delete stations
  for(int i=0; i<NSTATION; i++)
    delete stations[i];
  delete []stations;
}
