//:ts=3  -*- C++ -*-

// Simulation of an FDDI-Tokenring
// mz, Stephan Mehrholz (cg)

// CNCL Includes
#include <iostream.h>
#include <CNCL/QueueFIFO.h>
#include <CNCL/sim.h>
#include <CNCL/FiboG.h>
#include <CNCL/NegExp.h>
#include <CNCL/Moments.h>
#include <CNCL/Batches.h>
#include <CNCL/DiscUniform.h>
#include <CNCL/String.h>


// C++ Includes
#include <stdlib.h>
#include <GetOpt.h>

// Local Include
//#include "Job.h"
#include <CNCL/Job.h>

bool ezd = FALSE;      // TRUE if animation is recommended
bool thton = FALSE;    // TRUE if Token Holding Timer is recommended
bool trton = FALSE;    // TRUE if Token Rotation Timer is recommended

void usage(void)
{
 cerr << "This is the CNCL FDDI Simulation\n"
      << "Usage: fddi -d -h [-l] [-s] [-L] [-Q] [-T]\n\n"
      << "option \tparameters\tcorresponding action\n"
      << "\t(mandatory)\n"
      << "------ \t----------\t--------------------\n"
      << "  -d  \t\t\tchoose animation with ezd on the X display\n"
      << "\t\t\t\tdefault: no animation\n"
      << "  -h  \t\t\tdisplay this message\n"
      << "  -l \t<double>  \tchange the load (load > 0)\n"
      << "\t\t\t\tdefault: RHO = 0.3\n"
      << "  -s \t<integer> \t# of stations in the ring, (1..30)\n"
      << "\t\t\t\tdefault: NSTATION = 10\n"
      << "  -t \t<double>  \tchange the ttrt\n"
      << "\t\t\t\tdefault: TTRT = 0.004 s\n"
      << "  -L \t<double>  \tchange the avarage packet length\n"
      << "\t\t\t\tdefault: PL = 5000 Bit\n"
      << "  -Q \t<integer> \tchange the length of all queues, (1..100)\n"
      << "\t\t\t\tdefault: QL = 10\n"
      << "  -T \t<string>  \tactivate Timer \tTHT with <THT>, TRT with <TRT>\n"
      << "\t\t\t\t\tTHT and TRT with <ALL>\n"
      << "\t\t\t\tdefault: no Timer display\n"
      << endl;
 exit (0);
}

// variables for FDDI, ring parameter
// ----------------------------------
long    NSTATION = 10;          // default # of stations
long          QL = 10;          // default Queue Length
CNSimTime   TTRT = 0.004;       // default Target Token Rotation Timer [s]
double        CM = 200000000.0; // propagation speed of the Media [m/s]
double        RV = 100000000.0; // Ring Velocity (Transferrate) [Bit/s]
double        RL = 100000.0;    // Ring Length [m]
double        TL = 88.0;        // Token Length [Bit]
double        PL = 5000.0;      // default Paket Length [Bit]
double      SLAT = 60.0;        // Station LATency [Bit]
double        CP = 1.0;         // coefficient of variation
double      RLAT = 0.0;         // RingLATency
double TOKENTIME = 0.0;         // TIME for TOKEN from Station [i] to [i+1]
double    LAMBDA;               // packet generating rate
double       RHO = 0.3;         // default load


// variables for Simulation
// ------------------------
enum { EV_JOB, EV_INIT, EV_TOKEN, EV_SEND }; // CNEvent types for simulation
int aktstation = 0;             // # of active station
long global_packet_counter = 0; // counts # of all packets

CNRNG *rng;                     // base random generator
CNBatches *bat_waiting_time;    // batch means evaluation waiting time
ofstream bat_outfile;           // file for Batch Means data
double theo_waiting_time=0.0;   // theoretical waiting time

// Array of <names> for EZD Objects, (hack)
CNString names[] = {
 "q01", "q02", "q03", "q04", "q05", "q06", "q07", "q08", "q09", "q10",
 "q11", "q12", "q13", "q14", "q15", "q16", "q17", "q18", "q19", "q20",
 "q21", "q22", "q23", "q24", "q25", "q26", "q27", "q28", "q29", "q30"
};


// CNClasses
// --------------
class Station : public CNEventHandler
{
private:
  int       no;                    // station #
  CNJob       *job;                  // served job
  CNQueueFIFO queue;               // CNQueue
  CNRandom  &rnd_packet_length;    // distribution of service time b
  CNRandom  &rnd_packet_rate;      // distribution of packet arrival
  CNMoments mom_waiting_time;      // evaluation  waiting time
  CNMoments mom_service_time;      //             service time
  CNMoments mom_rotation_time;     //            rotation time
  CNMoments mom_packet_rate;       //            packet_rate
  enum {ST_WAITING, ST_SERVING };
  CNSimTime ttrt;                  // Target Token Rotation Timer
  CNSimTime trt;                   // Token Rotation Timer
  CNSimTime tht;                   // Token Holding Timer
  CNSimTime ta;                    // packet arrival time
  long      late_count;             
  long      packet_counter; 
  long      lost_packet_counter;
  CNSimTime rotation_time;         // time one trip around the ring
  CNSimTime ring_time;             // ring_time-last_ring_time yields
  CNSimTime last_ring_time;        //                rotation_time
public:
  inline void rotation_time_check();
  inline void trt_check(CNSimTime delay);
  virtual void event_handler(const CNEvent *ev);
  virtual void print(ostream &strm = cout) const;
  virtual void dump (ostream &strm = cout) const;
  void eval_job(CNJob *job);
  void print_results();
  Station(CNRandom &rnd1, CNRandom &rnd2, int ii) : 
          job(NIL),
          rnd_packet_length(rnd1), 
          rnd_packet_rate(rnd2), 
          mom_waiting_time("waiting time T_w"), 
          mom_service_time("service time T_s"),
          mom_rotation_time("rotation time T_c"),
          mom_packet_rate("packet rate T_A")
  {
    state(ST_WAITING);             // default state() of all stations
    no = ii;                       // identifikation # of the station
    // initialize all trt, as if the token had been send through an empty ring
    // e.g. #0 has NSTATION*TOKENTIME, #9 has 1*TOKENTIME
    trt  = (NSTATION-ii)*TOKENTIME;
    ttrt = TTRT;                   // global Value TTRT

    trt=0.0;            
    tht=0.0;            
    late_count=0;
    packet_counter=0;
    lost_packet_counter = 0;
    rotation_time=0.0;  
    ring_time=0.0;      
    last_ring_time=0.0; 
    ta = 0.0;
  }
};

// array of pointer of Station 
Station **stations;


// member function for Station
// ---------------------------
void Station::rotation_time_check()
{
  last_ring_time = ring_time;
  ring_time = now();
  rotation_time = ring_time - last_ring_time;
  mom_rotation_time.put(rotation_time);
}

void Station::trt_check(CNSimTime delay)
{
  for(int z=0;z<NSTATION;z++)
  {
    stations[z]->trt += delay;
    if(stations[z]->trt > stations[z]->ttrt)
    {
      stations[z]->late_count++;
      stations[z]->trt = 0.0;
    } // if(trt>ttrt)..
  } // for ..
}

void Station::event_handler(const CNEvent *ev)
{
  CNSimTime ts;  // service time



  switch(state())
  {
    case ST_WAITING:
    {
      // station is waiting for the TOKEN
      switch(ev->type())
      {
        case EV_INIT:
        {
          // send EV_JOB TO SELF
          ta = rnd_packet_rate();
          mom_packet_rate.put(ta);
          send_delay(new CNEvent(EV_JOB, stations[no], new CNJob), ta);
        }
        break; // EV_INIT

        case EV_JOB:
        // Incoming Job, put into queue
        {
          packet_counter++;
          if(queue.length() < QL)
          {
            CNJob *job;
            job = (CNJob *)ev->object();
            job->in = now();
            queue.put(job);
          }
          else // queue is full!
          {
            lost_packet_counter++;
          }
          // generate packets, until Batch Means Error is satisfied
          if(bat_waiting_time->status()==CNStatistics::END)
          {
            scheduler()->stop();
            return;
          }
          else
          {
            global_packet_counter++;
            // send EV_JOB TO SELF
            ta = rnd_packet_rate();
            mom_packet_rate.put(ta);
            send_delay(new CNEvent(EV_JOB, stations[no], new CNJob), ta);
          }
        }
        break; // EV_JOB

        case EV_TOKEN:
          // Station gets TOKEN, has right to send!
          {
            // check the rotation time of the token
            rotation_time_check();
            // set trt=0.0 if(late_count==0)
            if(late_count==0)
            {
              tht=trt; // Token holding timer = token rotation timer
              trt=0.0; // token rotation timer = NULL
            }
            // Send first paket!
            // packets in queue ??
            if((!queue.empty()) && (late_count==0))
            {
              job = (CNJob *)queue.get();
              job->start = now();
              job->length = rnd_packet_length();
              // CNRandom service time
              ts = (1.0 * job->length)/RV;          // [bit] / [bit/s] = [s]
              // tht + service time
              tht +=ts;
              // CNMoments for service time
              mom_service_time.put(ts);
              trt_check(ts);
              send_delay(new CNEvent(EV_SEND), ts);

              state(ST_SERVING);
            }
            else
            {
              // if late_count != 0 or !queue.empty()
              // even if the late_count is zero and is set to zero again!
              late_count = 0;

              // queue empty, send TOKEN to stations[aktstation+1]
              // TOKENTIME is the time for the TOKEN to reach next station!
              aktstation = (aktstation+1) % NSTATION;
              trt_check(TOKENTIME);
              send_delay(new CNEvent(EV_TOKEN, stations[aktstation],0.), TOKENTIME);
              // change state()
              state(ST_WAITING);
            }
          }
          break; // EV_TOKEN

        case EV_SEND:
          {
            error("Station: ", "Event 'EV_SEND' in state(ST_WAITING) NOT allowed!");
          }
          break; // EV_SEND

        default:
          error("Station: ", "illegal event in event_handler(), state ST_WAITING");
          ev->print();
          break;
        } // switch(ev->type() in state ST_WAITING
    } // ST_WAITING
    break;


    case ST_SERVING:
    {
    // station has TOKEN, sending until ( (tht>ttrt) !! (queue.empty()) )
      switch(ev->type())
      {
        case EV_INIT:
        {
            error("Station: ","Event 'EV_INIT' in state(ST_SERVING) NOT allowed!");
        }
        break; // EV_INIT

        case EV_JOB:
        {
          // Incoming Job, if(queue.length() < QL) put into queue
          packet_counter++;
          if(queue.length() < QL)
          {
            CNJob *job;
            job = (CNJob *)ev->object();
            job->in = now();
            queue.put(job);
          }
          else // queue is full! packet get's lost ..
          {
            lost_packet_counter++;
          }
          if(bat_waiting_time->status() == CNStatistics::END)
          {
            scheduler()->stop();
            return;
          }
          else
          {
            global_packet_counter++;
            // send EV_JOB TO SELF
            ta = rnd_packet_rate();
            mom_packet_rate.put(ta);
            send_delay(new CNEvent(EV_JOB, stations[no], new CNJob), ta);
          }
        }
        break; // EV_JOB

        case EV_TOKEN:
          {
            error("Station: ","Event 'EV_TOKEN' in state(ST_SERVING) NOT allowed!");
          }
          break; // EV_TOKEN

        case EV_SEND:
          // station is sending until queue empty   [ OR tht>ttrt ]
          {
            // transmitted job
            job->out = now();
            // Evaluate job
            eval_job(job);
            delete job;
            job = NIL;
        
            if((!queue.empty()) && (tht < ttrt))
            {
              job = (CNJob *)queue.get();
              job->start = now();
              job->length = rnd_packet_length();
              // CNRandom service time
              ts = (1.0 * job->length)/RV;         // [bit] / [bit/s] = [s]
              // tht + service_time
              tht += ts;
              // CNMoments for service time
              mom_service_time.put(ts);
              trt_check(ts);
              send_delay(new CNEvent(EV_SEND), ts);
            }
            else
            {
              // queue empty, send TOKEN to stations[aktstation+1]
              // TOKENTIME is the time for the TOKEN to reach next station!
              aktstation = (aktstation+1) % NSTATION;
              trt_check(TOKENTIME);
              send_delay(new CNEvent(EV_TOKEN, stations[aktstation],0.), TOKENTIME);
              // change state() 
              state(ST_WAITING);
            }
          }
          break; // EV_SEND

        default:
          error("Station: ", "illegal event in state ST_SERVING");
          break;
      } // switch(ev->type() in state ST_SERVING
    } // ST_SERVING
    break;

    default:
    {
      error("Station: ", "illegal state in event_handler(const CNEvent *ev) ");
      cout << get_state();
    }
    break;
  } // switch(state())
}

void Station::print(ostream &) const
{
}

void Station::dump(ostream &strm) const
{
  strm << "Station { state=" << state() << " no=" << no << " }" << endl;
}

void Station::eval_job(CNJob *job)
{
    bat_waiting_time->put(job->start - job->in);
    mom_waiting_time.put(job->start - job->in);
}

void Station::print_results()
{
    cout << "--- SERVER --- " << no << "\n" 
         << "lost packets: "  << lost_packet_counter << "\n"
         << mom_rotation_time << "\n"
         << mom_waiting_time  << "\n"
         << mom_service_time  << endl;
}


main(int argc, char **argv)
{
  // parse commandline arguments
  GetOpt getopt(argc, argv, "dhl:s:t:L:Q:T:");
  int    opt;
	
  while ((opt = getopt()) != EOF)
    switch(opt)
      {
      case 'd':
        ezd=TRUE;
        break;
	   case 'h':
        usage();
        break;
      case 'l':
        RHO = atof(getopt.optarg);
        if(RHO==0.0) usage();
        break;
      case 's':
        NSTATION = atoi(getopt.optarg);
        if((NSTATION==0)||(NSTATION>30)) usage();
        break;
      case 't':
        TTRT = atof(getopt.optarg);
        break;
      case 'L':
        PL = atof(getopt.optarg);
        break;
      case 'Q':
        QL = atoi(getopt.optarg);
        if((QL==0)||((ezd)&&(QL>100))) usage();
        break;
	   case 'T':
        if((CNString(getopt.optarg)=="THT")||(CNString(getopt.optarg)=="tht"))
          thton = TRUE;
        if((CNString(getopt.optarg)=="TRT")||(CNString(getopt.optarg)=="trt"))
          trton = TRUE;
        if((CNString(getopt.optarg)=="ALL")||(CNString(getopt.optarg)=="all"))
        {
          thton = TRUE;
          trton = TRUE;
        }
        break;
	   }

  cout << "Type fddi -h <RETURN> for help on fddi\n\n"
       << "Parameter\n~~~~~~~~~\n"
       << "     ezd=" << (  ezd==TRUE ? "True" : "False")<<"\n"
       << "   thton=" << (thton==TRUE ? "True" : "False")<<"\n"
       << "   trton=" << (trton==TRUE ? "True" : "False")<<"\n"
       << "     RHO=" << RHO << "\n"
       << "NSTATION=" << NSTATION << "\n" 
       << "    TTRT=" << TTRT << "\n"
       << "      PL=" << PL << "\n"
       << "      QL=" << QL << "\n"
       << "\nstarting simulation! ...\n"
       << endl;

  // compute values depending on the parameters of the Ring
  LAMBDA    = RHO / (PL/RV * NSTATION);
  RLAT      = RL/CM * RV  +  NSTATION * SLAT;
  TOKENTIME = RL / (NSTATION * CM)  +  TL / RV;
  // compute theoretical waiting time
  theo_waiting_time = ((RLAT+NSTATION*TL)/RV*(1-RHO/NSTATION))/(2*(1-RHO))+
      (RHO*(CP*CP+1)*PL/RV)/(2*(1-RHO)); // theoretical waiting time in [s]
  cout << "   LAMBDA=" << LAMBDA << "\n"
       << "     RLAT=" << RLAT << "\n"
       << "TOKENTIME=" << TOKENTIME << "\n"
       << "      T_w=" << theo_waiting_time << "\n"
       << endl;
       

  rng   = new CNFiboG;
  CNNegExp rnd_packet_rate(1.0/LAMBDA, rng);        // RANDOM # of packet arrival
  CNNegExp rnd_packet_length(1.0 * PL, rng);        // RANDOM # of packet length
  if(ezd)
   bat_waiting_time=new CNBatches(0.0,0.004,0.5,10,100,20,95,"waiting time");
  else
   bat_waiting_time=new CNBatches(0.0,0.005,0.2,10,10000,20,95,"waiting time");
// bat_waiting_time=new CNBatches(0.0,0.0025,0.02,10,300000,20,95,"waiting time");
                    // (BOTTOM, TOP, ERROR, NOG, SOG, NOI, CONF, NAME)	
  ofstream bat_outfile("fddi.btm", ios::out);



  // create Stations
  stations = new Station *[NSTATION];
  for(int i=0; i<NSTATION; i++)
    stations[i] = new Station(rnd_packet_length, rnd_packet_rate, i);

  // scheduler
  CNEventScheduler scheduler;

  // send EV_JOB with delay to ALL Stations!
  for(int i=0; i<NSTATION; i++)
    scheduler.send_event(new CNEvent(EV_INIT, stations[i], 0.0));

  // start simulation with EV_TOKEN
  scheduler.start(new CNEvent(EV_TOKEN, stations[aktstation], 0.0));

  // print results
  for(int i=0; i<NSTATION; i++)
    stations[i]->print_results();

  // print batch means results
  //bat_outfile << bat_waiting_time << endl;
  bat_waiting_time->print(bat_outfile);

  // delete stations
  for(int i=0; i<NSTATION; i++)
    delete stations[i];
  delete [] stations;
}
