// -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * Class: CNArray2Long  --- 2 dimensional long array class
 *
 * Created from
 * $Id: T_array2.cP,v 0.31 1996-08-07 18:11:22+02 steppler Exp $
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include "Array2Long.h"



/***** Private functions *****************************************************/

void CNArray2Long::init(size_t r, size_t c)
{
    size_t i, j;
    
    arows = r;
    acols = c;
    
    array = new CNArrayLong[r];
    for (i = 0; i < r; i++)
    {
	array[i].size(c);
	for(j = 0; j < c; j++)
	    array[i][j] = def_val;
    }
}



void CNArray2Long::delete_array()
{
    arows = acols = 0;
    if(array)
	delete [] array;
    array = NIL;
}



/***** Constructors **********************************************************/

CNArray2Long::CNArray2Long(size_t r, size_t c, long def)
{
    def_val = def;
    init(r, c);
}



CNArray2Long::CNArray2Long(const CNArray2Long &a)
{
    size_t i, j;
    
    init(a.arows, a.acols);
    for (i = 0; i < arows; i++)
	for (j = 0; j < acols; j++)
	    array[i][j] = a.array[i][j];
}


CNArray2Long::~CNArray2Long()
{
    CNArray2Long::delete_array();
}



CNArray2Long &CNArray2Long::operator = (const CNArray2Long &a)
{
    size_t i, j;
    
    delete_array();
    init(a.arows, a.acols);
    for (i = 0; i < arows; i++)
	for (j = 0; j < acols; j++)
	    array[i][j] = a.array[i][j];

    return *this;
}
    


/***** public functions ******************************************************/

void CNArray2Long::size(size_t r, size_t c)  // Set size of array
{
    size_t i, j;
    
    if (array == NIL)
    {
	init(r, c);
	return;
    }
    if(r == arows && c == acols)	// Nothing to do
	return;
    if(r == arows)			// Resize rows only
    {
	for(i = 0; i < arows; i++)
	    array[i].size(c);
	acols = c;
	return;
    }
    
    CNArrayLong *temp = array;

    array = new CNArrayLong[r];
    for (i = 0; i < r; i++)
    {
	array[i].size(c);
	for(j = 0; j < c; j++)
	    if(i<arows && j<acols)
		array[i][j] = temp[i][j];
	    else
		array[i][j] = def_val;
    }
    delete [] temp;
    arows = r;
    acols = c;
}



/***** Default I/O member function for CNCL classes **************************/

// CNNormal output
void CNArray2Long::print(ostream &strm) const
{
    strm << "[ " << endl;
    for(size_t i=0; i<arows; i++)
    {
	strm << "  " << array[i];
	if(i != arows-1)
	    strm << " ,";
	strm << endl;
    }
    strm << "]" << endl;
}

// Debug output
void CNArray2Long::dump(ostream &strm) const
{
    strm << "CNArray2Long { $Revision: 0.31 $" << endl;
    strm << "rows=" << arows << " cols=" << acols
	 << " array=" << hex << (unsigned long)(array) << dec << endl;
    strm << "values=" << endl << this << endl;
    strm << "}" << endl;
}



/***** CNCL stuff for type ***************************************************/

// Describing object for class CNArray2Long
static CNClass CNArray2Long_desc("CNArray2Long", "$Revision: 0.31 $",
			    CNArray2Long::new_object);

// "Type" for type checking functions
CNClassDesc CN_ARRAY2LONG = &CNArray2Long_desc;
