// -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 *
 * $Id: T_array3.hP,v 1.2 1996-08-07 18:11:25+02 steppler Exp $
 * Class: CNArray3Float  --- 3 dimensional float array class
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#ifndef __Array3Float_h
#define __Array3Float_h


#include <CNCL/Class.h>
#include <CNCL/Param.h>
#include <CNCL/Array3.h>
#include <CNCL/Array2Float.h>

extern CNClassDesc CN_ARRAY3FLOAT;	// CNClass CNArray3Float description


/*
 * The class CNArray3Float
 */


class CNArray3Float : public CNArray3
{
  public:	/***** Constructors ******************************************/
    CNArray3Float()          : CNArray3(0,0,0), array(NIL), def_val(0) {}
    CNArray3Float(CNParam *) : CNArray3(0,0,0), array(NIL), def_val(0) {}
    CNArray3Float(size_t r, size_t c, size_t d, float def=0);
    CNArray3Float(const CNArray3Float &a);		// Copy constructor

    ~CNArray3Float();

    CNArray3Float &operator = (const CNArray3Float &a);
    
  public:	/***** Public interface **************************************/
    virtual void size(size_t r, size_t c, size_t d);

    void put(int r, int c, int d, float value)// Put value into array
    {
#   ifndef NO_RANGE_CHECK
        if(r<0 || arows<=(size_t)r || 
           c<0 || acols<=(size_t)c || 
           d<0 || adepth<=(size_t)d)
            range_error(r, c, d);
	else
#   endif
	    array[r][c][d] = value;
    }

    float get(int r, int c, int d) const	// Get value from array
    {
#   ifndef NO_RANGE_CHECK
        if(r<0 || arows<=(size_t)r || 
           c<0 || acols<=(size_t)c || 
           d<0 || adepth<=(size_t)d)
	{   range_error(r, c, d); return 0; }
	else
#   endif
	    return array[r][c][d];
    }

    CNArray2Float& operator[] (int index)	// Operator []
    {
#   ifndef NO_RANGE_CHECK
	if (index<0 || arows<=(size_t)index)
	    range_error(index);
#   endif
	return array[index];
    }

  private:	/***** Internal private members ******************************/
    CNArray2Float *array;		// Array of array of array of float
    float def_val;			// Default value
    
    void init(size_t r, size_t c, size_t d);		// Initialize array
    void delete_array();                // Delete array

  public:	/***** Member functions required by CNCL *********************/
    virtual CNClassDesc class_desc() const	// CNClass description
    { return CN_ARRAY3FLOAT; }
            
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    { return desc == CN_ARRAY3FLOAT ? TRUE : CNArray3::is_a(desc); }
        
    static CNArray3Float *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
	return (CNArray3Float *)obj;
#   else
	return (CNArray3Float *)( !obj || obj->is_a(CN_ARRAY3FLOAT)
	       ? obj : fatal_type(obj->class_desc(), CN_ARRAY3FLOAT) );
#   endif
    }
    
    static CNObject *new_object(CNParam *param = NIL) // Object creation
    { return param ? new CNArray3Float(param) : new CNArray3Float; }
    
    // Print/debug output
    virtual void print(ostream &strm = cout) const;
    virtual void dump (ostream &strm = cout) const;
};

#endif /**__Array3Float_h**/
