// -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * Class: CNArrayChar  --- 1 dimensional char array class
 *
 * Created from
 * $Id: T_array.hP,v 0.31 1996-08-07 18:11:20+02 steppler Exp $
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#ifndef __ArrayChar_h
#define __ArrayChar_h


#include <CNCL/Class.h>
#include <CNCL/Param.h>
#include <CNCL/Array.h>

extern CNClassDesc CN_ARRAYCHAR;	// CNClass CNArrayChar description


/*
 * The class CNArrayChar
 */


class CNArrayChar : public CNArray
{
  public:	/***** Public interface **************************************/
    virtual void size(size_t sz=0);	// Set size of array

    void put(int index, char value)	// Put value into array
    {
#   ifndef NO_RANGE_CHECK
	if(index<0 || asize<=(size_t)index)
	    range_error(index);
	else
#   endif
	    array[index] = value;
    }

    char get(int index) const		// Get value from array
    {
#   ifndef NO_RANGE_CHECK
	if (index<0 || asize<=(size_t)index)
	{   range_error(index); return 0; }
	else
#   endif
	    return array[index];
    }

    char& operator[] (int index)	// Operator []
    {
#   ifndef NO_RANGE_CHECK
	if (index<0 || asize<=(size_t)index)
	    range_error(index);
#   endif
	return array[index];
    }

  private:	/***** Internal private members ******************************/
    friend class CNPArrayChar;
    
    char *array;			// Array of char
    char def_val;			// Default value
    
    void init(size_t sz);		// Initialize array
    void delete_array();                // Delete array

  public:	/***** Constructors ******************************************/
    CNArrayChar()          : CNArray((size_t)0), array(NIL), def_val(0) {}
    CNArrayChar(CNParam *) : CNArray((size_t)0), array(NIL), def_val(0) {}
    CNArrayChar(size_t sz, char def=0);
    CNArrayChar(const CNArrayChar &a);		// Copy constructor

    ~CNArrayChar();

    CNArrayChar &operator = (const CNArrayChar &a);
    

  public:	/***** Member functions required by CNCL *********************/
    virtual CNClassDesc class_desc() const	// CNClass description
    { return CN_ARRAYCHAR; }
            
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    { return desc == CN_ARRAYCHAR ? TRUE : CNArray::is_a(desc); }
        
    static CNArrayChar *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
	return (CNArrayChar *)obj;
#   else
	return (CNArrayChar *)( !obj || obj->is_a(CN_ARRAYCHAR)
	       ? obj : fatal_type(obj->class_desc(), CN_ARRAYCHAR) );
#   endif
    }
    
    static CNObject *new_object(CNParam *param = NIL) // Object creation
    { return param ? new CNArrayChar(param) : new CNArrayChar; }
    
    // Print/debug output
    virtual void print(ostream &strm = cout) const;
    virtual void dump (ostream &strm = cout) const;
};

#endif /**__ArrayChar_h**/
