//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: DLList.c,v 0.35 1996-08-07 17:54:59+02 steppler Exp $
 *
 * Class: CNDLList --- Double Linked List
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 * 
 * As an exception to this rule you may use this template to generate
 * your own classes. This does not cause these classes to be covered by
 * the GNU Library General Public License. This exception does not
 * however invalidate any other reasons why the resulting program must be
 * covered by the GNU Library General Public License.
 *****************************************************************************/

#include <CNCL/Class.h>

#include "DLList.h"
#include <CNCL/Object.h>

#include <limits.h>

CNDLList::CNDLList():  CNSLList()
{}

CNDLList::CNDLList(CNParam *p): CNSLList(p)
{}

CNDLList::CNDLList(const CNDLList &) : CNSLList()
{
    fatal(NIL, "can't copy CNDLObject object");
}

CNDLList::~CNDLList()
{
    delete_all();
}

CNDLObject *CNDLList::first() const
{
    return (CNDLObject *)pfirst;
}

CNDLObject *CNDLList::next(CNDLObject *link) const
{
    return (CNDLObject *)link->pnext;
}

CNDLObject *CNDLList::last() const
{
    return (CNDLObject*)plast;
}

CNDLObject *CNDLList::prev(CNDLObject *link) const
{
    return link->pprev;
}



bool CNDLList::ok()
{
    int v = TRUE;
    if (pfirst)
    {
	CNDLObject *p;
	long num = LONG_MAX;      // Lots of chances to find h!

	for(p=CNDLObject::cast_from_object(pfirst); p && v && num>0;
		      p = CNDLObject::cast_from_object(p->pnext))
	{
	    if(p->pprev)
		v &= p->pprev->pnext == p;
	    if(p->pnext)
		v &= CNDLObject::cast_from_object(p->pnext)->pprev == p;
	}
	v &= num > 0;
    }
    if(!v)
	error(NIL, "CNDLList: ", "invariant failure");
    return v;
}



/*
 * Remove node from list, do NOT delete DLObject
 */
CNDLObject *CNDLList::remove_object(CNDLObject *pos)
{
    CNDLObject *p, *n;
    
    p = pos->pprev;
    n = CNDLObject::cast_from_object(pos->pnext);
    
    if(p)
	p->pnext = n;
    else
	pfirst   = n;
    if(n)
	n->pprev = p;
    else
	plast    = p;
    
    count--;
    
    return n;
}



/*
 * Remove node from list, delete DLObject
 */
CNDLObject *CNDLList::delete_object(CNDLObject *pos)
{
    CNDLObject *p, *n;
    
    p = pos->pprev;
    n = CNDLObject::cast_from_object(pos->pnext);
    
    if(p)
	p->pnext = n;
    else
	pfirst   = n;
    if(n)
	n->pprev = p;
    else
	plast    = p;
    
    delete pos;
    count--;
    
    return n;
}

CNDLObject *CNDLList::insert_before(CNDLObject *pos, CNObject *obj)
{
    return insert_before(pos, new CNDLObject(obj));
}

CNDLObject *CNDLList::insert_before(CNDLObject *pos, CNObject &obj)
{
    return insert_before(pos, new CNDLObject(&obj));
}

CNDLObject *CNDLList::insert_before(CNDLObject *pos, CNDLObject *obj)
{
    CNDLObject *p;
    
    p = pos->pprev;
    if(p)
	p->pnext = obj;
    else
	pfirst   = obj;
    obj->pprev = p;
    obj->pnext = pos;
    pos->pprev = obj;
    count++;
    
    return obj;
}


CNDLObject *CNDLList::insert_after(CNDLObject *pos, CNObject *obj)
{
    return insert_after(pos, new CNDLObject(obj));
}

CNDLObject *CNDLList::insert_after(CNDLObject *pos, CNObject &obj)
{
    return insert_after(pos, new CNDLObject(&obj));
}

CNDLObject *CNDLList::insert_after(CNDLObject *pos, CNDLObject *obj)
{
    CNDLObject *n;
    
    n = CNDLObject::cast_from_object(pos->pnext);
    if(n)
	n->pprev = obj;
    else
	plast   = obj;
    obj->pnext = n;
    obj->pprev = pos;
    pos->pnext = obj;
    count++;
    
    return obj;
}



CNDLObject *CNDLList::append(CNObject *obj)
{
    return append(new CNDLObject(obj));
}

CNDLObject *CNDLList::append(CNObject &obj)
{
    return append(new CNDLObject(&obj));
}

CNDLObject *CNDLList::prepend(CNObject *obj)
{
    return prepend(new CNDLObject(obj));
}

CNDLObject *CNDLList::prepend(CNObject &obj)
{
    return prepend(new CNDLObject(&obj));
}


CNDLObject *CNDLList::append(CNDLObject *link)
{
    if(pfirst)
    {
	plast->pnext = link;
	link ->pprev = CNDLObject::cast_from_object(plast);
	link ->pnext = NIL;
	plast        = link;
    }
    else
    {
	pfirst = plast = link;
	link->pnext = link->pprev = NIL;
    }
    count++;

    return link;
}

CNDLObject *CNDLList::prepend(CNDLObject *link)
{
    if (pfirst)
    {
	CNDLObject::cast_from_object(pfirst)->pprev = link;
	link  ->pnext = pfirst;
	link  ->pprev = NIL;
	pfirst        = link;
    }
    else
    {
	pfirst = plast = link;
	link->pnext = link->pprev = NIL;
    }
    count++;

    return link;
}



/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNDLList::print(ostream &strm) const
{
    CNDLObject *p;

    for(p=CNDLObject::cast_from_object(first()); p; p = CNDLObject::cast_from_object(p->next()))                    
	strm << p;
}

// Debug output
void CNDLList::dump(ostream &strm) const
{
    CNDLObject *p;

    strm << "CNDLList { "
	 << "$Revision: 0.35 $  count=" << count << endl;
    for(p=CNDLObject::cast_from_object(first()); p; p=CNDLObject::cast_from_object(p->next()))
	p->dump(strm);
    strm << "}" << endl;
}

CNObject *CNDLList::new_object(CNParam *param = NIL) 
{
    return param ? new CNDLList(param) : new CNDLList;
}



// IOStream operator <<
ostream &operator << (ostream &strm, const CNDLList &obj)
{
    obj.print(strm);
    return strm;
}

ostream &operator << (ostream &strm, const CNDLList *obj)
{
    if(obj)
	obj->print(strm);
    else
	strm << "(NIL)";
    return strm;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNDLList
static CNClass CNDLList_desc("CNDLList", "$Revision: 0.35 $",
			    CNDLList::new_object);

// "Type" for type checking functions
CNClassDesc CN_DLLIST = &CNDLList_desc;
