//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: EventHeapSched.c,v 0.31 1996-08-07 17:56:34+02 steppler Exp $
 *
 * CNClass: CNHeapEventScheduler --- CNEvent scheduler
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 ***************************************************************************/

#include "EventHeapSched.h"


/*
 * Contructors
 */
CNEventHeapSched::CNEventHeapSched()
	: CNEventBaseSched()
{
    heap_n = 0;
    heap_l =100;
    heap = new CNEvent* [heap_l];
    for (long l = heap_n; l < heap_l; l++)
	heap[l] = NIL;
}


CNEventHeapSched::CNEventHeapSched(CNParam *)
	: CNEventBaseSched()
{
    heap_n = 0;
    heap_l = 100;
    heap = new CNEvent* [heap_l];
}


/*
 * Destructor
 */
CNEventHeapSched::~CNEventHeapSched()
{
    delete[] heap;
}


/*
 * Add events to list
 */
void CNEventHeapSched::add_event(CNEvent *ev)
{
    if (stat) stat->put(heap_n);
    ev->issued(simtime);
    heap_n++;
    if (heap_n >= heap_l) {  // resize heap
	heap_l *= 10;
	CNEvent** new_heap = new CNEvent* [heap_l];
	for (long i=0; i<heap_n; i++) new_heap[i] = heap[i];
	delete [] heap;
	heap = new_heap;
    };
    heap[heap_n] = ev;
    heap_upheap(heap_n);
}



/*
 * Delete events from list
 */
void CNEventHeapSched::delete_event(CNEventID id)
{
    for (long i=1; i<=heap_n; i++)
    {
	if (heap[i]->ev_id == id)
	{
	    CNEvent *v = heap[i];
	    heap[i] = heap[heap_n--];
	    if (v->after(heap[i])) {
		heap_upheap(i);
	    } else {
		heap_downheap(i);
	    };
	    delete v;
	    return;
	}
    }
}



/*
 * Delete events from list
 */
void CNEventHeapSched::delete_events(CNEventHandler *evh, bool to)
{
    bool done = FALSE;
    while (!done)
    {
	done = TRUE;
	for (long i=heap_n; i>0; i--)
	{
	    if ( (to && (evh == heap[i]->ev_to)) ||
		 ((!to) && (evh == heap[i]->ev_from)) ) 
		{
		    done = FALSE;
		    CNEvent *v = heap[i];
		    heap[i] = heap[heap_n--];
		    if (v->after(heap[i])) {
			heap_upheap(i);
		    } else {
			heap_downheap(i);
		    };
		    delete v;
		    i--;
		}
	}
    }
}



/*
 * Peek at next event
 */
CNEvent *CNEventHeapSched::peek_event()
{
    return (heap_n > 0) ? heap[1] : 0;
}


CNEvent *CNEventHeapSched::peek_event(CNEventID id)
{
    for (long i=1; i<=heap_n; i++) {
	if (heap[i]->ev_id == id) return heap[i];
    };
    return 0;
}



/*
 * Get next event
 */
CNEvent *CNEventHeapSched::next_event()
{
    return heap_remove();
}



/*
 * Stop scheduler by setting flag
 */
void CNEventHeapSched::stop()
{
    stop_flag = TRUE;			  // Set stop flag
    while (heap_n) delete heap[heap_n--]; // and delete all events in the list
}



/*
 * Heap: upheap()
 */
void CNEventHeapSched::heap_upheap(long k)
{
    CNEvent* v = heap[k];
    while ( (k>1) &&  heap[k/2]->after(v) ) {
	heap[k] = heap[k/2];
	k /= 2;
    };
    heap[k] = v;
}


/*
 * Heap: downheap()
 */
void CNEventHeapSched::heap_downheap(long k)
{
    CNEvent* v = heap[k];
    long j;
    while (k <= heap_n/2) {
	j = k+k;
	if ( (j<heap_n) && heap[j]->after(heap[j+1]) ) j++;
	if ( heap[j]->after(v) ) break;
	heap[k] = heap[j];
	k = j;
    };
    heap[k] = v;
}


/*
 * Heap: remove()
 */
CNEvent *CNEventHeapSched::heap_remove()
{
    CNEvent *result = NIL;
    if (heap_n) {
	result = heap[1];
	heap[1] = heap[heap_n--];
	heap_downheap(1);
    };
    return result;
}


/*
 * create event-iterator
 */
CNEventIterator *CNEventHeapSched::create_iterator()
{
    return new CNEventHIterator(this);
}




/***** Default I/O member function for CNCL classes **************************/

// CNNormal output
void CNEventHeapSched::print(ostream &strm) const
{
    for (long i=1; i<=heap_n; i++)
	heap[i]->print(strm);
}

// Debug output
void CNEventHeapSched::dump(ostream &strm) const
{
    strm << "CNEventHeapSched { $Revision: 0.31 $" << endl;
    for (long i=1; i<=heap_n; i++)
	heap[i]->dump(strm);
    strm << "}" << endl;
}



/***** CNCL stuff for type information and exemplar objects ******************/

// Describing object for class CNEventHeapSched
static CNClass CNEventHeapSched_desc("CNEventHeapSched",
				     "$Revision: 0.31 $",
				     CNEventHeapSched::new_object);

// "Type" for type checking functions
CNClassDesc CN_EVENTHEAPSCHED = &CNEventHeapSched_desc;
