//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: EZDDiag.c,v 0.30 1996-08-07 17:57:10+02 steppler Exp $
 *
 * Class: EZDDiag --- x-y diagram as an EZDObject
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include "EZDDiag.h"

/*
 * Initialize diagram
 */
void EZDDiag::initialize(int /*x*/, int /*y*/, int w, int h, int s)
{
    width   = w;
    height  = h;
    frame = FALSE;
    
    valx    = 0;
    valstep = s;
    valn    = width / s;
    
    vals    = new int [valn];
    for(int i=0; i<valn; vals[i++]=0 ) ;
    
}

/* 
 * Destructor
 */ 
EZDDiag::~EZDDiag()
{
    if(vals) delete vals;
}




/*
 * Add a new value to the diagram at the current x position
 */
void EZDDiag::draw_value(int x)
{
    if(vals[x] < 0)
    {
	if(x > 0)
	    fill_rectangle((x-1)*valstep, 0,
				2*valstep, height, color_bar);
	else
	    fill_rectangle(x*valstep, 0, valstep, height, color_bar);
    }
    else
	switch(valstyle)
	{
	case HISTO:
	    fill_rectangle(x*valstep, vals[x],
				valstep, height-vals[x], color_draw);
	    break;
	case DOT:
	    point(x*valstep, vals[x], color_draw);
	    break;
	case LINE:
	    if(x>0 && vals[x-1] >= 0)
		line((x-1)*valstep, vals[x-1],
			  x*valstep, vals[x], color_draw);
	    else
		point(x*valstep, vals[x], color_draw);
	    break;
	}
}



void EZDDiag::add(int v)
{

    if(v >= 0)
	v = height - v;			// Reverse y axis

    if(valx >= valn)			// Scroll diag to the left
    {
	int step = (int)SCROLL_STEP>valn ? valn : (int)SCROLL_STEP;
	int i;
	
	/* Shift */
	for(i=0; i<valn-step; i++)
	    vals[i] = vals[i+step];
	valx -= step;
    }
    vals[valx] = v;
    redraw();
    valx++;
    
}


/*
 * Set the diagram style
 */
void EZDDiag::style(EZDDiag::Style s)
{
    valstyle = s;
}

/*
 * Set the drawing color
 */
void EZDDiag::set_color_draw(CNString c)
{
    color_draw = c;
}

/*
 * Set the color for drawing of negative colors
 */
void EZDDiag::set_color_bar(CNString c)
{
    color_bar = c;
}


/*
 * Redraw function
 */
void EZDDiag::redraw()
{
    start();
    for(int i=0; i<valx; i++)
    {
	draw_value(i);
    }
    end();
    draw_now();
}

/*
 * Clear function
 */
void EZDDiag::clear()
{
    for(int i=0; i<valn; vals[i++]=0 ) ;
    valx = 0;
    
    redraw();
}

/*
 * set frame
 */	
void EZDDiag::set_frame()
{
	frame = TRUE;
	rectangle(0, 0, width, height, "black");
}

/***** Default I/O member function for CNCL classes **************************/

// Normal output
void EZDDiag::print(ostream &strm) const
{
    strm << "..." << endl;
}

// Debug output
void EZDDiag::dump(ostream &strm) const
{
    strm << "EZDDiag { $Revision: 0.30 $ ..."
	 << " }" << endl;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class EZDDiag
static CNClass EZDDiag_desc("EZDDiag", "$Revision: 0.30 $",
			    EZDDiag::new_object);

// "Type" for type checking functions
CNClassDesc CN_EZDDIAG = &EZDDiag_desc;

