// -*- C++ -*-

// GIST -- Graphical Interactive Simulation Result Tool

//   CNCL interface classes

//   1992,93 Martin Horneffer

#include "gist_ivimpl.hh"
#include <CNCL/Select.h>
#include <stdio.h>

// handle strings put to an ostream so they can easily be read
// and don't get accidentally splitted

class Quote {
    const char *str_;
  public:
    Quote(const char *str) { str_ = str; };
    ostream& write(ostream& out) const;
};
inline ostream& operator<<(ostream& os, const Quote& q) {return q.write(os);};
ostream& Quote::write(ostream& out) const {
    const char *s;
    char c;
    for (s = str_; (c = *s); s++) {
	switch (c) {
	  case ' ' :  out << "\\_";  break;
	  case '\n':  out << "\\n";  break;
	  case '\\':  out << "\\\\"; break;
	  default: if (!iscntrl(c))  out << c; break;
	};
    };
    return out;
};


// interface to InterViews-GIST

CommIVImpl::CommIVImpl(const char *host, unsigned short port, bool abort,
		       bool do_magic, long magic)
  : id(0), port_(port), abort_(abort)
{ 
    ostrstream o;
    o << host << ends;
    host_ = o.str();
    if (port > 1) {
	pipe = 0;
	file = 0;
	out_ = socket = new SocketStream(host, port);
	in_ = socket;
	fd = socket->filedesc();
    } else if (port == 1) {
	socket = 0;
	pipe = 0;
	out_ = file = new ofstream("gist.log");
	in_ = &cin;
	fd = 0;
    } else {
	socket = 0;
	file = 0;
	pipe = new CNPipe;
	if (pipe->open("igist")) {
	    CNCL::msg() <<"cannot start iv-gist." <<ends;
	    if (abort_) CNCL::fatal(); else CNCL::warning();
	};
	out_ = &pipe->out();
	in_ = &pipe->in();
	fd = pipe->fd_in();
	
	char buf[128];
	pipe->in().getline(buf, sizeof(buf));

	if (do_magic) {
	    char fn[64];
	    post_filename(fn, magic);
	    ofstream post(fn);
	    post << pipe->get_pid() << endl;
	    post << "localhost" << endl;
	};
    };
};
CommIVImpl::~CommIVImpl() {
    if (socket) delete socket;
    if (pipe) delete pipe;
    if (file) delete file;
    delete [] host_;
};
int CommIVImpl::ok() { 
    return ( out_ && !out_->bad() ); 
};
const char *CommIVImpl::gist_type() { 
    return "iv"; 
};
int CommIVImpl::newid() { 
    return ++id; 
};
void CommIVImpl::send(const char *msg) {
    if ( out_ && in_ ) {

// check input
	int stop=0;
	char linebuf[64];
	CNSelect selector;
	selector.add_read(fd);
	int sel = selector.select(0,0);
	while ( stop || in_->rdbuf()->in_avail() ||
		((sel>0) && selector.test_read(fd)) ) {
	    sel = 0;
	    in_->getline(linebuf,64);
	    if (!in_->good()){
		CNCL::msg() << "cannot read from igist, probably terminated." <<ends;
		if (abort_) CNCL::fatal(); else CNCL::warning();
		in_ = 0;
		CNCL::info("continuing without GIST..");
		return;
	    };
	    if ( strncmp(linebuf,"stop",4) == 0 ) {
		stop = 1;
		CNCL::info("simulation stopped. waiting..");
	    } else if ( strncmp(linebuf,"cont",4) == 0 ) {
		stop = 0;
		CNCL::info("simulation continued.");
	    };
	};

// do output
	(*out_) << msg;
	out_->flush();
	if ( !out_->good() ) {
	    CNCL::msg() << "cannot send to igist." <<ends;
	    if (abort_) CNCL::fatal(); else CNCL::warning();
	    out_ = 0;
	    CNCL::info("continuing without GIST..");
	    return;
	};
    
    };
};

WorldIVImpl::WorldIVImpl(CommImpl *comm, const char *name, 
			 float xmin, float xmax, float ymin, float ymax, 
			 double st, int m)
  : GISTWorldImpl(name) 
{
    comm_ = (CommIVImpl*)comm;
    lasttime = st;
    mode = m;
    id = comm_->newid();
    str = new ostrstream;
    str->precision(8);
    out()
	<<"GIST2\n"
        <<"create "<< id <<" world "<< st
        <<" "<< xmin <<" "<< xmax <<" "<< ymin <<" "<< ymax 
        <<" "<< Quote(name) <<"\n";
    flush();
};
WorldIVImpl::~WorldIVImpl() {
    out() <<"call "<< id <<" flush "<< lasttime <<"\n"
	  <<"call "<< id <<" die\n"<< ends;
    flush();
    delete str;
};
void WorldIVImpl::flush() {
    out() << ends;
    comm_->send( str->str() );
    delete str->str();
    delete str;
    str = new ostrstream;
};
void WorldIVImpl::flush(double time) { 
    out() <<"call "<< id <<" flush "<< time <<"\n";
    flush();
    lasttime = time;
};
void WorldIVImpl::check_flush() {
    if (mode==0) flush();
};
void WorldIVImpl::check_flush(double time) {
    if (mode==1) {
	lasttime = time;
    } else if (mode==2) {
	if (time > lasttime) {
	    flush(time);
	};
    } else {
	flush(time);
    };
};
const char *WorldIVImpl::gist_type() { 
    return "iv"; 
};

ObjectIVImpl::ObjectIVImpl(GISTWorldImpl *world,
			   const char *name)
  : GISTObjectImpl(name) 
{
    world_ = (WorldIVImpl*)world;
    id = world_->comm_->newid();
    world_->out()
	<<"call "<< world_->id <<" create "<< id 
	<<" object "<< Quote(this->name()) <<"\n";
    world_->check_flush();
};
ObjectIVImpl::~ObjectIVImpl() {
    world_->out() <<"call "<< id <<" die\n";
};
const char *ObjectIVImpl::gist_type() { 
    return "iv"; 
};


AttributeIVImpl::AttributeIVImpl(GISTObject *object) {
    obj_ = (ObjectIVImpl*)object->impl();
    world_ = obj_->world_;
    id = world_->comm_->newid();
};
AttributeIVImpl::~AttributeIVImpl() {
    world_->out() <<"call "<< id <<" die\n";
};

TextAttributeIVImpl::TextAttributeIVImpl(GISTObject *object, const char *name)
  : GISTTextAttributeImpl(object,name), AttributeIVImpl(object) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" attr text "<< Quote(this->name()) <<"\n";
};
void TextAttributeIVImpl::put(double time, const char *data) {
    world_->out() <<"call "<< id <<" put "<< time <<" "<< Quote(data) <<"\n";
    world_->check_flush(time);
};

ValueAttributeIVImpl::ValueAttributeIVImpl(GISTObject *object, const char *name)
  : GISTValueAttributeImpl(object,name), AttributeIVImpl(object) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" attr value "<< Quote(this->name()) <<"\n";
};
void ValueAttributeIVImpl::put(double time, float data) {
    world_->out() <<"call "<< id <<" put "<< time <<" "<< data <<"\n";
    world_->check_flush(time);
};

CoordAttributeIVImpl::CoordAttributeIVImpl(GISTObject *object, const char *name)
  : GISTCoordAttributeImpl(object,name), AttributeIVImpl(object) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" attr coord "<< Quote(this->name()) <<"\n";
};
void CoordAttributeIVImpl::put(double time, float x, float y) {
    world_->out() <<"call "<< id <<" put "<< time <<" "<< x <<" "<< y <<"\n";
    world_->check_flush(time);
};

ValueArrayAttributeIVImpl::ValueArrayAttributeIVImpl(GISTObject *object, const char *name, unsigned short arraydim)
  : GISTValueArrayAttributeImpl(object,name), AttributeIVImpl(object) 
{
    dim = arraydim;
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" attr valuearray "<< Quote(this->name()) <<" "<< dim <<"\n";
};
void ValueArrayAttributeIVImpl::put(double time, const float data[]) {
    world_->out() <<"call "<< id <<" put "<< time; 
    unsigned short i = dim;
    while(i--) world_->out() <<" "<< *data++;
    world_->out() <<"\n";
    world_->check_flush(time);
};

CoordArrayAttributeIVImpl::CoordArrayAttributeIVImpl(GISTObject *object, const char *name,unsigned short arraydim)
  : GISTCoordArrayAttributeImpl(object,name,arraydim), AttributeIVImpl(object) 
{
    dim = arraydim;
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" attr coordarray " << Quote(this->name()) <<" "<< dim <<"\n";
};
void CoordArrayAttributeIVImpl::put(double time, const float data[]) {
    world_->out() <<"call "<< id <<" put "<< time; 
    unsigned short i = dim*2;
    while(i--) world_->out() <<" "<< *data++;
    world_->out() <<"\n";
    world_->check_flush(time);
};


GraphicIVImpl::GraphicIVImpl(GISTObject *object) {
    obj_ = (ObjectIVImpl*)object->impl();
    world_ = obj_->world_;
    id = world_->comm_->newid();
};
GraphicIVImpl::~GraphicIVImpl() {
    world_->out() <<"call "<< id <<" die\n";
};
int GraphicIVImpl::byte(float v) {
    int res = int(256*v);
    if (res < 0 ) res = 0;
    if (res > 255) res = 255;
    return res;
};
const char *GraphicIVImpl::xcolor_name(float red, float green, float blue) { 
    sprintf(colorbuf, "#%02x%02x%02x", byte(red), byte(green), byte(blue));
    return colorbuf; 
};

PictureIVImpl::PictureIVImpl(GISTObject *parent,
			     const char *name,
			     int prio,
			     GISTTextAttributeImpl *vis,
			     GISTWorldImpl *world)
  : GISTPictureImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out() <<"call "<< obj_->id <<" create "<< id 
      <<" picture"<<" prio "<< prio;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
};
const char *PictureIVImpl::color_name(float red, float green, float blue) { 
    return xcolor_name(red, green, blue);
};
void PictureIVImpl::set_color(const char *color) {
    world_->out() <<"call "<< id
	<<" color "<< Quote(color) <<"\n";
};
void PictureIVImpl::set_color(GISTTextAttributeImpl *attr) {
    world_->out() <<"call "<< id
	<<" color_a "<< ((TextAttributeIVImpl*)attr)->id <<"\n";
};
void PictureIVImpl::set_brush(float thickness, unsigned short pattern) {
    world_->out() <<"call "<< id
	<<" brush "<< thickness <<" "<< pattern <<"\n";
};
void PictureIVImpl::pen_down() {
    world_->out() <<"call "<< id <<" pendown\n";
};
void PictureIVImpl::pen_up() {
    world_->out() <<"call "<< id <<" penup\n";
    world_->check_flush();
};
void PictureIVImpl::pen_up_fill() {
    world_->out() <<"call "<< id <<" penupfill\n";
    world_->check_flush();
};
void PictureIVImpl::close_path() {
    world_->out() <<"call "<< id <<" closepath\n";
    world_->check_flush();
};
void PictureIVImpl::set_pos(float x, float y) {
    world_->out() <<"call "<< id <<" setpos "<< x <<" "<< y <<"\n";
};
void PictureIVImpl::set_pos(GISTCoordAttributeImpl *attr, int steadiness) {
    world_->out() <<"call "<< id
	<<" setpos_c "<< ((CoordAttributeIVImpl*)attr)->id;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    world_->out() <<"\n";
};
void PictureIVImpl::set_path(GISTCoordArrayAttributeImpl *attr, int steadiness) {
    world_->out() <<"call "<< id <<" setpos_ca "
	<< ((CoordArrayAttributeIVImpl*)attr)->id;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    world_->out() <<"\n";
};
void PictureIVImpl::move_rel(float dx, float dy) {
    world_->out() <<"call "<< id <<" move "
	<< dx <<" "<< dy <<"\n";
};
void PictureIVImpl::move_rel(float dx, GISTValueAttributeImpl *dy, int steadiness) {
    world_->out() <<"call "<< id <<" move_y "
	<< dx <<" "<< ((ValueAttributeIVImpl*)dy)->id;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    world_->out() <<"\n";
};
void PictureIVImpl::move_rel(GISTValueAttributeImpl *dx,float dy, int steadiness) {
    world_->out() <<"call "<< id <<" move_x "
	<< ((ValueAttributeIVImpl*)dx)->id <<" "<< dy;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    world_->out() <<"\n";
};
void PictureIVImpl::move_rel(GISTValueAttributeImpl *dx,GISTValueAttributeImpl *dy, int steadiness) {
    world_->out() <<"call "<< id <<" move_xy "
	<< ((ValueAttributeIVImpl*)dx)->id
	<<" "<< ((ValueAttributeIVImpl*)dy)->id;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    world_->out() <<"\n";
};
void PictureIVImpl::move_dir(float dx, GISTValueAttributeImpl *dy, int steadiness) {
    world_->out() <<"call "<< id <<" move_w "
	<< dx <<" "<< ((ValueAttributeIVImpl*)dy)->id;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    world_->out() <<"\n";
};
void PictureIVImpl::move_dir(GISTValueAttributeImpl *dx,float dy, int steadiness) {
    world_->out() <<"call "<< id <<" move_r "
	<< ((ValueAttributeIVImpl*)dx)->id <<" "<< dy;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    world_->out() <<"\n";
};
void PictureIVImpl::move_dir(GISTValueAttributeImpl *dx,GISTValueAttributeImpl *dy, int steadiness) {
    world_->out() <<"call "<< id <<" move_rw "
	<< ((ValueAttributeIVImpl*)dx)->id
	<<" "<< ((ValueAttributeIVImpl*)dy)->id;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    world_->out() <<"\n";
    world_->check_flush();
};
void PictureIVImpl::polyline(GISTCoordArrayAttributeImpl *a, int steadiness) {
    world_->out() <<"call "<< id <<" polyline "
	<< ((CoordArrayAttributeIVImpl*)a)->id;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    world_->out() <<"\n";
    world_->check_flush();
};    
void PictureIVImpl::polygon(GISTCoordArrayAttributeImpl *a, int steadiness) {
    world_->out() <<"call "<< id <<" polygon "
	<< ((CoordArrayAttributeIVImpl*)a)->id;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    world_->out() <<"\n";
};
void PictureIVImpl::text(const char *text,float size,const char *font,const char *style) {
    world_->out() <<"call "<< id <<" text size "<< size;
    if (font) world_->out() <<" font "<< Quote(font);
    if (style) world_->out() <<" style "<< Quote(style);
    world_->out() <<" text "<< Quote(text) <<"\n";
    world_->check_flush();
};
void PictureIVImpl::text(GISTTextAttributeImpl *text,float size,const char *font,const char *style) {
    world_->out() <<"call "<< id <<" text size "<< size;
    if (font) world_->out() <<" font "<< Quote(font);
    if (style) world_->out() <<" style "<< Quote(style);
    world_->out() <<" text_a "<< ((TextAttributeIVImpl*)text)->id <<"\n";
    world_->check_flush();
};
void PictureIVImpl::text(GISTValueAttributeImpl *text,float size,int steadiness,int precision,long flags,const char *font,const char *style) {
    world_->out() <<"call "<< id <<" text size "<< size
	<<" prec "<< precision
	<<" flags "<< flags;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    if (font) world_->out() <<" font "<< Quote(font);
    if (style) world_->out() <<" style "<< Quote(style);
    world_->out() <<" value_a "<< ((ValueAttributeIVImpl*)text)->id <<"\n";
    world_->check_flush();
};
void PictureIVImpl::draw_image(const char *name,float x_align,float y_align) {
    world_->out() <<"call "<< id
	<<" image "<< x_align <<" "<< y_align
	<<" "<< Quote(name) <<"\n";
    world_->check_flush();
};
void PictureIVImpl::draw_image(GISTTextAttributeImpl *name,float x_align, float y_align) {
    world_->out() <<"call "<< id
	<<" image_a "<< x_align <<" "<< y_align
	<<" "<< ((TextAttributeIVImpl*)name)->id <<"\n";
    world_->check_flush();
};

const char *ImageIVImpl::color_name(float red, float green, float blue) { 
    return xcolor_name(red, green, blue);
};
ImageIVImpl::ImageIVImpl(GISTObject *parent,
			 const char *name,
			 const char *image, 
			 float x, float y,
			 float width, float height,
			 float hcenter, float vcenter,
			 int prio,
			 GISTTextAttributeImpl *vis,
			 GISTWorldImpl *world,
			 float minsize)
  : GISTGraphicImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out()
	<<"call "<< obj_->id <<" create "<< id <<" image"
	<<" imagename "<< Quote(image)
	<<" position "<< x <<" "<< y
	<<" extension "<< width <<" "<< height
	<<" center "<< hcenter <<" "<< vcenter
	<<" prio "<< prio;
    if (minsize != 0.0) world_->out() <<" minsize "<< minsize;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
    world_->check_flush();
};
ImageIVImpl::ImageIVImpl(GISTObject *parent,
			 const char *name,
			 const char *image, 
			 GISTCoordAttributeImpl *position,
			 float width, float height,
			 float hcenter, float vcenter,
			 int prio,
			 GISTTextAttributeImpl *vis,
			 GISTWorldImpl *world,
			 int steadiness,
			 float minsize)
  : GISTGraphicImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out()
	<<"call "<< obj_->id <<" create "<< id <<" image"
	<<" imagename "<< Quote(image)
	<<" posattr "<< ((CoordAttributeIVImpl*)position)->id
	<<" extension "<< width <<" "<< height
	<<" center "<< hcenter <<" "<< vcenter
	<<" prio "<< prio;
    if (minsize != 0.0) world_->out() <<" minsize "<< minsize;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
    world_->check_flush();
};
ImageIVImpl::ImageIVImpl(GISTObject *parent,
			 const char *name,
			 GISTTextAttributeImpl *image, 
			 float x, float y,
			 float width, float height,
			 float hcenter, float vcenter,
			 int prio,
			 GISTTextAttributeImpl *vis,
			 GISTWorldImpl *world,
			 float minsize)
  : GISTGraphicImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out()
	<<"call "<< obj_->id <<" create "<< id <<" image"
	<<" imageattr "<< ((TextAttributeIVImpl*)image)->id
	<<" position "<< x <<" "<< y
	<<" extension "<< width <<" "<< height
	<<" center "<< hcenter <<" "<< vcenter
	<<" prio "<< prio;
    if (minsize != 0.0) world_->out() <<" minsize "<< minsize;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
    world_->check_flush();
};
ImageIVImpl::ImageIVImpl(GISTObject *parent,
			 const char *name,
			 GISTTextAttributeImpl *image, 
			 GISTCoordAttributeImpl *position,
			 float width, float height,
			 float hcenter, float vcenter,
			 int prio,
			 GISTTextAttributeImpl *vis,
			 GISTWorldImpl *world,
			 int steadiness,
			 float minsize)
  : GISTGraphicImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out()
	<<"call "<< obj_->id <<" create "<< id <<" image"
	<<" imageattr "<< ((TextAttributeIVImpl*)image)->id
	<<" posattr "<< ((CoordAttributeIVImpl*)position)->id
	<<" extension "<< width <<" "<< height
	<<" center "<< hcenter <<" "<< vcenter
	<<" prio "<< prio;
    if (minsize != 0.0) world_->out() <<" minsize "<< minsize;
    if (steadiness != 1) world_->out() <<" steadiness "<< steadiness;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
    world_->check_flush();
};

const char *CurveIVImpl::color_name(float red, float green, float blue) { 
    return xcolor_name(red, green, blue);
};
CurveIVImpl::CurveIVImpl(GISTObject *parent,
			 const char *name,
			 GISTValueAttributeImpl *value, 
			 float x, float y, int type,
			 float width, float height,
			 float hcenter, float vcenter,
			 const char *foreg, const char *backg,
			 int prio,
			 GISTTextAttributeImpl *vis,
			 GISTWorldImpl *world,
			 float minsize)
  : GISTGraphicImpl(parent,name), CurveIVImpl::GraphicIVImpl(parent) 
{
    world_->out()
	<<"call "<< obj_->id <<" create "<< id <<" curve "
	<< ((ValueAttributeIVImpl*)value)->id <<" type "<< type
	<<" position "<< x <<" "<< y
	<<" extension "<< width <<" "<< height
	<<" center "<< hcenter <<" "<< vcenter
	<<" foreg "<< Quote(foreg)
	<<" backg "<< Quote(backg)
	<<" prio "<< prio;
    if (minsize != 0.0) world_->out() <<" minsize "<< minsize;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
    world_->check_flush();
};
void CurveIVImpl::hint(const char*text) {
    world_->out() <<"call "<< id <<" hint "<< Quote(text) <<"\n";
};

const char *DiagramIVImpl::color_name(float red, float green, float blue) { 
    return xcolor_name(red, green, blue);
};
DiagramIVImpl::DiagramIVImpl(GISTObject *parent,
			     const char *name,
			     GISTCoordArrayAttributeImpl *value, 
			     float x, float y, int type,
			     float width, float height,
			     float hcenter, float vcenter,
			     const char *foreg, const char *backg,
			     int prio,
			     GISTTextAttributeImpl *vis,
			     GISTWorldImpl *world,
			     float minsize)
  : GISTGraphicImpl(parent,name), GraphicIVImpl(parent) 
{
    world_->out()
	<<"call "<< obj_->id <<" create "<< id <<" diagram "
	<< ((CoordArrayAttributeIVImpl*)value)->id <<" type "<< type
	<<" position "<< x <<" "<< y
	<<" extension "<< width <<" "<< height
	<<" center "<< hcenter <<" "<< vcenter
	<<" foreg "<< Quote(foreg)
	<<" backg "<< Quote(backg)
	<<" prio "<< prio;
    if (minsize != 0.0) world_->out() <<" minsize "<< minsize;
    if (vis) world_->out() <<" vis "<< ((TextAttributeIVImpl*)vis)->id;
    if (world) world_->out() <<" world "<< ((WorldIVImpl*)world)->id;
    world_->out() <<" name "<< Quote(this->name()) <<"\n"; 
    world_->check_flush();
};
void DiagramIVImpl::hint(const char*text) {
    world_->out() <<"call "<< id <<" hint "<< Quote(text) <<"\n";
};
