// -*- C++ -*-

//    SocketBuf derived from libg++ 2.3 filebuf implementation

//    This is part of the iostream library, providing input/output for C++.
//    Copyright (C) 1991, 1992 Per Bothner.
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <iostreamP.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include <stdlib.h>
#include <unistd.h>

#include "SocketBuf.h"


#define CLOSED_FILEBUF_FLAGS \
  (_IO_IS_FILEBUF+_IO_NO_READS+_IO_NO_WRITES+_IO_TIED_PUT_GET)

void SocketBuf::init()
{
    _IO_file_init(this);
}

SocketBuf::SocketBuf() : streambuf(CLOSED_FILEBUF_FLAGS)
{
    _IO_file_init(this);
}

SocketBuf::SocketBuf(int fd) : streambuf(CLOSED_FILEBUF_FLAGS)
{
    _IO_file_init(this);
    _IO_file_attach(this, fd);
}

SocketBuf::~SocketBuf()
{
    if (!(xflags() & _IO_DELETE_DONT_CLOSE))
	close();

    _un_link();
}

SocketBuf* SocketBuf::attach(int fd)
{
    return (SocketBuf*)_IO_file_attach(this, fd);
}

/*
 * Allocate a socket buffer with seperate parts for reading and writing
 */

int SocketBuf::doallocate()
{
    char *p;
    
    p = new char[2 * BUFSIZE];

    setb(p, p + 2*BUFSIZE, 0);
    setp(p, p + BUFSIZE);
    setg(p + BUFSIZE, p + BUFSIZE, p + BUFSIZE);

// Set socket buffer to line buffered
    _flags |= _IO_LINE_BUF;

    return 1;
}

int SocketBuf::overflow(int c)
{
    if (xflags() & _IO_NO_WRITES) // SET ERROR
	return EOF;

    // Allocate a buffer if needed.
    allocbuf();

    if (c == EOF)
	return do_flush();

    if (pptr() >= epptr() ) // Buffer is really full
	if (do_flush() == EOF)
	    return EOF;

    xput_char(c);

    if (linebuffered() && c == '\n')
	if (do_flush() == EOF)
	    return EOF;

    return (unsigned char)c;
}

int SocketBuf::underflow()
{
    if (xflags() & _IO_NO_READS)
	return EOF;

    if (gptr() < egptr())
	return *(unsigned char*)gptr();

    // Allocate a buffer if needed.
    allocbuf();

    // FIXME This can/should be moved to __streambuf ??
    if (xflags() & (_IO_LINE_BUF|_IO_UNBUFFERED)) {
	// Flush all line buffered files before reading.
	streambuf::flush_all_linebuffered();
    }

    streamsize count = sys_read(base()+BUFSIZE, BUFSIZE);
    if (count <= 0) {
	if (count == 0)
	    xsetflags(_IO_EOF_SEEN);
	else
	    xsetflags(_IO_ERR_SEEN), count = 0;
    }
    setg(base()+BUFSIZE, base()+BUFSIZE, base()+BUFSIZE+count);
    if (count == 0)
	return EOF;
    return *(unsigned char*)gptr();
}

int SocketBuf::do_write(const char *data, int to_do)
{
    if (to_do == 0)
	return 0;

    streamsize count = sys_write(data, to_do);
    if (_cur_column)
	_cur_column = _IO_adjust_column(_cur_column - 1, data, to_do) + 1;
    if (count != to_do)
	return EOF;

    if (xflags() & _IO_UNBUFFERED)
	setp(base(), base());
    else
	setp(base(), base()+BUFSIZE);

    return 0;
}

int SocketBuf::sync()
{
    if (pptr() > pbase())
	if (do_flush()) return EOF;

    return 0;
}

SocketBuf* SocketBuf::close()
{
    if (!is_open())
	return NULL;

    // This flushes as well as switching mode.
    if (pptr() > pbase())
	if(overflow(EOF) == EOF)
	    return NULL;

    int status = sys_close();

    // Free buffer.
    char *p = base();
    setb(NULL, NULL, 0);
    setg(NULL, NULL, NULL);
    setp(NULL, NULL);
    delete [] p;

    _un_link();
    _flags = _IO_MAGIC|CLOSED_FILEBUF_FLAGS;
    _fileno = EOF;

    return status < 0 ? (SocketBuf *)NULL : this;
}

streamsize SocketBuf::sys_read(char* buf, streamsize size)
{
    for (;;) {
	streamsize count = ::read(fd(), buf, size);
	if (count != -1 || errno != EINTR)
	    return count;
    }
}

streamsize SocketBuf::sys_write(const char *buf, streamsize n)
{
    streamsize to_do = n;
    while (to_do > 0) {
	streamsize count = ::write(fd(), buf, to_do);
	if (count == EOF) {
	    if (errno == EINTR)
		continue;
	    else {
		_flags |= _IO_ERR_SEEN;
		break;
	    }
	}
	to_do -= count;
	buf += count;
    }
    n -= to_do;

    return n;
}

int SocketBuf::sys_close()
{
    return ::close(fd());
}
