//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: GetOpt.h,v 0.30 1996-08-07 17:59:08+02 steppler Exp $
 *
 * Class: CNGetOpt --- Interface to GNU getopt()
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

/* Partly from GNU getopt.h: */

/* Declarations for getopt.
   Copyright (C) 1989, 1990, 1991, 1992, 1993 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef __CNGetOpt_h
#define __CNGetOpt_h

#include <stdlib.h>

#include <CNCL/Object.h>		// Base class

extern CNClassDesc CN_GETOPT;	// Class CNGetOpt description


/*
 * The class CNGetOpt
 */

class CNGetOpt : public CNObject
{
  public:	/***** Struct definition *************************************/
    enum ParamType { NOPARAM=0, WPARAM=1, OPTPARAM=2 };

    /* Describe the long-named options requested by the application.
       The LONG_OPTIONS argument to getopt_long or getopt_long_only is a vector
       of `struct option' terminated by an element containing a name which is
       zero.
       
       The field `has_arg' is:
       NOPARAM    	  (or 0) if the option does not take an argument,
       WPARAM             (or 1) if the option requires an argument,
       OPTPARAM           (or 2) if the option takes an optional argument.
       
       If the field `flag' is not NULL, it points to a variable that is set
       to the value given in the field `val' when the option is found, but
       left unchanged if the option is not found.
       
       To have a long-named option do something other than set an `int' to
       a compiled-in constant, such as set a value from `optarg', set the
       option's `flag' field to zero and its `val' field to a nonzero
       value (the equivalent single-letter option character, if there is
       one).  For long options that have a zero `flag' field, `getopt'
       returns the contents of the `val' field.  */
    struct option
    {
	char *name;
	/* has_arg can't be an enum because some compilers complain about
	   type mismatches in all the code that assumes it is an int.  */
	int has_arg;
	int *flag;
	int val;
    };

    
  public:	/***** Constructors ******************************************/
    CNGetOpt()
	    : i_argc(0), i_argv(NIL), i_index(0),
	      i_char_options(NIL), i_long_options(NIL), n_long(0),
	      i_optarg(NIL), i_optind(0), i_opterr(1), i_optopt('?')        {}
    CNGetOpt(CNParam *)
	    : i_argc(0), i_argv(NIL), i_index(0),
	      i_char_options(NIL), i_long_options(NIL), n_long(0),
	      i_optarg(NIL), i_optind(0), i_opterr(1), i_optopt('?')        {}

    CNGetOpt(int argc, char **argv, char *copts=NIL)
	    : i_argc(argc), i_argv(argv), i_index(0),
	      i_char_options(copts), i_long_options(NIL), n_long(0),
	      i_optarg(NIL), i_optind(0), i_opterr(1), i_optopt('?')        {}


  public:	/***** Public interface **************************************/
    void   set_args        (int argc, char **argv);
    void   set_char_options(char *copts);
    void   set_long_options(option *lopts);
    void   add_long_option (char *lopt, ParamType pt, char copt);
  
    int    getopt();
    int    getopt(int argc, char *const *argv, const char *optstring);
    int    getopt_long(int argc, char *const *argv, const char *options,
		       const struct option *long_options, int *opt_index);
    int    getopt_long_only(int argc, char *const *argv, const char *options,
			    const struct option *long_options, int *opt_index);
 
    int    operator() ()   { return getopt(); }
     
    char  *optarg() 	   { return i_optarg; }
    double optarg_double() { return atof(i_optarg); }
    int    optarg_int()    { return atoi(i_optarg); }
    int    optind() 	   { return i_optind; }
    int    optopt() 	   { return i_optopt; }
    void   opterr(int err) { i_opterr = err; }
    

  private:	/***** Internal private members ******************************/
    int i_argc;
    char **i_argv;
    int i_index;
    char *i_char_options;
    option *i_long_options;
    int n_long;
    
    enum { MAXLONGOPTIONS=32 };

  private:	/***** Stuff from original getopt.h / getopt.c ***************/

    /* For communication from `getopt' to the caller.
       When `getopt' finds an option that takes an argument,
       the argument value is returned here.
       Also, when `ordering' is RETURN_IN_ORDER,
       each non-option ARGV-element is returned here.  */
    char *i_optarg;

    /* Index in ARGV of the next element to be scanned.
       This is used for communication to and from the caller
       and for communication between successive calls to `getopt'.
       On entry to `getopt', zero means this is the first call; initialize.
       When `getopt' returns EOF, this is the index of the first of the
       non-option elements that the caller should itself scan.
       Otherwise, `optind' communicates from one call to the next
       how much of ARGV has been scanned so far.  */
    int i_optind;

    /* Callers store zero here to inhibit the error message `getopt' prints
       for unrecognized options.  */
    int i_opterr;

    /* Set to an option character which was unrecognized.  */
    int i_optopt;

    /* Internal stuff */
    char *nextchar;
    enum { REQUIRE_ORDER, PERMUTE, RETURN_IN_ORDER } ordering;
    int first_nonopt;
    int last_nonopt;

    /* Internal functions */
    void exchange (char **argv);
    int _getopt_internal(int argc, char *const *argv, const char *optstring,
			 const struct option *longopts, int *longind,
			 int long_only                                      );
    



  public:	/***** Member functions required by CNCL *********************/
    virtual CNClassDesc class_desc() const	// CNClass description
    { return CN_GETOPT; }
            
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    { return desc == CN_GETOPT ? TRUE : CNObject::is_a(desc); }
        
    static CNGetOpt *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
	return (CNGetOpt *)obj;
#   else
	return (CNGetOpt *)( !obj || obj->is_a(CN_GETOPT)
	       ? obj : fatal_type(obj->class_desc(), CN_GETOPT) );
#   endif
    }
    
    static CNObject *new_object(CNParam *param = NIL) // Object creation
    { return param ? new CNGetOpt(param) : new CNGetOpt; }
};

#endif /**__CNGetOpt_h**/
