//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: IniFile.c,v 0.32 1996-08-07 17:59:12+02 steppler Exp $
 *
 * Class: CNIniFile --- .ini-style config file
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include <stdlib.h>

#include <fstream.h>

#include <CNCL/Class.h>

#include "IniFile.h"



static CNString ret_FALSE = "FALSE";
static CNString ret_TRUE  = "TRUE";



/*****************************************************************************/
/*
 * Internal class CNIniEntry
 */

extern CNClassDesc CN_INIENTRY;	// Class IniEntry description


/*
 * The class IniEntry
 */

class CNIniEntry : public CNObject
{
  public:	/***** Public interface **************************************/
    CNIniEntry() : t(EMPTY), l(), r() {}
    
    enum Type { COMM, EMPTY, SECTION, ENTRY, ENTRYNOEQ };

    Type t;
    CNString l, r;

  public:	/***** Member functions required by CNCL *********************/
    virtual CNClassDesc class_desc() const	// CNClass description
    { return CN_INIENTRY; }
            
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    { return desc == CN_INIENTRY ? TRUE : CNObject::is_a(desc); }
        
    static CNIniEntry *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
	return (CNIniEntry *)obj;
#   else
	return (CNIniEntry *)( !obj || obj->is_a(CN_INIENTRY)
	       ? obj : fatal_type(obj->class_desc(), CN_INIENTRY) );
#   endif
    }
    
    // Print/debug output
    virtual void print(ostream &strm = cout) const;
};



/***** Output *****/

void CNIniEntry::print(ostream &strm) const
{
    switch(t)
    {
    case ENTRY:
	strm << l << "=" << r << endl;
	break;
    case SECTION:
	strm << "[" << l << "]" << endl;
	break;
    case COMM:
    case EMPTY:
    case ENTRYNOEQ:
    default:
	strm << l << endl;
	break;
    }
}



/***** CNCL stuff for type information *****/

// Describing object for class IniEntry
static CNClass CNIniEntry_desc("CNIniEntry", "$Revision: 0.32 $", NIL);

// "Type" for type checking functions
CNClassDesc CN_INIENTRY = &CNIniEntry_desc;



/*****************************************************************************/


/*
 * Constructors
 */
CNIniFile::CNIniFile()
	: CNNamed(""), ini_section(""), ini_list(), ini_trav(), ini_sec()
{}


CNIniFile::CNIniFile(CNParam *)
	: CNNamed(""), ini_section(""), ini_list(), ini_trav(), ini_sec()
{}


CNIniFile::CNIniFile(CNStringR name)
	: CNNamed(name), ini_section(""), ini_list(), ini_trav(), ini_sec()
{
    read();
}

CNIniFile::~CNIniFile()
{
    ini_list.delete_all_w_obj();
}


/*
 * Read .ini file into list
 */
int CNIniFile::read(CNStringR name)
{
    set_name(name);
    return read();
}


int CNIniFile::read()
{
    CNString in, l, r;
    CNIniEntry *p;
    int pos;

    
    ini_list.delete_all_w_obj();

    CNString temp(get_name());
    ifstream istr(temp);
    if(!istr)
    {
	error("CNIniFile: ", "can't open ", get_name());
	return CNERROR;
    }

    while(TRUE)
    {
	istr >> in;
	if(istr.eof())
	    break;
	
	p = new CNIniEntry;

	in.strip_crlf();
	in.strip_space();

	switch(in[0])
	{
	case '#':			// Comment
	case ';':
	    p->t = CNIniEntry::COMM;
	    p->l = in;
	    break;

	case '\0':			// Empty line
	    p->t = CNIniEntry::EMPTY;
	    break;

	case '[':			// Section
	    p->t = CNIniEntry::SECTION;
	    in.del(0, 1);
	    if(in[-1] == ']')
		in.del(-1, 1);
	    in.strip_space();
	    p->l = in;
	    break;

	default:			// Normal entry
	    pos = in.upsearch('=');
	    if(pos == CNERROR)
	    {
		p->t = CNIniEntry::ENTRYNOEQ;
		p->l = in;
	    }
	    else 
	    {
		p->t = CNIniEntry::ENTRY;
		p->l = in.before(pos);
		p->r = in.after(pos+1);
	    }
	    p->l.strip_space();
	    p->r.strip_space();
	    break;
	}
	
	ini_list.append(p);
    }

    return 0;
}



/*
 * Write .ini file
 */
int CNIniFile::write(CNStringR name)
{
    set_name(name);
    return write();
}


int CNIniFile::write()
{
    CNString temp(get_name());
    ofstream out(temp);
    if(!out)
    {
	error("CNIniFile: ", "can't open ", get_name());
	return CNERROR;
    }

    out << *this;

    return 0;
}



/*
 * Read entries
 */
void CNIniFile::set_section(CNStringR section)
{
    CNObject *p;
    CNIniEntry *pi;

    ini_trav.reset(ini_list);
    while((p = ini_trav++))
    {
	pi = CNIniEntry::cast_from_object(p);
	if(pi->t == CNIniEntry::SECTION  &&  pi->l == section)
	{
	    ini_sec = ini_trav;
	    return;
	}
    }
}


CNStringR CNIniFile::get_section()
{
    return ini_section;
}


CNStringR CNIniFile::get_entry(CNStringR name, bool first)
{
    CNObject *p;
    CNIniEntry *pi;

    if(ini_sec.position() == NIL)
	return ret_FALSE;

    if(first)
	ini_trav = ini_sec;
    while((p = ++ini_trav))
    {
	pi = CNIniEntry::cast_from_object(p);
	if(pi->t == CNIniEntry::ENTRY && pi->l == name)
	    return pi->r;
	if(pi->t == CNIniEntry::ENTRYNOEQ && pi->l == name)
	    return ret_TRUE;
    }

    return ret_FALSE;
}


double CNIniFile::get_double(CNStringR name, bool first)
{
    CNString s;

    s = get_entry(name, first);
    if(s == ret_FALSE)
	return -1;
    else
	return atof(s);
}


int CNIniFile::get_int(CNStringR name, bool first)
{
    CNString s;

    s = get_entry(name, first);
    if(s == ret_FALSE)
	return -1;
    else
	return atoi(s);
}



/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNIniFile::print(ostream &strm) const
{
    CNDLIterator trav(ini_list);
    CNObject *p;
    
    while((p = trav++))
	p->print(strm);
}

// Debug output
void CNIniFile::dump(ostream &strm) const
{
    print(strm);
}

// IOStream operator <<
ostream &operator << (ostream &strm, const CNIniFile &obj)
{
    obj.print(strm);
    return strm;
}

ostream &operator << (ostream &strm, const CNIniFile *obj)
{
    if(obj)
	obj->print(strm);
    else
	strm << "(NIL)";
    return strm;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNIniFile
static CNClass CNIniFile_desc("CNIniFile", "$Revision: 0.32 $",
			      CNIniFile::new_object);

// "Type" for type checking functions
CNClassDesc CN_INIFILE = &CNIniFile_desc;
