//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: ParseArgs.h,v 1.2 1996-08-07 17:59:24+02 steppler Exp $
 *
 * Class: CNParseArgs --- Argument Parser
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 * 
 * As an exception to this rule you may use this template to generate
 * your own classes. This does not cause these classes to be covered by
 * the GNU Library General Public License. This exception does not
 * however invalidate any other reasons why the resulting program must be
 * covered by the GNU Library General Public License. 
 *****************************************************************************/

#ifndef __CNParseArgs_h
#define __CNParseArgs_h


#include <CNCL/Object.h>		// Base class

extern CNClassDesc CN_PARSEARGS;	// Class CNParseArgs description

/* 
 * Classes used for argument parsing to CNParseArgs
 */
enum BlankHandling {NO_BLANK, BLANK}; // handling of blanks after option

struct OptionDescription {
   char * option_string;    // option string to be parsed
   BlankHandling blank;
   char * default_argument; // default argument 
   char * usage;            // optional string for usage
   char * argument_string;  // contains the argument on return
};

struct CNParseArgsList {
   int number_of_args; // number of elements of args_array 
   OptionDescription * args_array; // array containing description
			           // of options 
   int mandatory_args;     // number of args that have to be given
   int positional_args;    // number of args that may be given without
			   // option but in correct positional order
   void (*usage_func)(int argc, char **argv);   // Pointer to a user
			   // supplied usage function 
};
   


/*
 * The class CNParseArgs
 */

class CNParseArgs : public CNObject
{

/****************** Constructors ******************************************/
  public:
    CNParseArgs();
    CNParseArgs(CNParam *);
    CNParseArgs(int argc, char **argv, CNParseArgsList &args_list);
    char * operator[](int index);
    
/****************** Public interface **************************************/
  public:
    static void usage_screen(char * prog_name, CNParseArgsList &args); 
    void override_nil(int first_arg, int last_arg = 0);
//    static const char * const empty_string; 

/****************** Internal private members ******************************/
  private:
    const int number_args;
    const char **args_ptr;
    const CNParseArgsList * option_list;


/****************** Member functions required by CNCL *********************/
  public:
    virtual CNClassDesc class_desc() const	// CNClass description
    { return CN_PARSEARGS; }
            
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    { return desc == CN_PARSEARGS ? TRUE : CNObject::is_a(desc); }
        
    static CNParseArgs *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
	return (CNParseArgs *)obj;
#   else
	return (CNParseArgs *)( !obj || obj->is_a(CN_PARSEARGS)
		? obj : fatal_type(obj->class_desc(), CN_PARSEARGS) );
#   endif
    }
    
    static CNObject *new_object(CNParam *param = NIL) // Object creation
    { return param ? new CNParseArgs(param) : new CNParseArgs; }
    
    // Print/debug output
    virtual void print(ostream &strm = cout) const;
    virtual void dump (ostream &strm = cout) const;
};


ostream &operator << (ostream &strm, const CNParseArgs &obj);
ostream &operator << (ostream &strm, const CNParseArgs *obj);

// Usefull template to read arguments given as strings into
// (e.g. numeric) values.  

template <class T> void assign_from_string(const char *string, T &input) {
// Reads input from `string' and assigns the value to the referent of 
// `input'. The string stream object is only created as a temporary
// and will be destructed as the statement is finished.
  istrstream(string, strlen(string)) >> input;
// Use of temporary is preferred to explicit creation and explicit
// call to d'tor:
//   (*new istrstream(string, strlen(string)) >> input).~istream();
}

/* Syntax Rules: 
 * =============
 * If an option is given, but no argument, CNParseArgs returns a white space
 * in `args_list.args_array[i].argument_string'. 
 * 
 * If an option is left out, the default argument is returned by 
 * CNParseArgs.
 *
 * Thus it is possible to distinguish an option not specified from
 * one specified but without an argument.
 * 
 * If `args_list.args_array[i].blank' equals `BLANK' the argument has
 * to be separated from the option string by white space characters. If
 * it equals `NO_BLANK' the option must be followed immediately by the
 * argument without any separating character. Such an argument must
 * not occur as the prefix of any other argument in the list. This is
 * checked by the program. Besides the list of options is not
 * checked for ambiguous specification. If an option is specified more
 * than once the result is undefined in `args_list.args_array'. 
 * 
 * A `-' as the first character of an argument has to be given as
 * `--'. This is necessary to allow the specification of negative
 * numbers as arguments. The pointer which has returned in
 * `args_list.args_array[i].argument_string' is pointing to the second
 * `-'. Thus the argument can directly be parsed semantically.
 * 
 * The first `mandatory_args' arguments of 'args_list.args_array' have
 * to be specified, otherwise the program exits (as long as `usage_func'
 * is `NIL') and reports on the program's usage of the
 * parameters given in `args_list'. If the function_pointer
 * `usage_func' is given, no report is given but `usage_func()'
 * it is called.
 *
 * The number `positional_args' of arguments may be specified without
 * an option string and anywhere in the input string but in the correct
 * positional order.
 * ---If positional options are at first given with option string and later
 * are specified without option string the first specification is overwritten.
 *
 *
 * Implementation constraints: 
 * ==========================
 * 
 * It is not possible to specify an empty string `""' for a positional
 * argument without specifying the option.
 * 
 * Depending on which number of `mandatory_args' and `positional_args'
 * is smaller this number of the first arguments is as well mandatory
 * as positional. It is not possible to specify the positional and
 * mandatory property separately for these arguments.
 *   */


#endif /**__CNParseArgs_h**/
