//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: Ref.c,v 1.4 1996-08-07 17:59:28+02 steppler Exp $
 *
 * Class: CNRef --- Base class for classes with reference counting
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include <CNCL/Class.h>
#include <strstream.h>

#include "Ref.h"



/*
 * static member
 */
bool CNRef::debug_ref  = FALSE;
bool CNRef::debug_list = FALSE;


void CNRef::set_debug(bool r, bool l)
{
    debug_ref  = r;
    debug_list = l;
}



/*
 * Constructor
 */
CNRef::CNRef()
	: ref_count(0)
{}



/*
 * Reference/dereference
 */
void CNRef::ref()
{
    ref_count++;

    if (debug_ref)
    {
	// ... debug output ...
	ostrstream str;

	str << "CNRef::ref(),\t\tthis = " << hex << this
	    << ", count = " << dec << ref_count << ends;
	CNCL::info(str.str());
    }
}

void CNRef::ref(CNRef* aRef)
{
    if (aRef)
	aRef->ref();
}

void CNRef::deref()
{
    if (ref_count == 0)
    {
	CNCL::fatal(NIL, "trying to deref() an object with count = 0!");
	/**NOT REACHED**/
	return;
    }
    
    ref_count--;
    
    if (debug_ref)
    {
	// ... debug output ...
	ostrstream str;

	str << "CNRef::deref(),\t\tthis = " << hex << this
	    << ", count = " << dec << ref_count;
	if (!ref_count)
	    str << " --> deleted";
	str << ends;
	CNCL::info(str.str());
    }

    // Delete this object, if reference count is 0
    if (ref_count == 0)
	delete this;
}

void CNRef::deref(CNRef* aRef)
{
    if (aRef)
	aRef->deref();
}

/*
 * Decrease reference count without deleting referenced object
 */
void CNRef::decrease()
{
    if (!ref_count)
    {
	CNCL::fatal(NIL, "trying to decrease() an object with count = 0!");
	/**NOT REACHED**/
	return;
    }
    ref_count--;
    if (debug_ref)
    {
	// ... debug output ...
	ostrstream str;

	str << "CNRef::decrease(),\tthis = " << hex << this
	    << ", count = " << dec << ref_count;
	str << ends;
	CNCL::info(str.str());
    }
}

void CNRef::decrease(CNRef* aRef)
{
    if (aRef)
	aRef->decrease();
}

/*
 * Decrease reference count if non zero
 * Delete object if nescessary
 */
void CNRef::del(CNRef* aRef)
{
    if (aRef)
    {
	if (aRef->ref_count)
	    aRef->deref();
	else
	{
	    if (debug_ref)
	    {
		// ... debug output ...
		ostrstream str;

		str << "CNRef::del(),\t\tthis = " << hex << aRef
		    << ", count = 0 --> deleted";
		str << ends;
		CNCL::info(str.str());
	    }
	    delete aRef;
	}
    }
}

/*
 * Get reference count
 */
unsigned long CNRef::get_count() const
{
    return ref_count;
}
