//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: BatchMeans.h,v 1.4 1996-08-07 18:02:44+02 steppler Exp $
 *
 * Class: CNBatchMeans --- Batch Means Evaluation
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 * 
 * As an exception to this rule you may use this template to generate
 * your own classes. This does not cause these classes to be covered by
 * the GNU Library General Public License. This exception does not
 * however invalidate any other reasons why the resulting program must be
 * covered by the GNU Library General Public License. 
 *****************************************************************************/

#ifndef __CNBatchMeans_h
#define __CNBatchMeans_h


#include <CNCL/Statistics.h>		// Base class

#include <math.h>
#include <limits.h>
#include <float.h>

#include <iostream.h>
#include <iomanip.h>

#include <CNCL/minmax.h>


extern CNClassDesc CN_BATCHMEANS;	// Class CNBatchMeans description


/*
 * The class CNBatchMeans
 */

class CNBatchMeans : public CNStatistics
{
public:	/***** Constructors ******************************************/
    CNBatchMeans(double, double, long, long, long,
		 short = 95, const char* = NIL, const char* = NIL);
    CNBatchMeans(double, double, long, long, double,
		 short = 95, const char* = NIL, const char* = NIL);
    CNBatchMeans(CNParam *) {}

    ~CNBatchMeans();
    
private:
    CNBatchMeans()          {}

public:	/***** Public interface **************************************/

    void put(double);

    unsigned long trials() const { return nrv; }
    double min()           const { return min_val; }
    double max()           const { return max_val; }
    
    double mean()     const;
    double variance() const;
    double sigma()    const;
    
    Phase status() const { return phase; }
    bool end()     const { return end_reached; }

    void reset();
    void print(CNStatistics::Type = CNStatistics::DF, ostream& = cout) const;

public:
    struct resultline
    {
	double x;            // left interval limit
	double fx, gx, rh;   // f(x), g(x), rel.H
	double ferr, gerr;   // rel.error
	double fconf, gconf; // confidence interval
    };
    
public:
    long min_index()   const { return 0; }
    long max_index()   const { return long(intervals - 1); }
    long groups_done() const { return long(act_group); }
    
    const struct resultline *get_result(long);
    
    void change_error(double ne) { max_err = ne; }

    double p(double);  // p(x)
    double f(double);  // f(x)
    double g(double);  // g(x)

    double bayes_err()       const;  // relative bayes error for mean
    double mean_confidence() const;  // confidence intervall for mean
    double correlation()     const;  // corr. coeff. 1st order of y_i
    
private:
    /* a single batch (size_of_group) */
    struct Group
    {
	unsigned long *histo;        // histogram 
	unsigned long lower, higher; // values out of range
	double sum_x;                // sum of x_i
	double qsum_x;               // square sum of x_i
	unsigned long remaining, // num. values still needed in act. batch
	    sog, inter;
	
	void reset();
	Group(unsigned long, unsigned long);
	~Group();
    };

    /* evaluation for every interval */
    struct Interval
    {
	unsigned long hits;
	double sum_rh, qsum_rh;
	double sum_f,  qsum_f;
	double sum_g,  qsum_g;
	
	void reset();
	Interval();
    };
    
private:	/***** Internal private members ******************************/
    
    unsigned long nrv;         // number of trials
    double min_val, max_val;   // extrema

    /* --- Parameters of Evaluation --- */
    double bottom, top;        // range of evaluation
    unsigned long intervals;   // number of intervals
    unsigned long size_of_groups,
	number_of_groups;      
    double max_err;            // limiting bayes error
    bool fixed_length;         // flag: use error or sog * nog?
    short confidence;          // confidence interval
    /* --------------------------------- */
    
    Phase phase;
    bool end_reached;
    
    unsigned long wasted_left, wasted_right; // values outside of limits
    double int_step;                         // size of each interval

    unsigned long act_group;  // number of actual batch
    Group *actual_group;      // current batch
    Interval *result;         // cummulative results
    
    double sum_y;        // sum of y_i (= sum_xi / s_o_group = gr_mean)
    double qsum_y;       // square of y_i
    double sum_gr_var;   // sum of group variances
    double qsum_gr_var;  // square sum of gr_var 

    double *y_i; // need to hold them all to calculate correlation coefficient
    resultline line;
    
private:
    void eval_group();
    double calc_confidence(unsigned long, double)    const; 
    double calc_error(unsigned long, double, double) const;
    double calc_var(unsigned long, double, double)   const;
		    
public:	/***** Member functions required by CNCL *********************/
    virtual CNClassDesc class_desc() const	// CNClass description
    { return CN_BATCHMEANS; }
            
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    { return desc == CN_BATCHMEANS ? TRUE : CNStatistics::is_a(desc); }
        
    static CNBatchMeans *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
	return (CNBatchMeans *)obj;
#   else
	return (CNBatchMeans *)( !obj || obj->is_a(CN_BATCHMEANS)
	       ? obj : fatal_type(obj->class_desc(), CN_BATCHMEANS) );
#   endif
    }
    
    static CNObject *new_object(CNParam *param = NIL) // Object creation
    { return param ? new CNBatchMeans(param) : new CNBatchMeans; }
    
    // Print/debug output
    virtual void dump (ostream &strm = cout) const;
    virtual void print(ostream &strm = cout) const;
};



ostream &operator << (ostream &strm, const CNBatchMeans &obj);
ostream &operator << (ostream &strm, const CNBatchMeans *obj);

#endif /**__CNBatchMeans_h**/
