//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_                C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_                          N E T W O R K S
 * |_        |_  |_  |_  |_        |_                                C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_                      L I B R A R Y
 *
 * $Id: DLREF.c,v 0.34 1996-08-07 18:02:55+02 steppler Exp $
 *
 * Class: CNDLREF --- Discrete LRE (LRE III) for distribution function
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include "DLREF.h"


CNDLREF::CNDLREF(double * xvalues, long level, double eerror, 
		 double pre_first, const char* llrename, const char* ttext, 
		 double fmin, unsigned long mmax_nrv)
	: CNDLRE(xvalues, level, eerror, pre_first, llrename, ttext,
		 mmax_nrv)
{
    cur_level_index = index_max - 1;
    Fmin = fmin;
}

CNDLREF::CNDLREF(double xxmin, double xxmax, double iint_size, double eerror, 
		 double pre_first, const char* lrename, const char* ttext,
		 double fmin, unsigned long mmax_nrv)
	: CNDLRE(xxmin, xxmax, iint_size, eerror, pre_first, lrename, ttext,
		 mmax_nrv)
{
    cur_level_index = index_max - 1;
    Fmin = fmin;
}



void CNDLREF::put( double cur_rv )
{
    if( nrv+1 < max_nrv )
    {
	cur_index = get_index( cur_rv );
	if( cur_index == NO_INDEX )
        {
	    CNCL::warning("Wrong x value in DLREF::put !");
	    return;
        }
      
	nrv++;
	h++;
	rv_sum        += cur_rv;
	rv_square_sum += cur_rv*cur_rv;
	if (max_value < cur_rv) max_value = cur_rv; 
	if (min_value > cur_rv) min_value = cur_rv;
      
	if( cur_index == LOWER ) 
        {
	    wasted_left++;
	    pre_rv = xmin-1.0;
	    pre_index = index_min;
	    return ;
        }
	else if( cur_index == GREATER )
        {
	    wasted_right++;
	    if( pre_rv < cur_rv ) 
		for(long i=pre_index; i<index_max-1 ; i++) (x[ i ].c)++;
	    pre_rv    = xmax+1.0;
	    pre_index = index_max-1;
	    return ;
        }
	else if( pre_rv < cur_rv ) 
	    for(long i=pre_index; i<cur_index; i++) (x[ i ].c)++;
	x[ cur_index ].h++;
      
	// check if ready
	if (h >= n0)
        {
	    phase = rtc();
	    h = 0;
        }
      
	// save current values
	pre_rv    = cur_rv;
	pre_index = cur_index;
    }
    else
    {
	phase = END;
	end_reached = TRUE; 
    }
}


void CNDLREF::print( ostream &output ) const
{
    if (nrv == 0) return;

    double rho=0., sigrho=0., d=0.;
    double uf;
    double nf = double(nrv);
    double vf = wasted_left;
    double F  = vf/nf;
    double cf = 0.0; 
    long i;
  
    // header
    output << setprecision(4);
    output.setf(ios::left, ios::fixed);
    output << "#LRE RESULT (THIS IS A MAGIC LINE)" << endl;
    output << "#---------------------------------------------------------------------------" << endl;
    output << "#    Discrete Lre --- distribution function" << endl;
    output << "#---------------------------------------------------------------------------" << endl;
    output << "#Name: " << name << endl;
    output << "#Text: " << text << endl;
    output << "#Default values: max.rel. error = " << setw(4) << reler_max*100 
	   << "%  X_min: " << xmin << "  X_max: " << xmax << endl;
    output << "#trials: " << nrv ;
    output << "\tmean: " << mean() << "\tvariance: " << variance() << endl;
    output << "#trials < X_min: " << wasted_left << "\ttrials > X_max: " 
	   << wasted_right << endl;
    switch (reason)
    {
    case OK:
	output << "#All levels calculated." << endl;
	break;
    case MIN:
	output << "#Evaluated till Fmin = " << Fmin << "." << endl;
	break;
    case LAST:
	output << "#Last level couldn't be calculated." << endl;
	break;
    default:
	break;
    }
    output << "#---------------------------------------------------------------------------" << endl;
    output << "#F(x)           x               rel.error"
	   << "       rho             sigrho" << endl;
  
    // results
    output << setprecision(7);
    for(i = index_min; i < index_max-1; i++)
    {
	vf += double( x[ i ].h ); 
	cf  = double( x[ i ].c );
	if( nf < 1000.0 || vf < 100.0 || (vf-cf) < 10.0 ) // status != END
        {
	    output << "#";
	    output << setiosflags(ios::scientific);
	    output << F*base;
	    output << " " << setw(15) << x[i].x
		   << endl;
	    F = vf/nf;
	    output << "#";
	    output << F*base;
	    output << " " << setw(15) << x[i].x
		   << endl;
        }
	else
        {
	    output << setiosflags(ios::scientific);
	    output << F*base;
	    output << " " << setw(15) << x[i].x;       
	    output << setw(16) << d;
	    output << setw(16) << rho << endl;
	    F = vf/nf;
	    rho = ((F == 1.0) || (vf == 0.0)) 
		? 0.0 : 1.0 - cf/vf/( 1.0 - F );
	    uf = nf - vf;
	    sigrho = ((vf == 0.0) || (uf == 0.0)) ? 0.0 
		: sqrt( cf*(((1 - cf/vf)/(vf*vf)) + ((1 - cf/uf)/(uf*uf))));
	    output << F*base;
	    output << " " << setw(15) << x[i].x;       
	    d = ((vf == 0.0) || (rho == 1.0)) ? 0.0
		: sqrt( (1.0 - vf/nf)/vf * (1.0 + rho)/(1.0 - rho));
	    output << setw(16) << d;
	    output << setw(16) << rho;
	    output << setw(16) << sigrho << endl;
        }
    }
    output << F*base;
    output << " " << setw(15) << x[i].x;       
    output << setw(16) << d;
    output << setw(16) << rho << endl;
    vf += double( x[ i ].h ); F = vf/nf;
    output << F*base;
    output << " " << setw(15) << x[i].x << endl;       
}


/*
  run-time control function; it checks wether large sample conditions are
  fulfilled and measured error is lower than provided one; the number of
  values to collect before next rtc-call is estimated (hopefully not too
  stupid) based on the measured data.
*/
CNStatistics::Phase CNDLREF::rtc()
{
    if( nrv < 1000 ) return INITIALIZE;
    if (cur_f_lev() <= Fmin)
    {
	end_reached = TRUE;
	reason = MIN;
	return END;
    }
    else 
    {
	double cf, d, rho;
	double nf = double( nrv );
	double vf = double( nrv - wasted_right);
	double rq = reler_max * reler_max;
	long i, k;
	for (i = index_max - 1; i > cur_level_index; i--) vf -= x[i].h;
	i = cur_level_index; // --i would be wrong with only two x-levels
	while( i > index_min+1 )
        {
	    cf = x[i-1].c;
	    if (vf<100.0) 
	    { 
		cur_level_index = i;
		k = CNmax( (i - index_min) / 3, 1l );
		for (i = cur_level_index-1; i >= (cur_level_index-k); i--)
		    vf -= x[i].h;
		n0 = CNmin(long (1.5 * nf * (100.0 / vf - 1)), long (nf));
		n0 = CNmax( n0, (cur_level_index - index_min) * 10 );
		n0 = CNmax( n0, 100l );
		return ITERATE;
	    }
	    else if ((vf-cf)<10)
	    {
		cur_level_index = i;
		k = CNmax( (i - index_min) / 3, 1l );
		for (i = cur_level_index-1; i >= (cur_level_index-k); i--)
		    vf -= x[i].h;
		cf = x[i].c; // `i' is one too low here
		n0 = CNmin(long (1.5 * nf * (10.0 / (vf-cf) - 1)), long (nf));
		n0 = CNmax( n0, (cur_level_index - index_min) * 10 );
		n0 = CNmax( n0, 100l );
		return ITERATE;
	    }   
	    else
            {
		rho = ((vf == 0.0) || (vf == nf)) ? 0.0
		    : 1.0 - cf/vf/(1.0 - vf/nf);
		d = ((vf == 0.0) || (rho == 1.0)) ? 0.0
		    : ( 1.0 - vf/nf)/vf * (1.0+rho) / (1.0-rho);
		if( d > rq ) 
                { 
		    cur_level_index = i;
		    k = CNmax( (i - index_min) / 3, 1l );
		    for (i = cur_level_index-1; i >= (cur_level_index-k); i--)
			vf -= x[i].h;
		    cf = x[i].c; // `i' is one too low here
		    rho = ((vf == 0.0) || (vf == nf)) ? 0.0
			: 1.0 - cf/vf/(1.0 - vf/nf);
		    d = ((vf == 0.0) || (rho == 1.0)) ? 0.0
			: ( 1.0 - vf/nf)/vf * (1.0+rho) / (1.0-rho);
		    n0 = CNmin(long (1.5 * nf / 2 * (d - rq) / rq), long (nf));
		    n0 = CNmax( n0, (cur_level_index - index_min) * 10 );
		    n0 = CNmax( n0, 100l );
		    return ITERATE;
                }
            }
	    vf -= x[i].h; 
	    i--;
        }

	// i is now index_min+1
	cur_level_index = i; 
	vf -= x[i].h; cf = x[i-1].c;
	if( vf<100.0 )
	{
	    n0 = CNmin(long (1.2 * nf * (100.0 / vf - 1)), long (nf));
	    n0 = CNmax( n0, 100l );
	    return ITERATE;
	}
	if( ((vf-cf)==0.0) && (min_value==xmin))
        {
	    CNCL::warning(
		"Relative error of last level can`t be calculated: \n",
		"CNDLREF::status set to END !!\n");
	    end_reached = TRUE;
	    reason = LAST;
	    return END;
        }
	else 
        {
	    rho = ((vf == 0.0) || (vf == nf)) ? 0.0
		: 1.0 - cf/vf/(1.0 - vf/nf);
	    d = ((vf == 0.0) || (rho == 1.0)) ? 0.0
		: ( 1.0 - vf/nf)/vf * (1.0+rho) / (1.0-rho);
	    if( d > rq )
	    {
		n0 = CNmin(long (1.2 * nf / 2 * (d - rq) / rq), long (nf));
		n0 = CNmax( n0, 100l );
		return ITERATE;
	    }
        }
	end_reached = TRUE;
	reason = OK;
	return END;
    }
}



double CNDLREF::cur_f_lev()          // fuss (19.06.93)
{
    if( nrv < 1000 ) return 1.0;
    else 
    {
	double vf = double( nrv )-wasted_right;
	//double cur_x = x[ cur_level_index ].x;
	for( long i=index_max-1; i>=cur_level_index; i--) 
	    vf -= double( x[ i ].h );
	return vf/double( nrv );
    }
}


double CNDLREF::f( double xt )
{
    if( nrv < 1000 ) return 1.0;
    else
    {
	long i;
	double vf = double( nrv ) - double( x[ index_min ].h )-wasted_right;
	double nf = double( nrv );
	for( i=index_max-1; ( i>index_min ) && ( x[i].x != xt ); i-- )
	    vf -= x[ i+1 ].h;
	if( x[ i ].x == xt ) return vf/nf;
	else return -1.0;
    }     
}


const struct CNDLRE::resultline *CNDLREF::get_result( long index )
{
    if ((index < min_index()) || (index > max_index()))
    {
	warning("CNDLREF::get_result():", "index out of range.");
	return NIL;
    }

    double nf = double(nrv);
    double vf = wasted_left;
    long i;
    for(i = index_min; i <= index; i++)
	vf += double( x[ i ].h );
    double F = vf/nf;
    double cf = double( x[ index ].c );
    if ( nf < 1000.0 || vf < 100.0 || (vf-cf) < 10.0 )
    {
	line.rho    = 0.0;
	line.sigrho = 0.0;
	line.relerr = 0.0;
    }
    else
    {
	double rho = ((F == 1.0) || (vf == 0.0)) 
	    ? 0.0 : 1.0 - cf/vf/( 1.0 - F );
	double uf = nf - vf;
	line.sigrho = ((vf == 0.0) || (uf == 0.0)) ? 0.0 
	    : sqrt( cf*(((1 - cf/vf)/(vf*vf)) + ((1 - cf/uf)/(uf*uf))));
	line.relerr = ((vf == 0.0) || (rho == 1.0)) ? 0.0
	    : sqrt( (1.0 - vf/nf)/vf * (1.0 + rho)/(1.0 - rho));
	line.rho = rho;
    }
    line.vf = F * base;
    line.nx = x[ index ].h;
    line.x  = x[ index ].x;

    return &line;
}



void CNDLREF::change_error( double ne )
{
    if (ne < reler_max) cur_level_index = index_max - 1;
    reler_max = ne;
    phase = rtc();
    h = 0;
}



/***** CNCL stuff for type information and exemplar objects ******************/

// Describing object for class CNDLREF
static CNClass CNDLREF_desc("CNDLREF", "$Revision: 0.34 $", NIL);

// "Type" for type checking functions
CNClassDesc CN_DLREF = &CNDLREF_desc;
