//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: DLREP.c,v 0.35 1996-08-07 18:03:00+02 steppler Exp $
 *
 * Class: CNDLREP --- Discrete LRE (LRE III), probability function
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include <CNCL/Class.h>

#include "DLREP.h"


CNDLREP::CNDLREP(double * xvalues, long level, double eerror, 
		 double pre_first, const char* llrename, const char* ttext, 
		 bool force_rminusa_ok, unsigned long mmax_nrv)
	: CNDLRE(xvalues, level, eerror, pre_first, llrename, ttext, mmax_nrv)
{
    if( pre_first < xmin ) pre_index = index_min-1;
    else if( pre_first > xmax ) pre_index = index_max;
    force_cf = force_rminusa_ok;
}

CNDLREP::CNDLREP(double xxmin, double xxmax, double iint_size, double eerror, 
		 double pre_first, const char* lrename, const char* ttext,
		 bool force_rminusa_ok, unsigned long mmax_nrv)
	: CNDLRE(xxmin, xxmax, iint_size, eerror, pre_first, lrename, ttext,
		 mmax_nrv) 
{
    if( pre_first < xmin ) pre_index = index_min-1;
    else if( pre_first > xmax ) pre_index = index_max;
    force_cf = force_rminusa_ok;
}



void CNDLREP::put( double cur_rv )
{
    if( nrv+1 < max_nrv )
    {
	cur_index = get_index( cur_rv );
	if( cur_index == NO_INDEX )
        {
	    CNCL::warning("Warning: Wrong x value in DLREP::put !");
	    return;
        }
      
	nrv++;
	h++;
	rv_sum        += cur_rv;
	rv_square_sum += cur_rv*cur_rv;
	if (max_value < cur_rv) max_value = cur_rv; 
	if (min_value > cur_rv) min_value = cur_rv;
 
	if( cur_index == LOWER )
	{
	    wasted_left++;
	    if ((pre_index >= index_min) && (pre_index < index_max))
		x[ pre_index ].c++;
	    pre_rv    = xmin-1.0;
	    pre_index = index_min-1;
	    return;
	}
	else if( cur_index == GREATER )
	{
	    wasted_right++;
	    if ((pre_index >= index_min) && (pre_index < index_max))
		x[ pre_index ].c++;
	    pre_rv    = xmax+1.0;
	    pre_index = index_max;
	    return;
	}
	else if ((pre_index != cur_index) 
		 && (pre_index >= index_min) && (pre_index < index_max))
	    x[ pre_index ].c++;
	x[ cur_index ].h++;

        // check if ready
	if (h >= n0) 
        {
	    phase = rtc();
	    h     = 0;
        }
      
	// save current values
	pre_rv    = cur_rv;
	pre_index = cur_index;
    }
    else
    {
	phase       = END;
	end_reached = TRUE; 
    }
}


void CNDLREP::print( ostream &output ) const
{
    if (nrv == 0) return;

    double rho=0., sigrho=0., d=0.;
    double uf;
    double nf = double(nrv);
    double vf;
    double P;
    double cf;
    long i;
  
    // header
    output << setprecision(4);
    output.setf(ios::left, ios::fixed);
    output << "#LRE RESULT (THIS IS A MAGIC LINE)" << endl;
    output << "#---------------------------------------------------------------------------" << endl;
    output << "#    Discrete Lre --- probability function" << endl;
    output << "#---------------------------------------------------------------------------" << endl;
    output << "#Name: " << name << endl;
    output << "#Text: " << text << endl;
    output << "#Default values: max.rel. error = " << setw(4) << reler_max*100 
	   << "%  X_min: " << xmin << "  X_max: " << xmax << endl;
    output << "#trials: " << nrv ;
    output << "\tmean: " << mean() << "\tvariance: " << variance() << endl;
    output << "#trials < X_min: " << wasted_left << "\ttrials > X_max: " 
	   << wasted_right << endl;
    if (!force_cf)
	output << "#large sample condition r-a < 10 not forced."
	       << endl;
    output << "#---------------------------------------------------------------------------" << endl;
    output << "#P(x)           x               rel.error"
	   << "       rho             sigrho" << endl;
  
    // results
    output << setprecision(7);
    for(i = index_min; i <= index_max-1; i++)
    {
	vf = double( x[ i ].h ); 
	cf = double( x[ i ].c );
	if( nf < 1000.0 || vf < 100.0 ) /*|| (vf-cf) < 10.0 )*/ // status != END
        {
	    output << setiosflags(ios::scientific);
	    P = vf/nf;
	    output << "#";
	    output << P*base;
	    output << " " << setw(15) << x[i].x
		   << endl;
        }
        else
        {
	    output << setiosflags(ios::scientific);
	    P = vf/nf;
	    rho = ((P == 1.0) || (vf == 0.0)) 
		? 0.0 : 1.0 - cf/vf/( 1.0 - P );
	    uf = nf - vf;
	    sigrho = ((vf == 0.0) || (uf == 0.0)) ? 0.0 
		: sqrt( cf*(((1 - cf/vf)/(vf*vf)) + ((1 - cf/uf)/(uf*uf))));
	    output << P*base;
	    output << " " << setw(15) << x[i].x;       
	    d = ((vf == 0.0) || (rho == 1.0)) ? 0.0
		: sqrt( (1.0 - vf/nf)/vf * (1.0 + rho)/(1.0 - rho));
	    output << setw(16) << d;
	    output << setw(16) << rho;
	    output << setw(16) << sigrho << endl;
        }
    }
}


/*
  run-time control function; it checks wether large sample conditions are
  fulfilled and measured error is lower than provided one; the number of
  values to collect before next rtc-call is estimated (hopefully not too
  stupid) based on the measured data.
*/
CNStatistics::Phase CNDLREP::rtc()
{
    if( nrv < 1000 ) return INITIALIZE;
    else 
    {
	double cf, d, rho;
	double nf = double( nrv );
	double vf;
	double rq = reler_max * reler_max;
	long i = index_min;

	bool warn_vf_minus_cf_lower_10 = FALSE;      /* olly 08.95 */
	bool vf_minus_cf_always_zero   = TRUE;
      
	while( i < index_max-1 )
	{
	    vf = x[i].h;
	    cf = x[ i ].c;
	    if( vf<100.0 )      /* || (vf-cf)<10 ) */ /* olly 08.95 */ 
	    { 
		cur_level_index = i;
		n0 = CNmin(long (1.2 * nf * (100.0 / vf - 1)), long (nf));
		n0 = CNmax( n0, (index_max - index_min - 1) * 10 );
		n0 = CNmax( n0, 100l );
		return ITERATE;
	    }
	    else
	    {
		if( (vf-cf)<10 )
		    if (force_cf)
		    {
			cur_level_index = i;
			n0 = CNmin(long (1.2 * nf * (10.0 / (vf-cf) - 1)),
				   long (nf));
			n0 = CNmax( n0, (index_max - index_min - 1) * 10 );
			n0 = CNmax( n0, 100l );
			return ITERATE;
		    }
		    else
			warn_vf_minus_cf_lower_10 = TRUE;
		if( (vf-cf)>0 )
		    vf_minus_cf_always_zero   = FALSE;
	      
		rho = ((vf == 0.0) || (vf == nf)) ? 0.0
		    : 1.0 - cf/vf/(1.0 - vf/nf);
		d = ((vf == 0.0) || (rho == 1.0)) ? 0.0
		    : ( 1.0 - vf/nf)/vf * (1.0+rho) / (1.0-rho);
		if( d > rq ) 
		{ 
		    cur_level_index = i;
		    n0 = CNmin(long (2.0 * nf / 2 * (d - rq) / rq), long (nf));
		    n0 = CNmax( n0, 100l );
		    return ITERATE;
		}
	    }
	    i++;
        }
	if (warn_vf_minus_cf_lower_10)
	    warning("CNDLREP::rtc: ",
		    "large sample condition (r-a>=10) not fulfilled.");
	if (vf_minus_cf_always_zero)
	    warning("CNDLREP::rtc: ",
		    "large sample condition (r-a>=10) not fulfilled;\n",
		    "\tr-a yields always zero.");
      
	end_reached = TRUE; 
	return END;
    }
}



const struct CNDLRE::resultline *CNDLREP::get_result( long index )
{
    if ((index < min_index()) || (index > max_index()))
    {
	warning("CNDLREP::get_result():", "index out of range.");
	return NIL;
    }

    double nf = double(nrv);
    double vf = 0.0;
    long i;
    for(i = index_min; i <= index; i++)
	vf = double( x[ i ].h );
    double P = vf/nf;
    double cf = double( x[ index ].c );
    if ( nf < 1000.0 || vf < 100.0 || (vf-cf) < 10.0 )
    {
	line.rho    = 0.0;
	line.sigrho = 0.0;
	line.relerr = 0.0;
    }
    else
    {
	double rho = ((P == 1.0) || (vf == 0.0)) 
	    ? 0.0 : 1.0 - cf/vf/( 1.0 - P );
	double uf = nf - vf;
	line.sigrho = ((vf == 0.0) || (uf == 0.0)) ? 0.0 
	    : sqrt( cf*(((1 - cf/vf)/(vf*vf)) + ((1 - cf/uf)/(uf*uf))));
	line.relerr = ((vf == 0.0) || (rho == 1.0)) ? 0.0
	    : sqrt( (1.0 - vf/nf)/vf * (1.0 + rho)/(1.0 - rho));
	line.rho = rho;
    }
    line.vf = P * base;
    line.nx = x[ index ].h;
    line.x  = x[ index ].x;

    return &line;
}


void CNDLREP::change_error( double ne )
{
    reler_max = ne;
    phase = rtc();
    h = 0;
}



/***** Default I/O member function for CNCL classes **************************/


// Debug output
void CNDLREP::dump(ostream &strm) const
{
    strm << "CNDLREP { $Revision: 0.35 $ ..."
	 << " }" << endl;
}

// IOStream operator <<
ostream &operator << (ostream &strm, const CNDLREP &obj)
{
    obj.print(strm);
    return strm;
}

ostream &operator << (ostream &strm, const CNDLREP *obj)
{
    if(obj)
	obj->print(strm);
    else
	strm << "(NIL)";
    return strm;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNDLREP
static CNClass CNDLREP_desc("CNDLREP", "$Revision: 0.35 $",
			    CNDLREP::new_object);

// "Type" for type checking functions
CNClassDesc CN_DLREP = &CNDLREP_desc;
