// -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: Histogram.c,v 1.2 1996-08-07 18:03:04+02 steppler Exp $
 *
 * Class: CNHistogram --- Reduces data to a histogram
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 * 
 * As an exception to this rule you may use this template to generate
 * your own classes. This does not cause these classes to be covered by
 * the GNU Library General Public License. This exception does not
 * however invalidate any other reasons why the resulting program must be
 * covered by the GNU Library General Public License.
 *****************************************************************************/

#include <CNCL/Class.h>

#include "Histogram.h"
#include<fstream.h>
#include<stdio.h>
#include<stdlib.h>
#include<float.h>
#include<math.h>
#include<ctype.h>


/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNHistogram::print(ostream &strm) const
{
   strm << *this << endl;
}

// Debug output
void CNHistogram::dump(ostream &strm) const
{
    strm << "CNHistogram { $Revision: 1.2 $ ..."
	 << " }" << endl;
}

// IOStream operator <<
ostream &operator << (ostream &strm, const CNHistogram &obj)
{
    obj.print_header(strm);
    obj.print_histo(strm);
    return strm;
}

ostream &operator << (ostream &strm, const CNHistogram *obj)
{
   if (obj) {
	strm << *obj;
   } else {
	strm << "(NIL)";
   }
   return strm;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNHistogram
static CNClass CNHistogram_desc("CNHistogram", "$Revision: 1.2 $",
			    CNHistogram::new_object);

// "Type" for type checking functions
CNClassDesc CN_HISTOGRAM = &CNHistogram_desc;





/********************** constructor **************************************/
CNHistogram::CNHistogram(char* aName, double l_border,
			 double u_border, short int number_ob,
			 double normalisation, char* aDescription)
    : CNStatistics(aName, aDescription)
{
  lower_border        = l_border;
  upper_border        = u_border;
  number_of_intervals = number_ob;
  norm = normalisation;

  interval_width = (double) (upper_border-lower_border)/number_of_intervals;
  number_of_overflows  = 0;
  number_of_underflows = 0;
  v_mean               = 0;
  v_variance           = 0;
  sum                  = 0;
  sum_sqr              = 0;
  counter              = 0;

// Use init_extreme flag instead
  init_extremes = true;

  field_ptr = new int [number_of_intervals];
  field_ptr_pdf = new double [number_of_intervals];
  if ((field_ptr == NIL) || (field_ptr_pdf == NIL ))
    {
    cout << "class CNHistogram: memory allocation error\n";
    exit(1);
    }
  
  for (int i = 0; i < number_of_intervals; i++) field_ptr[i] = 0;

}

CNHistogram::CNHistogram()
    : v_mean(0.0),
      v_variance(0.0),
      sum(0.0),
      sum_sqr(0.0),
      lower_border(0.0),
      upper_border(0.0),
      interval_width(0.0),
      number_of_underflows(0),
      number_of_overflows(0),
      number_of_intervals(0),
      counter(0),
      norm(0.0),
      init_extremes(true),
      smallest_x(0.0),
      largest_x(0.0),
      field_ptr(NIL),
      field_ptr_pdf(NIL)
{}

CNHistogram::CNHistogram(CNParam*)
    : v_mean(0.0),
      v_variance(0.0),
      sum(0.0),
      sum_sqr(0.0),
      lower_border(0.0),
      upper_border(0.0),
      interval_width(0.0),
      number_of_underflows(0),
      number_of_overflows(0),
      number_of_intervals(0),
      counter(0),
      norm(0.0),
      init_extremes(true),
      smallest_x(0.0),
      largest_x(0.0),
      field_ptr(NIL),
      field_ptr_pdf(NIL)
{}

/************************ destructor **********************************
 */
CNHistogram::~CNHistogram() {
  delete []field_ptr;
  delete []field_ptr_pdf;
}


/*********************** member mean **********************************
 */
double CNHistogram::mean() const {
  return (v_mean);
}

/*********************** member variance ******************************
 */
double CNHistogram::variance() const {
  return(v_variance);
}

/*********************** member trials ********************************
 */
unsigned long CNHistogram::trials() const {
   return (counter);
}
 
/*********************** member min ***********************************
 */
double CNHistogram::min() const {
   return (smallest_x);
}

/*********************** member max ***********************************
 */
double CNHistogram::max() const {
   return (largest_x);
}

/*********************** member status  *******************************
 */
CNStatistics::Phase CNHistogram::status() const {
   if (counter == 0) {
      return INITIALIZE;
   } else {
      return ITERATE;
   }
}   

/*********************** member set_norm ******************************
 */
void CNHistogram::set_norm(double normalization) {
   norm = normalization;
}

/*********************** member put ***********************************
 */ 
void CNHistogram::put(double x) {
  counter++;  // Histogramm now contains one value more
  sum = sum + x;
  v_mean = (double) sum / counter;
  sum_sqr = sum_sqr + x*x;

  if (counter == 1) {
    v_variance = 0;
  } else {
    v_variance = (double) 1 / (counter - 1) * ( sum_sqr -
	      counter*v_mean*v_mean); 
  }
    
// Compare value against the extremes 
  if (init_extremes) { // initialize extremal values
    smallest_x = largest_x = x;
    init_extremes = false;  
  } else if (x < smallest_x) { 
    smallest_x = x;

  } else if (x > largest_x) {
    largest_x = x;
  }

  if (x <= lower_border)  {
    number_of_underflows++;
  } else if (x > upper_border) { 
    number_of_overflows++;
  } else {
    int histo_field; // Index of the histogramm bin where x falls into!

    histo_field = ( int )((x - lower_border)/interval_width);
    // check if the value falls exactly to the interval border. 
    // If yes it belongs to the lower(!) bin, therefore decrement
    // `histo_field' by 1
    if (histo_field * interval_width == x - lower_border) histo_field--;
    // Count the value in the appropriate field of the histogramm
    field_ptr[histo_field]++;
  }
}

/*********************** member insert ***********************************
 */
void CNHistogram::insert(double x, int y)
{
  counter +=  y; // Histogram now contains y values more
  
  sum = sum + x*y;
  v_mean = (double) sum / counter;
  sum_sqr = sum_sqr + x*x*y*y;
  
// Compare value against the extremes 
  if (init_extremes) {
    smallest_x = largest_x = x;
    init_extremes = false;  
  } else if (x < smallest_x) { 
    smallest_x = x;
  } else if (x > largest_x) {
    largest_x = x;
  }

  if (counter == 1) { 
     v_variance = 0;
  } else {
     v_variance = (double) 1 / (counter - 1) * ( sum_sqr -
	       counter*v_mean*v_mean); 
  }    

  // Insert value `x' for `y' times to the appropriate slot! 
  if (x <=  lower_border) { 
    number_of_underflows += y;
  } else if (x > upper_border) {
    number_of_overflows  += y;
  } else {
    int histo_field; // Index of the histogramm bin where x falls into!

    histo_field = ( int )((x - lower_border)/interval_width);
    // check if the value falls exactly to the interval border. 
    // If yes it belongs to the lower(!) bin, therefore decrement
    // `histo_field' by 1
    if (histo_field * interval_width == x - lower_border) histo_field--;
    field_ptr[histo_field] += y;
  }  
}

/************************** member print_header  **********************
 */
void CNHistogram::print_header(ostream &out_str, char *prefix) const {
  
   double sigma=sqrt(v_variance);
   
   out_str << prefix;
   out_str << "********** statistics of " << name << endl;
   out_str << prefix;
   out_str << "Name of Histogram                    : " << name << endl;
   out_str << prefix;
   out_str << "Description                          : " << text << endl;
   out_str << prefix;
   out_str.precision(4);
   out_str.setf(ios::right);
   out_str << "lower border                         : ";
   out_str.width(9); out_str << lower_border << endl;
   out_str << prefix;
   out_str << "upper border                         : ";
   out_str.width(9); out_str << upper_border << endl;
   out_str << prefix;
   out_str << "number of intervals                  : ";
   out_str.width(9); out_str << number_of_intervals << endl;
   out_str << prefix;
   out_str << "interval width                       : ";
   out_str.width(9); out_str << interval_width << endl;
   out_str << prefix;
   out_str << "number of values entered             : ";
   out_str.width(9); out_str << counter << endl;
   out_str << prefix;
   out_str << "mean                                 : ";
   out_str.width(9); out_str << v_mean << endl;
   out_str << prefix;
   out_str << "Deviation of mean (sigma/sqrt(n-1))  : ";
   out_str.width(9); out_str << sigma/sqrt((double) counter - 1.) << " (" <<
			100.*sigma/sqrt((double) counter-1.) / v_mean << " %)"                              << endl; 
   out_str << prefix;
   out_str << "Standard deviation (sigma)           : ";
   out_str.width(9); out_str << sigma << endl;
   out_str << prefix;
   out_str << "Empiric variance (sigma^2)           : ";
   out_str.width(9); out_str << v_variance << endl;
   out_str << prefix;
   out_str << (smallest_x <= lower_border ? "!! U" : "No"
	       " u") << "nderflows." << endl;   
   out_str << prefix;
   out_str << "Minimum (" << smallest_x << ") <=  x <= " <<
      lower_border << ": " << number_of_underflows << " (" << 100.0 *
      (double) number_of_underflows/counter << "%)" << endl;
   out_str << prefix;
   out_str << (largest_x > upper_border ? "!! O" : 
                           "No o") << "verflows." << endl;
   out_str << prefix;   
   out_str << upper_border  << " < x <= Maximum (" << largest_x << "): "                  << number_of_overflows << " (" << 100.0 *
              (double) number_of_overflows/counter << "%)" << endl << endl; 
   out_str << prefix;
   out_str << "======================================================" << endl;
}


/************************** member print_histo ************************/
void CNHistogram::print_histo(ostream &out_str) const {

   int i = 0;
   if (!counter) {    
      cout << "Histogram is empty <press any key>";
      getchar();
      return;
   }

   out_str.precision(4);
   out_str.setf(ios::right|ios::fixed);
   out_str.width(11);
   out_str << "minimum" << " < x <= ";
   out_str.width(11); out_str << (double) lower_border;
   out_str.width(11); out_str << number_of_underflows / ((norm != 0.0) ?                                        counter : 1.0) * ((norm != 0.0) ? norm : 1.0);
   out_str.width(11); out_str << (double) number_of_underflows/counter << endl;
  
   for (i=0; i<number_of_intervals; i++) {
      out_str.width(11); out_str << i*interval_width+lower_border;
      out_str << " < x <= ";
      out_str.width(11); out_str << (i+1)*interval_width+lower_border;
      out_str.width(11); out_str << (field_ptr[i] / ((norm != 0.0) ?                                    counter : 1.0) * ((norm != 0.0) ? norm : 1.0));
      out_str.width(11); out_str << (double) field_ptr[i]/counter << endl;
   }

   out_str.width(11); out_str << (double) upper_border << " < x <= " ;
   out_str.width(11); out_str << "maximum";
   out_str.width(11); out_str << number_of_overflows / ((norm != 0.0) ?                                         counter : 1.0) * ((norm != 0.0) ? norm : 1.0);
   out_str.width(11); out_str << (double) number_of_overflows/counter << endl;
}

 
/************************** member data_out *****************************
*/
double * CNHistogram::data_out( void ) {
   for (int i = 0; i < number_of_intervals; i++) 
      //field_ptr_pdf[i] = (double) field_ptr[i]/counter;
      field_ptr_pdf[i] = (double) field_ptr[i];
 
   return(field_ptr_pdf);
}


/************************** member pdf_out *************************/
double * CNHistogram::pdf_out( void ) {
   int i;
   double sum = 0.0;
   for (i = 0; i < number_of_intervals; i++) 
      sum += field_ptr[i];
   if ( sum > 1.e-8 ){
      for (i = 0; i < number_of_intervals; i++) 
         field_ptr_pdf[i] = 100. * field_ptr[i] / sum;         // in %%%%
      for (i = 1; i < number_of_intervals-1; i++) 
         field_ptr_pdf[i] = (
	    0.25 *  field_ptr_pdf[i-1] +
	    0.5  *  field_ptr_pdf[i]   +
	    0.25 *  field_ptr_pdf[i+1] );
   }  
 return(field_ptr_pdf);
}


/************************** member get_number_of_intervalls *********/
short int CNHistogram::get_number_of_intervalls( void ) {
  return( number_of_intervals ); 
}


/************************** member compare_histo ********************/
double CNHistogram::compare( CNHistogram *second_histo ) {
    double *first_data, *second_data;
    short int first_length, second_length;
    double sum=0.0;

    first_data   = this->pdf_out();  
    second_data  = second_histo->pdf_out();
    first_length   = this->get_number_of_intervalls();  
    second_length  = second_histo->get_number_of_intervalls();

    if ( first_length == second_length ){
       //diff_data = new double [first_length];
       // for (int k=0;k<first_length;k++)
       //sum += diff_data[k] = ( first_data[k] - second_data[k] ) * (
       //first_data[k] - second_data[k] ) / first_data[k]; 
       //if (first_data[k] > 0.0 )
       //      sum +=  ( first_data[k] - second_data[k] ) * (
       //      first_data[k] - second_data[k] ); 

       for (int k=0;k<first_length;k++)
              sum +=  ( first_data[k] - second_data[k] ) * (
		 first_data[k] - second_data[k] ); 
        return( sum );
    } else {
       cout << "Histograms can't be compared: number of intervalls differ!"
	    << endl;
       exit(0);
   }
}


/********************* member plot_histo ****************************/

void CNHistogram::plot_histo(ostream &out_str, 
			      int drawing_style /* = 0 */) { 

// Writes dots into `out_str` to be used as input values for gnuplot
// to draw histogram.  If `drawing_style == 0` all vertical lines are
// drawn to the value of the next step (Step-histogram), otherwise all
// vertical lines are drawn to y=0 (Box-histogram).

   if (!counter) {
      cout << "Histogram is empty <press any key>";
      getchar();
      exit (1);
   }

   print_header (out_str,"# ");   // Prints statistic informations with
				  // comment sign '#' in front

   if (smallest_x <= lower_border) {
      out_str << "#  Comment next three lines to undisplay underflow"
	 " bin!" << endl;
      out_str << " " << smallest_x << " " << 0.0 << endl;  
      out_str << " " << smallest_x << " " << number_of_underflows << endl; 
      out_str << " " << lower_border << " " << number_of_underflows <<
	 endl;  
   }

   out_str << lower_border << " " << 0.0 << endl;  

   for (int loop = 0; loop < number_of_intervals; loop++){
      out_str << loop * interval_width + lower_border << " " << 
	 field_ptr[loop] / ((norm != 0.0) ? counter : 1.0)
	 * ((norm != 0.0) ? norm : 1.0)   << endl; 
      out_str << (loop+1) * interval_width + lower_border << " " << 
	 field_ptr[loop] / ((norm != 0.0) ? counter : 1.0)
	 * ((norm != 0.0) ? norm : 1.0)    << endl; 
      // Comment the lines drawing down to zero for step plots except
      // for the last vertical line which has alwaws to be drawn down to
      // zero. Otherwise the values will fall out of the plot ;-)   
      if (drawing_style == 0 && loop != number_of_intervals -1 )
	 out_str << "# "; // Write comment character  
      else if (loop != number_of_intervals - 1) 
	 out_str <<" "; //Otherwise indent to allow patching of
      //output file by editor!
      out_str << (loop+1) * interval_width + lower_border << " " <<
	 0.0 << endl;  
   }

   if (largest_x > upper_border) {
      out_str << "#  Comment next three lines to undisplay overflow"
	 " bin!" << endl;
      out_str << " " << upper_border << " " << number_of_overflows /                 ((norm != 0.0) ? counter : 1.0) * ((norm != 0.0) ? norm : 1.0) << endl;  
      out_str << " " << largest_x << " " << number_of_overflows /                    ((norm != 0.0) ? counter : 1.0) * ((norm != 0.0) ? norm : 1.0)  << endl;  
      out_str << " " << largest_x << " " << 0.0 << endl;  
   }
}



/********************* member plot_file ****************************/

void CNHistogram::plot_file (char *out_file_name, int drawing_style) {
 
  ofstream output_file(out_file_name);
  if (output_file == 0) {
     cout << "Error: couldn't open output file: `" << out_file_name 
	  << "'. Exiting!" << endl;  
     return; // Nichts auf file geschrieben!
  } 
  plot_histo(output_file, drawing_style);
}


/********************* member print_file ****************************/

void CNHistogram::print_file (char *out_file_name) {
 
  ofstream output_file(out_file_name);
  if (output_file == 0) {
     cout << "Error: couldn't open output file: `" << out_file_name 
	  << "'. Exiting!" << endl;  
     return; // Nichts auf file geschrieben!
  } 
  print_header(output_file);
  print_histo(output_file);
}


CNHistogram * CNHistogram::fill_histo_from_file(char * infile, double
		lower_bound, double upper_bound, short int
		number_of_bins, double normalisation, short int
		use_data_column) {
  ifstream input_data(infile);
  double value;
  int skip_loop;
  char character;

  if (!input_data) {
      cout << "Error opening file `" << infile <<"'. Exiting!" << endl;
      exit(-1);
  }

  CNHistogram &histo = *new CNHistogram(infile, lower_bound, upper_bound,
			number_of_bins, normalisation); 

  while (!input_data.eof()){
    // Read items until specified column
    for (skip_loop = 0; skip_loop < use_data_column; skip_loop++) {
      // Skip whitespace character until next column but don't skip
      // newline! If a new line occurs the actual line does not
      // contain as many columns as requested => print error message
      // and exit!
      while (input_data.get(character) && isspace(character)
	     && '\n' != character) ;
      // Check for newline character as long as end of file has not
      // been reached! 
      if ('\n' == character && !input_data.eof()) {
	  // newline character occured before specified column could
	  // be read => This is an error!
	  cout << "Error in fill_histo_from_file(): Not that many data"
	      "columns available in file `" << infile
	       <<"'.\n Number of data columns read: "<< skip_loop
	       << "\n Data column requested: " << use_data_column
	       << "\n Exiting!\n";
	  exit(-1);
      } else if (input_data.eof()) {
	  // End of file reached, finish data input
	  break;
      } else {
	  // `character' contains data, put back to stream!
	  input_data.putback(character);
      }
      input_data >> value;
    }

    if (!input_data.eof())  { 
      // Write value to histogramm
      histo.put(value);
    }
    // skip remaining characters until end of line
    do {
      input_data.get(character);
    } while (character != '\n' && !input_data.eof());
  }
  return &histo;
}
