// -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: Histogram.h,v 1.2 1996-08-07 18:03:06+02 steppler Exp $
 *
 * Class: CNHistogram --- Reduces data to a histogram
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 * 
 * As an exception to this rule you may use this template to generate
 * your own classes. This does not cause these classes to be covered by
 * the GNU Library General Public License. This exception does not
 * however invalidate any other reasons why the resulting program must be
 * covered by the GNU Library General Public License. 
 *****************************************************************************/

#ifndef __CNHistogram_h
#define __CNHistogram_h


#include <CNCL/Statistics.h>		// Base class

extern CNClassDesc CN_HISTOGRAM;	// Class CNHistogram description


/*
 * The class CNHistogram
 */


class CNHistogram : public CNStatistics  {
 private:
   double v_mean;		// mean
   double v_variance;		// variance
   double sum;			// Sum of all values
   double sum_sqr;		// Sum of squared values
  
   double lower_border;
   double upper_border;	
   double interval_width;

   int number_of_underflows;	// Counter for underflows
   int number_of_overflows;	// Counter for owerflows
  
   short int number_of_intervals;	// Number of classes
   long  int counter;			// Number of input values 
   double norm;                 // Normalise output of histogram

   bool init_extremes;   // flag to assign initial values
   double smallest_x;    // smallest input value
   double largest_x;     // largest x
  
   int *field_ptr;		// Pointer to field of histogram-values
   double *field_ptr_pdf;	// Pointer to field pdf
  
 public:
   CNHistogram();
   CNHistogram(CNParam *);
   CNHistogram(char*, double, double, short int, double = 0.0, char* = NIL);
   ~CNHistogram();

   unsigned long  trials() const;  // Number of values in histogram  
   double         min() const;     // Largest input value
   double         max() const;     // Smallest input value
   double   mean() const;	   // Returns mean
   double   variance() const;	   // Returns variance
   void     put(double);	   // Put value into field
   void     set_norm (double);      // Set normalization
   void     insert(double, int );  // Put value into field y times
   void     print_header(ostream &, char * = "") const;  // Print header of
				                         // array
   void     print_histo(ostream &) const;                // Print array
   double * data_out( void );  
   double * pdf_out( void );  
   short int      get_number_of_intervalls( void ); 
   double         compare( CNHistogram* );
   void           plot_histo(ostream &, int = 0);
   void           plot_file(char*, int);
   void           print_file(char*);
   Phase          status() const; // Query lifecycle phase
   bool           end () const {return FALSE;}; // Always returns FALSE as
   // histogram evaluation never reaches an end
   void           pdf_write( char* );

   // Dummy-function for CNCLStatistics :
   void           reset () {}; // Reset histogram for newstart (for
                               // later implementation if necessary)

// Static member function to read histogram from file!
    static CNHistogram * fill_histo_from_file(char * infile,
	       double lower_bound, double upper_bound, short int
	       number_of_bins, double normalisation, short int
	       use_data_column);


   /***** Member functions required by CNCL *********************/
   virtual CNClassDesc class_desc() const	// CNClass description
   { return CN_HISTOGRAM; }
            
   virtual bool is_a(CNClassDesc desc) const	// Type checking
   { return desc == CN_HISTOGRAM ? TRUE : CNStatistics::is_a(desc); }
        
   static CNHistogram *cast_from_object(CNObject *obj) // Safe type cast
   {
#   ifdef NO_TYPE_CHECK
      return (CNHistogram *)obj;
#   else
      return (CNHistogram *)( !obj || obj->is_a(CN_HISTOGRAM)
	      ? obj : fatal_type(obj->class_desc(), CN_HISTOGRAM) );
#   endif
   }
    
   static CNObject *new_object(CNParam *param = NIL) // Object creation
   { return param ? new CNHistogram(param) : new CNHistogram; }
    
   // Print/debug output
   virtual void print(ostream &strm = cout) const;
   virtual void dump (ostream &strm = cout) const;
};


ostream &operator << (ostream &strm, const CNHistogram &obj);
ostream &operator << (ostream &strm, const CNHistogram *obj);


#endif /**__CNHistogram_h**/

