//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: MomentsTime.c,v 1.3 1996-08-07 18:03:18+02 steppler Exp $
 *
 * Class: CNMomentsTime --- Simple stat. evaluation:
 *                          mean, variance, skewness, etc.
 *                          time-weighted
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 * 
 * As an exception to this rule you may use this template to generate
 * your own classes. This does not cause these classes to be covered by
 * the GNU Library General Public License. This exception does not
 * however invalidate any other reasons why the resulting program must be
 * covered by the GNU Library General Public License.
 *****************************************************************************/

#include "MomentsTime.h"

#include <math.h>
#include <limits.h>
#include <float.h>

CNMomentsTime::CNMomentsTime(const char* aName, const char* aDescription)
    : CNStatistics(aName, aDescription),
      sum(0), sqsum(0), cbsum(0), n(0),
      min_value(DBL_MAX), max_value(-DBL_MAX), prev_time(0)
{}

void CNMomentsTime::put(double x_i, CNSimTime put_time)
{
    double square = x_i * x_i * (put_time - prev_time) ;
    sum += x_i * (put_time - prev_time);
    sqsum += square ;
    cbsum += x_i * square ;
    if (x_i > max_value)
	max_value = x_i ;
    if (x_i < min_value)
	min_value = x_i ;
    n++ ;
    prev_time = put_time ;
    
}

void CNMomentsTime::put(double x)
{
    cout << "Use put(double x_i, CNSimTime put_time)  to put "
	 << x << " instead !" << endl;
}

void CNMomentsTime::reset()
{
    n = 0 ;
    sum = sqsum = cbsum = 0 ;
    min_value = DBL_MAX ;
    max_value = -DBL_MAX;
    prev_time = 0 ;
    
}

double CNMomentsTime::mean() const
{
    return prev_time ? (sum / prev_time) : 0. ;
}

double CNMomentsTime::variance() const
{
    return prev_time ? (sqsum - sum * sum / prev_time) / (prev_time) : 0. ;
}

double CNMomentsTime::relative_variance() const
{
    return prev_time == 0 || sum == 0.0 ?
	0.0 : variance() * prev_time * prev_time / sum / sum ;
}

double CNMomentsTime::M_2() const
{
    return prev_time ? sqsum / prev_time : 0. ;
}

double CNMomentsTime::M_3() const
{
    return prev_time ? cbsum / prev_time : 0. ; 
}

double CNMomentsTime::Z_3() const
{
    if (prev_time == 0.0)
	return 0.0;

    return (prev_time * cbsum +
	    sum * (2.0 * sum * sum / prev_time - 3.0 * sqsum)) /
	(prev_time) / (prev_time);
}

double CNMomentsTime::skewness() const
{
    if (prev_time == 0.0)
	return 0.0;
    
    double var = variance();
    if (var == 0)
	return 0.;
   return Z_3() / (var * sqrt(var));
}

double CNMomentsTime::deviation() const
{
     return prev_time ? sqrt(variance()) : 0. ;
}

double CNMomentsTime::relative_deviation() const
{
    if (prev_time == 0.0 || mean() == 0.0)
	return 0.0;
    else
	return sqrt(variance()) / mean() ;
}
    
/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNMomentsTime::print(ostream &strm) const
{
    strm << "Evaluation " << name << " (CNMomentsTime)" << endl;
    strm << "Description        : " << text                 << endl;
    strm << "Trials             : " << n                    << endl;
    strm << "Mean               : " << mean()               << endl;
    strm << "Variance           : " << variance()           << endl;
    strm << "Relative variance  : " << relative_variance()  << endl;
    strm << "Deviation          : " << deviation()          << endl;
    strm << "Relative deviation : " << relative_deviation() << endl;
    strm << "Skewness           : " << skewness()           << endl;
    strm << "Minimum            : " << min()                << endl;
    strm << "Maximum            : " << max()                << endl;
}

// Debug output
void CNMomentsTime::dump(ostream &strm) const
{
    strm << "CNMomentsTime { $Revision: 1.3 $ ..."
	 << " }" << endl;
    strm << "name = " << name << endl;
    strm << "text = " << text << endl;
    strm << "sum = " << sum << "  sqsum = " << sqsum
	 << "  cbsum = " << cbsum << endl;
    strm << "n = " << n << "  min_value = " << min_value
	 << "  max_value = " << max_value << endl;
    strm << "Evaluation " << name << " (CNMomentsTime)" << endl;
    strm << "Trials             : " << n                    << endl;
    strm << "Mean               : " << mean()               << endl;
    strm << "Variance           : " << variance()           << endl;
    strm << "Relative variance  : " << relative_variance()  << endl;
    strm << "Deviation          : " << deviation()          << endl;
    strm << "Relative deviation : " << relative_deviation() << endl;
    strm << "Skewness           : " << skewness()           << endl;
    strm << "Minimum            : " << min()                << endl;
    strm << "Maximum            : " << max()                << endl;
    strm << " }" << endl;
}

// IOStream operator <<
ostream &operator << (ostream &strm, const CNMomentsTime &obj)
{
    obj.print(strm);
    return strm;
}

ostream &operator << (ostream &strm, const CNMomentsTime *obj)
{
    if(obj)
	obj->print(strm);
    else
	strm << "(NIL)";
    return strm;
}


/***** CNCL stuff for type information ***************************************/

// Describing object for class CNMomentsTime
static CNClass CNMomentsTime_desc("CNMomentsTime", "$Revision: 1.3 $",
			    CNMomentsTime::new_object);

// "Type" for type checking functions
CNClassDesc CN_MOMENTSTIME = &CNMomentsTime_desc;



















