// -*- C++ -*-
// Program for testing class CNHistogram
// Input to main is: 
//     input_file_name
//     output_file_name
//     lower_bound
//     upper_bound
//     number_of_bins
//     drawing_style
//     use_data_column
// The input file consists of lines each containing at least
// 'use_data_column' floating point numbers separated by
// whitespace. The data for the histogramm is read from column
// no. 'use_data_column' (default is 1)
 
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#include <CNCL/Histogram.h>
#include <CNCL/ParseArgs.h>


// Description of arguments recognized by `cnhisto'
const int arg_desc_size = 8;

OptionDescription arg_desc[arg_desc_size] = { 
// {"'option'", BLANK/NO_BLANK, "'default_argument'", "'usage'"}
   {"i", BLANK, NIL, "input file name (no default!)"},
   {"o", BLANK, "histo.dat", "output file name (histo.dat)"},
   {"l", BLANK, "0", "lower border (0)"},
   {"u", BLANK, "10000", "upper border (10000)"},
   {"n", BLANK, "200", "number of bins (200)"},
   {"P", BLANK, "0", "drawing style (0(default)=steps, else=boxes))"},
   {"c", BLANK, "1", "data column of input file (1)"},
   {"r", BLANK, "1", "normalization value (0, 1)"} 
};

const int mandatory = 1;
const int positional = 5;

// Forward declartion of `user_function' to allow specification in
// initialization of `args'
static void user_function(int argc, char **argv);

CNParseArgsList args = {
    arg_desc_size, arg_desc, mandatory, positional, user_function
//    arg_desc_size, arg_desc, mandatory, positional, NIL
};

// Definition of user supplied function for printing usage text 
static void user_function(int, char **argv) {
    cout << "Here comes the usage you should obey!"
         << endl;

// Call to static member function of CNParseArgs in order to print
// usage text from information provided in `args' is also possible
    CNParseArgs::usage_screen(argv[0], args);
//    exit(-1);
}


static void screen_output(CNParseArgsList &args_list) {
    cout << "\n****************\n";
    cout << "Option\tArgument" << endl;
    for (int loop = 0; loop < arg_desc_size; loop++) {
	cout << " " <<args_list.args_array[loop].option_string << "\t "
	     << (args_list.args_array[loop].argument_string == NIL ?
		 " (nothing specified)" :
		 args_list.args_array[loop].argument_string ) 
	     << endl;
    }
    cout << "****************\n";
} 


int main(int argc, char **argv) {

    char *input_file_name, *output_file_name;
    double upper_bound, lower_bound, normalisation;
    short int number_of_bins, use_data_column;
    int drawing_style;
    OptionDescription * aa_ptr;
    
    CNParseArgs parse_args(argc, argv, args);
    screen_output(args); // Control output
    parse_args.override_nil(0,6);
    screen_output(args); // Control output
    
    
    aa_ptr = args.args_array;

    input_file_name = parse_args[0];
    output_file_name = parse_args[1];

    assign_from_string(parse_args[2], lower_bound);
    assign_from_string(parse_args[3], upper_bound);
    assign_from_string(parse_args[4], number_of_bins);
    assign_from_string(parse_args[5], drawing_style);
    assign_from_string(parse_args[6], use_data_column);

    if (aa_ptr[7].argument_string == NIL) {
	// Option rel not specified, no normalisation
	normalisation = 0.0;
    } else {
	assign_from_string(aa_ptr[7].argument_string, normalisation);
    }

    cout << "Sampling data for histogram from file `" << input_file_name
	 << "'\n using column " << use_data_column << ".\n";

    cout << "Gnuplot output will be written to file `" << output_file_name
	 << "'.\n\n";
    
    CNHistogram * main_histo = CNHistogram::fill_histo_from_file(
	input_file_name, lower_bound, upper_bound, number_of_bins,
	normalisation, use_data_column); 

//    main_histo->print (cout);
    cout << main_histo;
    main_histo->plot_file (output_file_name, drawing_style);

    delete main_histo;
    main_histo = NIL;

    cout << main_histo << endl;
    
    return 0;
}
