//  libeasy, C++ library to encapsulate things and make life easy.
//  Copyright (C) 2000 Hans Dijkema 
//
//  This program/library is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License as published 
//  by the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program/library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program/library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// This software is maintained by Hans Dijkema.
// See the Makefile.lsm for your primary and secundary site. 
// email me at hnmdijkema@softhome.net
//
#ifndef __HERROR__CXX
#define __HERROR__CXX

#ifdef WIN32
#include <vc6.hxx>
#endif

//doc
//    library : libeasy.a
//    module  : herror
//    include : herror.hxx
//    link    : -leasy
//
// Versie Wie              Datum       Wijziging/Opdracht/Omschrijving
// --------------------------------------------------------------------------
// 1.00   HNM Dijkema      27-12-1999  Algemeen bruikbare library ten behoeve
//                                     van C++ projecten.
// 1.01   HNM Dijkema      29-12-1999  Toevoegen functionaliteit
//
//h2 Revisies
// Versie Revisieinformatie
// --------------------------------------------------------------------------
// 1.00   Initiele versie
// 1.01   THROW_FATAL3(), THROW_ERROR3() en THROW_WARNING3() toegevoegd.
// 1.11   THROW_FATAL4(), THROW_ERROR4() en THROW_WARNING4() toegevoegd.
//
//h1 Doel van de library
//
//  Module herror implementeert een methode voor het afhandelen van fouten
//  in een programma. Onderscheid wordt gemaakt tussen fatale fouten, 
//  gewone fouten en waarschuwingen.
//  
//  Zie verder de functie definities.
//  
//end
  
////////////////////////////////////////////////////////////////////
// definitions
////////////////////////////////////////////////////////////////////

//h1 Constanten en variabelen

#define HERROR_MSG_LEN	4096

//def
#define HERROR_FATAL    12
#define HERROR_WARN      4
#define HERROR_WARNING   HERROR_WARN
#define HERROR_ERROR     8
#define HERROR_NONE      0
//end

////////////////////////////////////////////////////////////////////
// global variables
////////////////////////////////////////////////////////////////////

extern char HERROR_MSG[HERROR_MSG_LEN],HERROR_MSG_I[HERROR_MSG_LEN];

////////////////////////////////////////////////////////////////////
// classes
////////////////////////////////////////////////////////////////////

//h1classes en functies

//sec
//h2 class checkp_out
//
// Deze class implementeert een virtuele functie print(), welke
// volgens de printf() methode werkt. De default implementatie schrijft
// naar de stderr.
//
//end

class checkp_out 
{
  public:
//def
    virtual void print(char *format,...);
//
// pre : -
// post: fprintf(stderr,format,...)
//end
};


//def
//h2 class checkpoint
//
// Deze class implementeert een methode om vlaggetjes te zetten
// in een programma. Bij de foutafhandeling wordt een backtrace
// van deze vlaggetjes met behulp van class checkp_out geprint.
//end

class checkpoint
{
  private:
//def
    checkpoint        *next;
    char              *file,*function;
    int                line;
//end
  public:
//def
    checkpoint(const char *file,const int line,const char *function);
//
// pre : 'file' is de source waar checkpoint in wordt gezet
//       'line', is de regel die van toepassing is.
//       'function', de functie waarbinnen de checkpoint wordt gezet.
// post: Variabele van checkpoint geinstantieerd, variabelen
//       'file', 'function' en 'line' zijn gezet; en de verwijzing naar de
//       vorige checkpoint staat in variabele 'next'.
// opm.: static variabele 'top' wordt gebruikt om de stapel van
//       checkpoints op te bouwen.
//end
//def
   ~checkpoint();
//
// pre : Stapelgewijze verwerking van checkpoints, d.w.z geen
//       willekeurige verwerking daarvan. checkpoint instantiatie
//       in volgorde 1 2 3, wordt gedealloceerd in volgorde 3 2 1.
// post: static variable 'top' heeft de waarde van variable 'next'
//       gekregen.
//end
  public:
//def
    void print(void);
// 
// pre : -
// post: Vanaf dit checkpoint zijn alle checkpoints die met variable
//       'next' te bereiken zijn geprint.
//       Hierdoor ontstaat een backtrace van checkpoints die in het
//       programma gepasseerd zijn.
//end
};

////////////////////////////////////////////////////////////////////
// checkpoint macro's
////////////////////////////////////////////////////////////////////

//def
#define CHECKPOINT checkpoint chkp##__LINE__(__FILE__,__LINE__,__FUNCTION__)
//
// Met deze macro wordt een variabele van type checkpoint 
// gedeclareerd. Toepassing in een programma is als volgt:
//
//    {CHECKPOINT;
//       <code>
//       if (...) {CHECKPOINT; <code> }
//       etc.
//    }
//end

////////////////////////////////////////////////////////////////////
// error functions
////////////////////////////////////////////////////////////////////

//def
void herror_handle(char *msg);
void herror_checkpoints(char ind,int exitcode);
//
// interne error handling functies
//
// herror_handle drukt interpreteert msg (zie THROW_HERROR() verderop).
// Indien NUM>HERROR_WARN, dan wordt het programma verlaten met exit(NUM).
//
// herror_checkpoints drukt de checkpoints af, indien ind=='F' of
// (ind=='E' en niet exitcode==0).
//end

////////////////////////////////////////////////////////////////////
// error macro's
////////////////////////////////////////////////////////////////////

//h2 Error macros
//def
#define THROW_HERROR(IND,NUM,msg) \
        { \
            sprintf(HERROR_MSG,"%c%d:%s:%s[%d]:%s\n", IND, NUM, \
                       __FILE__,__FUNCTION__,(int) __LINE__,msg); \
                       herror_checkpoints(IND,NUM); \
            throw(HERROR_MSG); \
        }
//
// Interne macro voor afhandeling van programma fouten.
// IND is de fout-indicator (char).
// NUM is de te gebruiken exitcode.
// msg is het bericht (char *).
//
//   Een boodschap:
//      "%c%d:%s:%s[%d]L:%s\n", IND, NUM, 
//                             __FILE__, __FUNCTION__, __LINE__,
//                              msg
//   wordt samengesteld.
//   De backtrace van checkpoints wordt gemaakt.
//   Vervolgens wordt een throw() gedaan met de boodschap.
//end

//def
#define THROW_HERROR2(IND,NUM,msg,var) \
        { \
            sprintf(HERROR_MSG_I,msg,var); \
            THROW_HERROR(IND,NUM,HERROR_MSG_I); \
        }
//
// Interne macro die verder als THROW_HERROR() is, maar
// met de uitzondering dat er eerste een sprintf(buffer,msg,var)
// wordt gedaan, waarbij msg dus de 'format' variabele van de
// printf functie vertegenwoordigt.
//end

//def
#define THROW_HERROR3(IND,NUM,msg,v1,v2) \
        { \
            sprintf(HERROR_MSG_I,msg,v1,v2); \
            THROW_HERROR(IND,NUM,HERROR_MSG_I); \
        }
//
// Interne macro die verder als THROW_HERROR() is, maar
// met de uitzondering dat er eerste een sprintf(buffer,msg,var)
// wordt gedaan, waarbij msg dus de 'format' variabele van de
// printf functie vertegenwoordigt.
//end

//def
#define THROW_HERROR4(IND,NUM,msg,v1,v2,v3) \
        { \
            sprintf(HERROR_MSG_I,msg,v1,v2,v3); \
            THROW_HERROR(IND,NUM,HERROR_MSG_I); \
        }
//
// Interne macro die verder als THROW_HERROR() is, maar
// met de uitzondering dat er eerste een sprintf(buffer,msg,var)
// wordt gedaan, waarbij msg dus de 'format' variabele van de
// printf functie vertegenwoordigt.
//end


//def
#define THROW_FATAL(msg)           THROW_HERROR('F',HERROR_FATAL,msg)
#define THROW_FATAL2(msg,var)      THROW_HERROR2('F',HERROR_FATAL,msg,var)
#define THROW_FATAL3(msg,v1,v2)    THROW_HERROR3('F',HERROR_FATAL,msg,v1,v2)
#define THROW_FATAL4(msg,v1,v2,v3) THROW_HERROR4('F',HERROR_FATAL,msg,v1,v2,v3)
#define THROW_ERROR(msg)           THROW_HERROR('E',HERROR_ERROR,msg)
#define THROW_ERROR2(msg,var)      THROW_HERROR2('E',HERROR_ERROR,msg,var)
#define THROW_ERROR3(msg,v1,v2)    THROW_HERROR3('E',HERROR_ERROR,msg,v1,v2)
#define THROW_ERROR4(msg,v1,v2,v3) THROW_HERROR4('E',HERROR_ERROR,msg,v1,v2,v3)
#define THROW_WARNING(msg)         THROW_HERROR('W',HERROR_WARN,msg)
#define THROW_WARNING2(msg,var)    THROW_HERROR2('W',HERROR_WARN,msg,var)
#define THROW_WARNING3(msg,v1,v2)  THROW_HERROR3('W',HERROR_WARNING,msg,v1,v2)
#define THROW_WARNING4(msg,a,b,c)  THROW_HERROR4('W',HERROR_WARNING,msg,a,b,c)
//end

//def
#define HERROR_CATCH catch (char *msg) \
        { \
            herror_handle(msg); \
        }
//
// pre : er is een C++ try { } block opgezet.
//       De msg moet van formaat boodschap zijn, zoals bij
//       THROW_HERROR(IND,NUM,msg).
// post: er is een catch(char *msg) block opgezet.
//       In dit block wordt herror_handle(msg) aangeroepen.
//end


//sec
//h1 Verwijzingen
//   n.v.t.

//h1 Gegevensbenadering
//   n.v.t.

//h1 Voorbeeld programma(s)
//h2 tst_herror
//  #include <herror.hxx>
//  #include <stdio.h>
//  
//  int f(int n)
//  {
//    if (n<0) {
//     THROW_FATAL("n<0");
//  /*   try{
//      THROW_FATAL("n<0");
//     } 
//     HERROR_CATCH;
//     */
//    }
//    else if (n==5) {
//     try {
//      THROW_WARNING("n==5");
//     }
//     HERROR_CATCH;
//     f(n-1);
//    }
//    else {
//      CHECKPOINT;
//      f(n-1);
//    }
//  }
//  
//  int main()
//  {
//    try{
//      CHECKPOINT;
//  
//      f(10);
//    }
//    HERROR_CATCH;
//  
//  return 0;
//  }
//  
//end

#endif

