#include <assert.h>

#include <qapp.h>
#include <qpixmap.h>
#include <qobjcoll.h>
#include <qfiledlg.h>
#include <qfileinf.h>
#include <qtstream.h>
#include <qregexp.h>
#include <qlayout.h>
#include <qobjcoll.h>
#include <qtooltip.h>
#include <qmenubar.h>
#include <qkeycode.h>
#include <qmsgbox.h>
#include <qmenudta.h>

#include <kmsgbox.h>
#include <kapp.h>
#include <kconfig.h>

#include "TMainWindow.h"

#include "TLayoutWindow.h"
#include "TFileIO.h"
#include "TQuestionDialog.h"
#include "TConnectionDlg.h"
#include "TEditWindow.h"
#include "TDataListBoxDlg.h"
#include "TToolWindow.h"
#include "QfruLibraryDlg.h"

#include "pics/pointer.xpm"

/*!
  \class TMainWindow TMainWindow.h
  \brief The TMainWindow class is the top level window for MAWC

  TMainWindow is the top level window for MAWC. It contains the main menu
  as well as the available QObjects which can be used.
  */

/*!
  Constructor for TMainWindow
  */
TMainWindow::TMainWindow(TStore *st, QWidget *, const char *name): KTopLevelWidget(name) {
  QPixmap p;

  initMetaObject();

  menu = NULL;
  layoutWidgets = NULL;
  fSelectedButton = NULL;
  fPointerButton = NULL;
  fStore = NULL;
  fToolWindow = NULL;
  fPropertyWidget = NULL;
  fActiveLayoutWindow = NULL;
  fConnectionWindow = NULL;
  fEditWindow = NULL;
  fClipBoard.setAutoDelete(TRUE);




  loadSettings();
  initGui();


  fPropertyWidget = new TPropertyWidget(NULL, "PropertyEditor");
  CHECK_PTR(propertyWidget());
  fPropertyWidget->store(st);
  fToolWindow = new TToolWindow(NULL, "Tool Window");
  fToolWindow->setMainWindow(this);

  
  setStore(st);
  loadGeometry();
}


/*!
  Destructor for TMainWindow
  */
TMainWindow::~TMainWindow() {
}






void TMainWindow::loadSettings() {
	KConfig *config = kapp->getConfig();
	config->setGroup( "Settings" );
	QString entry;

	fDefaultGridSize = config->readNumEntry("DefaultGridSize", 4);

	entry = config->readEntry("Toolbar", "top");
	showToolbar = true;
	if ( entry == "top" )
		toolbarPos = KToolBar::Top;
	else if ( entry == "left" )
		toolbarPos = KToolBar::Left;
	else if ( entry == "right" )
		toolbarPos = KToolBar::Right;    
	else if ( entry == "bottom" )
		toolbarPos = KToolBar::Bottom;    
	else if ( entry == "floating" )
		toolbarPos = KToolBar::Floating;    
	else
		showToolbar = false;

	entry = config->readEntry("Menubar", "top");
	showMenubar = true;
	if ( entry == "top" )
		menubarPos = KMenuBar::Top;
	else if ( entry == "bottom" )
		menubarPos = KMenuBar::Bottom;    
	else if ( entry == "floating" )
		menubarPos = KMenuBar::Floating;    
	else
		showMenubar = false;


/*    entry = config->readEntry("Statusbar", "top");
    showStatusbar = true;
    if ( entry == "top" )
	statusbarPos = KStatusBar::Top;
    else if ( entry == "bottom" )
	statusbarPos = KStatusBar::Bottom;    
    else if ( entry == "floating" )
	statusbarPos = KStatusBar::Floating;    
    else
	showStatusbar = false;
	*/


	config->setGroup( "Paths" );
	if (config) {
		fOpenDialogPath = config->readEntry("OpenDialog");
		fSaveDialogPath = config->readEntry("SaveDialog");
		fOpenCodePath = config->readEntry("OpenCode");
		fSaveCodePath = config->readEntry("SaveCode");
	}
}




void TMainWindow::initGui() {
	setCaption("MAWC");

	initMenu();
	initToolbar();
	initStatusBar();

	initView();
}


void TMainWindow::initStatusBar() {
//    statusBar = new KStatusBar( this );
// 
//    statusBar->insertItem( (char*)klocale->translate("MAWC"), 1 );
//    
//    statusBar->show();
//    setStatusBar( statusBar );
//    if ( !showStatusbar )
//		statusBar->enable( KStatusBar::Hide );
}



void TMainWindow::initMenu() {
	//
	// FILE
	//
	QPopupMenu *fileMenu = new QPopupMenu;
	fileMenu->insertItem("New", this, SLOT(newFile()), CTRL+Key_N);
	fileMenu->insertItem("Open", this, SLOT(openFile()), CTRL+Key_O);
	fileMenu->insertItem("Save", this, SLOT(saveFile()), CTRL+Key_S);
	fileMenu->insertItem("Save As...", this, SLOT(saveFileAs()), CTRL+Key_A);
	fileMenu->insertSeparator();
	fileMenu->insertItem("Quit", this, SLOT(quit()), CTRL+Key_Q);
	
	//
	// View
	//
	QPopupMenu *viewMenu = new QPopupMenu;
	viewMenu->insertItem("Properties", this, SLOT(showProperties()) );
	viewMenu->insertItem("Alignment", this, SLOT(showAlignment()) );
	viewMenu->insertItem("Connections", this, SLOT(showConnections()) );
	viewMenu->insertItem("Source Code", this, SLOT(editCode()));

	fWindowMenu = new QPopupMenu;
	viewMenu->insertItem("Windows", fWindowMenu);
	connect(viewMenu, SIGNAL(activated(int)), SLOT(showWindow(int)) );
	connect(viewMenu, SIGNAL(activatedRedirect(int)), SLOT(showWindow(int)) );

	//
	// Preferences
	//
    moptions = new QPopupMenu;
    CHECK_PTR( moptions );
    moptions->setCheckable(true);
    moptions->insertItem( klocale->translate("Show &Menubar"),
						  this, SLOT(slotShowMenubar()) );
    moptions->insertItem( klocale->translate("Show &Toolbar"),
						  this, SLOT(slotShowToolbar()) );
    moptions->insertSeparator();
	moptions->insertItem("Grid Size...", this, SLOT(gridSize()) );
	fShowGridId = moptions->insertItem("Show Grid", this, SLOT(gridShow()) );
	moptions->setItemChecked(fShowGridId, TRUE);
	fSnapToGridId = moptions->insertItem("Snap to Grid", this, SLOT(gridSnap()) );
	moptions->setItemChecked(fSnapToGridId, TRUE);
    moptions->insertSeparator();
	moptions->insertItem("Lib Directory", this, SLOT(setLibraryDirectory()) );
    moptions->insertItem( klocale->translate("Sa&ve Settings"),
						  this, SLOT(saveSettings()) );

    moptions->setItemChecked( moptions->idAt( 0 ), showMenubar );
    moptions->setItemChecked( moptions->idAt( 1 ), showToolbar );


    QPopupMenu *help = new QPopupMenu;
    help->insertItem( klocale->translate("&Help"), this, SLOT(slotHelp()) );
    help->insertSeparator();
    help->insertItem( klocale->translate("&About..."), this, SLOT(slotAbout()) );
    help->insertItem( klocale->translate("&About..."), this, SLOT(slotAboutQT()) );


	// 
	// MAIN 
	//
	menu = new KMenuBar(this, "Menu");	
	menu->insertItem( klocale->translate("File"), fileMenu);
	menu->insertItem( klocale->translate("View"), viewMenu);
	menu->insertItem( klocale->translate("Preferences"), moptions);
    menu->insertSeparator();
    menu->insertItem( klocale->translate("&Help"), help );
	if (KMenuBar::Floating == menubarPos)
		menu->setMenuBarPos(KMenuBar::Top);
	else
		menu->setMenuBarPos(menubarPos);
	setMenu(menu);

}


void TMainWindow::initView() {
	QWidget *view = new QWidget(this, "View Frame");

	tabCtrl = new QfruTabCtrl(view, "Base Panel");

	currentPanel = new QWidget(tabCtrl, "General");
	new QHBoxLayout(currentPanel, 0, -1, "layout");

	tabCtrl->addTab(currentPanel, "General");
	panelList.append(currentPanel);

	QPixmap pix((const char **)pointer_xpm);
	fPointerButton = new TPushButton(pix, view, "Select Object");
	fPointerButton->setToggleButton(TRUE);
	fPointerButton->adjustSize();
	fPointerButton->setFixedSize(fPointerButton->size());
	QToolTip::add(fPointerButton, "Select Object");
	connect(fPointerButton, SIGNAL(clicked(int,int)), SLOT(updateToggles(int,int)));


	QGridLayout *grid = new QGridLayout(view, 1, 2);
	grid->setColStretch(1, 1);
	grid->addWidget(fPointerButton, 0, 0, AlignCenter);
	grid->addWidget(tabCtrl, 0, 1);
	grid->activate();

	setView(view);
}




void TMainWindow::initToolbar() {
    QString file, path;
    QPixmap pixmap;

    toolbarButtons = new KToolBar(this, "kfmwin-toolbar");
    path = kapp->kdedir() + "/share/toolbar/";
    
    pixmap.load(path + "filenew.xpm");
    toolbarButtons->insertButton(pixmap, 0, SIGNAL( clicked() ), this, 
								 SLOT( newFile() ), true, 
								 klocale->translate("Open"));

    pixmap.load(path + "fileopen.xpm");
    toolbarButtons->insertButton(pixmap, 0, SIGNAL( clicked() ), this, 
								 SLOT( openFile() ), true, 
								 klocale->translate("Open"));

    pixmap.load(path + "filefloppy.xpm");
    toolbarButtons->insertButton(pixmap, 0, SIGNAL( clicked() ), this, 
								 SLOT( saveFile() ), true, 
								 klocale->translate("Save"));

    toolbarButtons->insertSeparator();
    pixmap.load(path + "exit.xpm");
    toolbarButtons->insertButton(pixmap, 0, SIGNAL( clicked() ), this, 
								 SLOT( quit() ), true, 
								 klocale->translate("Quit"));

    toolbarButtons->insertSeparator();
    pixmap.load(path + "help.xpm");
    toolbarButtons->insertButton(pixmap, 0, SIGNAL( clicked() ), this, 
								 SLOT( slotHelp() ), true, 
								 klocale->translate("Help"));



	
	addToolBar(toolbarButtons);
	toolbarButtons->setBarPos(toolbarPos);
	toolbarButtons->show();
	toolbarButtons->setFullWidth(TRUE);
}



void TMainWindow::loadGeometry() {
  KConfig *config = KApplication::getKApplication()->getConfig();
  config->setGroup( "WindosPos" );
 
  if (config) {
    QString str;
    str = config->readEntry(className());
    if (str.isEmpty() || str=="")
      resize(400,70);
    else {
      int a,b,c,d;
      if (4 != sscanf(str.data(), "%d %d %d %d", &a, &b, &c, &d))
	resize(400,70);
      else
	setGeometry(a,b,c,d);
    }
  }
}

/*!
  The close event checks for any unsaved files before closing. It
  also deletes any child windows, and finally saves the geometry
  for use the next time around.
  */
void TMainWindow::closeEvent(QCloseEvent *ev) {
  QListIterator<TLayoutWindow> it(fLayoutWindowList);
  for ( ; it.current() ; ++it) {
    if (it.current()->modified()) {
      QString tmp;
      tmp.sprintf("Dialog %s not saved!\nDo you wish to save it?", it.current()->name());
      int i = QMessageBox::warning(this, "Dialog not saved", 
				   tmp, "Yes",
				   "No", "Cancel", 0, 2);
      switch (i) {
      case 0:
	it.current()->save();
	break;
      case 1:
	//accept = TRUE;
	break;
      case 2:
	ev->ignore();
	break;
      }
    }
  }

  if (fPropertyWidget)
    delete fPropertyWidget;
  if (fToolWindow)
    delete fToolWindow;
  if (fConnectionWindow)
    delete fConnectionWindow;
  if (fEditWindow)
    delete fEditWindow;

  saveSettings();

  ev->accept();
}


/*!
  Sets the current TStore to be 's'

  It creates a toggle button for each and every TStoreType entry
  in the TStore.
  */
void TMainWindow::setStore(TStore *s) {
  fStore = s;

  QListIterator<TLayoutWindow> bit(fLayoutWindowList);
  for ( ; bit.current() ; ++bit)
    bit.current()->store(s);

  QDictIterator<TIntLibraryInfo> it(s->dictionary());
  for ( ; it.current() ; ++it) {
    if (it.current()->isCreatable) {
		// look to see which panel this needs to be added to...
		QString panelname = it.current()->panelName;
		if (panelname.isNull() || panelname.isEmpty())
			panelname = "General";
		cout << "Adding: " << it.currentKey() << " to: " << panelname << endl;
		
		QObjectListIt panelIt(panelList);
		for ( ; panelIt.current() ; ++panelIt)
			if (panelname == panelIt.current()->name())
				break;
		QWidget *panel = (QWidget*)panelIt.current();

		// if we have not found the tab, create it
		if (!panel) {
			panel = new QWidget(tabCtrl, panelname);
			new QHBoxLayout(panel, 0, -1, "layout");
			panelList.append(panel);
			tabCtrl->addTab(panel, panelname);
		}
		

		TPushButton *btn = new TPushButton(it.current()->pixmap, panel, 
										   it.currentKey());
		btn->setToggleButton(TRUE);
		btn->show();
		btn->adjustSize();
		btn->setMaximumSize(btn->size());
		btn->setMinimumSize(btn->size());
		QToolTip::add(btn, it.currentKey());
		
		connect(btn, SIGNAL(clicked(int,int)), SLOT(updateToggles(int,int)));


		int max = panel->height();
		if (max < btn->height() )
			max = btn->height();
		panel->setFixedHeight(max);

		QObjectList *olist = panel->queryList(0, "layout");
		assert (0!=olist->count());
		QHBoxLayout *blayout = (QHBoxLayout *)olist->first();
		blayout->addWidget(btn);
		blayout->activate();
	}
  }
  tabCtrl->adjustSize();
  tabCtrl->resize(tabCtrl->size());
  updateRects();
};



/*!
  Sets the multi-use state of the selected button, and then
  updates all of the controls
  */
void TMainWindow::updateToggles(int, int state) {
  bool multi = state & ShiftButton;
  multiSelect(multi);
  updateToggles((TPushButton*)sender());
}


/*!
  Ensures that there is only ever one toggle button active bt deselecting
  all other toggle buttons
  */
void TMainWindow::updateToggles(TPushButton *aSelected) {
  if (aSelected) {
    if (!aSelected->isOn()) {
      fPointerButton->setOn(TRUE);
      fSelectedButton = fPointerButton;
    }
    else
      fSelectedButton = aSelected;

    bool oldState = fSelectedButton->isOn();

    QObjectList *list = queryList("TPushButton");
    QObjectListIt it(*list);
    for ( ; it.current() ; ++it) {
      ((TPushButton*)it.current())->setOn(FALSE);
    }
    delete list;

    fSelectedButton->setOn(oldState);

    if (!fSelectedButton->isOn())
      fSelectedButton = NULL;
  }
}  


/*!
  Sets the current control to be created back to the selection
  pointer unless we are in multi-select mode.
  */
void TMainWindow::haveCreatedObject(const QRect &) {
  if (multiSelect())
    return;

  updateToggles(fPointerButton);
}


/*!
  returns the classname of the type we currently have as the 
  selected control.

  returns an empty string if there is not a control selected
  */
QString TMainWindow::typeToCreate() {
  if (fSelectedButton) {
    if (fSelectedButton==fPointerButton) {
      return ("");
    }
    else {
      return(fSelectedButton->name());
    }
  }
  else {
    return ("");
  }
}



/*!
  Creates a new TLayoutWindow and places it on the desktop
  */
void TMainWindow::newFile() {
	TDataListBoxDlg *dlg = new TDataListBoxDlg(NULL, "Base Class");
	QDictIterator<TIntLibraryInfo> it(store()->dictionary());
	for ( ; it.current() ; ++it) {
		if (it.current()->isBaseClass) {
			dlg->insertItem(it.currentKey(), 0, TRUE);
		}
	}

	if (dlg->exec()) {
		TStoreTypeCreateFn createType = fStore->type(dlg->text());
		
		TLayoutWindow *lw = newLayoutWindow(createType);

		if (lw)
			lw->show();
	}
	delete dlg;
}


TLayoutWindow *TMainWindow::newLayoutWindow(TStoreTypeCreateFn createFn) {
  static int count=0;
  QString str;
  TLayoutWindow *boundWindow=NULL;

  str = str.sprintf("Widget%d", ++count);
  
  boundWindow = new TLayoutWindow(this, NULL, str);
  boundWindow->setMainWindow(this);

  if (createFn)
	  boundWindow->create(NULL, createFn);
  
  fLayoutWindowList.append(boundWindow);
  boundWindow->gridSize(fDefaultGridSize);
  boundWindow->setSnapToGrid(moptions->isItemChecked(fSnapToGridId));
  boundWindow->setGridVisible(moptions->isItemChecked(fShowGridId));
  boundWindow->show();
  
  connect(boundWindow, SIGNAL(sigCreatedObject(const QRect &)), 
		  this, SLOT(haveCreatedObject(const QRect &)));
  setActiveLayoutWindow(boundWindow);
  
  fWindowMenu->insertItem(str);

  return boundWindow;
}


/*!
  Opens an existing Dialog file and loads it into a new TLayoutWindow
  */
void TMainWindow::openFile() {
  QFileDialog *fd = new QFileDialog(fOpenDialogPath, "*.dlg", NULL, 
				    "OpenFile", TRUE);
  fd->setCaption("Open Dialog");

  if (fd->exec() == QDialog::Accepted) {
    fOpenDialogPath = fd->dirPath();
    QString filename = fd->selectedFile();
    delete fd;

    QFileInfo info(filename);
    
    newLayoutWindow(0);
    if (fActiveLayoutWindow->open(info))
		fActiveLayoutWindow->show();
	else {
		delete fActiveLayoutWindow;
		fLayoutWindowList.remove(fActiveLayoutWindow);
		fActiveLayoutWindow = NULL;
	}
  }
  else
    delete fd;
}

/*!
  Saves the currently active TLayoutWindow
  */
void TMainWindow::saveFile() {
  if (!fActiveLayoutWindow) {
    KMsgBox::message(NULL, "Save Error", "There is no active window to save.");
    return;
  }
  fActiveLayoutWindow->save();
}


/*!
  Saves the currently active TLayoutWindow under a new name
  */
void TMainWindow::saveFileAs() {
  if (!fActiveLayoutWindow) {
    KMsgBox::message(NULL, "Save Error", "There is no active window to save.");
    return;
  }
  fActiveLayoutWindow->saveAs();
}




/*!
  Opens an editing window which holds the code for all of the 
  currently opened TLayoutWindows
  */
void TMainWindow::editCode() {
   if (fActiveLayoutWindow)
     fActiveLayoutWindow->editCode();
   else
     showEditWindow();
}



/*!
  Exits the application
  */
void TMainWindow::quit() {
  close();
}


/*!
  Sets the default gridsize 
  */
void TMainWindow::gridSize() {
  QString tmp;
  tmp.setNum(fDefaultGridSize);

  TQuestionDialog *qd = new TQuestionDialog("Grid Size:", tmp);

  if (qd->exec()) {
    fDefaultGridSize = atoi(qd->text());
    QListIterator<TLayoutWindow> it(fLayoutWindowList);
    for ( ; it.current() ; ++it)
      it.current()->gridSize(fDefaultGridSize);
  }
  delete qd;

}


/*!
  Toggles whether or not to show the grid
  */
void TMainWindow::gridShow() {
  bool oldState = moptions->isItemChecked(fShowGridId);

  QListIterator<TLayoutWindow> it(fLayoutWindowList);
  for ( ; it.current() ; ++it)
    it.current()->setGridVisible(!oldState);
  menu->setItemChecked(fShowGridId, !oldState);
}

/*!
  Toggles snap-to-grid on and off
  */
void TMainWindow::gridSnap() {
  bool oldState = moptions->isItemChecked(fSnapToGridId);

  QListIterator<TLayoutWindow> it(fLayoutWindowList);
  for ( ; it.current() ; ++it)
    it.current()->setSnapToGrid(!oldState);
  menu->setItemChecked(fSnapToGridId, !oldState);
}

/*!
  Shows the Property Window
  */
void TMainWindow::showProperties() { 
  fPropertyWidget->show();
  fPropertyWidget->raise();
};

/*!
  Shows the Alignment palette
  */
void TMainWindow::showAlignment() { 
  fToolWindow->show(); 
  fToolWindow->raise(); 
};

/*!
  Show the Connections window
  */
void TMainWindow::showConnections() { 
  connectionWindow()->show(); 
  connectionWindow()->raise(); 
};

/*!
  Show the Code Editing Window
  */
void TMainWindow::showEditWindow() { 
  editWindow()->show(); 
  editWindow()->raise(); 
};


/*!
  returns the Connection window, creating it if it is not already created
  */
TConnectionWindow *TMainWindow::connectionWindow(void) { 
  if (!fConnectionWindow)
    fConnectionWindow = new TConnectionWindow(NULL, "Connections");
  return fConnectionWindow; 
};

/*!
  returns the edit window, creating it if it is not already created
  */
TEditWindow *TMainWindow::editWindow(void) { 
  if (!fEditWindow)
    fEditWindow = new TEditWindow(NULL, "Source Code");
  return fEditWindow; 
};

/*!
  Sets the currently active TLayoutWindow to bw
  */
void TMainWindow::setActiveLayoutWindow(TLayoutWindow *bw) {
  fActiveLayoutWindow = bw;
  fToolWindow->setLayoutWindow(bw);
}


/*!
  Removes all data from the internal clipboard which is used for Cut/Copy/Paste
  */
void TMainWindow::clearClipBoard(void) {
  fClipBoard.clear();
}

/*!
  Add a tree to the clipboard. This does not affect any other trees which might
  already be on the clip board
*/
void TMainWindow::addToClipBoard(TStoreType *tree) {
  fClipBoard.append(tree);
}


/*!
  Sets up the library directory. Pops up a dialog asking for a directory to be 
  typed in
  */
void TMainWindow::setLibraryDirectory() {
	KConfig *config = KApplication::getKApplication()->getConfig();
	config->setGroup( "Directories" );
 
	if (config) {
		QStrList strlist;
		config->readListEntry("Libraries", strlist);
    
		QfruLibraryDlg *libdlg = new QfruLibraryDlg();
		libdlg->list(strlist);
		if (libdlg->exec()) {
			strlist = libdlg->list();
			config->writeEntry("Libraries", strlist);
			config->sync();
		}
		delete libdlg;
	}

}


/*!
  Show the window relating to the menu 'id'.

  Used to show the windows on the window list
  */
void TMainWindow::showWindow(int id) {
  if (sender()->inherits("QPopupMenu")) {
    const char *str =  ((QPopupMenu*)sender())->text(id);
    if (str) {
      QListIterator<TLayoutWindow> it(fLayoutWindowList);
      for ( ; it.current() ; ++it) {
	if (0==strcmp(str, it.current()->name()))
	  break;
      }
      if (it.current())
	it.current()->show();
    }
  }
}



void TMainWindow::setAlignTop() {
	fActiveLayoutWindow->setAlignTop();
}

void TMainWindow::setAlignLeft(){
	fActiveLayoutWindow->setAlignLeft();
}
void TMainWindow::setAlignRight(){
	fActiveLayoutWindow->setAlignRight();
}
void TMainWindow::setAlignBottom(){
	fActiveLayoutWindow->setAlignBottom();
}
void TMainWindow::setAlignWidths(){
	fActiveLayoutWindow->setAlignWidths();
}
void TMainWindow::setAlignHeights(){
	fActiveLayoutWindow->setAlignHeights();
}
void TMainWindow::setAlignVCentres(){
	fActiveLayoutWindow->setAlignVCentres();
}
void TMainWindow::setAlignHCentres(){
	fActiveLayoutWindow->setAlignHCentres();
}
void TMainWindow::setAlignVSpread(){
	fActiveLayoutWindow->setAlignVSpread();
}
void TMainWindow::setAlignHSpread(){
	fActiveLayoutWindow->setAlignHSpread();
}
void TMainWindow::setPosHCentre(){
	fActiveLayoutWindow->setPosHCentre();
}
void TMainWindow::setPosVCentre(){
	fActiveLayoutWindow->setPosVCentre();
}




void TMainWindow::saveSettings()
{
	KConfig *config = kapp->getConfig();
	config->setGroup( "Settings" );
    
	QString entry;

	if ( !showToolbar )
		config->writeEntry( "Toolbar", "hide" );
	else if ( toolbarButtons->barPos() == KToolBar::Top )
		config->writeEntry( "Toolbar", "top" );
	else if ( toolbarButtons->barPos() == KToolBar::Bottom )
		config->writeEntry( "Toolbar", "bottom" );
	else if ( toolbarButtons->barPos() == KToolBar::Left )
		config->writeEntry( "Toolbar", "left" );
	else if ( toolbarButtons->barPos() == KToolBar::Right )
		config->writeEntry( "Toolbar", "right" );
	else if ( toolbarButtons->barPos() == KToolBar::Floating )
		config->writeEntry( "Toolbar", "floating" );

/*  if ( !showStatusbar )
      config->writeEntry( "Statusbar", "hide" );
  else
      config->writeEntry( "Statusbar", "bottom" );*/

	if ( !showMenubar )
		config->writeEntry( "Menubar", "hide" );
	else if ( menu->menuBarPos() == KMenuBar::Top )
		config->writeEntry( "Menubar", "top" );
	else if ( menu->menuBarPos() == KMenuBar::Bottom )
		config->writeEntry( "Menubar", "bottom" );
	else if ( menu->menuBarPos() == KMenuBar::Floating )
		config->writeEntry( "Menubar", "floating" );


	config->setGroup( "WindosPos" );
    
	QString str;
	str.sprintf("%d %d %d %d", x(), y(), width(), height());
	config->writeEntry(className(), str);

	config->setGroup( "Paths" );
	config->writeEntry("OpenDialog", fOpenDialogPath);
	config->writeEntry("SaveDialog", fSaveDialogPath);
	config->writeEntry("OpenCode", fOpenCodePath);
	config->writeEntry("SaveCode", fSaveCodePath);

	config->sync();


	config->sync();
}



void TMainWindow::slotShowMenubar()
{
    showMenubar = !showMenubar;
    moptions->setItemChecked( moptions->idAt( 0 ), showMenubar );
    if (showMenubar)
		menu->show();
    else
		menu->hide();
    updateRects();
    // TODO: does not work yet
}

void TMainWindow::slotShowToolbar() {
    showToolbar = !showToolbar;
    moptions->setItemChecked( moptions->idAt( 1 ), showToolbar );
    if ( !showToolbar )
		toolbarButtons->enable( KToolBar::Hide );
    else
		toolbarButtons->enable( KToolBar::Show );
    updateRects();
    resizeEvent( 0L );
}

void TMainWindow::slotAboutQT() {
    QMessageBox::aboutQt( this, "About Qt" );
}

void TMainWindow::slotAbout() {
    KMsgBox::message( this, "About MAWC", "Mawc v0.1.1\n(c) 1998 Chris Ward (frugal@fysh.org)" );
}

void TMainWindow::slotHelp() {
  kapp->invokeHTMLHelp( "", "" );
}


void TMainWindow::slotTabSelected(int) {
	updateToggles(fPointerButton);
};
