#include <qscrbar.h>
#include <qpainter.h>

#include "TBoxLayoutTable.h"


TBoxLayoutTable::TBoxLayoutTable(QBoxLayout::Direction direction,
				 QWidget *parent, 
				 const char *name): QTableView (parent, name) {
  initMetaObject();

  setFocusPolicy(QWidget::ClickFocus);

  setFrameStyle(QFrame::Raised | QFrame::Panel);
  setNumCols(4);
  setTableFlags(Tbl_vScrollBar | Tbl_clipCellPainting);

  QFontMetrics fm(font());
  setCellHeight(fm.height() + 4);

  fCurrentRow = -1;

  fEdit = new QLineEdit(this);
  fEdit->resize(cellWidth(2), cellHeight() );
  fEdit->hide();
  fEdit->installEventFilter(this);

  fCombo = new QComboBox(this);
  fCombo->resize(cellWidth(3), cellHeight());
  fCombo->hide();
  fCombo->installEventFilter(this);

  fDirection = direction;
  switch (direction) {
  case QBoxLayout::TopToBottom:
  case QBoxLayout::BottomToTop:
    fCombo->insertItem("AlignLeft");
    fCombo->insertItem("AlignCenter");
    fCombo->insertItem("AlignRight");
    break;
  case QBoxLayout::LeftToRight:
  case QBoxLayout::RightToLeft:
    fCombo->insertItem("AlignTop");
    fCombo->insertItem("AlignCenter");
    fCombo->insertItem("AlignBottom");
    break;
  }

}



int TBoxLayoutTable::cellWidth(int col) {
  float w = viewWidth();
  int rVal = 0;

  switch (col) {
  case 0:
  case 1:
    rVal = (int)(w * 0.3); break;
  case 2:
    rVal = (int)(w * 0.1); break;
  case 3:
    rVal = (int)(w * 0.3); break;
  }
  return rVal;
}



void TBoxLayoutTable::paintCell(QPainter *p, int row, int col) {
  int w = cellWidth(col);
  int h = cellHeight();

  if (row == fCurrentRow) {
    p->setPen(colorGroup().dark());
    p->drawLine(0,0,w,0);
    p->drawLine(0,0,0,h);
    p->setPen(colorGroup().light());
    p->drawLine(w-1,h-1,w,0);
    p->drawLine(w-1,h-1,0,h-1);
  }

  TBoxLayoutDataItem *item = fList.at(row);
  if (!item)
    return;

  QString str;
  switch (col) {
  case 0:
    switch (item->type) {
    case TBoxLayoutDataItem::Widget:
      str = "Widget"; break;
    case TBoxLayoutDataItem::Layout:
      str = "Layout"; break;
    case TBoxLayoutDataItem::Stretch:
      str = "Stretch"; break;
    case TBoxLayoutDataItem::Strut:
      str = "Strut"; break;
    case TBoxLayoutDataItem::Space:
      str = "Space"; break;
    }
    break;
  case 1:
    if (item->type == TBoxLayoutDataItem::Widget ||
	item->type == TBoxLayoutDataItem::Layout)
      str = item->objectTree->object()->name();
    break;
  case 2:
    str.setNum(item->size);
    break;
  case 3:
    if (item->type == TBoxLayoutDataItem::Widget) {
      switch (item->alignment) {
      case AlignCenter:
	str = "AlignCenter"; break;
      case AlignTop:
	str = "AlignTop"; break;
      case AlignBottom:
	str = "AlignBottom"; break;
      case AlignLeft:
	str = "AlignLeft"; break;
      case AlignRight:
	str = "AlignRight"; break;
      }
    }
    break;
  }

  p->setPen(colorGroup().text());
  p->drawText(2, 2, w-2, h-2, AlignCenter, str);
}



void TBoxLayoutTable::mouseReleaseEvent(QMouseEvent *ev) {
  int col = findCol(ev->pos().x());
  int row = findRow(ev->pos().y());
  int rowPos;
  int colPos;

  if ((col==-1) || (row==-1))
    return;
  
  if (!rowYPos(row, &rowPos))
    return;
  if (!colXPos(col, &colPos))
    return;

  TBoxLayoutDataItem *item = fList.at(row);
  if (!item)
    return;

  setCurrentRow(row);

  QString s;
  switch (col) {
  case 2:
    s.setNum(item->size);
    fEdit->move(colPos, rowPos);
    fEdit->resize(cellWidth(2), cellHeight() );
    fEdit->setText(s);
    fEdit->show();
    fEdit->setFocus();
    fEdit->selectAll();
    break;
  case 3:
    fCombo->move(colPos, rowPos);
    fCombo->resize(cellWidth(3), cellHeight() );
    fCombo->show();
    fCombo->setFocus();

    switch (item->alignment) {
    case AlignTop:
    case AlignLeft:
      fCombo->setCurrentItem(0);
      break;
    case AlignCenter:
      fCombo->setCurrentItem(1);
      break;
    case AlignBottom:
    case AlignRight:
      fCombo->setCurrentItem(2);
      break;
    }
    break;
  }
  return;
}



void TBoxLayoutTable::setCurrentRow(int index) {
  cout << "Setting Current Row from: " << fCurrentRow << " to: " << index << endl;
  if (fCurrentRow != index) {
    int oldRow = fCurrentRow;

    if (index<0 || index>=(int)fList.count())
      fCurrentRow = -1;
    else
      fCurrentRow = index;
    
    for (int i=0; i<4 ; i++) {
      updateCell(oldRow, i);
      updateCell(fCurrentRow, i);
    }
  }
}


bool TBoxLayoutTable::eventFilter(QObject *obj, QEvent *ev) {
  TBoxLayoutDataItem *item = fList.at(fCurrentRow);
  if (!item)
    return FALSE;

  switch (ev->type()) {
  case Event_FocusOut:
    if (obj==fEdit) {
      cout << "Object = fEdit" << endl;
      QString str = fEdit->text();
      item->size = str.toInt();
    
      fEdit->hide();
      updateCell(fCurrentRow, 2);
    }
    else if (obj==fCombo) {
      QString str = fCombo->currentText();
      if (str == "AlignCenter")
	item->alignment = AlignCenter;
      else if (str == "AlignTop")
	item->alignment = AlignTop;
      else if (str == "AlignBottom")
	item->alignment = AlignBottom;
      else if (str == "AlignLeft")
	item->alignment = AlignLeft;
      else if (str == "AlignRight")
	item->alignment = AlignRight;
      fCombo->hide();
      updateCell(fCurrentRow, 3);
    }
    break;
  }
  return FALSE;
}


void TBoxLayoutTable::addWidget(TStoreType *widget, int stretch, int alignment) {
  cout << "Adding Widget" << endl;
  TBoxLayoutDataItem *item = new TBoxLayoutDataItem;

  item->type = TBoxLayoutDataItem::Widget;
  item->objectTree = widget;
  item->size = stretch;
  item->alignment = alignment;

  fList.append(item);

  setNumRows(fList.count());
  setCurrentRow(fList.count()-1);
}


void TBoxLayoutTable::addSpacing(int spacing) {
  cout << "Adding Spacing" << endl;
  TBoxLayoutDataItem *item = new TBoxLayoutDataItem;

  item->type = TBoxLayoutDataItem::Space;
  item->objectTree = NULL;
  item->size = spacing;
  item->alignment = 0;

  fList.append(item);
  setNumRows(fList.count());

  setCurrentRow(fList.count()-1);
}  

void TBoxLayoutTable::addStretch(int stretch) {
  cout << "Adding Stretch" << endl;
  TBoxLayoutDataItem *item = new TBoxLayoutDataItem;

  item->type = TBoxLayoutDataItem::Stretch;
  item->objectTree = NULL;
  item->size = stretch;
  item->alignment = 0;

  fList.append(item);

  setNumRows(fList.count());
  setCurrentRow(fList.count()-1);
}  

void TBoxLayoutTable::addStrut(int strut) {
  cout << "Adding Strut" << endl;
  TBoxLayoutDataItem *item = new TBoxLayoutDataItem;

  item->type = TBoxLayoutDataItem::Strut;
  item->objectTree = NULL;
  item->size = strut;
  item->alignment = 0;

  fList.append(item);

  setNumRows(fList.count());
  setCurrentRow(fList.count()-1);
}  



void TBoxLayoutTable::addLayout(TStoreType *layoutTree, int stretch) {
  cout << "Adding Layout" << endl;

  cout << "Layout item->objectTree = " << (void*) layoutTree << endl;
  cout << "Layout item->objectTree->object = " << (void*) layoutTree->object() << endl;
  cout << "Layout item->objectTree->Name() = " << layoutTree->object()->name() << endl;
  
  TBoxLayoutDataItem *item = new TBoxLayoutDataItem;

  item->type = TBoxLayoutDataItem::Layout;
  item->objectTree = layoutTree;
  item->size = stretch;
  item->alignment = 0;

  fList.append(item);

  setNumRows(fList.count());
  setCurrentRow(fList.count()-1);
}  



void TBoxLayoutTable::deleteItem() {
  fList.remove(fCurrentRow);
  setNumRows(fList.count());

  cout << "Deleting Item: " << fCurrentRow << endl;

  if ( ((int)fList.count()) >= fCurrentRow )
    setCurrentRow(fCurrentRow-1);
  repaint();
}
